(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author: Till Mossakowski         		                           *)
(* Date: 1998/2000				 			   *)
(* Purpose of this file: ML yacc parser for CASL			   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* This module just builds the CASL parser from the ML yacc grammar
   in the ML yacc specific way.
   Note that we provide a separate function for parsing annotations,
   using (a different section of) the same ML yacc grammar (working
   with some initial string to distiguish annotations from
   CASL libraries).
*)

structure CASLParser :
sig
val parse : bool -> string ->  AS.LIB_DEFN
val CASLParse : string -> AS.LIB_DEFN
val CASLParsefile : string -> AS.LIB_DEFN
val ParseAnnos : AS.ANNO list -> AS.ANNO list
end
= struct

open Global Utils;

structure CASLLrVals =
	CASLLrValsFun(structure Token = LrParser.Token);
structure CASLLex =
	CASLLexFun(structure Tokens = CASLLrVals.Tokens);
structure CASLParser=
	Join
	    (structure ParserData = CASLLrVals.ParserData
	     structure Lex = CASLLex
	     structure LrParser = LrParser);
(*****************************************************************************) 
(*************************** CASL parser functions ***************************)
(*****************************************************************************) 
(* Use Parse(f,s) to parse string s coming from filename f.
   Let f be the empty string if there is no filename
   
   Use ParseStream(f,g) to parse more efficiently (especially for large files)
   f is a filename as above
   g is a function taking the number of characters to read and returning
        any positive number of characters (or the empty string in case of EOF)
        Typically, g should return one storage block of the file system
   Oops, this requires to the read the whole file in advance anyway:
   Lexer_line_lengths has to be set to the line lengths!
   (May be improved later on.)
*)



fun ParseStream (file:string, getinput:int->string) =  
     let val lexer = CASLParser.makeLexer getinput
         val filemsg = if file = "" then "" else file^":"
(*         val print_error = fn (s,t,_) =>
			      let
				  val (row,col) = t
			      in
				  (print ("*** "^filemsg^(Int.toString(row))^
					  "."^(Int.toString(col)) ^
					  ", "^s^"\n");())
			      end
*)
	 fun print_error (s,(row,col),_) =
	     (print ("*** "^filemsg^(Int.toString row)^
		     "."^(Int.toString col)^", "^s^"\n");())
         val (result,lexer') = CASLParser.parse(0,lexer,print_error,())
     in result
     end;

fun get_line_lengths str =
    map String.size (split_lines str)
     
fun Parse (str:string) =  
    let val input = ref str
        fun getinput _ = fst(!input,input:="")
        val x = (Lexer_line_lengths := get_line_lengths str); 
    in
    ParseStream (!Global.cur_file,getinput)
    end;


fun CASLParse (str:string) = 
    case Parse str of
      AS.lib_defn_value ld => ld 
      | AS.anno_value _ => raise ERR ("Parse error for library")
fun parse html s =
     CASLParse s

fun CASLParsefile file = CASLParse (read_without_cr file);


fun strip_group (c::str) = 
    if c = "(" then
	let val rstr = rev(str);
	in 	
	  if take(2,rstr) = ["%",")"] then
	    rev(drop(2,rstr))
	  else 
	      str
	end
    else
	c::str
fun ParseAnno(AS.unparsed_anno str) =
    if take(8,explode str) = ["%","d","i","s","p","l","a","y"]
    then AS.display_anno (AS.make_sid "",
			  implode(strip_group(drop(8,explode str))))
	 (* TODO: Strip of optional LPAREN and CLOSE_ANNO *)
    else (case Parse ("= * %!"^str) of  (* Add magic signs to recognize anno *)
          AS.anno_value an => an
          | AS.lib_defn_value _ => raise ERR ("Parse error for annotation"))
  | ParseAnno (AS.pos_ANNO(r,an)) = AS.pos_ANNO(r,ParseAnno an)
  | ParseAnno an = an
      
fun ParseAnnos (ans:AS.ANNO list) =
    map ParseAnno ans
    
end
