(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Authors: Pascal Schmidt                                                 *)
(*          Klaus Lttich                                                  *)
(* Date: 2002    				 			   *)
(* Purpose of this file: Polytypic generation of abstract syntax           *)
(*			 						   *)
(*									   *)
(* *********************************************************************** *)

(* This module mimicks some kind of polytypic programming for CASL's abstract
   syntax and its translation to various interface formats such as ATerms and XML.
   The asbtract syntax is generated from a datastructure describing the
   datatypes (this roughly corresponds to functors polytypic programming).
   The module automatically generates as_types.sml.

   Future work: also automatically generate interface modules.

   line_number durch region ersetzen
   op_symb und pred_symb aktualisieren
   label_anno mit string statt ID
   Mixfix als annotation fuer Applikationen (um x+y und (op +:sxs->s)(x,y)
     zu unterscheiden)

*)

(*****************************************************************************)

(* $Id: asgen.sml,v 1.1 2004/02/13 11:29:09 7till Exp $ *)

structure asgen
=
struct

  (* ----------------------------------------------------------------- *)
  (* type definitions                                                  *)
  (* ----------------------------------------------------------------- *)

  datatype 'a option = None | Some of 'a

  type type_name = string

  datatype type_id = option of type_id
                   | list of type_id
                   | comp of (type_id list)
                   | single of (type_name)

  type type_def = string * type_id

  type datatype_cons = string * type_id option
  
  type datatype_def = string * datatype_cons list

  datatype define = dd of datatype_def
                  | td of type_def
                  | an of datatype_def
                  | wt of type_def

  (* ----------------------------------------------------------------- *)
  (* printing                                                          *)
  (* ----------------------------------------------------------------- *)

  fun spaces c "" = ""
    | spaces c s  = c ^ (spaces c ((String.implode o tl o String.explode) s))

  fun p_type_name x = x

  fun p_type_list []     = ""
    | p_type_list [h]    = h
    | p_type_list (h::t) = h ^ " * " ^ (p_type_list t)

  fun p_type_id (option i) = (p_type_id i) ^ " option"
    | p_type_id (list i)   = (p_type_id i) ^ " list"
    | p_type_id (comp l)   = "(" ^ (p_type_list ((map p_type_id) l)) ^ ")"
    | p_type_id (single n) = p_type_name n

  fun p_type_def t (n,i) = "  " ^ t ^ " " ^ n ^ " = " ^ (p_type_id i) ^ "\n\n" 

  fun p_datatype_cons (c,None)   = c ^ "\n"
    | p_datatype_cons (c,Some i) = c ^ " of " ^ (p_type_id i) ^ "\n"
  
  fun p_cons_name t n c = "  " ^ t ^ " " ^ n ^ " = " ^ (p_datatype_cons c)

  fun p_cons_cont t n c = " " ^ (spaces " " t) ^ "| " 
                          ^ (p_datatype_cons c)

  fun p_datatype_def x (n,[])     = ""
    | p_datatype_def x (n,(h::t)) = (p_cons_name x n h) ^ (String.concat ((map
                                     (p_cons_cont x n)) t)) ^ "\n"

  fun p_define (dd x) = p_datatype_def "datatype" x
    | p_define (an x) = p_datatype_def "and" x
    | p_define (td x) = p_type_def "type" x
    | p_define (wt x) = p_type_def "withtype" x

  (* ----------------------------------------------------------------- *)
  (* helper functions                                                  *)
  (* ----------------------------------------------------------------- *)

  fun unique_id(a,ids) = 
      if List.exists (fn(x)=>a=x) ids then
	  unique_id(a ^ "i", ids)
      else
	  if a = "o" then (* avoiding an "o" as variable *)
	      unique_id(a ^ "i", ids)
	  else
	      a

  fun conv_at_cons c = (implode o 
			(map (fn(x)=>if x= #"_" then #"-" else x)) o
			explode) c


  fun patt_list_un ids nil = ("",ids)
    | patt_list_un (ids) ([a]) = 
      let 
	  val (id,nids) = mk_ids_un ids a 
	  (*val uid = unique_id(id,ids) *)
      in
	  (id,nids)
      end

    | patt_list_un ids (a::t) = 
      let
	  val (id,nids) = mk_ids_un ids a 
	  (*val uid = unique_id(id,ids) *)
	  val (str,nnids) = (patt_list_un (nids) t)
      in
	  (id ^ ", " ^ str, nnids) 
      end

  and mk_ids_un ids (option i) = 
      let 
	  val (str,nids) = (mk_ids_un ids i)
	  val uid = unique_id(str^"_opt",ids)
      in
	  (uid,ids@[uid])
      end
    | mk_ids_un ids (list i)   = 
      (case i of
	   (comp _) =>  let 
	       val uid = unique_id("comp_lst",ids)
	   in
	       (uid, ids@[uid])
	   end
	 | (_) =>  let
	       val (str,nids) = (mk_ids_un ids i) 
	       val uid = unique_id(str^"_lst",ids)
	   in 
	       (uid,ids@[uid])
	   end
		   )
    | mk_ids_un ids (comp l)   = 
      let
	  val (str,nids) = (patt_list_un ids l)
      in
	  ("(" ^ str ^ ")",nids)
      end
    | mk_ids_un ids (single n) = 
      let 
	  val nid = unique_id(((str o Char.toLower o hd o explode) n),ids)
      in
	  (nid,ids@[nid])
      end

  fun mk_id_string x = 
      let
	  val (str,ids) = mk_ids_un [] x
      in
	  str
      end

  fun get_id_list x =
      let
	  val (str,ids) = mk_ids_un [] x
      in
	  ids
      end

  (* ----------------------------------------------------------------- *)
  (* converting as_to_aterms.sml                                       *)
  (* ----------------------------------------------------------------- *)

  fun conv_at_func_name n = "at_" ^ n

  fun as2at_cons_patt (c,None) = c
    | as2at_cons_patt (c,Some i) = c ^ " " ^ (mk_id_string i) 

  fun as2at_mk_comp_fn kw ids n t = 
      let 
	  val (args,nids) = mk_ids_un ids t
	  val terms = as2at_mk_terms2 kw nids (List.length(ids)) t
      in
	  "(fn("^args^")=> AAppl(AId(\"\"),["^terms^"],[]))"
      end

  and as2at_mk_comp_terms kw ids n nil = ""
    | as2at_mk_comp_terms kw ids n [a] = "(" ^ (as2at_mk_terms2 kw ids n a) ^ 
					 ")"
    | as2at_mk_comp_terms kw ids n (a::t) = 
      "(" ^ 
      (as2at_mk_terms2 kw ids n a) ^ 
      "),\n  " ^ (spaces " " kw) ^ " " ^
      (as2at_mk_comp_terms kw ids (n+1) t)

  and as2at_mk_terms2 kw ids n (option t) = 
      "case " ^ List.nth(ids,n) ^ " of\n" ^ 
      "  " ^ (spaces " " kw) ^ "    None => AFun(AId(\"None\"),[])\n" ^ 
      (*"[AList([],[])],[])\n" ^  The empty Alist ist needed to fulfill
				  the aterm specification *)    
      "  " ^ (spaces " " kw) ^ "  | Some opt_val => AAppl(AId(\"Some\"),[" ^
      (case t of
	   (single tn) => "(" ^ (conv_at_func_name tn) ^ " opt_val)" 
	 | _ => "" 
         (* This is not complete, but it fullfills the necessities *)
	 (* Warning: This is not enough if more than two option types 
	    are used in one pattern !!!*)
		) ^
      "],[])\n  " ^ (spaces " " kw) ^ " " 
	   
    | as2at_mk_terms2 kw ids n (list t) =
      (case t of
	   (comp l)  => ("AList(((map " ^ as2at_mk_comp_fn kw ids (n+1) t ^ 
			 ") " ^
			 List.nth(ids,n) ^
			 "),[])")
			
	 | (single tp) => ("AList(((map " ^ 
			   (conv_at_func_name tp) ^ ") " ^
			   List.nth(ids,n) ^
			   "),[])")
         (* The following two cases may do wrong (unexpected) things *)
	 | (option tp) => ("AList(((map " ^
			   (as2at_mk_terms2 kw ids (n+1) tp) ^ ") " ^
			   List.nth(ids,n) ^
			   "),[])")
			  
	 | (list lt)   => ("AList(((map " ^ 
			   (as2at_mk_terms2 kw ids (n+1) lt) ^ ") " ^
			   List.nth(ids,n) ^
			   "),[])")
			  )
      
    | as2at_mk_terms2 kw ids n (comp l) = "\n  " ^ (spaces " " kw) ^ " " ^
					  (as2at_mk_comp_terms kw ids n l)
					        
    | as2at_mk_terms2 kw ids n (single name) = 
      (conv_at_func_name name) ^ " " ^ List.nth(ids,n) 

  fun as2at_mk_terms kw x = 
      let
	  val (ids) = (get_id_list x)
      in 
	  as2at_mk_terms2 kw ids 0 x
      end

  fun as2at_type_def kw (dn,i) =  
      let
	  val args = mk_id_string i
	  val terms = as2at_mk_terms kw i
      in
	  "  " ^ kw ^ " " ^ (conv_at_func_name dn) ^ " " ^
	  args ^ " =\n  " ^ (spaces " " kw) ^ 

	  (case i of
	       (comp _) =>
	       " AAppl(AId(\"\"),[" ^ terms ^ "],[])\n\n"
	     | _ => terms^ "\n\n")
      end

  fun as2at_cons_term kw (c,None) = 
      "AFun(AId(\"" ^ (conv_at_cons c) ^ "\"),[])\n"
    | as2at_cons_term kw (c,Some i) = 
      " AAppl(AId(\"" ^ (conv_at_cons c) ^"\"),[" ^ 
      (as2at_mk_terms kw i) ^ "],[])\n"


  fun as2at_cons_first kw fdn h = "  " ^ kw ^ " " ^ fdn ^ 
				 " (" ^ (as2at_cons_patt h) ^ ") =\n" ^
				 "  " ^ (spaces " " kw) ^
				 (as2at_cons_term kw h)
				 
  fun as2at_cons_cont kw fdn h = " " ^ (spaces " " kw) ^ "| " ^ fdn ^ 
				 " (" ^ (as2at_cons_patt h) ^ ") =\n" ^
				 "  " ^ (spaces " " kw) ^ 
				 (as2at_cons_term kw h)

  fun as2at_datatype_def kw (dn,[]) = ""
    | as2at_datatype_def kw (dn,(h::t)) =
      let 
	  val fdn = conv_at_func_name(dn);
      in
	  (as2at_cons_first kw fdn h) ^
	  (String.concat ((map (as2at_cons_cont kw fdn)) t)) ^
	  "\n"
      end

  fun as2at_define (dd x) = as2at_datatype_def "fun" x
    | as2at_define (an x) = as2at_datatype_def "and" x
    | as2at_define (td x) = as2at_type_def "fun" x
    | as2at_define (wt x) = as2at_type_def "and" x 

  val asconv_preamble = 
      "(* autogenerated by asgen.sml -- do not edit *)\n\n"^
      "structure LibraryOption :\n"^
      "sig\n"^
      "  datatype 'a option = None | Some of 'a\n" ^
      "end =\n"^
      "struct\n"^
      "  open Library\n" ^ 
      "end\n\n"^
      "structure AT_gen =\n"^
      "struct\n\n"^
      "  local open LibraryOption AS ATDef in\n\n" ^
      "\n(* Bool conversion functions *)\n" ^
      "  exception BOOL_CONV\n" ^
      "  fun at_bool true = AFun(AId(\"true\"),[])\n" ^
      "    | at_bool false = AFun(AId(\"false\"),[])\n\n" ^
      "  fun as_bool (AFun (ab,_)) =\n" ^
      "     (case ab of\n" ^
      "          (AId \"true\")  => true\n" ^
      "        | (AId \"false\") => false\n" ^
      "        | _ => raise BOOL_CONV)\n" ^
      "    | as_bool _ = raise BOOL_CONV\n\n"^
      "\n(* Integer conversion functions *)\n" ^
      "  exception INT_CONV\n" ^
      "  fun at_int i = AFun(ANumber(i),[])\n\n" ^
      "  fun as_int (AFun (an,_)) =\n" ^ 
      "      (case an of\n" ^
      "           (ANumber i) => i\n"^
      "         | _ => raise INT_CONV)\n" ^
      "    | as_int _ = raise INT_CONV\n\n" ^
      "\n(* String conversion functions *)\n" ^
      "  exception STRING_CONV\n" ^
      "  fun escape s =\n" ^
      "    let\n" ^
      "        fun esca y = implode([#\"\\\\\",y])\n" ^
      "    in\n" ^
      "      String.translate (fn(x)=>\n" ^
      "                        if ord(x) < 32 then\n" ^
      "                         (case x of \n" ^
      "                            #\"\\n\" => esca #\"n\"\n" ^
      "                          | #\"\\t\" => esca #\"t\"\n" ^
      "                          | #\"\\r\" => esca #\"r\"\n" ^
      "                          | _ => raise STRING_CONV)\n" ^
      "                        else\n" ^
      "                         (case x of \n" ^
      "                            #\"\\\\\" => esca x\n" ^
      "                          | #\"\\\"\" => esca x\n" ^
      "                          | _ => implode([x]))) s\n" ^
      "    end\n\n" ^
(*      
      "      String.translate (fn(x)=>\n" ^
      "                          if ord(x) < 32 then\n" ^
      "                              \"\\\\\" ^\n" ^ 
      "                              (StringCvt.padLeft #\"0\" 3\n" ^ 
      "                                                 (Int.fmt StringCvt.OCT\n" ^ 
      "                                                          (ord(x))))\n" ^
      "                          else\n" ^ 
      "                              if x = #\"\\\\\" orelse x = #\"\\\"\" then\n" ^
      "                                  implode([#\"\\\\\",x])\n" ^ 
      "                              else\n" ^
      "                                  implode([x])) s\n\n" ^
*)
      "  fun unescape s =\n" ^
      "  let\n" ^
      "     fun unesca x = \n" ^
      "         case x of\n" ^
      "           #\"n\" => #\"\\n\"\n" ^
      "         | #\"r\" => #\"\\r\"\n" ^
      "         | #\"t\" => #\"\\t\"\n" ^
      "         | #\"\\\\\" => x\n" ^
      "         | #\"\\\"\" => x\n" ^
      "         | _ =>  raise STRING_CONV\n;" ^
      "     fun unescape_sub c_lst sub =\n" ^
      "         if Substring.size(sub) < 2 then\n" ^
      "             (implode(c_lst)) ^ Substring.string(sub)\n" ^
      "         else\n" ^
      "             (let\n" ^ 
      "                 val (fc,rem) = (valOf(Substring.getc(sub)))\n" ^
      "              in\n" ^
      "                 (if fc = #\"\\\\\" then\n" ^
      "                    (let\n"^
      "                         val (sc,rem2) = (valOf(Substring.getc(rem)))\n"^
      "                     in\n"^
      "                         unescape_sub (c_lst@[unesca sc]) rem2\n"^
      "                     end)\n" ^
      "                  else\n"^
      "                      unescape_sub (c_lst@[fc]) rem)\n"^
      "              end)\n" ^
(*
      "      if Substring.size(sub) < 2 then\n" ^
      "          (implode(c_lst)) ^ Substring.string(sub)\n" ^
      "      else\n" ^
      "          (let\n" ^ 
      "               val (fc,rem) = (valOf(Substring.getc(sub)))\n" ^
      "           in\n" ^
      "               (if fc = #\"\\\\\" then\n" ^
      "                    (if Char.isDigit(valOf(Substring.first(rem)))\n"^
      "                     then\n"^
      "                         (let\n"^
      "                             val (digits,suff) = Substring.splitAt(rem,3)\n"^
      "                          in\n"^
      "                              unescape_sub\n"^ 
      "                                  (c_lst @\n"^
      "                                   [(chr(valOf(StringCvt.scanString\n"^
      "                                                   (Int.scan StringCvt.OCT)\n"^
      "                                                   (Substring.string(digits)))))])\n"^
      "                                  suff\n"^
      "                          end)\n"^
      "                     else\n"^
      "                         (let\n"^
      "                              val (sc,rem2) = (valOf(Substring.getc(rem)))\n"^
      "                          in\n"^
      "                              unescape_sub (c_lst@[sc]) rem2\n"^
      "                          end))\n" ^
      "                else\n"^
      "                    unescape_sub (c_lst@[fc]) rem)\n"^
      "            end)\n\n"^
*)
      "    in\n"^ 
      "      unescape_sub [] (Substring.all s)\n"^
      "    end\n\n"^
      "  fun at_string s = AFun(AString(escape(s)),[])\n\n" ^
      "  fun as_string (AFun ((AString (cs)),_)) = unescape(cs)\n" ^
      "    | as_string _ = raise STRING_CONV\n\n" ^
      "\n"
			
  val asconv_postamble = "  end\nend\n"

  (* ----------------------------------------------------------------- *)
  (* converting aterms_to_as                                           *)
  (* ----------------------------------------------------------------- *)

  fun conv_as_func_name n = "as_" ^ n

  fun at2as_type_exception dn = dn ^ "_CONV"

  fun at2as_mk_ids i =
      let 
	  val ids = (get_id_list i)
      in
	  hd(ids) ^ String.concat((map (fn(x)=>", "^x)) (tl(ids))) 
      end

  fun at2as_mk_comp_fn kw exc ids n t =
      let 
	  val (_,nids) = mk_ids_un ids t
	  val nnids = List.drop(nids,List.length(ids))
	  val terms = at2as_mk_terms2 kw exc nids (List.length(ids)) t
      in
	  "(fn(AAppl(AId(\"\"),[" ^ 
	  (hd(nnids) ^ String.concat((map (fn(x)=>", "^x)) (tl(nnids)))) ^ 
	  "],_))=> (" ^ terms ^ ")\n" ^
	  "  " ^ (spaces " " kw) ^ "  | _ => raise " ^ exc ^ ")"
      end

  and at2as_mk_comp_terms kw exc ids n nil = ""
    | at2as_mk_comp_terms kw exc ids n [a] = 
      "(" ^ (at2as_mk_terms2 kw exc ids n a) ^ ")"
    | at2as_mk_comp_terms kw exc ids n (a::t) = 
      "(" ^ 
      (at2as_mk_terms2 kw exc ids n a) ^ 
      "),\n  " ^ (spaces " " kw) ^ " " ^
      (at2as_mk_comp_terms kw exc ids (n+1) t)
      

  and at2as_mk_terms2 kw exc ids n (single name) = 
      (conv_as_func_name name) ^ " " ^ List.nth(ids,n)   
    | at2as_mk_terms2 kw exc ids n (option opt) = 
      "(case " ^ List.nth(ids,n) ^ " of\n" ^
      "  " ^ (spaces " " kw) ^ "    AFun(AId(\"None\"),_) => (None)\n" ^
      "  " ^ (spaces " " kw) ^ "  | AAppl(AId(\"Some\"),[opt_val],_) => " ^
      (case opt of
	   (single name) => "(Some (" ^ 
			    (at2as_mk_terms2 kw exc ["opt_val"] 0 opt) ^ "))"
	 | _ => "") ^ (* This is NOW enough, but might be a problem 
                         in the future *)
      "\n" ^
      "  " ^ (spaces " " kw) ^ "  | _ => raise " ^ exc ^ ")"

    | at2as_mk_terms2 kw exc ids n (comp tup) = 
      at2as_mk_comp_terms kw exc ids n tup 
    | at2as_mk_terms2 kw exc ids n (list lst) = 
      "(case " ^ List.nth(ids,n) ^ " of\n" ^
      "  " ^ (spaces " " kw) ^ "    AList(lst_val,_) => ((map " ^ 
      (case lst of
	   (single name) => (conv_as_func_name name) 
	 | (comp tup) => (at2as_mk_comp_fn kw exc (ids@["lst_val"]) (n+1) lst)
 	   (* The other cases are irrelevant now *)
	 | _ => "") ^ ") lst_val)\n" ^
      "  " ^ (spaces " " kw) ^ "  | _ => raise " ^ exc ^ ")"
  
  fun at2as_mk_terms kw exc i =
      let 
	  val ids = (get_id_list i)
      in
	  at2as_mk_terms2 kw exc ids 0 i
      end

  fun at2as_type_def kw (dn,i) =  
      let
	  val exc = at2as_type_exception dn
	  val args = at2as_mk_ids i
	  val terms = at2as_mk_terms kw exc i
      in
	  ("  exception " ^ exc ^ "\n",
	   "  " ^ kw ^ " " ^ (conv_as_func_name dn) ^ " " ^
	   (case i of
		(comp _) =>
		" (AAppl(AId(\"\"),[" ^ args ^ "],_))"
		^ " =\n  " ^ (spaces " " kw) ^ 
		"(" ^ terms ^ ")\n" ^ " " ^ (spaces " " kw) ^
		"| " ^ (conv_as_func_name dn) ^ " _ = raise " ^ exc ^ "\n\n"
	      | _ => args ^ " =\n  " ^ (spaces " " kw) ^ terms^ "\n\n")
	   )
      end

  fun at2as_cons_term kw exc (c,None) = c ^ "\n"
    | at2as_cons_term kw exc (c,Some i) =
      "(" ^ c ^ " (" ^ (at2as_mk_terms kw exc i) ^ "))\n"
      

  fun at2as_cons_patt (c,None) = "AFun (AId (\"" ^ (conv_at_cons c) ^ "\"),_)"
    | at2as_cons_patt (c,Some i) = 
      "AAppl (AId (\"" ^ (conv_at_cons c) ^ "\"),[" ^ 
      (at2as_mk_ids i) ^ 
      "],_)"

  fun at2as_cons_first kw fdn exc h = 
      "  " ^ kw ^ " " ^ fdn ^ " (" ^ (at2as_cons_patt h) ^") =\n" ^
      "  " ^ (spaces " " kw) ^ " " ^ (at2as_cons_term kw exc h) 

  fun at2as_cons_cont kw fdn exc h = 
      " " ^ (spaces " " kw) ^ "| " ^ fdn ^ 
      " (" ^ (at2as_cons_patt h) ^") =\n" ^
      "  " ^ (spaces " " kw) ^ " " ^ (at2as_cons_term kw exc h) 

  fun at2as_datatype_def kw (dn,[]) = ("","")
    | at2as_datatype_def kw (dn,(h::t)) = 
      let 
	  val fdn = conv_as_func_name dn
	  val exc = at2as_type_exception dn
      in
	  ("  exception " ^ exc ^ "\n",
	   (at2as_cons_first kw fdn exc h) ^
	   (String.concat ((map (at2as_cons_cont kw fdn exc)) t)) ^
	   " " ^ (spaces " " kw) ^ 
	   "| " ^ fdn ^ " _ = raise " ^ exc ^ "\n\n")
      end

  fun at2as_define (dd x) = at2as_datatype_def "fun" x
    | at2as_define (an x) = at2as_datatype_def "and" x
    | at2as_define (td x) = at2as_type_def "fun" x
    | at2as_define (wt x) = at2as_type_def "and" x 


  fun at2as_mk_ex_fns2 (exl,fnl) nil = (exl,fnl)
    | at2as_mk_ex_fns2 (exl,fnl) [a] = 
      let 
	  val (nex,nfn) = at2as_define a
      in
	  (exl@[nex],fnl@[nfn])
      end 
    | at2as_mk_ex_fns2 (exl,fnl) (h::t) = 
      let 
	  val (nex,nfn) = at2as_define h
      in
	  at2as_mk_ex_fns2 (exl@[nex],fnl@[nfn]) t
      end

  fun at2as_mk_ex_fns tdefs = at2as_mk_ex_fns2 (nil,nil) tdefs
			      
      

  (* ----------------------------------------------------------------- *)
  (* AS definitions for CASL                                           *)
  (* ----------------------------------------------------------------- *)

  val preamble = "(* autogenerated by asgen.sml, do not edit *)\n\n" ^
                 "structure AS_types =\n" ^
                 "struct\n\n" ^
		 "  local open Library in\n\n"
  val postamble = "\n  end\nend\n"

  val (typedefs:define list) = [
    td ("srcpos",comp ([single "int",single "int"])),
    
    td ("region",comp ([single "srcpos",single "srcpos"])),

    td ("var_id",comp ([single "string", single "int"])),
  
    td ("LINE_NUMBER",option (single "int")),

    dd ("token_or_place",[("token",Some (single "string")),
                          ("place",None)]),

    td ("SIMPLE_ID", comp ([single "string", single "LINE_NUMBER"])),

    td ("TOKEN_OR_MIXFIX",comp ([list (single "token_or_place"),
                                 single "string",
                                 single "LINE_NUMBER"])),

    dd ("ID",[("simple_id",Some (single "TOKEN_OR_MIXFIX")),
              ("compound_id",Some (comp ([single "TOKEN_OR_MIXFIX",
                                          list (single "ID")]))),
              ("var_ID", Some (single "var_id"))]),

    td ("SORT",single "ID"),

    td ("OP_NAME",single "ID"),

    td ("PRED_NAME",single "ID"),

    td ("VAR",single "SIMPLE_ID"),

    td ("SPEC_NAME",single "SIMPLE_ID"),

    td ("ARCH_SPEC_NAME",single "SIMPLE_ID"),

    td ("UNIT_NAME",single "SIMPLE_ID"),

    td ("AUTHOR", single "string"),

    td ("DATE",single "string"),

    td ("SITE",single "string"),

    td ("SUBLANGUAGE",single "string"),

    dd ("ANNO",[("comment",Some (single "string")),
                ("comment_line",Some (single "string")),
                ("label_anno",Some (single "ID")),
                ("unparsed_anno",Some (single "string")),
                ("number_anno",Some (single "ID")),
                ("floating_anno",Some (comp ([single "ID",single "ID"]))),
                ("string_anno",Some (comp ([single "ID",single "ID"]))),
                ("list_anno",Some (comp ([single "ID",single "ID",
                                          single "ID"]))),
                ("display_anno",Some (comp ([single "ID",single "string"]))),
                ("prec_anno",Some (comp ([single "bool",list (single "ID"),
                                          list (single "ID")]))),
                ("lassoc_anno",Some (list (single "ID"))),
                ("rassoc_anno",Some (list (single "ID"))),
                ("conservative",None),
                ("definitional",None),
                ("mono",None),
                ("pos_ANNO",Some (comp ([single "region",single "ANNO"]))),
                ("implies",None)]),

    td ("URL",single "string"),

    td ("PATH_NAME",single "string"),

    dd ("LIB_ID",[("url",Some (single "URL")),
                  ("pos_LIB_ID",Some (comp ([single "region",single "LIB_ID"]))),
                  ("path_name",Some (single "PATH_NAME"))]),

    dd ("VERSION",[("version",Some (list (single "string"))),
                   ("pos_VERSION",Some (comp ([single "region",single "VERSION"])))]),

    dd ("LIB_NAME",[("lib",Some (single "LIB_ID")),
                    ("pos_LIB_NAME",Some (comp ([single "region",single "LIB_NAME"]))),
                    ("versioned_lib",Some (comp ([single "LIB_ID",
                                                  single "VERSION"])))]),
    
    dd ("SORTS",[("sorts",Some (list (single "SORT"))),
                 ("pos_SORTS",Some (comp ([single "region",single "SORTS"])))]),

    td ("VAR_DECL",comp ([list (single "VAR"),single "SORT"])),

    dd ("ARG_DECL",[("arg_decl",Some (single "VAR_DECL")),
                    ("pos_ARG_DECL",Some (comp ([single "region",single "ARG_DECL"])))]),

    dd ("OP_TYPE",[("total_op_type",Some (comp ([single "SORTS",
                                                 single "SORT"]))),
                   ("pos_OP_TYPE",Some (comp ([single "region",single "OP_TYPE"]))),
                   ("partial_op_type",Some (comp ([single "SORTS",
                                                   single "SORT"])))]),

    dd ("OP_HEAD",[("total_op_head",Some (comp ([list (single "ARG_DECL"),
                                                 single "SORT"]))),
                   ("pos_OP_HEAD",Some (comp ([single "region",single "OP_HEAD"]))),
                   ("partial_op_head",Some (comp ([list (single "ARG_DECL"),
                                                   single "SORT"])))]),

    dd ("OP_SYMB",[("op_symb",Some (comp ([single "OP_NAME",
                                           option (single "OP_TYPE")]))),
                   ("pos_OP_SYMB",Some (comp ([single "region",single "OP_SYMB"]))),
                   ("var_OP_SYMB", Some (single "var_id"))]),

    dd ("PRED_TYPE",[("pred_type",Some (single "SORTS")),
                     ("pos_PRED_TYPE",Some (comp ([single "region",single "PRED_TYPE"])))]),

    dd ("PRED_HEAD",[("pred_head",Some (list (single "ARG_DECL"))),
                     ("pos_PRED_HEAD",Some (comp ([single "region",single "PRED_HEAD"])))]),

    dd ("PRED_SYMB",[("pred_symb",Some (comp ([single "PRED_NAME",
                                            option (single "PRED_TYPE")]))),
                     ("pos_PRED_SYMB",Some (comp ([single "region",single "PRED_SYMB"]))),
                     ("var_PRED_SYMB", Some (single "var_id"))]),

    dd ("TERM",[("var_or_const",Some (single "SIMPLE_ID")),  (* only for parsing purposes *)
                ("qual_var",Some (comp ([single "VAR",single "SORT"]))),
                ("application",Some (comp ([single "OP_SYMB",
                                            single "TERMS"]))),
                ("sorted_term",Some (comp ([single "TERM",single "SORT"]))),
                ("cast",Some (comp ([single "TERM",single "SORT"]))),
                ("conditional",Some (comp ([single "TERM",single "FORMULA",
                                            single "TERM"]))),
                ("pos_TERM",Some (comp ([single "region",single "bool",single "TERM"]))),
                ("var_TERM", Some (single "var_id")),
                ("unparsed_term",Some (single "string"))]), (* only for parsing purposes *)

    an ("TERMS",[("terms",Some (list (single "TERM"))),
                 ("pos_TERMS",Some (comp ([single "region",single "TERMS"])))]),

    an ("QUANTIFIER",[("forall",None),
                      ("exists",None),
                      ("pos_QUANTIFIER",Some (comp ([single "region",single "QUANTIFIER"]))),
                      ("exists_uniquely",None)]),

    an ("FORMULA",[("quantification",Some (comp ([single "QUANTIFIER",
                                                  list (single "VAR_DECL"),
                                                  single "FORMULA"]))),
                   ("pred_quantification",Some (comp ([single "QUANTIFIER",
                                                  list (comp [single "ID",single "PRED_TYPE"]),
                                                  single "FORMULA"]))), (* only for output of coding to SOL= *)
                   ("conjunction",Some (list (single "FORMULA"))),
                   ("disjunction",Some (list (single "FORMULA"))),
                   ("implication",Some (comp ([single "FORMULA",
                                               single "FORMULA"]))),
                   ("equivalence",Some (comp ([single "FORMULA",
                                               single "FORMULA"]))),
                   ("negation",Some (single "FORMULA")),
                   ("atom",Some (single "ATOM")),
                   ("pos_FORMULA",Some (comp ([single "region",single "bool",single "FORMULA"]))),
                   ("var_FORMULA", Some (single "var_id")),
                   ("sort_gen_ax",Some (comp ([list (single "SORT"),
                                               list (single "OP_SYMB")]))),
                   ("sort_cogen_ax",Some (comp ([list (single "SORT"),
                                               list (single "OP_SYMB")]))), (* only for CoCASL *)
                   ("sort_cofree_ax",Some (comp ([list (single "SORT"),
                                               list (single "OP_SYMB")]))),  (* only for CoCASL *)
                   ("unparsed_formula",Some (single "string"))]),   (* only for parsing purposes *)

    an ("ATOM",[("ttrue",None),
                ("ffalse",None),
                ("predication",Some (comp ([single "PRED_SYMB",
                                            single "TERMS"]))),
                ("definedness",Some (single "TERM")),
                ("existl_equation",Some (comp ([single "TERM",
                                                single "TERM"]))),
                ("strong_equation",Some (comp ([single "TERM",
                                                single "TERM"]))),
                ("membership",Some (comp ([single "TERM",single "SORT"])))]),

    td ("L_FORMULA",comp ([single "FORMULA",list (single "ANNO")])),

    dd ("COMPONENTS",[("total_select",Some (comp ([list (single "OP_NAME"),
                                                   single "SORT"]))),
                      ("partial_select",Some (comp ([list (single "OP_NAME"),
                                                     single "SORT"]))),
                      ("pos_COMPONENTS",Some (comp ([single "region",single "COMPONENTS"]))),
                      ("sort_component",Some (single "SORT"))]),

    dd ("ALTERNATIVE",[("total_construct",Some (comp ([single "OP_NAME",
                                                list (single "COMPONENTS")]))),
                       ("partial_construct",Some (comp ([single "OP_NAME",
                                                list (single "COMPONENTS")]))),
                       ("pos_ALTERNATIVE",Some (comp ([single "region",single "ALTERNATIVE"]))),
                       ("subsort",Some (list (single "SORT")))]),

    td ("L_ALTERNATIVE",comp ([single "ALTERNATIVE",list (single "ANNO")])),

    dd ("DATATYPE_DECL",[("datatype_decl",Some (comp ([single "SORT",
                                                list (single "L_ALTERNATIVE"),
                                                list (single "ANNO")]))),
                         ("pos_DATATYPE_DECL",Some (comp ([single "region",single "DATATYPE_DECL"]))),
                         ("var_DATATYPE_DECL", Some (single "var_id"))]),

    dd ("OP_ATTR",[("associative",None),
                   ("commutative",None),
                   ("idempotent",None),
                   ("pos_OP_ATTR",Some (comp ([single "region",single "OP_ATTR"]))),
                   ("unit_op_attr",Some (single "TERM"))]),

    dd ("OP_ITEM",[("op_decl",Some (comp ([list (single "OP_NAME"),
                                           single "OP_TYPE",
                                           list (single "OP_ATTR")]))),
                   ("pos_OP_ITEM",Some (comp ([single "region",single "OP_ITEM"]))),
                   ("op_defn",Some (comp ([single "OP_NAME",single "OP_HEAD",
                                           single "TERM",list (single "ANNO")]
                                          )))]),

    td ("L_OP_ITEM",comp ([single "OP_ITEM",list (single "ANNO")])),

    dd ("PRED_ITEM",[("pred_decl",Some (comp ([list (single "PRED_NAME"),
                                               single "PRED_TYPE"]))),
                     ("pos_PRED_ITEM",Some (comp ([single "region",single "PRED_ITEM"]))),
                     ("pred_defn",Some (comp ([single "PRED_NAME",
                                               single "PRED_HEAD",
                                               single "L_FORMULA",
                                               list (single "ANNO")])))]),

    td ("L_PRED_ITEM",comp ([single "PRED_ITEM",list (single "ANNO")])),

    dd ("SORT_ITEM",[("sort_decl",Some (list (single "SORT"))),
                     ("subsort_decl",Some (comp ([list (single "SORT"),
                                                  single "SORT"]))),
                     ("subsort_defn",Some (comp ([single "SORT",
                                                  single "VAR",single "SORT",
                                                  single "FORMULA",
                                                  list (single "ANNO")]))),
                     ("pos_SORT_ITEM",Some (comp ([single "region",single "SORT_ITEM"]))),
                     ("iso_decl",Some (list (single "SORT")))]),

    td ("L_SORT_ITEM",comp ([single "SORT_ITEM",list (single "ANNO")])),

    dd ("SIG_ITEMS",[("sort_items",Some (comp ([list (single "L_SORT_ITEM"),
                                                list (single "ANNO")]))),
                     ("op_items",Some (comp ([list (single "L_OP_ITEM"),
                                              list (single "ANNO")]))),
                     ("pred_items",Some (comp ([list (single "L_PRED_ITEM"),
                                                list (single "ANNO")]))),
                     ("pos_SIG_ITEMS",Some (comp ([single "region",single "SIG_ITEMS"]))),
                     ("datatype_items",Some (comp ([list (single
                                  "DATATYPE_DECL"),list (single "ANNO")]))),
                     ("var_SIG_ITEMS", Some (single "var_id"))]),



    dd ("BASIC_ITEMS",[("sig_items",Some (single "SIG_ITEMS")),
                       ("free_datatype",Some (comp ([list (single
                                  "DATATYPE_DECL"),list (single "ANNO")]))),
                       ("cofree_datatype",Some (comp ([list (single
                                  "DATATYPE_DECL"),list (single "ANNO")]))),
                       ("sort_gen",Some (comp ([list (single "SIG_ITEMS"),
                                                list (single "ANNO")]))),
                       ("sort_cogen",Some (comp ([list (single "SIG_ITEMS"),
                                                list (single "ANNO")]))),
                       ("var_items",Some (comp ([list (single "VAR_DECL"),
                                                 list (single "ANNO")]))),
                       ("local_var_axioms",Some (comp ([list (single
                                       "VAR_DECL"),list (single "L_FORMULA"),
                                       list (single "ANNO")]))),
                       ("pos_BASIC_ITEMS",Some (comp ([single "region",single "BASIC_ITEMS"]))),
                       ("axiom_items",Some (comp ([list (single "L_FORMULA"),
                                                   list (single "ANNO")]))),
                       ("var_BASIC_ITEMS", Some (single "var_id"))]),

    dd ("BASIC_SPEC",[("basic_spec",Some (list (single "BASIC_ITEMS"))),
                      ("pos_BASIC_SPEC",Some (comp ([single "region",single "BASIC_SPEC"]))),
                      ("var_BASIC_SPEC", Some (single "var_id"))]),

    dd ("TYPE",[("op_symb_type",Some (single "OP_TYPE")),
                ("pos_TYPE",Some (comp ([single "region",single "TYPE"]))),
                ("pred_symb_type",Some (single "PRED_TYPE"))]),

    dd ("SYMB",[("symb_id",Some (single "ID")),
                ("pos_SYMB",Some (comp ([single "region",single "SYMB"]))),
                ("qual_id",Some (comp ([single "ID",single "TYPE"])))]),

    dd ("SYMB_MAP",[("symb_map",Some (comp ([single "SYMB",single "SYMB"]))),
                    ("pos_SYMB_MAP",Some (comp ([single "region",single "SYMB_MAP"])))]),

    dd ("SYMB_OR_MAP",[("symb",Some (single "SYMB")),
                       ("symb_or_map",Some (single "SYMB_MAP")),
                       ("pos_SYMB_OR_MAP",Some (comp ([single "region",single "SYMB_OR_MAP"])))]),

    dd ("SYMB_KIND",[("implicitk",None),
                     ("sortsk",None),
                     ("opsk",None),
                     ("predsk",None),
                     ("pos_SYMB_KIND",Some (comp ([single "region",single "SYMB_KIND"])))]),

    td ("VIEW_NAME",single "SIMPLE_ID"),

    dd ("SYMB_MAP_ITEMS",[("symb_map_items",Some (comp ([single "SYMB_KIND",
                                            list (single "SYMB_OR_MAP")]))),
                          ("pos_SYMB_MAP_ITEMS",Some (comp ([single
                                            "region",single
                                                "SYMB_MAP_ITEMS"]))),
                          ("var_SYMB_MAP_ITEMS",Some (single "var_id"))]),

    dd ("FIT_ARG",[("fit_spec",Some (comp ([single "L_SPEC",
                                            list (single "SYMB_MAP_ITEMS")]))),
                   ("fit_view",Some (comp ([single "VIEW_NAME",
                                            list (single "FIT_ARG")]))),
                   ("pos_FIT_ARG",Some (comp ([single "region",single "FIT_ARG"]))),
		   ("var_FIT_ARG",Some (single "var_id"))]),

    an ("SYMB_ITEMS",[("symb_items",Some (comp ([single "SYMB_KIND",
                                                 list (single "SYMB")]))),
                      ("pos_SYMB_ITEMS",Some (comp ([single
                                                 "region",single
                                                 "SYMB_ITEMS"]))),
                      ("var_SYMB_ITEMS",Some (single "var_id"))]),

    an ("RENAMING",[("renaming",Some (list (single "SYMB_MAP_ITEMS"))),
                    ("pos_RENAMING",Some (comp ([single "region",single "RENAMING"]))),
		    ("var_RENAMING",Some (single "var_id"))]),

    an ("RESTRICTION",[("hide",Some (list (single "SYMB_ITEMS"))),
                       ("reveal",Some (list (single "SYMB_MAP_ITEMS"))),
                       ("pos_RESTRICTION",Some (comp ([single "region",single "RESTRICTION"]))),
		       ("var_RESTRICTION",Some (single "var_id"))]),

    an ("GENERICITY",[("genericity",Some (comp ([single "PARAMS",
                                                 single "IMPORTS"]))),
                      ("pos_GENERICITY",Some (comp ([single "region",single "GENERICITY"]))),
                      ("var_GENERICITY",Some (single "var_id"))]),

    an ("PARAMS",[("params",Some (list (single "L_SPEC"))),
                  ("pos_PARAMS",Some (comp ([single "region",single "PARAMS"])))]),

    an ("IMPORTS",[("imports",Some (list (single "L_SPEC"))),
                   ("pos_IMPORTS",Some (comp ([single "region",single "IMPORTS"])))]),

    an ("SPEC",[("basic",Some (single "BASIC_SPEC")),
                ("translation",Some (comp ([single "L_SPEC",
                                            single "RENAMING",
                                            list (single "ANNO")]))),
                ("reduction",Some (comp ([single "L_SPEC",
                                          single "RESTRICTION",
                                          list (single "ANNO")]))),
                ("union_spec",Some (list (comp ([single "L_SPEC",
                                                 list (single "ANNO")])))),
                ("extension",Some (list (comp ([single "L_SPEC",
                                                 list (single "ANNO")])))),
                ("free_spec",Some (comp ([single "L_SPEC",
                                          list (single "ANNO")]))),
                ("cofree_spec",Some (comp ([single "L_SPEC",
                                          list (single "ANNO")]))),
                ("local_spec",Some (comp ([single "L_SPEC",
                                           list (single "ANNO"),single "L_SPEC",
                                           list (single "ANNO")]))),
                ("closed_spec",Some (comp ([single "L_SPEC",
                                           list (single "ANNO")]))),
                ("spec_inst",Some (comp ([single "SPEC_NAME",
                                          list (single "FIT_ARG")]))),
                ("pos_SPEC",Some (comp ([single "region",single "bool",single "SPEC"]))),
		("var_SPEC", Some (single "var_id"))]),

    wt ("L_SPEC",comp ([single "SPEC",list (single "ANNO")])),

    dd ("VIEW_TYPE",[("view_type",Some (comp ([single "L_SPEC",single "L_SPEC"]
                                         ))),
                     ("pos_VIEW_TYPE",Some (comp ([single "region",single "VIEW_TYPE"])))]),

    dd ("ARCH_SPEC",[("basic_arch_spec",Some (comp ([list (comp ([
                                        single "UNIT_DECL_DEFN",list (single
                                        "ANNO")])),single "RESULT_UNIT",
                                        list (single "ANNO")]))),
                     ("named_arch_spec",Some (single "ARCH_SPEC_NAME")),
                     ("pos_ARCH_SPEC",Some (comp ([single "region",single "ARCH_SPEC"])))]),

    an ("UNIT_DECL_DEFN",[("unit_decl_case",Some (single "UNIT_DECL")),
                          ("unit_defn_case",Some (single "UNIT_DEFN")),
                          ("pos_UNIT_DECL_DEFN",Some (comp ([single "region",single "UNIT_DECL_DEFN"])))]),

    an ("UNIT_DECL",[("unit_decl",Some (comp ([single "UNIT_NAME",
                                               single "UNIT_SPEC",
                                               single "UNIT_IMPORTS"]))),
                     ("pos_UNIT_DECL",Some (comp ([single "region",single "UNIT_DECL"])))]),

    an ("UNIT_IMPORTS",[("unit_imports",Some (list (single "UNIT_TERM"))),
                        ("pos_UNIT_IMPORTS",Some (comp ([single "region",single "UNIT_IMPORTS"])))]),

    an ("UNIT_DEFN",[("unit_defn",Some (comp ([single "UNIT_NAME",
                                               single "UNIT_EXPRESSION"]))),
                     ("pos_UNIT_DEFN",Some (comp ([single "region",single "UNIT_DEFN"])))]),

    an ("UNIT_SPEC",[("unit_type_case",Some (single "UNIT_TYPE")),
                     ("spec_name_case",Some (single "SPEC_NAME")),
                     ("arch_spec_case",Some (comp ([single "ARCH_SPEC",
                                                    list (single "ANNO")]))),
                     ("closed",Some (single "UNIT_SPEC")),
                     ("pos_UNIT_SPEC",Some (comp ([single "region",single "UNIT_SPEC"])))]),

    an ("UNIT_TYPE",[("unit_type",Some (comp ([list (single "L_SPEC"),
                                               single "L_SPEC"]))),
                     ("pos_UNIT_TYPE",Some (comp ([single "region",single "UNIT_TYPE"])))]),

    an ("RESULT_UNIT",[("result_unit",Some (comp ([single "UNIT_EXPRESSION",
                                                   list (single "ANNO")]))),
                       ("pos_RESULT_UNIT",Some (comp ([single "region",single "RESULT_UNIT"])))]),

    an ("UNIT_EXPRESSION",[("unit_expression",Some (comp ([list (single
                                       "UNIT_BINDING"),single "UNIT_TERM"]))),
                           ("pos_UNIT_EXPRESSION",Some (comp ([single "region",single "UNIT_EXPRESSION"])))]),

    an ("UNIT_BINDING",[("unit_binding",Some (comp ([single "UNIT_NAME",
                                                     single "UNIT_SPEC"]))),
                        ("pos_UNIT_BINDING",Some (comp ([single "region",single "UNIT_BINDING"])))]),

    an ("UNIT_TERM",[("unit_translation",Some (comp ([single "UNIT_TERM",
                                                      single "RENAMING"]))),
                     ("unit_reduction",Some (comp ([single "UNIT_TERM",
                                                    single "RESTRICTION"]))),
                     ("amalgamation",Some (list (single "UNIT_TERM"))),
                     ("local_unit",Some (comp ([list (single "UNIT_DEFN"),
                                                single "UNIT_TERM"]))),
                     ("unit_appl",Some (comp ([single "UNIT_NAME",
                                             list (single "FIT_ARG_UNIT")]))),
                     ("pos_UNIT_TERM",Some (comp ([single "region",
                           (* GROUP-UNIT-TERM *)   single "bool",
						   single "UNIT_TERM"])))]),

    an ("FIT_ARG_UNIT",[("fit_arg_unit",Some (comp ([single "UNIT_TERM",
                                        list (single "SYMB_MAP_ITEMS")]))),
                        ("pos_FIT_ARG_UNIT",Some (comp ([single "region",single "FIT_ARG_UNIT"])))]),

    td ("L_ARCH_SPEC",comp ([single "ARCH_SPEC",list (single "ANNO")])),

    td ("ITEM_NAME",single "SIMPLE_ID"),

    dd ("ITEM_NAME_OR_MAP",[("item_name",Some (single "ITEM_NAME")),
                            ("item_name_map",Some (comp ([single "ITEM_NAME",
                                                   single "ITEM_NAME"]))),
                            ("pos_ITEM_NAME_OR_MAP",Some (comp ([single "region",single "ITEM_NAME_OR_MAP"])))]),

    dd ("LIB_ITEM",[("spec_defn",Some (comp ([single "SPEC_NAME",single
                                              "GENERICITY",single "L_SPEC",
                                              list (single "ANNO")]))),
                    ("view_defn",Some (comp ([single "VIEW_NAME",
                                              single "GENERICITY",
                                              single "VIEW_TYPE",list
                                              (single "SYMB_MAP_ITEMS"),list
                                              (single "ANNO")]))),
                    ("arch_spec_defn",Some (comp ([single "ARCH_SPEC_NAME",
                                                   single "L_ARCH_SPEC",
                                                   list (single "ANNO")]))),
                    ("unit_spec_defn",Some (comp ([single "SPEC_NAME",
                                                   single "UNIT_SPEC",
                                                   list (single "ANNO")]))),
                    ("download_items",Some (comp ([single "LIB_NAME", list
                                                   (single "ITEM_NAME_OR_MAP"),
                                                   list (single "ANNO")]))),
                    ("pos_LIB_ITEM",Some (comp ([single "region",single "LIB_ITEM"]))),
		    ("var_LIB_ITEM", Some (single "var_id"))]),

   dd ("LIB_DEFN",[("lib_defn",Some (comp ([single "LIB_NAME",list (single
                                         "LIB_ITEM"),list (single "ANNO")]))),
                   ("pos_LIB_DEFN",Some (comp ([single "region",single "LIB_DEFN"])))])
    ]

   val typedefstr = preamble ^ (String.concat ((map p_define) typedefs))
                    ^ postamble  

   fun write name = let
                      val stream = TextIO.openOut name
                    in
                      (TextIO.output (stream,typedefstr);
                       TextIO.closeOut stream)
                    end

   fun writeas () = write "as_types.sml"

  (* ----------------------------------------------------------------- *)
  (* writing aterm_conv_functions                                      *)
  (* ----------------------------------------------------------------- *)
   

   val atermconvstr = asconv_preamble ^ 
		      String.concat((map as2at_define) typedefs) ^
		      (let 
			   val (excs,funcs) = at2as_mk_ex_fns typedefs
		       in
			   String.concat(excs) ^ 
			   String.concat(funcs) 
		       end) ^
		      asconv_postamble

   fun write_conv name = 
       let
	   val stream = TextIO.openOut name
       in
           (TextIO.output (stream,atermconvstr);
            TextIO.closeOut stream)
       end

   fun writeasconv () = write_conv "at_gen.sml"      
end
