(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author: Till Mossakowski, University of Bremen			   *)
(* Date: 1998				 			           *)
(* Purpose of this file: Printing functions for basic specifications       *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* This module converts the CASL abstract syntax back into concrete input
   syntax. Infix symbols are handeled properly.
   Future work: improve priority handling of infix symbols.
                Layout should be oriented at the input position information.
*)

structure BasicPrint : sig
val quote_underscore : string -> string
val quote_underscores : string -> string
(*val indent_one : string -> string
val drop_first : string list -> string list
val drop_last : string list -> string list
val indent : string -> string *)
val print_list : ('a -> string) -> string -> 'a list -> string
val do_print_list  : TextIO.outstream
          -> (TextIO.outstream -> 'a -> unit)
             -> TextIO.vector -> 'a list -> unit
val print_SIMPLE_ID : AS.SIMPLE_ID -> string
val print_ID : AS.ID -> string
val print_IDs : AS.ID list -> string
val print_VAR : AS.VAR -> string
val print_SORTS1 : AS.SORT list -> string
val print_SORTS : AS.SORTS -> string
val print_OP_TYPE : AS.OP_TYPE -> string
val print_OP_SYMB : AS.OP_SYMB -> string
val print_PRED_TYPE : AS.PRED_TYPE -> string
val print_PRED_SYMB : AS.PRED_SYMB -> string
val print_vars : AS.VAR list -> string
val print_VAR_DECL : AS.VAR_DECL -> string
val print_VAR_DECLList : AS.VAR_DECL list -> string

val print_TERMList : AS.TERM list -> string
val print_TERM : AS.TERM -> string
val print_TERMS : AS.TERMS -> string
val print_PREDICATION : AS.PRED_SYMB * AS.TERMS -> string
val print_ATOM : AS.ATOM -> string
val print_QUANTIFIER : AS.QUANTIFIER -> string
val print_FORMULA : AS.FORMULA -> string
val print_FORMULAs : AS.FORMULA list -> string
val print_L_FORMULA : AS.L_FORMULA -> string
val print_L_FORMULAList : AS.L_FORMULA list -> string

val infix_TERMList : AS.TERM list -> string
val infix_TERM : AS.TERM -> string
val infix_TERMS : AS.TERMS -> string
val infix_PREDICATION : AS.PRED_SYMB * AS.TERMS -> string
val infix_ATOM : AS.ATOM -> string
val infix_FORMULA : AS.FORMULA -> string
val infix_FORMULAs : AS.FORMULA list -> string
val infix_L_FORMULA : AS.L_FORMULA -> string
val infix_L_FORMULAList : AS.L_FORMULA list -> string

val print_ANNO : AS.ANNO -> string
val print_ANNOS : AS.ANNO list -> string

val print_sort_entry : AS.ID * LocalEnv.sort_entry -> string
val print_fun_entry : AS.ID * LocalEnv.fun_entry -> string
val print_pred_entry : AS.ID * LocalEnv.pred_entry -> string
val print_local_list : LocalEnv.local_list -> string
val output_local_env : TextIO.outstream -> LocalEnv.local_env -> unit
val output_sign : TextIO.outstream -> LocalEnv.sign -> unit
val print_local_env : LocalEnv.local_env -> string
val print_sign : LocalEnv.sign -> string

val latex_SIMPLE_ID : AS.SIMPLE_ID -> string 
val latex_local_env : TextIO.outstream -> LocalEnv.local_env -> unit
val latex_sign : TextIO.outstream -> LocalEnv.sign -> unit

end
= struct

open Utils AS  LocalEnv TextIO;
infix mem;

(* MR: to print ccc variables *)
fun print_var_id (s,i) = s ^ Int.toString(i);

(* MR: ccc vars added to
  print_ID 
  print_FORMULA
  infix_FORMULA
  print_OP_SYMB1
  print_OP_SYMB
  print_PRED_SYMB1
  print_PRED_SYMB
*)

fun quote_underscore "_" = "'_"
  | quote_underscore x = x

val quote_underscores = concat o (map quote_underscore) o explode;

fun escape_latex "_" = "\\_"
  | escape_latex "^" = "\\power "
  | escape_latex x = x

val escape_latexs = concat o (map escape_latex) o explode;

fun indent_one "\n" = "\n  "
  | indent_one c = c
  
fun drop_first nil = nil
  | drop_first ("\n  "::l) = "\n"::l
  | drop_first (x::l) = x::drop_first l
  
fun drop_last l = 
    rev (drop_first (rev l))
   
fun indent str = implode (" "::" "::drop_last (map indent_one (explode str)))


fun print_list f delim nil = ""
  | print_list f delim (x::nil) = f x
  | print_list f delim (x::xs) = f x^delim^print_list f delim xs
  

fun do_print_list h f delim nil = ()
  | do_print_list h f delim (x::nil) = f h x
  | do_print_list h f delim (x::xs) = 
    (f h x; 
     output (h,delim);
     do_print_list h f delim xs)
  
  
fun print_SIMPLE_ID (id,no) = id

fun print_ID (simple_id (tks,id,no)) = id
  | print_ID (compound_id ((tks,id,no),nil)) = id
  | print_ID (compound_id ((tks,id,no),idlist))
    = id^"["^print_IDs idlist^"]"
  | print_ID (var_ID v) =  print_var_id v

and print_IDs nil = ""
  | print_IDs (id::nil) = print_ID id
  | print_IDs (id1::id2::rest) 
     = print_ID id1^","^print_IDs (id2::rest)
     
   
fun print_VAR (v,_) = v


fun print_ANNO (label_anno id) = " %("^print_ID id^")%"
  | print_ANNO (comment_line s) = "%%"^s
  | print_ANNO (comment s) = "%{"^s^"}%"
  | print_ANNO (unparsed_anno s) = s^" %[unparsed]% "
  | print_ANNO (number_anno id) = "%number "^print_ID id
  | print_ANNO (floating_anno (id1,id2)) = "%floating "^print_ID id1^", "^print_ID id2
  | print_ANNO (string_anno (id1,id2)) = "%string "^print_ID id1^", "^print_ID id2
  | print_ANNO (list_anno (id1,id2,id3)) = "%list "^print_ID id1^", "^print_ID id2^", "^print_ID id3
(* The following case is a hack corresponding to the hack of the parser *)
  | print_ANNO (display_anno (id,s)) = 
    "%display" ^ 
    (if valOf(Substring.first(Substring.all(s))) = #"(" then 
	 s 
     else
	 " " ^ s)
  | print_ANNO (prec_anno (less,ids1,ids2)) = 
    "%prec {"^print_IDs ids1^"} "^(if less then "<" else "<>")^" {"^print_IDs ids2^"}"
  | print_ANNO (lassoc_anno ids) = "%left_assoc "^print_IDs ids
  | print_ANNO (rassoc_anno ids) = "%right_assoc "^print_IDs ids
  | print_ANNO (conservative) = "%cons"
  | print_ANNO (definitional) = "%def"
  | print_ANNO (pos_ANNO (r,an)) = print_ANNO an
  | print_ANNO (implies) = "%implies"
  | print_ANNO _ = "%unknown_anno"			   
fun print_ANNOS annos = print_list print_ANNO "\n" annos

fun print_SORTS1 nil = ""
  | print_SORTS1 ([s]) = print_ID s
  | print_SORTS1 (s::xs) = print_ID s^"*"^print_SORTS1 xs

 
fun print_SORTS (sorts sl) = print_SORTS1 sl
  | print_SORTS (pos_SORTS (_,s)) = print_SORTS s

fun print_OP_TYPE (total_op_type (sorts [], res)) =
	print_SORTS1 [res]
  | print_OP_TYPE (total_op_type (arg, res)) =
	(print_SORTS arg)^"->"^(print_SORTS1 [res])
  | print_OP_TYPE (partial_op_type (sorts [], res)) =
	"?"^(print_SORTS1 [res])
  | print_OP_TYPE (partial_op_type (arg, res)) =
	(print_SORTS arg)^"->?"^(print_SORTS1 [res])
  | print_OP_TYPE (pos_OP_TYPE (_,opt)) =
    print_OP_TYPE opt
	
fun print_OP_SYMB1 (op_symb (n, Some t)) =
	(print_SORTS1 [n])^" : "^print_OP_TYPE t
  | print_OP_SYMB1 (op_symb (n,None)) = (print_SORTS1 [n])
  | print_OP_SYMB1 (pos_OP_SYMB (_,sy)) =
    print_OP_SYMB1 sy
  | print_OP_SYMB1 (var_OP_SYMB v)=
    print_var_id v

fun print_OP_SYMB (op_symb (n, Some t)) =
	"(op "^(print_SORTS1 [n])^":"^print_OP_TYPE t^")"
  | print_OP_SYMB (op_symb (n,None)) = (print_SORTS1 [n])
  | print_OP_SYMB (pos_OP_SYMB (_,sy)) =
    print_OP_SYMB sy
  | print_OP_SYMB (var_OP_SYMB v)=
    print_var_id v

fun print_PRED_TYPE (pred_type (sorts [])) = "()"
  | print_PRED_TYPE (pred_type t) =
	print_SORTS t
  | print_PRED_TYPE (pos_PRED_TYPE (_,pt)) =
    print_PRED_TYPE pt

fun print_PRED_SYMB1 (pred_symb(n,Some t)) =
	"pred "^(print_SORTS1 [n])^" : "^print_PRED_TYPE t
  | print_PRED_SYMB1 (pred_symb(n,None)) = print_SORTS1 [n]
  | print_PRED_SYMB1 (pos_PRED_SYMB (_,sy)) =
    print_PRED_SYMB1 sy
  | print_PRED_SYMB1 (var_PRED_SYMB v)=
    print_var_id v



fun print_PRED_SYMB (pred_symb(n,Some t)) =
	"(pred "^(print_SORTS1 [n])^":"^print_PRED_TYPE t^")"
  | print_PRED_SYMB (pred_symb(n,None)) = print_SORTS1 [n]
  | print_PRED_SYMB (pos_PRED_SYMB (_,sy)) =
    print_PRED_SYMB sy
  | print_PRED_SYMB (var_PRED_SYMB v)=
    print_var_id v

fun print_vars nil = ""
  | print_vars ((x,l)::nil) = x
  | print_vars ((x,l)::xs) = x^","^print_vars xs

fun print_VAR_DECL (vl,s) =
	print_vars vl^" : "^(print_SORTS1 [s])

fun print_VAR_DECLList vl = print_list print_VAR_DECL ";" vl



fun print_TERM ((var_or_const x):TERM):string = print_vars [x]
  | print_TERM (qual_var (x,s )) = print_vars [x]^":"^(print_SORTS1 [s])
  | print_TERM (application (f,t)) =
    let val ts = get_terms t
    in
	print_OP_SYMB f^
	(if ts=nil then "" else "(")
	^print_TERMS (terms ts)^
	(if ts=nil then "" else ")")
    end
  | print_TERM (sorted_term (t,s)) =
	print_TERM t^" : "^(print_SORTS1 [s])
  | print_TERM (conditional (t,f,t1)) =
	print_TERM t^" when "^(print_FORMULA f) ^" else "^print_TERM
	t1
  | print_TERM (var_TERM v) = print_var_id v
  | print_TERM (cast (t,s)) = 
	print_TERM t^" as "^(print_SORTS1 [s])
  | print_TERM (unparsed_term t) = t
  | print_TERM (pos_TERM (_,true,t)) = "("^print_TERM t^")"
  | print_TERM (pos_TERM (_,false,t)) = print_TERM t

and print_TERMS ts = print_list print_TERM "," (get_terms ts)

and print_PREDICATION (p,ts) =
    let val tl = get_terms ts
    in
	print_PRED_SYMB p
	^(if tl=nil then "" else "(")
	^print_TERMS (terms tl)^
	(if tl=nil then "" else ")")
    end

and print_ATOM ttrue = "true"
  | print_ATOM ffalse = "false"
  | print_ATOM (predication p) = print_PREDICATION p
  | print_ATOM (definedness t) = "def "^print_TERM t
  | print_ATOM (existl_equation (t1,t2)) =
	print_TERM t1^" =e= "^print_TERM t2
  | print_ATOM (strong_equation (t1,t2)) =
	print_TERM t1^" = "^print_TERM t2
  | print_ATOM (membership (t,s)) =
	print_TERM t^" in "^print_SORTS1 [s]
(*  | print_ATOM (pos_ATOM (_,a)) =
    print_ATOM a*)

and print_QUANTIFIER forall = "forall"
  | print_QUANTIFIER exists = "exists"
  | print_QUANTIFIER exists_uniquely = "exists!"
  | print_QUANTIFIER (pos_QUANTIFIER (_,q)) =
    print_QUANTIFIER q

and print_FORMULA (quantification (q,vdlist,phi)) =
	print_QUANTIFIER q^" "
	^(print_VAR_DECLList vdlist)^" . "
	^print_FORMULA phi
  | print_FORMULA (conjunction phis) =
	print_list print_FORMULA " /\\ " phis
  | print_FORMULA (disjunction phis) =
	print_list print_FORMULA " \\/ " phis
  | print_FORMULA (implication (phi1,phi2)) =
	print_FORMULA phi1^" => "^print_FORMULA phi2
  | print_FORMULA (equivalence (phi1,phi2)) =
	print_FORMULA phi1^" <=> "^print_FORMULA phi2
  | print_FORMULA (negation phi) =
	"not "^print_FORMULA phi
  | print_FORMULA (atom a) = print_ATOM a
  | print_FORMULA (var_FORMULA v) = print_var_id v
  | print_FORMULA (sort_gen_ax (sl,opl)) = 
     "generated { sorts "^print_list print_ID "; " sl^";  ops "
                 ^print_list print_OP_SYMB1 "; " opl
                 ^" }"
  | print_FORMULA (sort_cogen_ax (sl,opl)) = 
     "cogenerated { sorts "^print_list print_ID "; " sl^";  ops "
                 ^print_list print_OP_SYMB1 "; " opl
                 ^" }"
  | print_FORMULA (sort_cofree_ax (sl,opl)) = 
     "cofree %{unstructured}% { sorts "^print_list print_ID "; " sl^";  ops "
                 ^print_list print_OP_SYMB1 "; " opl
                 ^" }"
  | print_FORMULA (unparsed_formula f) = f
  | print_FORMULA (pos_FORMULA (_,true,f)) = "("^print_FORMULA f^")"
  | print_FORMULA (pos_FORMULA (_,false,f)) = print_FORMULA f

fun print_TERMList tl = print_list print_TERM "," tl

fun print_FORMULAs philist =
	print_list print_FORMULA "\n" philist

fun print_L_FORMULA (phi, annos) =
	print_FORMULA  phi ^"  " ^ print_ANNOS annos 

fun print_L_FORMULAList philist =
	print_list print_L_FORMULA "\n" philist


fun fill_in nil _ s = s
  | fill_in (token t::resttoks) ts s = fill_in resttoks ts (s^t)
  | fill_in (place::resttoks) nil s = s
  | fill_in (place::resttoks) (t::ts) s = fill_in resttoks ts (s^" "^t^" ")

fun is_underscore x = x = "_"

fun mk_torplist nil = nil
  | mk_torplist ("_" :: "_" :: rest) = place::mk_torplist rest
  | mk_torplist (l as _::_) = 
    let fun get_first_token l =
        let val (pre,rest) = take_prefix (not o is_underscore) l
        in case rest of
            nil => (pre,rest)
          | "_"::"_"::rest1 => (pre,rest)
          | "_"::rest1 => 
                  let val (pre2,rest2) = get_first_token rest1
                  in (pre@["_"]@pre2,rest2)
                  end
	  | _ => raise ERROR
        end
        val (suff,rest) = get_first_token l
    in
      token (implode suff)::mk_torplist rest
    end

  
fun infix_TERM ((var_or_const x):TERM):string = print_vars [x]
  | infix_TERM (qual_var (x,s )) = print_vars [x]
  | infix_TERM (application (opsy,t)) =
    let val ts = get_terms t
        val f = get_op_name opsy
        val ((_,s,_),c) = case f of
                (simple_id sid) => (sid,[])
              | (compound_id (sid,components)) => (sid,components)
        val torplist = mk_torplist (explode s)
    in
	if s = "inj$" then infix_TERMS (terms ts)
	else
	if place mem torplist then
	(fill_in torplist (map infix_TERM ts) "")
	else
	( print_ID f^
	  (if ts=nil then "" else "(")
	   ^infix_TERMS (terms ts)^
  	  (if ts=nil then "" else ")")
  	)
     end
  | infix_TERM (sorted_term (t,s)) =
	infix_TERM t^" : "^(print_SORTS1 [s])
  | infix_TERM (conditional (t,f,t1)) =
	infix_TERM t^" when "^(infix_FORMULA f) ^" else "^infix_TERM t1
  | infix_TERM (cast (t,s)) = 
	infix_TERM t^" as "^(print_SORTS1 [s])
  | infix_TERM (unparsed_term t) = t
  | infix_TERM (pos_TERM (_,true,t)) = "("^infix_TERM t^")"
  | infix_TERM (pos_TERM (_,false,t)) = infix_TERM t

and infix_TERMS ts = print_list infix_TERM "," (get_terms ts)


and infix_PREDICATION (pred_symb (p,_),ts) =
    let val ((_,s,_),c) = case p of
                (simple_id sid) => (sid,[])
              | (compound_id (sid,components)) => (sid,components)
        val torplist = mk_torplist (explode s)
        val tl = get_terms ts
    in
	if place mem torplist then
	(fill_in torplist (map infix_TERM tl) "")
	else
	( print_ID p^
	  (if tl=nil then "" else "(")
	   ^infix_TERMS (terms tl)^
  	  (if tl=nil then "" else ")")
  	)
    end
  | infix_PREDICATION (pos_PRED_SYMB(_,p),ts) =
    infix_PREDICATION (p,ts)

and infix_ATOM ttrue = "true"
  | infix_ATOM ffalse = "false"
  | infix_ATOM (predication p) = infix_PREDICATION p
  | infix_ATOM (definedness t) = "def "^infix_TERM t
  | infix_ATOM (existl_equation (t1,t2)) =
	infix_TERM t1^" =e= "^infix_TERM t2
  | infix_ATOM (strong_equation (t1,t2)) =
	infix_TERM t1^" = "^infix_TERM t2
  | infix_ATOM (membership (t,s)) =
	infix_TERM t^" in "^print_SORTS1 [s]

and add_brackets outer_prio inner_prio s =
    let val brackets = outer_prio>=inner_prio
    in
    (if brackets then "(" else "")
    ^
    s
    ^
    (if brackets then ")" else "")
    end
    
and infix_FORMULA (quantification (q,vdlist,phi)) =
    print_QUANTIFIER q^" "
    ^(print_VAR_DECLList vdlist)^" . "
    ^infix_FORMULA phi
  | infix_FORMULA (conjunction phis) =
    print_list infix_FORMULA " /\\ " phis
  | infix_FORMULA (disjunction phis) =
    print_list infix_FORMULA " \\/ " phis
  | infix_FORMULA (implication (phi1,phi2)) =
    infix_FORMULA phi1^" => "^infix_FORMULA phi2
  | infix_FORMULA (equivalence (phi1,phi2)) =
    infix_FORMULA phi1^" <=> "^infix_FORMULA phi2
  | infix_FORMULA (negation phi) =
    "not "^infix_FORMULA phi
  | infix_FORMULA (atom a) = infix_ATOM a
  | infix_FORMULA (sort_gen_ax (sl,opl)) = 
     "generated\n  { sorts "^print_list print_ID "; " sl^"  ops "
                 ^print_list print_OP_SYMB1 "; " opl
                 ^" }"
  | infix_FORMULA (sort_cogen_ax (sl,opl)) = 
     "cogenerated\n  { sorts "^print_list print_ID "; " sl^"  ops "
                 ^print_list print_OP_SYMB1 "; " opl
                 ^" }"
  | infix_FORMULA (sort_cofree_ax (sl,opl)) = 
     "cofree %{unstructured}% \n  { sorts "^print_list print_ID "; " sl^"  ops "
                 ^print_list print_OP_SYMB1 "; " opl
                 ^" }"
  | infix_FORMULA (unparsed_formula f) = f
  | infix_FORMULA (pos_FORMULA (_,true,f)) = "("^infix_FORMULA f^")"
  | infix_FORMULA (pos_FORMULA (_,false,f)) = infix_FORMULA f
  | infix_FORMULA (var_FORMULA v) = print_var_id v
  


fun infix_TERMList tl = print_list infix_TERM "," tl

fun infix_FORMULAs philist =
	print_list infix_FORMULA "\n" philist

fun strip_quant (quantification (forall,vdlist,phi)) =
    (vdlist,phi)
  | strip_quant (pos_FORMULA (_,_,phi)) =
    strip_quant phi
  | strip_quant phi = ([],phi)
  
fun infix_L_FORMULA (phi as (sort_gen_ax _), annos) =
         infix_FORMULA phi
  | infix_L_FORMULA (phi as (sort_cogen_ax _), annos) =
         infix_FORMULA phi
  | infix_L_FORMULA (phi as (sort_cofree_ax _), annos) =
         infix_FORMULA phi
  | infix_L_FORMULA (pos_FORMULA (_,_,phi),annos) =
    infix_L_FORMULA (phi,annos)
  | infix_L_FORMULA (phi,annos) =
    "  . "^  infix_FORMULA phi ^ "  " ^  print_ANNOS annos
(*    let val (vdlist,psi) = strip_quant phi
    in
      if null vdlist 
      then "  . "^  infix_FORMULA psi ^ "  " ^  print_ANNOS annos 
      else "forall "^print_VAR_DECLList vdlist^"\n"
           ^"  . "^infix_FORMULA psi ^"  " ^  print_ANNOS annos 
    end
*)
(*
fun split (pred: 'a->bool) : 'a list -> ('a list * 'a list) =
  let fun spl [] = ([],[])
        | spl (x :: xs) = 
          let val (l1,l2) = spl xs
          in if pred x then (x :: l1,l2) else (l1, x :: l2)
          end
  in spl end;

fun is_gen (sort_gen_ax _,_) = true
  | is_gen (sort_cogen_ax _,_) = true
  | is_gen (sort_cofree_ax _,_) = true
  | is_gen (pos_FORMULA (_,_,phi),_) =
    is_gen (phi,[])
  | is_gen _ = false
  
fun infix_L_FORMULAList philist =
	let val (genax,ax) = split is_gen philist
	in
	(if null ax then "" 
	 else "axioms\n  "^print_list infix_L_FORMULA "\n" ax)
	^
	print_list (infix_FORMULA 0 o fst) "\n  " genax
	end
*)

fun infix_L_FORMULAList philist = print_list infix_L_FORMULA "\n" philist

fun print_sort_entry (s,subs) =
    let val newsubs = remove s subs
    in
      (if null newsubs then ""
         else (print_list print_ID "," newsubs ^" < ")
      )^ print_ID s
    end
    
fun print_fun_entry (f,profiles) =
    print_list (fn prof => print_ID f^" : "^print_OP_TYPE prof) ";\n  " profiles

fun print_pred_entry (p,profiles) =
    print_list (fn prof => print_ID p^" : "^print_PRED_TYPE prof) ";\n  " profiles

fun print_local_list (srts,subsorts,vars,ops,preds) =
    (if length subsorts = 1 then "sort\n" else"sorts\n")^
    indent (print_list print_sort_entry ";\n" (subsorts)^"\n")^
    (if null ops then "" else if length ops = 1 then "op\n" else "ops\n")^
    indent (print_list print_fun_entry ";\n" (ops)^"\n")^
    (if null preds then "" else if length preds = 1 then "pred\n" else "preds\n")^
    indent (print_list print_pred_entry ";\n" (preds)^"\n")

fun output_local_env h (subsorts,vars,ops,preds) =
    let val subsortlist = Symtab_id.dest subsorts
        val oplist = Symtab_id.dest ops
        val predlist = Symtab_id.dest preds
    in 
    (
      if length subsortlist = 1 then output(h,"sort\n") else output(h,"sorts\n");
      do_print_list h (fn h => fn x => output(h,"  "^print_sort_entry x)) ";\n" subsortlist;
      if null oplist then () else 
         if length oplist = 1 then output(h,"\nop\n") else output(h,"\nops\n");
      do_print_list h (fn h => fn x => output(h,"  "^print_fun_entry x)) ";\n" oplist;
      if null predlist then () else 
         if length predlist = 1 then output(h,"\npred\n") else output(h,"\npreds\n");
      do_print_list h (fn h => fn x => output(h,"  "^print_pred_entry x)) ";\n" predlist;
     output(h,"\n")
    )
    end

fun output_sign h (lenv) =
    output_local_env h lenv

fun print_local_env (subsorts,vars,ops,preds) =
    let val subsortlist = Symtab_id.dest subsorts
        val oplist = Symtab_id.dest ops
        val predlist = Symtab_id.dest preds
    in 
    (
      (if length subsortlist = 1 then "sort\n" else "sorts\n")
      ^print_list (fn x => "  "^print_sort_entry x) ";\n" subsortlist
      ^(if null oplist then "" else 
         if length oplist = 1 then "\nop\n" else "\nops\n")
      ^print_list (fn x => "  "^print_fun_entry x) ";\n" oplist
      ^(if null predlist then "" else 
         if length predlist = 1 then "\npred\n" else "\npreds\n")
      ^print_list (fn x => "  "^print_pred_entry x) ";\n" predlist
      ^"\n"
    )
    end

fun print_sign lenv =
    print_local_env lenv


fun latex_ID id = escape_latexs (print_ID id)
fun latex_SIMPLE_ID id = escape_latexs (print_SIMPLE_ID id)

fun latex_SORTS1 nil = ""
  | latex_SORTS1 ([s]) = latex_ID s
  | latex_SORTS1 (s::xs) = latex_ID s^" \\times "^latex_SORTS1 xs
 
fun latex_SORTS sl = latex_SORTS1 (get_sorts sl)


fun latex_sort_entry h (s,subs) =
    output(h,"\\("^latex_ID s^"\\) & \\(\\geq\\) & \\( "^print_list latex_ID "," subs
    ^"\\) \\\\")


fun latex_OP_TYPE (total_op_type (sorts [], res)) =
	latex_SORTS1 [res]
  | latex_OP_TYPE (total_op_type (arg, res)) =
	(print_SORTS arg)^"\\tfun "^(print_SORTS1 [res])
  | latex_OP_TYPE (partial_op_type (sorts [], res)) =
	"?"^(latex_SORTS1 [res])
  | latex_OP_TYPE (partial_op_type (arg, res)) =
	(print_SORTS arg)^"\\pfun "^(print_SORTS1 [res])
  | latex_OP_TYPE (pos_OP_TYPE (_,opt)) =
    latex_OP_TYPE opt

fun latex_one_fun f h prof =
    output(h,"\\("^latex_ID f^"\\)&:&\\("^latex_OP_TYPE prof^"\\) \\\\")
fun latex_fun_entry h (f,profiles) =
    do_print_list h (latex_one_fun f) "\n" profiles

fun latex_PRED_TYPE (pred_type t) =
	latex_SORTS t
  | latex_PRED_TYPE (pos_PRED_TYPE (_,t)) =
    latex_PRED_TYPE t
 
fun latex_one_fun p h prof =
    output(h,"\\("^latex_ID p^"\\)&:&\\("^latex_PRED_TYPE prof^"\\) \\\\")
fun latex_pred_entry h (p,profiles) =
    do_print_list h (latex_one_fun p) "\n" profiles


fun latex_local_env h (subsorts,vars,ops,preds) =
   (output(h,"{\\bf sorts}\n\n\\begin{tabular}{lcl}\n");
    do_print_list h latex_sort_entry "\n" (Symtab_id.dest subsorts);
    output(h,"\n\\end{tabular}\n");
    output(h,"\n\n{\\bf ops}\n\n\\begin{tabular}{lcl}\n");
    do_print_list h latex_fun_entry "\n" (Symtab_id.dest ops);
    output(h,"\n\\end{tabular}\n");
    output(h,"\n\n{\\bf preds}\n\n\\begin{tabular}{lcl}\n");
    do_print_list h latex_pred_entry "\n" (Symtab_id.dest preds);
    output(h,"\n\\end{tabular}\n")
   )

fun latex_sign h (lenv) =
    latex_local_env h lenv

end
