(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author: Till Mossakowski, University of Bremen			   *)
(* Date: 1998				 			           *)
(* Purpose of this file: basic utilities                                   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* Basic utilities, based on Isabelle's utilities in src/Pure/library.ML
*)

structure Utils = struct

open Library;


(* file functions *)

fun read(fullpath)= 
    let fun read_me si so_far = if TextIO.endOfStream si then 
                       so_far before (TextIO.closeIn si)
      else read_me si (so_far ^TextIO.inputLine si)
    in  read_me(TextIO.openIn fullpath) ""
    end

fun del_cr c = if ord(c)=13 then None else Some c

val del_crs = implode o (mapfilter del_cr) o explode

fun read_without_cr(fullpath)= 
    let fun read_me si so_far = if TextIO.endOfStream si then 
                       so_far before (TextIO.closeIn si)
      else read_me si (so_far ^ (del_crs (TextIO.inputLine si)))
    in  read_me(TextIO.openIn fullpath) ""
    end

fun write(fullpath) s = 
    let val str = TextIO.openOut fullpath
    in
    (TextIO.output (str,s);
     TextIO.closeOut str)
    end
       
fun exists file =
    let val str = TextIO.openIn file
    in
    (TextIO.closeIn str; true)
    end
    handle _ => false


(** Diagnostic output *)

fun out s = (writeln ("Writing "^s);s)

(** Error output **)

val null_region = ((0,0),(0,0))

fun get_pos_FORMULA (AS.pos_FORMULA (r,_,_)) = r
  | get_pos_FORMULA _ = null_region

fun get_brack_FORMULA (AS.pos_FORMULA (_,b,_)) = b
  | get_brack_FORMULA _ = false

local
fun pr_pos (0,0) = ""
  | pr_pos (row,col) = Int.toString row^"."^Int.toString col
in
fun perr ((0,0),(0,0)) = "*** "^(!Global.cur_file)^": error: "
  | perr (pos,(0,0)) = 
    "*** "^(!Global.cur_file)^":"^pr_pos pos^", error: "
  | perr (pos1,pos2) =
    "*** "^(!Global.cur_file)^":"^pr_pos pos1^"-"^pr_pos pos2^", error: "
end


fun remove_pos_ANNO (AS.pos_ANNO (_,a)) = remove_pos_ANNO a
  | remove_pos_ANNO a = a    

fun remove_pos_SORTS (AS.pos_SORTS (_,s)) =
    remove_pos_SORTS s
  | remove_pos_SORTS x = x

fun remove_pos_OP_TYPE (AS.pos_OP_TYPE (_,t)) =
    remove_pos_OP_TYPE t
  | remove_pos_OP_TYPE (AS.total_op_type (s,t)) =
    AS.total_op_type(remove_pos_SORTS s,t)
  | remove_pos_OP_TYPE (AS.partial_op_type (s,t)) =
    AS.partial_op_type(remove_pos_SORTS s,t)

fun remove_pos_PRED_TYPE (AS.pos_PRED_TYPE (_,t)) =
    remove_pos_PRED_TYPE t
  | remove_pos_PRED_TYPE (AS.pred_type s) =
    AS.pred_type(remove_pos_SORTS s)


fun remove_pos_TYPE (AS.pos_TYPE (_,t)) =
    remove_pos_TYPE t
  | remove_pos_TYPE (AS.op_symb_type t) =
    AS.op_symb_type (remove_pos_OP_TYPE t)
  | remove_pos_TYPE (AS.pred_symb_type t) =
    AS.pred_symb_type (remove_pos_PRED_TYPE t)


fun remove_pos_SYMB (AS.pos_SYMB (_,sy)) = remove_pos_SYMB sy
  | remove_pos_SYMB (AS.qual_id (id,t)) = AS.qual_id (id,remove_pos_TYPE t)
  | remove_pos_SYMB sy = sy


(** These functions from library.ML are hidden by as.sml ***)

(*exists pred [x1, ..., xn] ===> pred x1 orelse ... orelse pred xn*)
fun exists' (pred: 'a -> bool) : 'a list -> bool =
  let fun boolf [] = false
        | boolf (x :: xs) = pred x orelse boolf xs
  in boolf end;

(*forall pred [x1, ..., xn] ===> pred x1 andalso ... andalso pred xn*)
fun forall' (pred: 'a -> bool) : 'a list -> bool =
  let fun boolf [] = true
        | boolf (x :: xs) = pred x andalso boolf xs
  in boolf end;



exception ZIP_ERROR

fun member (x,y::ys) = x=y orelse member (x,ys)
  | member (x,nil) = false

fun curried_member x y = member(y,x)

fun last (x::nil) = x
  | last (x::xs) = last xs
  | last nil = raise (ERR "last of empty list")

fun append (nil,ys) = ys
  | append (x::xs,ys) = x::append (xs,ys)

fun appendlists (x:'a list list):'a list = foldr append (x,nil)

fun logand (false,x) = false
  | logand (true,x) = x
  
fun forall1 x = foldr logand (x,true)

fun zip (nil,nil) = nil
  | zip ((x::xl),(y::yl)) = (x,y)::zip (xl,yl)
  | zip (_,_) = raise ZIP_ERROR


fun remove x nil = nil
  | remove x (y::ys) =
    if x=y then remove x ys
           else y::remove x ys

local
fun remove_dups1 (nil,l) = l
  | remove_dups1 (x::xs,l) = 
  	if member (x,l) then remove_dups1 (xs,l)
  	else remove_dups1 (xs, x::l)
in
fun remove_dups l = remove_dups1 (l,nil)
end


fun generalized_member eq (x,nil ) = false
  | generalized_member eq (x,y::rest) =
     eq(x,y) orelse generalized_member eq (x,rest)

local
fun remove_dups_eq1 member (nil,l) = l
  | remove_dups_eq1 member (x::xs,l) = 
  	if member (x,l) then remove_dups_eq1 member (xs,l)
  	else remove_dups_eq1 member (xs, x::l)
in
fun remove_dups_eq eq l = 
    remove_dups_eq1 (generalized_member eq) (l,nil)
end



(* Transform a list [l1,l2, ... ln] to
   (in sloppy Haskell notation) [[x1,x2, ... ,xn] | x1<-l1, x2<-l2, ... xn<-ln] *)

fun singleton x = [x]

fun permute nil = [nil]
  | permute [x] = map  singleton x
  | permute ((x::l):'a list list): 'a list list  = 
	let
                fun cons x y = x::y
		fun distribute (perms: 'a list list) (y: 'a) = map (cons y) perms
	in   appendlists (map (distribute (permute l)) x)
	end



fun nefoldl f (x::xs) = foldl f (x,xs)
  | nefoldl f nil = raise (ERR "nefoldl applied to empty list") 

(**************************************************************************************)
(*		Functions for comparing types/profiles 				      *)
(**************************************************************************************)

fun get_sort (t,s) = s

fun get_sorts (AS.sorts ts) = ts
  | get_sorts (AS.pos_SORTS (_,ts)) = get_sorts ts

fun get_terms (AS.terms ts) = ts
  | get_terms (AS.pos_TERMS (_,ts)) = get_terms ts

fun get_region_TERMS (AS.pos_TERMS (r,_)) = Some r
  | get_region_TERMS _ = None

fun get_op_name (AS.op_symb (n,_)) = n
  | get_op_name (AS.pos_OP_SYMB (_,sy)) =
    get_op_name sy

fun get_op_type (AS.op_symb (_,t)) = t
  | get_op_type (AS.pos_OP_SYMB (_,sy)) =
    get_op_type sy

fun get_pred_name (AS.pred_symb (n,_)) = n
  | get_pred_name (AS.pos_PRED_SYMB (_,sy)) =
    get_pred_name sy

fun get_pred_type_PRED_SYMB (AS.pred_symb (_,t)) = t
  | get_pred_type_PRED_SYMB (AS.pos_PRED_SYMB (_,sy)) =
    get_pred_type_PRED_SYMB sy

fun get_region_PRED_SYMB (AS.pos_PRED_SYMB (r,_)) = Some r
  | get_region_PRED_SYMB _ = None

fun get_pred_type (AS.pred_type t) = t
  | get_pred_type (AS.pos_PRED_TYPE (_,t)) =
    get_pred_type t

fun is_implicit (AS.implicitk) = true
  | is_implicit (AS.pos_SYMB_KIND (_,k)) =
    is_implicit k
  | is_implicit _ = false

fun eq_kind (AS.pos_SYMB_KIND (_,k1)) k2 =
    eq_kind k1 k2
  | eq_kind k1 (AS.pos_SYMB_KIND (_,k2)) =
    eq_kind k1 k2
  | eq_kind k1 k2 = k1=k2

(* Get argument sorts and result sort of a function type *)

fun get_args (AS.total_op_type (args,res)) = get_sorts args
  | get_args (AS.partial_op_type (args,res)) = get_sorts args
  | get_args (AS.pos_OP_TYPE (r,opt)) = get_args opt
fun get_res (AS.total_op_type (args,res)) = res
  | get_res (AS.partial_op_type (args,res)) = res
  | get_res (AS.pos_OP_TYPE (r,opt)) = get_res opt

fun get_args_pred (AS.pred_type args) = get_sorts args
  | get_args_pred (AS.pos_PRED_TYPE (r,opt)) = get_args_pred opt

fun get_genericity (AS.pos_GENERICITY (_,gen)) =
    get_genericity gen
  | get_genericity (AS.genericity (AS.pos_PARAMS (_,p),i)) =
    get_genericity (AS.genericity (p,i))
  | get_genericity (AS.genericity (p,AS.pos_IMPORTS (_,i))) =
    get_genericity (AS.genericity (p,i))
  | get_genericity (AS.genericity (AS.params p,AS.imports i)) =
    (p,i)
 
fun get_VIEW_TYPE (AS.pos_VIEW_TYPE(_,t)) =
    get_VIEW_TYPE t
  | get_VIEW_TYPE (AS.view_type(s,t)) = (s,t)

 (* **************************************
 Shouldn't it be also a version of the same function for
 OP-HEAD and all the like *************** *)


(**********************************************************************)
(*		Reflexive and transitive closure		      *)
(**********************************************************************)

(* An (ordering) relation is given by a list of pairs of form (elem,leq_elems),
   where leq_elems is the list of elements less than
   or equal to elem *)

(* Less than predicate *)

fun leq rel x y = 
     case assoc (rel,y) of
       Some less_than_y => x mem less_than_y
       | None => false
       
fun less rel x y = leq rel x y andalso not (x=y)

(* Find a minimal element in a relation *)

fun is_min (eq : 'a * 'a -> bool) ((x,[y]):'a * 'a list) = eq (x,y)
  | is_min eq (x,nil) = true
  | is_min eq (_,y::_) = false

fun find_min_in_rel (eq : 'a * 'a -> bool)
                    (rel : (('a * 'a list) list) ) : ('a * 'a list) option = 
    find_first (is_min eq) rel

(* Remove one element form a relation *)

local
fun remove_from_rel1 x (y,leqs) = 
    if x=y then None
    else Some (y,remove x leqs)
in
fun remove_from_rel x rel = mapfilter (remove_from_rel1 x) rel
end

(* Generate the identity relation for a given set *)

local
fun identity1 x = (x,[x])
in
fun identity elems = map identity1 elems
end

(* Generate the reflexive closure of a relation for a given set *)

local
fun reflexive_closure1 (x,leqs) =
    if x mem leqs then (x,leqs) else (x,x::leqs)
in
fun reflexive_closure rel = map reflexive_closure1 rel
end

local
fun binand (x,y) = x andalso y;
fun binor (x,y) = x orelse y;

fun conszip nil nil = nil
|   conszip (x::xs) (y::ys) = (x::y)::(conszip xs ys)
|   conszip _ _ = raise (ERR "conszip of list of different lengths")
fun mk_list x = [x]

fun transpose nil = nil
|   transpose [x] = map mk_list x
|   transpose (x::xs) = conszip x (transpose xs)

fun vec_mult v1 v2 =
	foldl binor (false,map binand (zip(v1,v2)));

fun one_vec m v =
map (vec_mult v) m;

fun multiply (m1:bool list list) (m2:bool list list):bool list list =
map (one_vec (transpose m1)) m2 ; 

fun iterate (m:bool list list):bool list list =
let val m1 = multiply m m
in if m = m1 then m
   else iterate m1
end;

fun make_matrix eq (ordering) =
let val elems = map fst ordering
    fun member1 [] s = false
      | member1 (x::rest) s = eq(x,s) orelse member1 rest s
    fun make_col (s,elems') = map (member1 elems') elems
in map make_col ordering
end 	

in

(* Compute the transitive closure of a relation *)
		 
fun transitive_closure_old eq (relation:(''a * ''a list) list) : (''a * ''a list) list =
let val elems = map fst relation
    fun get_entry (s,true) = [s]
    |   get_entry (s,false) = []
    fun make_entry (s,row) = (s,flat(map get_entry (zip(elems,row)))) 
in map make_entry (zip (elems,iterate (make_matrix eq relation)))
end


fun get_lessthan eq node (n1,n2) =
    if eq (n1,node) andalso not (eq(n2,node))
    then Some n2
    else None
    
fun get_lessthans eq edges node =
    (node,node::mapfilter (get_lessthan eq node) edges)
    
fun make_relation eq nodes edges =
    map (get_lessthans eq edges) nodes
        
fun get_edges1 (node,lessthans) =
    map (pair node) (remove node lessthans)

fun get_edges relation =
    flat (map get_edges1 relation)

fun try_to_add eq (n1,n2) (n3,n4) =
    if eq (n2,n3) then Some (n1,n4)
    else if eq(n1,n4) then Some (n3,n2)
    else None
    
fun iterate_edges1 eq nil added_edges =
    added_edges
  | iterate_edges1 eq (e::rest) added_edges =
    iterate_edges1 eq rest (mapfilter (try_to_add eq e) rest @ added_edges) 
    
fun iterate_edges eq edges =
    let val edges' = remove_dups (iterate_edges1 eq edges edges)
    in if length edges= length edges' then edges'
        (* since iterate_edges1 keeps all old edges,
           it suffices to compare the lengths of the lists *)
       else iterate_edges eq edges'
    end
    
fun transitive_closure eq (relation:(''a * ''a list) list) : (''a * ''a list) list =
    let val edges = get_edges relation
        val edges' = iterate_edges eq edges
        val nodes = map fst relation
    in make_relation eq nodes edges'
    end

end

end

