(* *********************************************************************** *)
(*									   *)
(* Project: CASL	  						   *)
(* Author: Till Mossakowski, University of Bremen	 		   *)
(* Date: 28 04 2000 				 			   *)
(* Purpose of this file: Environment for CASL 				   *)
(*									   *)
(* *********************************************************************** *)

(* Data structure for global environment, representing CASL structured
   specifications. This is based on local_env.sml, providing data
   structures for the local environment (signature).
   The structure of the global environment follows the notion
   of global environment in the CASL semantics (study note S-9).
   However, there is a difference: we do not (only) flatten the specifications
   into signatures, but rather keep the structure of the
   specification (while transforming it into a more institution-like
   form, i.e. symbol maps are replaced by the induced signature morphisms).
   The flattened local environment is kept at the outermost level
   of the global environment, but not at the inner nodes (except
   for nodes for basic specifications, of course), since
   this would consume too much space.

   A library environment is a table of global environments,
   one for each library. This table is kept during static analysis,
   in order to avoid re-reading of libraries.

   todo:
   This file should be made institution independent
   by turning the structure into a functor.
   
   Annotationsformat fuer Globales Environment
   - Konstruktor- und Selektorinfos als Annotationen
   - Sharing von Namen ber de Brujin-Indizes
   - zustzlich erzeugte Signaturelemente und Formeln mit Ursprung kennzeichnen
*)

structure GlobalEnv 		:
sig  

type ext_signature = (LocalEnv.sign * LocalEnv.sign * AS.L_FORMULA list)
(* intended use: (flattened env, env of hidden symbols, formulas) *)

val empty_ext_signature : ext_signature

val ext_signature_union : ext_signature * ext_signature -> ext_signature

type Sort_map = AS.SORT Symtab_id.table
type Fun_map = (AS.OP_TYPE * AS.OP_NAME * bool) list Symtab_id.table
			(* The third field is true iff the target symbol is
                           total *)
type Pred_map = (AS.PRED_TYPE * AS.PRED_NAME) list Symtab_id.table


type morphism	  = (Sort_map * Fun_map * Pred_map)

val eq_mor : morphism * morphism -> bool

datatype spec_env = 	  basic_env of (LocalEnv.sign * AS.L_FORMULA list)
			| translate_env of (spec_env * morphism)
			| derive_env  of (spec_env * morphism)
			| union_env   of spec_env list
			| extension_env of (spec_env * AS.ANNO list) list
			| free_spec_env of spec_env
			| cofree_spec_env of spec_env
			| closed_spec_env of spec_env
			| spec_inst_env of (AS.SPEC_NAME * spec_env * morphism * spec_env list)
			  (* intended use: (name of gen spec, env for body, fitting morphism, env for actual args) *)
                        | dummy_spec_env

and   spec_lenv = 	SPEC_ENV of  (LocalEnv.sign * LocalEnv.sign * spec_env)
                          (* intended use: flattened env, (flattened) hidden env, structured env *)
                          
val empty_spec_lenv: spec_lenv
val ENV        : spec_lenv ref

val empty_morphism : morphism

type genericity_env     = (spec_lenv * spec_lenv list * LocalEnv.sign)
(* intended use: 
   (union of list of envs for imports, 
    list of envs for formal parameters,
    signature union of all imports and all formal parameters) *)

(* Semantics July 99, p. 141 *)
type arch_genv_entry = ArchTypes.arch_sig;
(* p. 142 *)
type unit_genv_entry = ArchTypes.unit_sig;

datatype global_entry =   
    spec_defn_env of (genericity_env * spec_lenv)
  | view_defn_env of (genericity_env * spec_lenv * morphism * spec_lenv)
  | arch_spec_defn_env of arch_genv_entry
  | unit_spec_defn_env of unit_genv_entry
;
			
type global_env =	global_entry Symtab_sid.table * AS.ANNO list
val empty_global_env: global_env;

type f_genericity_env     = (ext_signature * ext_signature list)

datatype f_global_entry =   f_spec_defn_env of (f_genericity_env * ext_signature)
			| f_view_defn_env of (f_genericity_env * ext_signature * morphism * ext_signature)
			| f_arch_spec_defn_env
			| f_unit_spec_defn_env
			
type f_global_env =	f_global_entry Symtab_sid.table * AS.ANNO list
val f_empty_global_env: f_global_env;

val get_ids : global_env -> AS.SPEC_NAME -> AS.ID list
val get_annos : global_env ->  AS.ANNO list
val get_spec_lenv : global_env -> AS.SPEC_NAME -> spec_lenv
val get_ids_spec_lenv : spec_lenv -> AS.ID list

val isDummyLenv : spec_lenv -> bool


type lib_env = (global_env * AS.LIB_DEFN) Symtab_str.table

val empty_lib_env : lib_env

end = 
struct


open Utils AS LocalEnv BasicAnalysis ArchTypes;







type ext_signature = (sign * sign * L_FORMULA list)
val empty_ext_signature = (empty_signature,empty_signature,[])

fun ext_signature_union ((s1,h1,ax1),(s2,h2,ax2)) =
    (signature_union (s1,s2),signature_union (h1,h2),ax1@ax2)

type Sort_map = SORT Symtab_id.table
type Fun_map = (OP_TYPE * OP_NAME * bool) list Symtab_id.table
type Pred_map = (PRED_TYPE * PRED_NAME) list Symtab_id.table

type morphism	  = (Sort_map *Fun_map*Pred_map)

fun eq_tab (t1,t2) = Symtab_id.dest t1 = Symtab_id.dest t2
fun eq_mor ((s1,f1,p1),(s2,f2,p2)) = eq_tab (s1,s2) andalso eq_tab (f1,f2) andalso eq_tab (p1,p2) 

datatype spec_env = 	basic_env of (sign * AS.L_FORMULA list)
			| translate_env of (spec_env * morphism)
			| derive_env  of (spec_env * morphism)
			| union_env   of spec_env list
			| extension_env of (spec_env * AS.ANNO list) list
			| free_spec_env of spec_env
			| cofree_spec_env of spec_env
			| closed_spec_env of spec_env
			| spec_inst_env of (AS.SPEC_NAME * spec_env * morphism * spec_env list)
			  (* intended use: (name of gen spec, env for body, fitting morphism, env for actual args) *)
                        | dummy_spec_env

and   spec_lenv = 	SPEC_ENV of  (sign * sign * spec_env)
val   empty_spec_lenv = SPEC_ENV  (empty_signature, empty_signature, dummy_spec_env)
val ENV = ref empty_spec_lenv;

type genericity_env     = (spec_lenv * spec_lenv list * LocalEnv.sign)
(* intended use: 
   (list of envs for imports, 
    list of envs for formal parameters,
    signature union of all inports and all formal parameters) *)

(* Semantics July 99, p. 141 *)
type arch_genv_entry = arch_sig;
(* p. 142 *)
type unit_genv_entry = unit_sig;

datatype global_entry =   
    spec_defn_env of (genericity_env * spec_lenv)
  | view_defn_env of (genericity_env * spec_lenv * morphism * spec_lenv)
  | arch_spec_defn_env of arch_genv_entry
  | unit_spec_defn_env of unit_genv_entry
;
		
type global_env =	global_entry Symtab_sid.table * AS.ANNO list;

	 
val empty_global_env = (Symtab_sid.empty,[])

val empty_morphism = (Symtab_id.empty,Symtab_id.empty,Symtab_id.empty)

type f_genericity_env     = (ext_signature * ext_signature list)

datatype f_global_entry =   f_spec_defn_env of (f_genericity_env * ext_signature)
			| f_view_defn_env of (f_genericity_env * ext_signature * morphism * ext_signature)
			| f_arch_spec_defn_env
			| f_unit_spec_defn_env
			
type f_global_env =	f_global_entry Symtab_sid.table * AS.ANNO list

val f_empty_global_env = (Symtab_sid.empty,[])

fun get_ids (genv,_) sp_name =
    case Symtab_sid.lookup (genv,sp_name) of
       (Some (spec_defn_env(_,SPEC_ENV(Sigma,HSigma,_)))) => 
                LocalEnv.get_ids Sigma @ LocalEnv.get_ids HSigma
     | (Some (view_defn_env(_,SPEC_ENV(Sigma1,HSigma1,_),_,
                              SPEC_ENV(Sigma2,HSigma2,_))))
            => LocalEnv.get_ids Sigma1 @ LocalEnv.get_ids Sigma2
               @  LocalEnv.get_ids HSigma1 @ LocalEnv.get_ids HSigma2
     | _ => raise ERROR


fun get_ids_spec_lenv (SPEC_ENV(Sigma,_,_)) = 
    LocalEnv.get_ids Sigma

fun get_annos (genv,annos) = annos

fun get_spec_lenv (genv,_) sp_name =
    case Symtab_sid.lookup (genv,sp_name) of
       (Some (spec_defn_env(_,speclenv))) => speclenv
     | _ => raise ERROR

fun isDummyLenv (SPEC_ENV (_,_,dummy_spec_env)) = true
  | isDummyLenv _ = false


fun is_number_an (number_anno _) = true
         | is_number_an _ = false
fun get_number_an an = find_first is_number_an an
       
fun is_string_an (string_anno _) = true
         | is_string_an _ = false
fun get_string_an an = find_first is_string_an an

fun is_float_an (floating_anno _) = true
         | is_float_an _ = false
fun get_float_an an = find_first is_float_an an

fun is_list_an (list_anno _) = true
         | is_list_an _ = false
fun get_list_an an = find_first is_list_an an  
     
type lib_env = (global_env * LIB_DEFN) Symtab_str.table

val empty_lib_env = Symtab_str.empty
       
end


