(* *********************************************************************** *)
(*									   *)
(* Project: HOL-CASL 							   *)
(* Author: Till Mossakowski, University of Bremen		           *)
(* Date: 13.09.2001				 			   *)
(* Purpose of this file: Adapt Isabelle tactics for HOL-CASL               *)
(*									   *)
(* *********************************************************************** *)

(* This module adapts tactic like res_inst_tac or induct_tac
   for HOL-CASL.

   todo:
   case_tac: freie Variablen in Formel besser behandeln
    (momentan wird bei Vorhandesein nicht-uqivalenter Expansionen
     einfach eine mit minimaler Injektionen-Anzahl rausgesucht)
   + fr Datentypen wie in Isabelle

   Injektivitt mglichst nicht als quivalenz, sondern
     letztere beweisen (Problem: flex-flex-Paare)
*)
signature CASL_TACTIC =
sig
val read_instantiate_sg: Sign.sg -> (string*string)list -> thm -> thm
val read_instantiate  : (string*string)list -> thm -> thm
val instantiate_tac : (string * string) list -> tactic
val res_inst_tac
  : (string * string) list -> thm -> int -> thm -> thm Seq.seq   
val compose_inst_tac  : (string*string)list -> (bool*thm*int) ->
                          int -> tactic
val cut_inst_tac      : (string*string)list -> thm -> int -> tactic
val dres_inst_tac     : (string*string)list -> thm -> int -> tactic
val eres_inst_tac     : (string*string)list -> thm -> int -> tactic
val forw_inst_tac     : (string*string)list -> thm -> int -> tactic
val lift_inst_rule    : thm * int * (string*string)list * thm -> thm

val case_tac : string -> int -> tactic 
val data_case_tac : string -> int -> tactic 
val induct_tac : string -> int -> thm -> thm Seq.seq
val induct_thm_tac : thm -> string -> int -> thm -> thm Seq.seq   
val meta : thm -> thm
end

structure CASL_Tactic : CASL_TACTIC =
struct

fun is_not_casl state = 
    take(4,explode (Sign.name_of (#sign(Thm.rep_thm state)))) <> ["C","A","S","L"]

(******************* Instantiation tactics *****************************)

(*read terms, infer types*)
fun CASL_read_def_terms (sign, types, sorts) used freeze sTs =
  let
    val thn = Sign.name_of sign
    val opt_th = StructEncodeIsabelle.get_casltheory thn
    val syn = #syn (Sign.rep_sg sign);
    fun read (s, T) =
        let val T' = Sign.certify_typ sign T handle TYPE (msg, _, _) => error msg
        in case opt_th of
             Some th => (CASLThyRead.read_term T' th s, T')
           | None => (Syntax.read syn T' s, T')
        end
    val tsTs = map read sTs;
    in Sign.infer_types_simult sign types sorts used freeze tsTs 
  end;


(*read terms, infer types, certify terms*)
fun CASL_read_def_cterms (sign, types, sorts) used freeze sTs =
  let
    val (ts', tye) = CASL_read_def_terms (sign, types, sorts) used freeze sTs;
    val cts = map (cterm_of sign) ts'
      handle TYPE (msg, _, _) => error msg
           | TERM (msg, _) => error msg;
  in (cts, tye) end;

(*read term, infer types, certify term*)
fun CASL_read_def_cterm args used freeze aT =
  let val ([ct],tye) = CASL_read_def_cterms args used freeze [aT]
  in (ct,tye) end;

fun absent ixn =
  error("No such variable in term: " ^ Syntax.string_of_vname ixn);

fun inst_failure ixn =
  error("Instantiation of " ^ Syntax.string_of_vname ixn ^ " fails");

fun CASL_read_insts sign (rtypes,rsorts) (types,sorts) used insts =
let
    fun split([],tvs,vs) = (tvs,vs)
      | split((sv,st)::l,tvs,vs) = (case Symbol.explode sv of
                  "'"::cs => split(l,(Syntax.indexname cs,st)::tvs,vs)
                | cs => split(l,tvs,(Syntax.indexname cs,st)::vs));
    val (tvs,vs) = split(insts,[],[]);
    fun readT((a,i),st) =
        let val ixn = ("'" ^ a,i);
            val S = case rsorts ixn of Some S => S | None => absent ixn;
            val T = Sign.read_typ (sign,sorts) st;
        in if Sign.typ_instance sign (T, TVar(ixn,S)) then (ixn,T)
           else inst_failure ixn
        end
    val tye = map readT tvs;
    fun mkty(ixn,st) = (case rtypes ixn of
                          Some T => (ixn,(st,typ_subst_TVars tye T))
                        | None => absent ixn);
    val ixnsTs = map mkty vs;
    val ixns = map fst ixnsTs
    and sTs  = map snd ixnsTs
    val (cts,tye2) = CASL_read_def_cterms(sign,types,sorts) used false sTs;
    fun mkcVar(ixn,T) =
        let val U = typ_subst_TVars tye2 T
        in cterm_of sign (Var(ixn,U)) end
    val ixnTs = ListPair.zip(ixns, map snd sTs)
in (map (fn (ixn,T) => (ixn,ctyp_of sign T)) (tye2 @ tye),
    ListPair.zip(map mkcVar ixnTs,cts))
end;

(*Lift and instantiate a rule wrt the given state and subgoal number *)
fun lift_inst_rule (st, i, sinsts, rule) =
let val {maxidx,sign,...} = rep_thm st
    val (_, _, Bi, _) = dest_state(st,i)
    val params = Logic.strip_params Bi          (*params of subgoal i*)
    val params = rev(rename_wrt_term Bi params) (*as they are printed*)
    val paramTs = map #2 params
    and inc = maxidx+1
    fun liftvar (Var ((a,j), T)) = Var((a, j+inc), paramTs---> incr_tvar inc T)
      | liftvar t = raise TERM("Variable expected", [t]);
    fun liftterm t = list_abs_free (params,
                                    Logic.incr_indexes(paramTs,inc) t)
    (*Lifts instantiation pair over params*)
    fun liftpair (cv,ct) = (cterm_fun liftvar cv, cterm_fun liftterm ct)
    fun lifttvar((a,i),ctyp) =
        let val {T,sign} = rep_ctyp ctyp
        in  ((a,i+inc), ctyp_of sign (incr_tvar inc T)) end
    val rts = types_sorts rule and (types,sorts) = types_sorts st
    fun types'(a,~1) = (case assoc(params,a) of None => types(a,~1) | sm => sm)
      | types'(ixn) = types ixn;
    val used = add_term_tvarnames
                  (#prop(rep_thm st) $ #prop(rep_thm rule),[])
    val (Tinsts,insts) = CASL_read_insts sign rts (types',sorts) used sinsts
in Drule.instantiate (map lifttvar Tinsts, map liftpair insts)
                     (lift_rule (st,i) rule)
end;


fun read_instantiate_sg sg sinsts th =
    let val ts = types_sorts th;
        val used = add_term_tvarnames(#prop(rep_thm th),[]);
    in  instantiate (CASL_read_insts sg ts ts used sinsts) th  end;

(*Instantiate theorem th, reading instantiations under theory of th*)
fun read_instantiate sinsts th =
    read_instantiate_sg (#sign (rep_thm th)) sinsts th;


(*For forw_inst_tac and dres_inst_tac.  Preserve Var indexes of rl;
  increment revcut_rl instead.*)
fun make_elim_preserve rl =
  let val {maxidx,...} = rep_thm rl
      fun cvar ixn = cterm_of (Theory.sign_of ProtoPure.thy) (Var(ixn,propT));
      val revcut_rl' =
          instantiate ([],  [(cvar("V",0), cvar("V",maxidx+1)),
                             (cvar("W",0), cvar("W",maxidx+1))]) revcut_rl
      val arg = (false, rl, nprems_of rl)
      val [th] = Seq.list_of (bicompose false arg 1 revcut_rl')
  in  th  end
  handle Bind => raise THM("make_elim_preserve", 1, [rl]);



(*instantiate and cut -- for a FACT, anyway...*)
fun cut_inst_tac sinsts rule = res_inst_tac sinsts (make_elim_preserve rule);

(*forward tactic applies a RULE to an assumption without deleting it*)
fun forw_inst_tac sinsts rule = cut_inst_tac sinsts rule THEN' assume_tac;

(*dresolve tactic applies a RULE to replace an assumption*)
fun dres_inst_tac sinsts rule = eres_inst_tac sinsts (make_elim_preserve rule);

(*instantiate variables in the whole state*)
val instantiate_tac = PRIMITIVE o read_instantiate;

(*eresolve elimination version*)
fun eres_inst_tac sinsts rule i =
    compose_inst_tac sinsts (true, rule, nprems_of rule) i;


(*
(*Deletion of an assumption*)
fun thin_tac s = eres_inst_tac [("V",s)] thin_rl;
*)

(*compose version: arguments are as for bicompose.*)
fun compose_inst_tac sinsts (bires_flg, rule, nsubgoal) i st =
  if i > nprems_of st then no_tac st
  else st |>
    (compose_tac (bires_flg, lift_inst_rule (st, i, sinsts, rule), nsubgoal) i
     handle TERM (msg,_)   => (writeln msg;  no_tac)
          | THM  (msg,_,_) => (writeln msg;  no_tac));


(*"Resolve" version.  Note: res_inst_tac cannot behave sensibly if the
  terms that are substituted contain (term or type) unknowns from the
  goal, because it is unable to instantiate goal unknowns at the same time.

  The type checker is instructed not to freeze flexible type vars that
  were introduced during type inference and still remain in the term at the
  end.  This increases flexibility but can introduce schematic type vars in
  goals.
*)
fun res_inst_tac sinsts rule i state =
  if is_not_casl state
  then Tactic.res_inst_tac sinsts rule i  state
  else compose_inst_tac sinsts (false, rule, nprems_of rule) i state;

val old_case_tac = case_tac

fun case_tac a i state =
    if is_not_casl state
    then old_case_tac a i state
    else res_inst_tac [("P",a)] case_split_thm i state; 


(******************* Induction tactics *****************************)

(*Warn if the (induction) variable occurs Free among the premises, which
  usually signals a mistake.  But calls the tactic either way!*)
fun occs_in_prems tacf vars =
  SUBGOAL (fn (Bi, i) =>
           (if  exists (fn Free (a, _) => a mem vars)
                      (foldr add_term_frees (#2 (strip_context Bi), []))
             then warning "Induction variable occurs also among premises!"
             else ();
            tacf i));

fun prep_var (Var (ixn, _), Some x) = Some (implode (tl (explode (Syntax.string_of_vname ixn))), x)
  | prep_var _ = None;

fun prep_inst (concl, xs) =	(*exception LIST*)
  let val vs = InductMethod.vars_of concl
  in mapfilter prep_var (Library.drop (length vs - length xs, vs) ~~ xs) end;

fun find_typ var Bi =
  let val frees = map dest_Free (term_frees Bi)
      val params = Logic.strip_params Bi;
  in case assoc (frees @ params, var) of
       None => error ("No such variable in subgoal: " ^ quote var)
     | Some (ty as Type (_,_)) => ty
     | _ => error ("Cannot determine type of " ^ quote var)
  end;

fun get_concl (Const ("==>",_) $ _ $ concl) = get_concl concl
  | get_concl x = x;

fun is_induction_axiom ty (_,ax) =
    case get_concl ( #prop (Thm.rep_thm ax)) of
      (Const ("Trueprop",_) $ (Var _ $ (Var (_,ty1)))) => ty = ty1
    | _ => false;

fun search_induction_axiom th ty =
    snd (the (Library.find_first (is_induction_axiom ty) (caslaxioms_of th)));

fun get_induction_axiom tn ty =
    let val (thn,simple_tn) = take_prefix  (fn x => x<>".") (explode tn)
        val sort_name = implode (tl simple_tn)
        val ax_name = "ga_generated_"^sort_name
    in case StructEncodeIsabelle.get_casltheory (implode thn) of
         Some th => (get_caslaxiom th ax_name
                      handle _ => search_induction_axiom th ty)
       | _ => raise ERROR
    end;

fun gen_induct_tac (varss, opt_rule) i state =
  let
    val (_, _, Bi, _) = Thm.dest_state (state, i);
    val (rule, rule_name) = 
      (case opt_rule of
        Some r => (r, "Induction rule")
      | None =>
          let val ty = find_typ (hd (mapfilter I (flat varss))) Bi
              val Type (tn,_) = ty
          in (get_induction_axiom tn ty
              handle _ => error ("No induction axiom for "^tn),
              "Induction rule for sort " ^ tn) end);
    val concls = InductMethod.concls_of rule;
    val insts = flat (map prep_inst (concls ~~ varss)) handle LIST _ =>
      error (rule_name ^ " has different numbers of variables");
  in occs_in_prems (res_inst_tac insts rule) (map #2 insts) i state end;

fun induct_tac s i state =
    if is_not_casl state
    then DatatypePackage.induct_tac s i state
    else gen_induct_tac (map (Library.single o Some) (Syntax.read_idents s), None) i state;

fun induct_thm_tac th s i state =
    if is_not_casl state
    then DatatypePackage.induct_thm_tac th s i state
    else gen_induct_tac ([map Some (Syntax.read_idents s)], Some th) i state;


fun normalize rule thm =
    normalize rule (thm RS rule)
    handle _ => thm;

fun meta thm = (normalize mp (normalize spec thm));


fun data_case_tac var i = induct_tac var i THEN ALLGOALS (TRY o (rtac impI THEN' etac thin_rl))
 
end
