(* ----------------------------------------------------------------------- *)
(* generator.sml                          Pascal Schmidt <pharao90@tzi.de> *)
(* ----------------------------------------------------------------------- *)
(* AS to lstring conversion                                                *)
(* ----------------------------------------------------------------------- *)

(* $Id: generator.sml,v 1.1 2004/02/13 11:29:20 7till Exp $ *)

(* todo

Spezifikations-Referenzen fhren zu Hyperlinks (fr pdf-latex)

Idee fr die LaTeX-Umwandlung:

Datenstruktur:

datatype rstring = nl  (* new line created by position info *)
                 | block_nl (* new line created by block structure *)
                 | str of string              (* raw string *)
                 | pos_rstring of (region*rstring)    (* rstring with position info *)
                 | cat of rstring list        (* concatenation of rstrings *)
              

Die Umwandlung geschieht in fnf Schritten:
1. AS.LIB_DEFN -> rstring
2. rstring -> rstring
3. rstring -> rstring list
4. rstring list -> rstring list
5. rstring list -> string list 

Schritt 1 wandelt die AS in LaTeX.
Positions-Information wird beibehalten (d.h. in pos_rstring bersetzt).
Da, wo die LaTeX-Blockstruktur schon implizit ein Newline erzeugt
(z.B. mit \I), wird ein block_nl erzeugt. Es sollten generell
arrays und \\ vermieden werden (wo das nicht geht, natrlich auch
fr \\ ein block_nl ausgeben).

Schritt 2 eliminiert die Positions-Info.
Dazu wird der rstring rekursiv traversiert und dabei ein Zeilenzhler
mitgeschleift. Hat eine Positionsinfo eine hhere Zeilennr. als
der Zeilenzhler, so werden entsprechend viele nl's eingefgt,
und der Zeilenzhler wird neu gesetzt.
Die Positionsinfos werden gelscht.

Schritt 3 klopft den rstring zu einer Liste von rstring's flach,
die kein cat mehr enthalten.

Schritt 4 lscht in der Liste von rstrings nebeneinanderliegende Paare
nl,block_nl oder block_nl,nl.
Damit wird erreicht, dass die aufgrund der Blockstruktur erzeugten
newlines nicht noch einmal aufgrund der Positionsinfos erzeugt werden.

Schritt 5 ersetzt nl durch \\ und str s durch s und lscht block_nl.


LaTeX pretty printing von Formeln

  anpassen an neue as, Changes fuer neue Syntax (App C, App. D-4 )
  Zeilenumbruch nach regions (auer fr Formeln und Terme)
    
  Anpassen an neues casl.sty
  Neues Summary fuer Beispiele
  
  neue get_anno-Funktionen einbinden
  Bei Makellos Annos und Kommentare ausgeben

  Syntax, genv, spec-name durchfaedeln bzw. auf oberster Ebene spec_lenv
    (erst nach der geandertern Env-Struktur!)
  
  local-spec
           
  Eigenen Datentypen, bestimmte Teile (decls, ...) auf pretty printer umstellen

  Spaetere ISO-Latin-1-Ausgabe da wo es geht mitberuecksichtigen
  
  generated {type xxx} => generated type xxx
          
  Behandlung von links- und rechtsassoziativen Operationen
     (auch bei when-else)
     
                     
  Kurze spec-defns besser ausgeben
  
  Bei Operations-, pred-, datatyep-decls etc.:
     erst alle Namen formatieren, 
     maximale Laenge feststellen, (erstmal Stringlaenge nehmen)
        falls max. Laenge < 1/3 Gesamtlaenge
        dann Typen mit Restlaenge formatieren
        sonst Zeilen als Einheit formatieren (ohne &)
  
  spaeter: logische Laengen fuer LaTeX-Kommandos mit speziellem Zeichen
  und Umbruchpositionen (vor oder nach place). Beispiel:
  if a then 
  a else 
  b fi
  
  < 1 ,
  2,
  3 >
 

reveal versus with in Arrays (LaTeX-Ausgabe)

Kommentare - wohin gehoeren sie?
  sort s
  op f,g : Nat->Nat; %%ccc
     h : Nat-> Nat; 
     %% ...

LaTeX nach ASCII
(typischer Arbeitszyklus: 
  ASCII als erste Eingabe,
  bersetzung nach LaTeX, LaTeX-file bleibt Haupt-Quelle
  Fein-tuning in LaTeX
  Rckbersetzung nach ASCII fr Parsing
  
  Annotationen wie union <-> \cup in beiden Richtungen bercksichtigen)


*)


structure latex_generator
=
struct

  local open AS latex_regions latex_posget latex_types in

  exception LatexListShort
  exception LatexError

  (* --------------------------------------------------------------------- *)
  (* general functions                                                     *)
  (* --------------------------------------------------------------------- *)

  val dummy_region = (0,0,0,0)

  fun lesc [] = []
    | lesc ((#"/")::(#"/")::t) = (#"/") :: (lesc t)
    | lesc (h::t) = if (h=(#"_")) then ((String.explode "\\_") @ (lesc t))
                    else
                    if (h=(#"^")) then ((String.explode "\\symbol{94}") @ (lesc t))
                    else
                    if (h=(#"{")) then ((String.explode "\\{") @ (lesc t))
                    else
                    if (h=(#"}")) then ((String.explode "\\}") @ (lesc t))
                    else
                    if (h=(#"#")) then ((String.explode "\\#") @ (lesc t))
                    else
                    if (h=(#"%")) then ((String.explode "\\%") @ (lesc t))
                    else
                    if (h=(#"\n")) then (lesc t)
                    else
                    (h :: (lesc t))

  fun lescape s = (String.implode o lesc o String.explode) s

  (* --------------------------------------------------------------------- *)
  (* line breaking functions                                               *)
  (* --------------------------------------------------------------------- *)

  fun cond_newline r1 r2 = if (is_same_line r1 r2) then
                             [nop]
                           else
                             [lnewline,newline]


  (* --------------------------------------------------------------------- *)
  (* format a list of elements                                             *)
  (* --------------------------------------------------------------------- *)
  (* prefix each item with pre, postfix each item with post, seperator     *)
  (* between list elements is sep, pos_f useable to get region info of a   *)
  (* list element, ltx_f useable to convert list element to LaTeX          *)
  (* --------------------------------------------------------------------- *)

  fun list_newline [] last pre post sep pos_f ltx_f = [nop]
    | list_newline [h] last pre post sep pos_f ltx_f  =
                        (cond_newline last (pos_f h)) @
                        pre @ (ltx_f h) @ post
    | list_newline (h::t) last pre post sep pos_f ltx_f =
                        (cond_newline last (pos_f h)) @
                        pre @ (ltx_f h) @ post @ sep @
                        (list_newline t (pos_f h) pre post sep pos_f ltx_f)

  fun format_list [] pre post sep pos_f ltx_f = [nop]
    | format_list (h::t) pre post sep pos_f ltx_f =
                      list_newline (h::t) (pos_f h) pre post sep pos_f ltx_f

  (* --------------------------------------------------------------------- *)
  (* ID to LaTeX mapping                                                   *)
  (* --------------------------------------------------------------------- *)

  fun l_simple_id (s,_) = [verbatim (lescape s)]

  fun l_id (simple_id (t)) = (l_token_or_mixfix t)
    | l_id (compound_id (t,l)) = (l_token_or_mixfix t) @ [verbatim "["] @
                                 (l_comp_list l) @ [verbatim "]"] 

  and l_comp_list []     = raise LatexListShort
    | l_comp_list [h]    = (l_id h)
    | l_comp_list (h::t) = (l_id h) @ [verbatim ","] @ (l_comp_list t)

  and l_token_or_mixfix (l,_,_) = (map l_token_or_place) l
  
  and l_token_or_place (token (t)) = verbatim (lescape t)
    | l_token_or_place (place)     = verbatim (lescape "__")

  (* --------------------------------------------------------------------- *)
  (* ANNO to LaTeX mapping                                                 *)
  (* --------------------------------------------------------------------- *)

  fun l_anno_head x = [verbatim "\\begin{ANNOTELINES}"] @ x @ [newline] @
                      [verbatim "\\end{ANNOTELINES}",newline]

  fun l_comment_head x = [verbatim "\\begin{COMMENT}"] @ x @ [newline] @
                         [verbatim "\\end{COMMENT}",newline]

  fun l_anno (pos_ANNO (r,a))      = (l_anno a)
    | l_anno (label_anno (i))      = [nop]
    | l_anno (comment_line (c))    = [verbatim "\\COMMENTLINE{",verbatim
                                     (lescape c),verbatim "}",newline]
    | l_anno (comment (c))         = l_comment_head ([verbatim (lescape c)])
    | l_anno (unparsed_anno (u))   = [nop]
    | l_anno (number_anno (i))     = l_anno_head ([verbatim "{\\NUMBER}",
                                     newline] @ (l_id i))
    | l_anno (floating_anno (i,j)) = l_anno_head ([verbatim "{\\FLOATING}",
                                     newline] @ (l_id i) @ [verbatim ","] @ 
                                     (l_id j))
    | l_anno (string_anno (i,j))   = l_anno_head ([verbatim "{\\STRING}",
                                     newline] @ (l_id i) @ [verbatim ","] @
                                     (l_id j))
    | l_anno (list_anno (i,j,k))   = l_anno_head ([verbatim "{\\LIST}",
                                     newline] @ (l_id i) @ [verbatim ","] @
                                     (l_id j) @ [verbatim ","] @ (l_id k))
    | l_anno (display_anno (i,s))  = [nop]
    | l_anno (prec_anno (b,l1,l2)) = [nop]
    | l_anno (lassoc_anno (l))     = [nop]
    | l_anno (rassoc_anno (l))     = [nop]
    | l_anno (conservative)        = [nop]
    | l_anno (definitional)        = [nop]
    | l_anno (implies)             = [nop]

  fun l_annos l = format_list l [nop] [nop] [nop] pos_anno l_anno

  (* --------------------------------------------------------------------- *)
  (* lib_item to LaTeX mapping                                             *)
  (* --------------------------------------------------------------------- *)

  fun l_genericity x = [nop]
  
  fun l_spec x = [nop]
  
  fun l_spec_name x = [verbatim "\\NAMEDEFN{"] @ (l_simple_id x) @
                      [verbatim "}"]

  fun l_spec_defn (n,g,s,a) = [item,verbatim "\\SPEC "] @ 
                              (l_spec_name n) @ (l_genericity g) @
                              [verbatim " =",newline] @
                              (l_spec s) @ [newline] @ (l_annos a) @
                              [newline]

  fun l_lib_item (pos_LIB_ITEM (r,i))     = l_lib_item i
    | l_lib_item (arch_spec_defn (n,s,a)) = [nop]                 (* FIXME *)
    | l_lib_item (unit_spec_defn (n,s,a)) = [nop]                 (* FIXME *)
    | l_lib_item (download_items (n,l,a)) = [nop]                 (* FIXME *)
    | l_lib_item (view_defn (n,g,t,l,a))  = [nop]                 (* FIXME *)
    | l_lib_item (spec_defn (sd))         = l_spec_defn sd
  
  fun l_lib_items l = format_list l [nop] [nop]
                      [newline,newline,verbatim "\\bigskip",newline,newline]
                      pos_lib_item l_lib_item

  (* --------------------------------------------------------------------- *)
  (* lib_defn to LaTeX mapping                                             *)
  (* --------------------------------------------------------------------- *)

  fun l_lib_defn (pos_LIB_DEFN (_,x))          = l_lib_defn x
    | l_lib_defn (lib_defn (name,items,annos)) =
                  [verbatim "\\LIBRARYDEFN{"] @ (l_lib_name name) @
                  [newline,newline,verbatim "\\bigskip",newline,newline] @
                  (l_annos annos) @
                  [newline,newline,verbatim "\\bigskip",newline,newline] @
                  (l_lib_items items)

  and l_lib_name (pos_LIB_NAME (_,x))  = l_lib_name x
    | l_lib_name (lib (x))             = (l_lib_id x) @ [verbatim "}"]
    | l_lib_name (versioned_lib (x,v)) = let
                                           val px = pos_lib_id x
                                           val pv = pos_version v
                                         in
                                           (l_lib_id x) @
                                           [verbatim "}"] @
                                           (cond_newline px pv) @
                                           (l_version pv v)
                                         end

  and l_lib_id (pos_LIB_ID (_,x)) = (l_lib_id x)
    | l_lib_id (url (x))          = (l_url x)
    | l_lib_id (path_name (x))    = (l_path x)
    
  and l_url x = [verbatim (lescape x)]
  
  and l_path x = [verbatim (lescape x)]

  and l_version p (pos_VERSION (_,x)) = (l_version p x)
    | l_version p (version (l))       = [verbatim "\\VERSION "]
                                        @ (l_version_list l)
    
  and l_version_list []     = raise LatexListShort
    | l_version_list [h]    = [verbatim (lescape h)]
    | l_version_list (h::t) = [verbatim (lescape h),verbatim "."] @
                              (l_version_list t)
  (* --------------------------------------------------------------------- *)
  (* end of LaTeX mapping                                                  *)
  (* --------------------------------------------------------------------- *)

  end (* local open in *)

end
