(* ******************************************************************* *)
(*								       *)
(* Project : CATS 						       *)
(* Author: Till Mossakowski, University of Bremen		       *)
(* Date: 28.10.99 		 				       *)
(* Purpose of this file: Static analysis of structured specifications  *)
(*								       *)
(*								       *)
(* ******************************************************************* *)

(* The static analysis of structured specifications closely follows
   their semantics (see study note S-9). Page numbers refer to
   S-9, July 99. 

  todo:
  Auf Signaturerweiterung bei %implies testen (sonst gibt's Fehler)
  %cons-Axioms knnten auch ignoriert werden (nicht aber die Signatur)???
  Annotationen in strukturierten Specs weiterreichen
  Check number of acutal params in instantiation
  Problem, dass Analyse aus
     spec LocalExample =
     local
       Spec1
     within
       Spec2
     end
   eine extension statt einer local_spec macht. 

  view-test-1: s |-> bubu fehlt (CATS-Fehler?)

*)

structure StructuredAnalysis :
sig

val structured_analysis : 
     bool -> 
     LocalEnv.sign * GlobalEnv.global_env * AS.L_SPEC -> 
     GlobalEnv.spec_lenv * AS.L_SPEC * string list
     
val spec_or_view_defn_analysis 
  : bool
    -> GlobalEnv.global_env * AS.LIB_ITEM
       -> GlobalEnv.global_env * AS.LIB_ITEM * string list
       
end
=
struct

open Utils AS LocalEnv GlobalEnv BasicAnalysis Symbols Morphisms
    TextForm TextIO BasicPrint StructuredPrint 

(* The next two functions should belong to stst_symmaps.ml,
   and stst_symmaps.ml should work on signatures *)

fun sig_rsymbols(Sigma) =
   map (fn x => SYMBOL_RAW_SYMBOL(x)) (Stat_symmaps.sig_symbols(Sigma))

fun sig_symbols(Sigma,_) =
   Stat_symmaps.sig_symbols(Sigma)


(* These functions need to be implemented *)


fun sign_eq(Sigma1,Sigma2) = true


(* This is needed since we sometimes work on local_env instead of sign
   Should be fixed *)
   

(******** Static analysis of structured specifications *************)

(* S-9, July 99, p. 85 ff. *)

fun extend_morphism r n ((Sigma_A,an_A),sigma,(Sigma,_),(Sigma',an')) =
  let val x1 = if (!Global.test) then
      ( (*writeln "***** Extension of signature";
        writeln "**Formal parameter";
        output_sign stdOut Sigma;
        writeln "**Body";
        output_sign stdOut Sigma';
        writeln "**Actual parameter";
        output_sign stdOut Sigma_A;
        (writeln "**Morphism";print_morphism sigma);
        writeln (print_morphism sigma)*) )
      else ()
     (* val _ = writeln ("Calling morphism_extension_along_sig_extension");*)
      val ((sigma_Delta,Sigma_A_Delta),errs)
      = (morphism_extension_along_sig_extension
        (Sigma,Sigma',sigma,Sigma_A),[])
     handle (NO_MORPHISM_EXTENSION err) => 
         ((sigma,Sigma_A),
          ["in instantiation of "^print_SIMPLE_ID n^"\n"^err]) 
     (* val _ = writeln ("Leaving morphism_extension_along_sig_extension");*)
  in
  ((Sigma_A_Delta,an_A@an' (* an' should be renamed here !!!*)
          ),(sigma_Delta,[]),map (fn e => perr r^e) errs)
  end


(* S-9, July 99, p. 97 *)

fun apply r n (spec_defn_env((ienv,penvlist,Sigma_P),
                         slenv_B as SPEC_ENV(Sigma_B,_,senv_B)))
          args =
  let val Sigma_A_i = map fst args
      val sigma_i = map snd args
      val SPEC_ENV(Sigma_I,_,_) = ienv
      val id_Sigma_I = Stat_symmaps.inclusion_morphism Sigma_I
      fun get_Sigma (SPEC_ENV(Sigma,_,_)) = Sigma
      val mors = 
          (Sigma_I,id_Sigma_I)::
            zip(map get_Sigma penvlist,sigma_i)
      
   (*   val x0 = writeln "****Fitting Morphisms:"
      val x1 = writeln (print_list print_morphism "\n" (map snd mors)) *)
      
      val Sigma_A = foldl signature_union (empty_signature,Sigma_A_i)
   (*   val x1 = writeln "First actual parameter:"
      val x2 = print_sign stdOut (hd Sigma_A_i)
      val x3 = writeln "Union of  actual parameters:"
      val x4 = print_sign stdOut ( Sigma_A)*)
      
      val SPEC_ENV(Sigma_I,_,_) = ienv
      val errs1 = if compatible_morphisms mors then []
                  else [perr r^"Arguments not compatible"]
      val sigma_f = merge_morphisms mors
   (*   val x3 = writeln "****United fitting Morphism:"
      val x4 = writeln ( print_morphism  (  sigma_f))  *)
      val ((Sigma_A_B,_),sigma_f_B,errs2) =
           (extend_morphism r n ((Sigma_A,[]),sigma_f,(Sigma_P,[]),(Sigma_B,[])))
  in (Sigma_A_B,fst sigma_f_B,errs1@errs2)
  end
 | apply _ _ _ _ = raise ERROR


fun get_Sigma (SPEC_ENV(Sigma,_,env), sp, err_list) = Sigma
fun get_env   (SPEC_ENV(Sigma,_,env), sp, err_list) = env
fun get_errs  (env, sp, err_list) = err_list
fun get_spec  (env, sp, err_list) = sp
fun get_senv  (env, sp, err_list) = env



(* S-9, July 99, p. 88 *)

fun structured_analysis1 (html:bool) (r:region)
    (Sigma:LocalEnv.local_env,(genv,gan),(basic bspec,ans):L_SPEC):(spec_lenv * L_SPEC * string list) =
  let
     val (bspec',Sigma',Sigma_frag,Phi,errs) =
        basic_analysis html (bspec,(Sigma,gan))  
     (*val Sigma_new_frag = signature_diff(Sigma_frag,Sigma)      *) (* Should we remove duplicate symbols??? *)
  in (SPEC_ENV(Sigma',empty_signature,basic_env(Sigma_frag,Phi)),
      (basic bspec',ans),
      errs)	
  end  


(* S-9, July 99, p. 89 *)

| structured_analysis1 html reg
  (Sigma,Gamma,(translation(sp,r,ans1),ans2):L_SPEC) =
  let
     val (SPEC_ENV(Sigma',_,senv'),sp',errs1) =
         structured_analysis1 html reg (Sigma,Gamma,sp)
     val (sigma,(Sigma'',_),errs2) = 
         Symmaps_analysis.stat_renaming ((Sigma',[]),r,[])
     val errs2' = map (fn e => perr reg^e) errs2 

     (* do we have to check that sigma is the identity on |Sigma| ??? Yes! *)
     
  in (SPEC_ENV(Sigma'',empty_signature,translate_env (senv',sigma)),
      (translation(sp',r,ans1),ans2),
      errs1@errs2')
  end


(* S-9, July 99, p. 91 *)

| structured_analysis1 html reg
  (Sigma:LocalEnv.local_env,Gamma,(reduction(sp,r,ans1),ans2):L_SPEC) =
  let
     val (SPEC_ENV(Sigma',_,senv'),sp',errs1) =
         structured_analysis1 html reg (Sigma,Gamma,sp)
     val ((Sigma1,_),sigma,(Sigma'',_),errs2) = 
         Symmaps_analysis.stat_restriction((Sigma,[]),(Sigma',[]),r,[])
     val errs2' = map (fn e => perr reg^e) errs2 
  in (SPEC_ENV(Sigma'',empty_signature,derive_env (senv',sigma)),
      (reduction(sp',r,ans1),ans2),
      errs1@errs2')
  end


(* S-9, July 99, p. 92 *)

|  structured_analysis1 html r
   (Sigma,Gamma,(union_spec speclist,ans):L_SPEC) =
  let
    fun analyse1 sp = structured_analysis1 html r (Sigma,Gamma,sp)
    val env_list = map (analyse1 o fst) speclist
    val Sigma_list = map get_Sigma env_list
    val Sigma_n = foldl signature_union (empty_signature,Sigma_list)
  in (SPEC_ENV(Sigma_n,empty_signature,union_env(map get_env env_list)),
      (union_spec (zip (map get_spec env_list,map snd speclist)),ans),
      flat (map get_errs env_list))
  end


(* S-9, July 99, p. 93 *)

|  structured_analysis1 html r
   (Sigma,Gamma,(extension spec_list,ans):L_SPEC) =
  let
     fun analyse1 ((Sigma1,senvs,errs,spec_list),sp) = 
         let val (SPEC_ENV(Sigma2,_,senv2),sp2,errs2) = 
                  structured_analysis1 html r (Sigma1,Gamma,sp)
         in (Sigma2,senv2::senvs,errs@errs2,sp2::spec_list)
         end
     val (Sigma_n,senvs,errs,new_spec_list) = 
         foldl analyse1 ((Sigma,[],[],[]),map fst spec_list)
     val pan_list = map (CASLParser.ParseAnnos o snd) spec_list
     val pans = CASLParser.ParseAnnos ans
  in
     (SPEC_ENV(Sigma_n,empty_signature,
               extension_env(zip(rev senvs,pan_list))),
      (extension (zip (rev new_spec_list,pan_list)),pans),
      errs)
  end


(* S-9, July 99, p. 94 *)

|  structured_analysis1 html r
   (Sigma,Gamma,(free_spec (sp,ans1),ans2):L_SPEC) =
  let
     val (SPEC_ENV(Sigma1,_,senv1),sp1,errs) = 
         structured_analysis1 html r (Sigma,Gamma,sp)

  in  (SPEC_ENV(Sigma1,empty_signature,free_spec_env(senv1)),
       (free_spec (sp1,ans1),ans2),
       errs)	
  end

|  structured_analysis1 html r
   (Sigma,Gamma,(cofree_spec (sp,ans1),ans2):L_SPEC) =
  let
     val (SPEC_ENV(Sigma1,_,senv1),sp1,errs) = 
         structured_analysis1 html r (Sigma,Gamma,sp)

  in  (SPEC_ENV(Sigma1,empty_signature,cofree_spec_env(senv1)),
       (free_spec (sp1,ans1),ans2),
       errs)	
  end


(* S-9, July 99, p. 95 *)

|  structured_analysis1 html r
   (Sigma,Gamma,(local_spec(spec1,ans1,spec2,ans2),ans3):L_SPEC) = 
  let
     val (SPEC_ENV(Sigma',_,senv'),sp',errs') = 
         structured_analysis1 html r (Sigma,Gamma,spec1)
     val (SPEC_ENV(Sigma'',_,senv''),sp'',errs'') = 
         structured_analysis1 html r (Sigma',Gamma,spec2)
     val syms = Finset.remove_set(sig_symbols(Sigma,[]),sig_symbols(Sigma',[]))
     val ((Sigma1,_,sigma),errs3) = 
         (Stat_symmaps.cogenerated_signat (syms,Sigma'',[]),[])
         handle (STAT_EXCEPTION err) => ((Sigma'',[],id_morphism  Sigma''),[err])
     val err_syms = 
         Finset.remove_set(Stat_symmaps.sig_symbols(Sigma''),
              Finset.sum(Stat_symmaps.sig_symbols(Sigma'),
                         Stat_symmaps.sig_symbols(Sigma1)))
     val errs4 = if null err_syms then [] 
                 else [perr r^"Result specification depends on local symbols"] 
  in (SPEC_ENV(Sigma1,empty_signature,
               derive_env(extension_env[(senv',[]),(senv'',[])],sigma)),
      (local_spec (sp',ans1,sp'',ans2),ans3),
      errs'@errs''@errs3@errs4)
  end    


(* S-9, July 99, p. 95 *)

|  structured_analysis1 html r
  (Sigma,Gamma,(closed_spec (sp,ans1),ans2):L_SPEC) =
  let
     val (SPEC_ENV(Sigma',_,senv'),sp',errs) = 
         structured_analysis1 html r (empty_signature,Gamma,sp)
     val Sigma1 = signature_union(Sigma,Sigma')
  in  (SPEC_ENV(Sigma1,empty_signature,closed_spec_env senv'),
       (closed_spec (sp',ans1),ans2),
       errs)		
  end


(* S-9, July 99, p. 102 *)

|  structured_analysis1 html r
  (Sigma,Gamma,(spec_inst(n,arglist),ans):L_SPEC) =
 
   (case Symtab_sid.lookup(fst Gamma,n) of  
   
     Some (GS as spec_defn_env((ienv,penvlist,_),
                     slenv_B as SPEC_ENV(Sigma_B,_,senv_B))) =>
       if not(length(arglist)=length(penvlist)) then
          (SPEC_ENV(signature_union(Sigma,Sigma_B),
                    empty_signature,
                    spec_inst_env (n, senv_B, empty_morphism, [])),
           (spec_inst(n,arglist),ans),
           [perr r^"Too "^
            (if length(arglist)<length(penvlist) then "few" else "many")
            ^" actual parameters in instantiation of "^print_SIMPLE_ID n])
       else
       if null penvlist then
          (SPEC_ENV(signature_union(Sigma,Sigma_B),
                    empty_signature,
                    spec_inst_env (n, senv_B, empty_morphism, [])),
           (spec_inst(n,[]),ans),
           [])
       else                   
       let val SPEC_ENV(Sigma_I,_,_) = ienv
          (* val _ = writeln ("Calling analysis of spec_inst");*)
           val Sigma_both = signature_union(Sigma,Sigma_I)
          (* val _ = writeln ("Calling fit_arg_analysis");*)
           val args = 
               map (fit_arg_analysis html null_region
                          (Sigma,Sigma_I,Sigma_both,Gamma)) 
                   (zip (penvlist,arglist))
          (* val _ = writeln ("Leaving fit_arg_analysis");
           val _ = writeln ("Calling apply");*)
           val (Sigma',sigma_f,errs) = apply r n GS (map get_fit_arg args)
          (* val _ = writeln ("Leaving apply");
           val _ = writeln ("Leaving analysis of spec_inst");*)
           fun get_env (SPEC_ENV(_,_,env)) = env
           val penv' = union_env (map get_env penvlist)
           val ienv' = get_env ienv
       in (SPEC_ENV(signature_union(Sigma,Sigma'),
                    empty_signature,
                    spec_inst_env (n, (*extension_env[(ienv',[]),(penv',[]),(senv_B,[])]*)
                                   senv_B, 
                                   sigma_f, 
                                   map get_fit_senv args)),
           (spec_inst(n,map get_fit_arg_tree args),ans),
           flat (map get_fit_errs args) @ errs)
       end
       
     | _ => (writeln (perr r^"specification "^print_SIMPLE_ID n^" not found"); raise ERROR)
            (* (empty_spec_lenv,
             (spec_inst(n,arglist),ans),
             [perr r^" Specification "^print_SIMPLE_ID n^" not found"])  *)
       )        

  |  structured_analysis1 html _
     (Sigma,Gamma,(pos_SPEC(r,bra,sp),ans):L_SPEC) =
     let val (senv,(sp',ans'),errs) = 
         structured_analysis1 html r (Sigma,Gamma,(sp,ans))
     in (senv,(pos_SPEC(r,bra,sp'),ans'),errs)
     end


(* S-9, July 99, p. 103 *)

and fit_arg_analysis html r
        (Sigma,Sigma_I,Sigma_both,Gamma) 
        (SPEC_ENV(Sigma_P,_,_),fit_spec (sp,symbmap)) =
    let val (SPEC_ENV(Sigma_A,_,senv_A),sp_A,errs1) =
            structured_analysis1 html null_region (Sigma_I,Gamma,sp)
        val Sigma_I_u_A = signature_union(Sigma_I,Sigma_A)

        val (rsymmap,errs2) = 
            Symmaps_analysis.stat_symb_map_items_star
               (Stat_symmaps.sig_symbols(Sigma_P),symbmap,errs1)
        val (sigma,errs3) = 
            (Stat_symmaps.induced_from_to_morphism
              (rsymmap (*@ id_rsymmap (Sigma_I)*),
               Sigma_P,
               Sigma_I_u_A),
              [])
            handle (STAT_EXCEPTION s) => 
              (empty_morphism,[s])
        fun mk_err s = perr r^"Cannot construct fitting morphism:\n"^s 
          (* val _ = writeln ("Leaving induced_from_to_morphism");*)
    in (sigma,Sigma_A,fit_spec(sp_A,symbmap),senv_A,map mk_err (errs2@errs3))
    end

(* S-9, July 99, p. 108 *)

  | fit_arg_analysis html r
        (Sigma,Sigma_I,Sigma_both,Gamma) 
        (SPEC_ENV(Sigma_P,_,_),fit_view (vname,arglist)) =

      (*** Sigma_both really needed here ???? ***)

   (case Symtab_sid.lookup(fst Gamma,vname) of  
      
     Some (view_defn_env(gen as (ienv,penvlist,_),
                         slenv_s as SPEC_ENV(Sigma_s,_,_),
                         sigma,
                         slenv_t as SPEC_ENV(Sigma_t,_,senv_t))) =>
       if not(length(arglist)=length(penvlist)) then
          (sigma,
           Sigma_t,
           fit_view (vname,arglist),
           senv_t,
           [perr r^"Too "^
            (if length(arglist)<length(penvlist) then "few" else "many")
            ^" actual parameters in instantiation of "^print_SIMPLE_ID vname])
       else
       if null penvlist then
       let val Sigma_I_u_P = signature_union(Sigma_I,Sigma_P)
           val Sigma_both_u_t = signature_union(Sigma_both,Sigma_t)
       in ((* Composition with inclusion
               from Sigma_t to Sigma_both_u_t currently is not needed,
               since domain remains the same!!! *) sigma,
           Sigma_t,
           fit_view (vname,[]),
           senv_t,
           if sign_eq(Sigma_s,Sigma_I_u_P) then []
           else [perr r^"Fitting view source signature " 
                ^"does not match formal parameter"] 
          )
       end  
       else                   
       let val Sigma_I_u_P = signature_union(Sigma_I,Sigma_P)
           val SPEC_ENV(Sigma_I',_,_) = ienv
             (* Beware! 
                Sigma_I is the import from the enclosing instantiation
                Sigma_I' is the import from the generic view *)
           val Sigma_both_u_I' = signature_union(Sigma_both,Sigma_I')
           val args = 
               map (fit_arg_analysis html null_region
                          (Sigma_both,Sigma_I',Sigma_both_u_I',Gamma)) 
                   (zip (penvlist,arglist))
           val GS = spec_defn_env (gen,slenv_t)
           val (Sigma_A,sigma_f',errs) = 
               apply r vname GS (map get_fit_arg args)
       in (compose_morphisms(Sigma_s,sigma,sigma_f'),
           Sigma_A,
           fit_view (vname,arglist),
           spec_inst_env(vname, (* Here, we have no better name
                                   Thus, we have to abuse the name
                                   of the view as the name of
                                   its parameterized target spec *) 
                         senv_t,
                         sigma_f', 
                         map get_fit_senv args),
           flat (map get_fit_errs args) @ errs
           @ (if sign_eq(Sigma_s,Sigma_I_u_P) then []
              else [perr r^"Fitting view source signature " 
                    ^"does not match formal parameter"])
          )
       end
           
     | _ => (empty_morphism,
             empty_signature,
             fit_view (vname,arglist),
             dummy_spec_env,
             [perr r^print_SIMPLE_ID vname^" is not a view name"])
     )
  | fit_arg_analysis html _
        (Sigma,Sigma_I,Sigma_both,Gamma) 
        (senv,pos_FIT_ARG(r,fa)) =
    let val (mor,Sigma',fa',senv',errs) =
        fit_arg_analysis html r (Sigma,Sigma_I,Sigma_both,Gamma) (senv,fa)
    in (mor,Sigma',pos_FIT_ARG(r,fa'),senv',errs)
    end

    
      
and get_fit_arg (sigma,Sigma_A,fa,senv_A,errs) = (Sigma_A,sigma)
and get_fit_senv (sigma,Sigma_A,fa,senv_A,errs) = senv_A
and get_fit_arg_tree (sigma,Sigma_A,fa,senv_A,errs):FIT_ARG = fa
and get_fit_errs (sigma,Sigma_A,fa,senv_A,errs) = errs


(* S-9, July 99, p. 99 *)

fun genericity_analysis1 html reg
    (Gamma,params plist,imports ilist) =
  let 
     val (SPEC_ENV(Sigma_I,_,senv_I),
          (union_spec ilist1,_),
          errs_I) =
         structured_analysis1 html null_region 
         (empty_signature,
          Gamma,
          (union_spec (map (fn sp => (sp,[])) ilist),[]))
     fun analyse1 sp = structured_analysis1 html null_region (Sigma_I,Gamma,sp)
     val penv_list = map analyse1 plist
     val Sigma_P_list = map get_Sigma penv_list
     val Sigma_P = foldl signature_union (empty_signature,Sigma_P_list)
     val plist1 = map get_spec penv_list
  in (SPEC_ENV(Sigma_I,empty_signature,senv_I),
      Sigma_P,
      map get_senv penv_list,
      params plist1,imports (map fst ilist1),
      errs_I @ flat (map get_errs penv_list) @
        (if not (null ilist) andalso null plist
         then [perr reg^"Imports only allowed in presence of parameters"]
         else []))
  end
 | genericity_analysis1 html reg
    (Gamma,pos_PARAMS (r,pars),imps) =
   let val (senv,Sigma,penvs,pars',imps',errs) =
       genericity_analysis1 html reg (Gamma,pars,imps)
   in (senv,Sigma,penvs,pos_PARAMS (r,pars'),imps',errs)
   end
 | genericity_analysis1 html reg
    (Gamma,pars,pos_IMPORTS(r,imps)) =
   let val (senv,Sigma,penvs,pars',imps',errs) =
       genericity_analysis1 html reg (Gamma,pars,imps)
   in (senv,Sigma,penvs,pars',pos_IMPORTS(r,imps'),errs)
   end

fun genericity_analysis html r (Gamma,genericity(pars,imps)) =
    let val (senv,Sigma,penvs,pars',imps',errs) =
        genericity_analysis1 html r (Gamma,pars,imps)
    in (senv,Sigma,penvs,genericity(pars',imps'),errs)
    end
  | genericity_analysis html _ (Gamma,pos_GENERICITY(r,g)) =
    let val (senv,Sigma,penvs,g',errs) =
        genericity_analysis html r (Gamma,g)
    in (senv,Sigma,penvs,pos_GENERICITY(r,g'),errs)
    end


fun get_view_type (view_type (s,t)) = (s,t)
  | get_view_type (pos_VIEW_TYPE(_,vt)) =
    get_view_type vt


fun get_pos_VIEW_TYPE (pos_VIEW_TYPE (r,_)) = r
  | get_pos_VIEW_TYPE _ = ((0,0),(0,0))

(*** Analyse one spec_defn ***)

(* S-9, July 99, p. 98 *)

fun spec_or_view_defn_analysis1 html _
    (Gamma as (genv,ans),
     tree as (spec_defn (name,gen,sp,ans_sp))) =
  let val (senv_I,Sigma_P,senvs_P,gen1,errs1) =
           genericity_analysis html null_region (Gamma,gen)
      val SPEC_ENV(Sigma_I,_,_) = senv_I
      val (senv_B,sp1,errs2) =
           structured_analysis1 html null_region (Sigma_P,Gamma,sp)            
      val entry = 
          spec_defn_env ((senv_I,senvs_P,Sigma_P),senv_B)
      val genv' = Symtab_sid.update ((name,entry),genv)
  in ((genv',ans),
      spec_defn (name,gen1,sp1,ans_sp),
      errs1@errs2) 
  end


(*** Analyse one view_defn ***)

(* S-9, July 99, p. 106 *)

  | spec_or_view_defn_analysis1 html reg
     (Gamma as (genv,ans),
      tree as (view_defn (name,gen, 
               vt,
               vmap,
               ans_v))) =
  let val (source,target) = get_view_type vt
      val r = get_pos_VIEW_TYPE vt
      val (slenv_s as SPEC_ENV(Sigma_s,_,_),
           source_tree,
           errs1) 
         = structured_analysis1 html null_region 
           (empty_signature,
            Gamma,
            source)
      val (senv_I,Sigma_P,senvs_P,gen1,errs2) =
           genericity_analysis html null_region (Gamma,gen)
      val SPEC_ENV(Sigma_I,_,_) = senv_I
      val (slenv_t as SPEC_ENV(Sigma_t,_,_),
           target_tree,
           errs3) =
           structured_analysis1 html null_region (Sigma_P,Gamma,target)
      
      val Sigma_P_u_t = signature_union(Sigma_P,Sigma_t)     
      val (rsymmap,errs4) = 
          Symmaps_analysis.stat_symb_map_items_star
             (Stat_symmaps.sig_symbols(Sigma_s),vmap,[])
     (* val x1 = writeln "Source:"
      val x2 = print_sign stdOut Sigma_s
      val x3 = writeln "Target:"
      val x4 = print_sign stdOut Sigma_P_u_t*)
      val (sigma,errs5) = 
          (Stat_symmaps.induced_from_to_morphism
            (rsymmap,
             Sigma_s,
             Sigma_P_u_t),
            [])
          handle (STAT_EXCEPTION s) => 
            (empty_morphism,[perr reg^"Cannot construct view morphism:\n"^s])
             
      val entry = 
          view_defn_env ((senv_I,senvs_P,Sigma_P),
                         slenv_s,
                         sigma,
                         slenv_t)
      val genv' = Symtab_sid.update ((name,entry),genv)
  in ((genv',ans),
      view_defn (name,
                 gen1,
                 pos_VIEW_TYPE(r,view_type(source_tree,target_tree)),
                 vmap,
                 ans_v),
      errs1@errs2@errs3@errs4@errs5) 
  end
  
  | spec_or_view_defn_analysis1 html _
     (Gamma,pos_LIB_ITEM(r,li)) =
   let val (Gamma',li',errs) = 
            spec_or_view_defn_analysis1 html r (Gamma,li)
       in (Gamma',pos_LIB_ITEM(r,li'),errs)
       end

  | spec_or_view_defn_analysis1 _ _ _ = raise ERROR


fun structured_analysis html = structured_analysis1 html null_region

fun spec_or_view_defn_analysis html = spec_or_view_defn_analysis1 html null_region
end
