(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Authors: Till Mossakowski, University of Bremen                         *)
(*          Klaus Lttich, University of Bremen			           *)
(* Date: 2001				 			           *)
(* Purpose of this file: Printing functions for structured specifications  *)
(*			 						   *)
(*									   *)
(* *********************************************************************** *)

(* This module converts the CASL abstract syntax back into concrete input
   syntax, and outputs environments in human readable form *)

(* todo:

Im FCasEnv-Text nur das Delta ausgeben
Im Text-Env auch Axiome, und Spezifikationsnamen bei Parametern
File+Position fuer LIB-ITEMs in CasEnv
Volle Signatur-Morphismen ausgeben (als CATS-Option!)
*)

structure StructuredPrint : sig

(* additional print-functions added to the signature by MR (and LS) *)
(* these are needed within ccc *)

val print_LIB_ITEM : AS.LIB_ITEM -> string
val print_FIT_ARG  : AS.FIT_ARG -> string
val print_RENAMING : AS.RENAMING -> string
val print_SYMB_MAP_ITEMS_list: AS.SYMB_MAP_ITEMS list -> string
val print_BASIC_ITEMS: AS.BASIC_ITEMS -> string
val print_BASIC_SPEC: AS.BASIC_SPEC -> string
(* end of the additional functions *)

val print_SPEC : AS.SPEC -> string
val print_LIB_DEFN : AS.LIB_DEFN -> string
val print_LIB_ID : AS.LIB_ID -> string
val print_VERSION : AS.VERSION -> string
val print_LIB_NAME : AS.LIB_NAME -> string

val print_symbol : Symbols.symbol -> string
val print_symbol_set : Symbols.symbol Finset.set -> string
val print_symbol_pair : Symbols.symbol * Symbols.symbol -> string
val print_symbol_pair_set : (Symbols.symbol * Symbols.symbol) Finset.set -> string
val print_morphism : GlobalEnv.morphism -> string
val print_spec_env : GlobalEnv.spec_env -> string
val output_spec_lenv : TextIO.outstream -> GlobalEnv.spec_lenv -> unit 
val output_global_entry : TextIO.outstream -> GlobalEnv.global_entry -> unit
val output_global_env : string -> GlobalEnv.global_env -> AS.LIB_DEFN -> unit
val output_f_global_env : string -> GlobalEnv.f_global_env -> AS.LIB_DEFN -> unit
val print_spec_lenv : GlobalEnv.spec_lenv -> string 
val print_global_entry :  GlobalEnv.global_entry -> string
val print_global_env :  GlobalEnv.global_env -> AS.LIB_DEFN -> string
val print_f_global_env :  GlobalEnv.f_global_env -> AS.LIB_DEFN -> string
val latex_global_env : string -> GlobalEnv.global_env -> AS.LIB_DEFN -> unit
val latex_f_global_env : string -> GlobalEnv.f_global_env -> AS.LIB_DEFN -> unit
end
= struct

open Utils AS Symbols Finset BasicPrint GlobalEnv TextIO;

(************************** Abstract syntax *************************)

(* MR: basic function to print ccc-variables *)

fun print_var_id (s,i) = s ^ Int.toString(i);

(* MR,LS:
   added for the variables which may occur as parameters of 
   the print functions for
     FIT_ARG
     RENAMING
     RESTRICTION
     SPEC
     LIB_ITEM
     BASIC_ITEMS
     DATATYPE_DECL 
     SIG_ITEMS
     GENERICITY
     BASIC_SPEC
*)

fun print_ANNO_list nil = "" (* unparsed_anno : (unparsed) => 
                                solved: modified basic_print.sml *) 

  | print_ANNO_list ans = 
     print_list print_ANNO "\n" ans ^ "\n" 

fun print_SORT s = print_ID s

fun print_OP_NAME opn = print_ID opn

fun print_COMPONENTS (total_select (on_list,srt)) =
    print_list print_OP_NAME ", " on_list ^ ": " ^
    print_SORT srt

  | print_COMPONENTS (partial_select (on_list,srt)) = 
    print_list print_OP_NAME ", " on_list ^ ":? " ^
    print_SORT srt

  | print_COMPONENTS (sort_component srt) =
    print_SORT srt

  | print_COMPONENTS (pos_COMPONENTS (_,c)) =
    print_COMPONENTS c


fun print_COMPONENTS_list cl =
    print_list print_COMPONENTS "; " cl

fun print_ALTERNATIVE (total_construct (on,nil)) = 
    print_OP_NAME on

  | print_ALTERNATIVE (total_construct (on,comp_list)) = 
    print_OP_NAME on ^ "(" ^ 
    print_COMPONENTS_list comp_list ^ ")" 
    
  | print_ALTERNATIVE (partial_construct (on,comp_list)) = 
    print_OP_NAME on ^ "(" ^ 
    print_COMPONENTS_list comp_list ^ ")?"

  | print_ALTERNATIVE (subsort [srt]) = 
    "sort " ^ print_SORT srt
    
  | print_ALTERNATIVE (subsort srt_list) =
    "sorts " ^ print_list print_SORT ", " srt_list

  | print_ALTERNATIVE (pos_ALTERNATIVE (_,alt)) = 
    print_ALTERNATIVE alt


fun print_L_ALTERNATIVE (alt,ans) =
    print_ALTERNATIVE alt ^ print_ANNO_list ans

fun print_DATATYPE_DECL (datatype_decl (srt,l_alt_list,ans)) =
    print_SORT srt ^ " ::= " ^ print_ANNO_list ans ^
    print_list print_L_ALTERNATIVE " | " l_alt_list
    
  | print_DATATYPE_DECL (pos_DATATYPE_DECL (_,dd)) = 
    print_DATATYPE_DECL dd

  | print_DATATYPE_DECL (var_DATATYPE_DECL v) = 
    print_var_id v

fun print_DATATYPE_DECL_list ddecls =
    print_list print_DATATYPE_DECL "; " ddecls

fun print_SORT_list srt_list =
    print_list print_SORT ", " srt_list

fun print_SORT_ITEM (sort_decl srt_list) = 
    print_SORT_list srt_list
    
  | print_SORT_ITEM (subsort_decl (srt_list,srt)) =
    print_SORT_list srt_list ^ " < " ^
    print_SORT srt

  | print_SORT_ITEM (subsort_defn (new_srt,v,in_srt,form,ans)) =
    print_SORT new_srt ^ " = " ^ print_ANNO_list ans ^"{" ^
    print_VAR v ^ ": " ^ print_SORT in_srt ^ " . " ^
    infix_FORMULA form ^ "}"

  | print_SORT_ITEM (iso_decl srt_list) =
    print_list print_SORT " = " srt_list

  | print_SORT_ITEM (pos_SORT_ITEM (_,si)) = 
    print_SORT_ITEM si


fun print_L_SORT_ITEM_single (si,ans) =
    print_SORT_ITEM si ^ print_ANNO_list ans


fun print_L_SORT_ITEM_in_list (si,ans) =
    print_SORT_ITEM si ^ "; " ^ print_ANNO_list ans

fun print_ARG_DECL (arg_decl vd) =
    print_VAR_DECL vd

  | print_ARG_DECL (pos_ARG_DECL (_,ad)) =
    print_ARG_DECL ad


fun print_ARG_DECL_list ads =
    print_list print_ARG_DECL "; " ads

fun print_OP_TYPE (total_op_type ((sorts nil),srt)) =
    print_SORT srt

  | print_OP_TYPE (total_op_type (srts,srt)) =
    print_SORTS srts ^ " -> " ^ print_SORT srt

  | print_OP_TYPE (partial_op_type ((sorts nil),srt)) =
    "? " ^ print_SORT srt
    
  | print_OP_TYPE (partial_op_type (srts,srt)) = 
    print_SORTS srts ^ " ->? " ^ print_SORT srt

  | print_OP_TYPE (pos_OP_TYPE (_,ot)) = 
    print_OP_TYPE ot


fun print_OP_HEAD (total_op_head (nil,srt)) =
    ": " ^ print_SORT srt

  | print_OP_HEAD (total_op_head (ads,srt)) = 
    "(" ^ print_ARG_DECL_list ads ^ ") : " ^
    print_SORT srt

  | print_OP_HEAD (partial_op_head (nil,srt)) = 
    ":? " ^ print_SORT srt

  | print_OP_HEAD (partial_op_head (ads,srt)) = 
    "(" ^ print_ARG_DECL_list ads ^ ") :? " ^
    print_SORT srt

  | print_OP_HEAD (pos_OP_HEAD (_,oh)) = 
    print_OP_HEAD oh

    
fun print_OP_ATTR (associative) = "assoc"
  | print_OP_ATTR (commutative) = "comm"
  | print_OP_ATTR (idempotent)  = "idem"
  | print_OP_ATTR (unit_op_attr trm) =
    "unit " ^ print_TERM trm

  | print_OP_ATTR (pos_OP_ATTR (_,oa)) = 
    print_OP_ATTR oa


fun print_OP_ITEM (op_decl (opns,ot,nil)) =
    print_list print_OP_NAME ", " opns ^
    " : " ^ print_OP_TYPE ot

  | print_OP_ITEM (op_decl (opns,ot,oattrs)) = 
    print_list print_OP_NAME ", " opns ^
    ": " ^ print_OP_TYPE ot ^ ", " ^
    print_list print_OP_ATTR ", " oattrs

  | print_OP_ITEM (op_defn (opn,oph,trm,ans)) =
    print_OP_NAME opn ^ print_OP_HEAD oph ^
    " = " ^ print_ANNO_list ans ^
    infix_TERM trm

  | print_OP_ITEM (pos_OP_ITEM (_,oi)) = 
    print_OP_ITEM oi


fun print_L_OP_ITEM_single (oi,ans) =
    print_OP_ITEM oi ^ print_ANNO_list ans

fun print_L_OP_ITEM_in_list (oi,ans) =
    print_OP_ITEM oi ^ "; " ^ print_ANNO_list ans

fun is_mult_l_op (op_decl (nil,_,_),_) =
    false
  | is_mult_l_op (op_decl ([opn],_,_),_) = 
    false
  | is_mult_l_op (op_decl (opns,_,_),_) = 
    true
  | is_mult_l_op (pos_OP_ITEM (_,oi),a) =
    is_mult_l_op (oi,a)
  | is_mult_l_op (_,_) =
    false

fun print_PRED_NAME pn = print_ID pn

fun print_PRED_HEAD (pred_head nil) = ""
    
  | print_PRED_HEAD (pred_head ads) =
    "(" ^ print_ARG_DECL_list ads ^ ")"
    
  | print_PRED_HEAD (pos_PRED_HEAD (_,ph)) = 
    print_PRED_HEAD ph


fun print_PRED_TYPE (pred_type (sorts nil)) =
    "()"

  | print_PRED_TYPE (pred_type srts) =
    print_SORTS srts 

  | print_PRED_TYPE (pos_PRED_TYPE (_,pt)) = 
    print_PRED_TYPE pt


fun print_PRED_ITEM (pred_decl (pns,pt)) =
    print_list print_PRED_NAME ", " pns ^ ": " ^
    print_PRED_TYPE pt
    
  | print_PRED_ITEM (pred_defn (pn,ph,(form,form_ans),ans)) =
    print_PRED_NAME pn ^ print_PRED_HEAD ph ^ " <=> " ^
    print_ANNO_list ans ^ 
    infix_FORMULA form ^ " " ^ print_ANNO_list form_ans

  | print_PRED_ITEM (pos_PRED_ITEM (_,pi)) =
    print_PRED_ITEM pi


fun print_L_PRED_ITEM_single (pi,ans) =
    print_PRED_ITEM pi ^ print_ANNO_list ans

fun print_L_PRED_ITEM_in_list (pi,ans) =
    print_PRED_ITEM pi ^ "; " ^ print_ANNO_list ans

fun print_SIG_ITEMS (sort_items ([lsi],ans)) =
    "sort " ^ print_ANNO_list ans ^
    print_L_SORT_ITEM_single lsi 

  | print_SIG_ITEMS (sort_items (lsi_list,ans)) =
    "sorts " ^ print_ANNO_list ans ^
    print_list print_L_SORT_ITEM_in_list "\n" lsi_list ^ "\n"

  | print_SIG_ITEMS (op_items ([loi],ans)) =
    (if is_mult_l_op loi then 
	 "\nops " 
     else 
	 "\nop ") ^ 
    print_ANNO_list ans ^
    print_L_OP_ITEM_single loi ^ "\n"
    
  | print_SIG_ITEMS (op_items (loi_list,ans)) =
    "\nops " ^ print_ANNO_list ans ^
    print_list print_L_OP_ITEM_in_list "\n" loi_list ^ "\n"

  | print_SIG_ITEMS (pred_items ([lpi],ans)) = 
    "\npred " ^ print_ANNO_list ans ^
    print_L_PRED_ITEM_single lpi ^ "\n"

  | print_SIG_ITEMS (pred_items (lpi_list,ans)) = 
    "\npreds " ^ print_ANNO_list ans ^
    print_list print_L_PRED_ITEM_in_list "\n" lpi_list ^ "\n"

  | print_SIG_ITEMS (datatype_items ([dd],ans)) =
    "\ntype " ^ print_ANNO_list ans ^
    print_DATATYPE_DECL dd ^ "\n"

  | print_SIG_ITEMS (datatype_items (dd_list,ans)) =
    "\ntypes " ^ print_ANNO_list ans ^
    print_DATATYPE_DECL_list dd_list ^ "\n"

  | print_SIG_ITEMS (pos_SIG_ITEMS (_,sis)) =
    print_SIG_ITEMS sis

  | print_SIG_ITEMS (var_SIG_ITEMS v) =
    print_var_id v

fun print_SIG_ITEMS_list sis_list =
    print_list print_SIG_ITEMS " " sis_list

fun print_L_FORMULA_list lf_list =
    print_list infix_L_FORMULA "\n " lf_list

fun print_BASIC_ITEMS (sig_items si) =
    print_SIG_ITEMS si

  | print_BASIC_ITEMS (free_datatype ([ddecl],ans)) = 
    "free type " ^ print_ANNO_list ans ^ 
    print_DATATYPE_DECL ddecl    
    
  | print_BASIC_ITEMS (free_datatype (ddecls,ans)) = 
    "free types " ^ print_ANNO_list ans ^ 
    print_DATATYPE_DECL_list ddecls    

  | print_BASIC_ITEMS (cofree_datatype ([ddecl],ans)) = 
    "cofree type " ^  print_ANNO_list ans ^ 
    print_DATATYPE_DECL ddecl

  | print_BASIC_ITEMS (cofree_datatype (ddecls,ans)) = 
    "cofree types " ^ print_ANNO_list ans ^ 
    print_DATATYPE_DECL_list ddecls

  | print_BASIC_ITEMS (sort_gen ([sis],ans)) = 
    "generated " ^ 
    print_ANNO_list ans ^
    print_SIG_ITEMS sis

  | print_BASIC_ITEMS (sort_gen (sis_list,ans)) = 
    "generated { " ^ 
    print_ANNO_list ans ^
    print_SIG_ITEMS_list sis_list ^
    "\n}"

  | print_BASIC_ITEMS (sort_cogen ([sis],ans)) = 
    "cogenerated " ^ 
    print_ANNO_list ans ^
    print_SIG_ITEMS sis

  | print_BASIC_ITEMS (sort_cogen (sis_list,ans)) =
    "cogenerated {" ^ 
    print_ANNO_list ans ^
    print_SIG_ITEMS_list sis_list ^
    "\n}"

  | print_BASIC_ITEMS (var_items ([vd],ans)) = 
    "var " ^ print_ANNO_list ans ^
    print_VAR_DECL vd 
    
  | print_BASIC_ITEMS (var_items (vd_list,ans)) = 
    "vars " ^ print_ANNO_list ans ^
    print_VAR_DECLList vd_list 

  | print_BASIC_ITEMS (local_var_axioms (vd_list,lf_list,ans)) = 
    "forall " ^ print_ANNO_list ans ^
    print_VAR_DECLList vd_list ^ "\n" ^
    infix_L_FORMULAList lf_list

  | print_BASIC_ITEMS (axiom_items (lf_list,ans)) = 
    "\n " ^ print_ANNO_list ans ^
    print_L_FORMULA_list lf_list

  | print_BASIC_ITEMS  (pos_BASIC_ITEMS (_,bi))=
    print_BASIC_ITEMS bi

  | print_BASIC_ITEMS  (var_BASIC_ITEMS v)=
    print_var_id v

fun print_BASIC_SPEC (basic_spec nil) = 
    "{}"
  | print_BASIC_SPEC (basic_spec bi_list) =  
    print_list print_BASIC_ITEMS "\n" bi_list

  | print_BASIC_SPEC (pos_BASIC_SPEC (_, bs)) = 
    print_BASIC_SPEC bs

  | print_BASIC_SPEC (var_BASIC_SPEC (v)) =
    print_var_id v


fun print_TYPE (op_symb_type ost) =
    print_OP_TYPE ost

  | print_TYPE (pred_symb_type pst) = 
    print_PRED_TYPE pst

  | print_TYPE (pos_TYPE (_,t)) = 
    print_TYPE t


fun print_SPEC_NAME sn = print_SIMPLE_ID sn

fun print_VIEW_NAME vn = print_SIMPLE_ID vn

fun print_SYMB (symb_id sid) = print_ID sid
  
  | print_SYMB (pos_SYMB (_, sym)) = 
    print_SYMB sym
  
  | print_SYMB (qual_id (sid, t)) =
    print_ID sid ^ ": " ^ print_TYPE t


fun print_SYMB_MAP (symb_map (s1,s2)) =
    print_SYMB s1 ^ " |-> " ^ print_SYMB s2
    
  | print_SYMB_MAP (pos_SYMB_MAP (_,sm)) = 
    print_SYMB_MAP sm
    

fun print_SYMB_OR_MAP (symb s) =
    print_SYMB s
    
  | print_SYMB_OR_MAP (symb_or_map sm) = 
    print_SYMB_MAP sm

  | print_SYMB_OR_MAP (pos_SYMB_OR_MAP (_,som)) = 
    print_SYMB_OR_MAP som

 
fun print_SYMB_KIND (implicitk) = ""
  | print_SYMB_KIND (sortsk) = "sort" 
  | print_SYMB_KIND (opsk) = "op"
  | print_SYMB_KIND (predsk) = "pred"
  | print_SYMB_KIND (pos_SYMB_KIND (_,sk)) = print_SYMB_KIND sk


fun print_SYMB_MAP_ITEMS (symb_map_items (sk,[som])) =
    print_SYMB_KIND sk ^ " " ^
    print_SYMB_OR_MAP som

  | print_SYMB_MAP_ITEMS (symb_map_items ((implicitk),sym_map_list)) = 
    print_list print_SYMB_OR_MAP ", " sym_map_list

  | print_SYMB_MAP_ITEMS (symb_map_items (sk,sym_map_list)) =
    print_SYMB_KIND sk ^ "s " ^
    print_list print_SYMB_OR_MAP ", " sym_map_list

  | print_SYMB_MAP_ITEMS (pos_SYMB_MAP_ITEMS (_,smis)) =
    print_SYMB_MAP_ITEMS smis

  | print_SYMB_MAP_ITEMS (var_SYMB_MAP_ITEMS (v)) =
    print_var_id v


fun print_SYMB_ITEMS (symb_items(sk,[sym])) =
    print_SYMB_KIND sk ^ " " ^
    print_SYMB sym
  | print_SYMB_ITEMS (symb_items(sk,symb_list)) =
    print_SYMB_KIND sk ^ "s " ^
    print_list print_SYMB ", " symb_list 
    
  | print_SYMB_ITEMS (pos_SYMB_ITEMS (_,sis)) = 
    print_SYMB_ITEMS sis

  | print_SYMB_ITEMS (var_SYMB_ITEMS (v)) = 
    print_var_id (v)

fun print_SYMB_MAP_ITEMS_list smis  = 
    print_list print_SYMB_MAP_ITEMS ", " smis

fun print_RENAMING (renaming smi_list) = 
    print_list print_SYMB_MAP_ITEMS ", " smi_list

  | print_RENAMING (pos_RENAMING(_,renam)) =
    print_RENAMING renam

  | print_RENAMING (var_RENAMING(v)) =
    print_var_id v

fun print_RESTRICTION_and_ANNO_list ((hide si_list),ans) =
    "\nhide " ^ print_ANNO_list ans ^
    print_list print_SYMB_ITEMS ", " si_list

  | print_RESTRICTION_and_ANNO_list ((reveal smi_list),ans) = 
    "\nreveal " ^ print_ANNO_list ans ^ 
    print_list print_SYMB_MAP_ITEMS ", " smi_list

  | print_RESTRICTION_and_ANNO_list ((pos_RESTRICTION (_, res)),ans) =
    print_RESTRICTION_and_ANNO_list (res,ans)

  | print_RESTRICTION_and_ANNO_list ( var_RESTRICTION(v),ans) = 
    "\nrestriction " ^ print_ANNO_list ans ^ 
     print_var_id(v)

fun print_RESTRICTION res = print_RESTRICTION_and_ANNO_list (res,nil)

(*fun is_spec_inst (spec_inst (_,_),_) = true

  | is_spec_inst ((pos_SPEC (_,_,spec)),_) =
    is_spec_inst (spec,nil) (* 2nd parameter is irrelevant 
			       (ANNO list) *)

  | is_spec_inst (_,_) = false
 *)

fun print_FIT_ARG (fit_spec (l_spec,nil)) =
    print_L_SPEC l_spec 

  | print_FIT_ARG (fit_spec (l_spec,smis)) =
    print_L_SPEC l_spec ^ 
    " fit " ^ print_SYMB_MAP_ITEMS_list smis

  | print_FIT_ARG (fit_view (vn,fas)) = 
    " view " ^ print_VIEW_NAME vn ^ " " ^ print_FIT_ARG_list fas

  | print_FIT_ARG (pos_FIT_ARG (_,fa)) = 
    print_FIT_ARG fa

  | print_FIT_ARG (var_FIT_ARG (v)) = 
    print_var_id v

and print_FIT_ARG_list fas =
    print_list (fn(x)=> "[" ^ print_FIT_ARG x ^ "]")
	       " "
	       fas


and print_SPEC (basic spec) = 
    print_BASIC_SPEC spec

  | print_SPEC (translation (l_spec,ren,ans)) =
    print_L_SPEC l_spec ^
    "\nwith " ^ print_ANNO_list ans ^ print_RENAMING ren

  | print_SPEC (reduction (l_spec,restriction,ans)) =
    print_L_SPEC l_spec ^ 
    print_RESTRICTION_and_ANNO_list (restriction,ans)

  | print_SPEC (union_spec union_list) =
    print_list (fn(l_spec,ans)=> 
		  print_ANNO_list ans ^
		  print_L_SPEC l_spec) 
	       " and " 
	       union_list 

  | print_SPEC (extension ext_list) = 
    "{"^
    (
    print_list (fn(l_spec,nil)=>
		  "\n" ^ print_L_SPEC l_spec 

		  
		| (l_spec,ans)=>
		  print_ANNO_list ans ^
		  print_L_SPEC l_spec)
	       "\nthen "
	       ext_list
    ) ^ "}"

  | print_SPEC (free_spec (l_spec,ans)) = 
    "free " ^ print_ANNO_list ans ^
	 print_L_SPEC l_spec

  | print_SPEC (cofree_spec (l_spec,ans)) = 
    "cofree " ^  print_ANNO_list ans ^ 
    print_L_SPEC l_spec

  | print_SPEC (local_spec (l_spec1,ans1,l_spec2,ans2)) =
    "local " ^  print_ANNO_list ans1 ^
    print_L_SPEC l_spec1 ^ 
    "\nwithin " ^ print_ANNO_list ans2 ^ print_L_SPEC l_spec2
     
  | print_SPEC (closed_spec (l_spec,ans)) = 
    "closed " ^ print_ANNO_list ans ^ 
    print_L_SPEC l_spec 
    
  | print_SPEC (spec_inst (sn,fis)) = 
    print_SPEC_NAME sn ^
    print_FIT_ARG_list fis

  | print_SPEC (pos_SPEC (_, brace_flag, spec)) =
    (if brace_flag then "{" else "") ^
    print_SPEC spec ^
    (if brace_flag then "}" else "")

  | print_SPEC (var_SPEC (v)) =
    print_var_id (v)

and print_L_SPEC (spec,ans) =  print_ANNO_list ans ^ print_SPEC spec

fun print_PARAMS (params p) = 
    print_list (fn(l_spec)=>"[" ^ 
			      print_L_SPEC l_spec ^ 
			      "]")
	       " " 
	       p

  | print_PARAMS (pos_PARAMS (_,p)) =
    print_PARAMS(p)
		

fun print_IMPORTS (imports nil) =
    ""

  | print_IMPORTS (imports i) = 
    " given " ^ 
    print_list (print_L_SPEC)
	       ", " 
	       i 

  | print_IMPORTS (pos_IMPORTS (_,i)) = 
    print_IMPORTS i


fun print_GENERICITY (genericity (param,impor)) = 
    print_PARAMS param ^ print_IMPORTS impor

  | print_GENERICITY (pos_GENERICITY(_,gen)) = 
    print_GENERICITY gen

  | print_GENERICITY (var_GENERICITY v) = 
    print_var_id v

fun print_VIEW_TYPE (view_type (spec1,spec2)) = 
    print_L_SPEC spec1 ^ " to " ^ print_L_SPEC spec2

  | print_VIEW_TYPE (pos_VIEW_TYPE (_,vt)) = 
    print_VIEW_TYPE vt

fun print_UNIT_NAME un = print_SIMPLE_ID un

fun print_UNIT_TYPE (unit_type (nil,lspec)) =
    print_L_SPEC lspec  

  | print_UNIT_TYPE (unit_type (lspecs,lspec)) = 
    print_list print_L_SPEC " * " lspecs ^ 
    " -> " ^ 
    print_L_SPEC lspec  

  | print_UNIT_TYPE (pos_UNIT_TYPE (_,ut)) =
    print_UNIT_TYPE ut

fun print_ARCH_SPEC_NAME asn = print_SIMPLE_ID asn

fun is_named_arch_spec (named_arch_spec _) = true
  | is_named_arch_spec (basic_arch_spec (_,_,_)) = false
  | is_named_arch_spec (pos_ARCH_SPEC (_,ars)) =
    is_named_arch_spec ars

fun print_FIT_ARG_UNIT (fit_arg_unit (ut,nil)) =
    print_UNIT_TERM ut  

  | print_FIT_ARG_UNIT (fit_arg_unit (ut,smis)) =
    print_UNIT_TERM ut ^ 
    " fit " ^ print_SYMB_MAP_ITEMS_list smis

  | print_FIT_ARG_UNIT (pos_FIT_ARG_UNIT (_,fau)) = 
    print_FIT_ARG_UNIT fau

and print_FIT_ARG_UNIT_list faus =
    print_list (fn(x)=> "[" ^ print_FIT_ARG_UNIT x ^ "]")  
	       " "
	       faus

and print_UNIT_IMPORTS (unit_imports nil) =
    ""
    
  | print_UNIT_IMPORTS (unit_imports uts) = 
    " given " ^ print_list print_UNIT_TERM ", " uts

  | print_UNIT_IMPORTS (pos_UNIT_IMPORTS (_,uis)) = 
    print_UNIT_IMPORTS uis

and print_UNIT_TERM (unit_translation (ut,rename)) = 
    print_UNIT_TERM ut ^ "\n with " ^ print_RENAMING rename

  | print_UNIT_TERM (unit_reduction (ut,restrict)) =
    print_UNIT_TERM ut ^ print_RESTRICTION restrict

  | print_UNIT_TERM (amalgamation uts) =
    print_list print_UNIT_TERM " and " uts

  | print_UNIT_TERM (local_unit (udfs,ut)) = 
    "local " ^ 
    print_list print_UNIT_DEFN "; " udfs ^
    " within " ^
    print_UNIT_TERM ut

  | print_UNIT_TERM  (unit_appl (un,faus)) =
    print_UNIT_NAME un ^ print_FIT_ARG_UNIT_list faus

  | print_UNIT_TERM (pos_UNIT_TERM (_,brace_flag,ut)) = 
    (* GROUP-UNIT-TERM => look in CASLsign/asgen.sml *) 
    (if brace_flag then " {\n" else "") ^
    print_UNIT_TERM ut ^
    (if brace_flag then "\n}" else "")

and print_UNIT_DECL (unit_decl (un,us,uimp)) = 
    print_UNIT_NAME un ^ ": " ^
    print_UNIT_SPEC us ^ 
    print_UNIT_IMPORTS uimp

  | print_UNIT_DECL (pos_UNIT_DECL (_,ud)) =
    print_UNIT_DECL ud

and print_UNIT_DEFN (unit_defn (un,uexp)) =
    print_UNIT_NAME un ^ " = " ^ print_UNIT_EXPRESSION uexp

  | print_UNIT_DEFN (pos_UNIT_DEFN (_,ud)) = 
    print_UNIT_DEFN ud

and print_UNIT_DECL_DEFN (unit_decl_case udc) = 
    print_UNIT_DECL udc

  | print_UNIT_DECL_DEFN (unit_defn_case udf) =
    print_UNIT_DEFN udf

  | print_UNIT_DECL_DEFN (pos_UNIT_DECL_DEFN (_,udd)) =
    print_UNIT_DECL_DEFN udd

and print_UNIT_SPEC (unit_type_case ut) =
    print_UNIT_TYPE ut

  | print_UNIT_SPEC (spec_name_case spn) =
    print_SPEC_NAME spn

  | print_UNIT_SPEC (arch_spec_case (ars,ans)) =
    (* GROUP-ARCH_SPEC {} wenn kein Name! *)
    if is_named_arch_spec ars then	
	"arch spec " ^(*{*) print_ARCH_SPEC ars (*}*)^ print_ANNO_list ans
    else
	"arch spec {\n" ^ print_ARCH_SPEC ars ^ 
	"\n} " ^ print_ANNO_list ans

  | print_UNIT_SPEC (closed us) =
    "closed " ^ print_UNIT_SPEC us

  | print_UNIT_SPEC (pos_UNIT_SPEC (_,us)) =
    print_UNIT_SPEC us

and print_RESULT_UNIT (result_unit (uexp,ans)) =
    print_UNIT_EXPRESSION uexp ^ print_ANNO_list ans 

  | print_RESULT_UNIT (pos_RESULT_UNIT (_,res_unit)) = 
    print_RESULT_UNIT res_unit

and print_UNIT_EXPRESSION (unit_expression (nil,u_term)) =
    print_UNIT_TERM u_term

  | print_UNIT_EXPRESSION (unit_expression (unit_bindings,u_term)) =
    "lambda " ^
    print_list print_UNIT_BINDING "; " unit_bindings ^
    "." ^
    print_UNIT_TERM u_term

  | print_UNIT_EXPRESSION (pos_UNIT_EXPRESSION (_,uexp)) =
    print_UNIT_EXPRESSION uexp

and print_UNIT_BINDING (unit_binding (un,us)) =
    print_UNIT_NAME un ^ ": " ^ print_UNIT_SPEC us

  | print_UNIT_BINDING (pos_UNIT_BINDING (_,ub)) = 
    print_UNIT_BINDING ub

and print_ARCH_SPEC (basic_arch_spec ([udd_ans],res_unit,ans_as)) =
    "unit " ^  print_ANNO_list ans_as ^
    print_UNIT_DECL_DEFN (#1 udd_ans) ^"; "^
    print_ANNO_list (#2 udd_ans) ^
    "\nresult " ^
    print_RESULT_UNIT res_unit

  | print_ARCH_SPEC (basic_arch_spec (udd_ans_list,res_unit,ans_as)) =
    "units " ^  print_ANNO_list ans_as ^
    (print_list (fn(unit_decl_def,ans_udd)=>
		   print_UNIT_DECL_DEFN unit_decl_def ^"; "^
		   print_ANNO_list ans_udd)
		"\n " 
		udd_ans_list) ^ 
    "\nresult " ^
    print_RESULT_UNIT res_unit 

  | print_ARCH_SPEC (named_arch_spec nas) =
    print_ARCH_SPEC_NAME nas

  | print_ARCH_SPEC (pos_ARCH_SPEC (_,ars)) =
    print_ARCH_SPEC ars


fun print_L_ARCH_SPEC (las,ans) =
    print_ANNO_list ans ^ print_ARCH_SPEC las

fun print_ITEM_NAME item_n = print_SIMPLE_ID item_n 

fun print_ITEM_NAME_OR_MAP (item_name item_n) =
    print_ITEM_NAME item_n
    
  | print_ITEM_NAME_OR_MAP (item_name_map(item_n1,item_n2)) =
    print_ITEM_NAME item_n1 ^ " |-> " ^ print_ITEM_NAME item_n2

  | print_ITEM_NAME_OR_MAP (pos_ITEM_NAME_OR_MAP (_, inom)) =
    print_ITEM_NAME_OR_MAP inom

fun print_ITEM_NAME_OR_MAP_list inoms =
    print_list print_ITEM_NAME_OR_MAP ", " inoms

(* datatype LIB_ITEM = spec_defn of (SPEC_NAME * GENERICITY * L_SPEC * ANNO list)
                    | view_defn of (VIEW_NAME * GENERICITY * VIEW_TYPE * SYMB_MAP_ITEMS list * ANNO list)
                    | arch_spec_defn of (ARCH_SPEC_NAME * L_ARCH_SPEC * ANNO list)
                    | unit_spec_defn of (SPEC_NAME * UNIT_SPEC * ANNO list)
                    | download_items of (LIB_NAME * ITEM_NAME_OR_MAP list * ANNO list)
                    | pos_LIB_ITEM of (region * LIB_ITEM)
*)


fun print_LIB_ID (url s) = s
  | print_LIB_ID (path_name p) = p
  | print_LIB_ID (pos_LIB_ID (_,lid)) =
    print_LIB_ID lid

fun print_VERSION (version versionlist) = 
    print_list (fn(x) => x) "." versionlist
  | print_VERSION (pos_VERSION (_,v)) =
    print_VERSION v

fun print_LIB_NAME (lib lid) = print_LIB_ID lid
  | print_LIB_NAME (versioned_lib (lid,v)) =
    print_LIB_ID lid ^ " version " ^ print_VERSION v
  | print_LIB_NAME (pos_LIB_NAME (_,ln)) =
    print_LIB_NAME ln


fun print_LIB_ITEM (spec_defn (spn,gen,lsp,ans)) =
    "\nspec " ^ print_SPEC_NAME spn ^ " " ^ print_GENERICITY gen ^ " = " ^
    print_ANNO_list ans ^ 
    print_L_SPEC lsp ^
    "\nend\n"

  | print_LIB_ITEM (view_defn (vn,gen,vt,nil,ans)) =
    "view " ^ print_VIEW_NAME vn ^ print_GENERICITY gen ^ " : " ^ 
    print_VIEW_TYPE vt ^
    print_ANNO_list ans ^ 
    "\nend\n"

  | print_LIB_ITEM (view_defn (vn,gen,vt,smis,ans)) =
    "view " ^ print_VIEW_NAME vn ^ print_GENERICITY gen ^ " : " ^ 
    print_VIEW_TYPE vt ^ " = " ^
     print_ANNO_list ans ^ 
    print_SYMB_MAP_ITEMS_list smis ^ "\nend\n"

  | print_LIB_ITEM (arch_spec_defn (arch_spec_name,l_arch_spec,annotations)) =
    "arch spec " ^ print_ARCH_SPEC_NAME arch_spec_name ^ " = " ^
    print_ANNO_list annotations ^ "\n" ^
    print_L_ARCH_SPEC l_arch_spec ^ "\n" ^
    "end\n"

  | print_LIB_ITEM (unit_spec_defn (spec_name,unit_spec,annotations)) =
    "unit spec " ^ print_SPEC_NAME spec_name ^ " =\n" ^
    print_ANNO_list annotations ^
    print_UNIT_SPEC unit_spec ^ "\n" ^
    "end\n"

  | print_LIB_ITEM (download_items (lib_name,item_name_or_maps,annotations)) =
    "from " ^ print_LIB_NAME lib_name ^ "\n" ^
    " get " ^ print_ITEM_NAME_OR_MAP_list item_name_or_maps ^
    "\nend\n" (* annotations are not allowed, 
                 they will be discarded from as_types.sml in the future*)
 
  | print_LIB_ITEM (pos_LIB_ITEM (_,li)) = 
    print_LIB_ITEM li

  | print_LIB_ITEM (var_LIB_ITEM (v)) = 
    print_var_id (v)

fun print_LIB_DEFN (lib_defn (libraryname,libraryitems,annotations)) =
    "library " ^ print_LIB_NAME libraryname ^ "\n" ^
    print_ANNO_list annotations ^
    print_list print_LIB_ITEM "\n" libraryitems
    
  | print_LIB_DEFN (pos_LIB_DEFN (_,lit)) =
    print_LIB_DEFN lit




(************************** Environments *************************)

fun print_symbol (SORT_SYMBOL s) = "sort "^print_ID s
  | print_symbol (TOTAL_FUN_SYMBOL (f,(args,res))) =
    "op "^print_ID f^":"^print_list print_ID " x " args^" -> "^print_ID res
  | print_symbol (PARTIAL_FUN_SYMBOL (f,(args,res))) =
    "op "^print_ID f^":"^print_list print_ID " x " args^" ->? "^print_ID res
  | print_symbol (PRED_SYMBOL (p,args)) =
    "pred "^print_ID p^":"^print_list print_ID " x " args

fun print_symbol_pair (sym1,sym2) =
    "("^print_symbol sym1^" , "^print_symbol sym2^")"
    
fun print_symbol_set (symset) =
    "{ "^print_list print_symbol ",\n  " symset^"\n}"
    
fun print_symbol_pair_set (symset) =
    "{ "^print_list print_symbol_pair ",\n  " symset^"\n}"
    
fun print_one_sort_mapping (s1,s2) =
    print_ID s1^" |-> "^print_ID s2
    
fun print_sort_map (smap:Sort_map) =
    print_list print_one_sort_mapping "," (Symtab_id.dest smap)

fun print_one_op_prof f1 (t,f2,_) =
    print_ID f1^":"^print_OP_TYPE t^"|->"^print_ID f2

fun print_one_op_mapping (f1,profs) =
    print_list (print_one_op_prof f1) ", " profs
    
fun print_op_map (fmap:Fun_map) =
    print_list print_one_op_mapping ", " (Symtab_id.dest fmap)

fun print_one_pred_prof p1 (t,p2) =
    print_ID p1^":"^print_PRED_TYPE t^"|->"^print_ID p2

fun print_one_pred_mapping (p1,profs) =
    print_list (print_one_pred_prof p1) ", " profs

fun print_pred_map (pmap:Pred_map) =
    print_list print_one_pred_mapping ", " (Symtab_id.dest pmap)

    
fun print_morphism (smap,fmap,pmap) =
    (if Symtab_id.is_empty smap then "" 
     else "sorts \n"^print_sort_map smap)^
    (if Symtab_id.is_empty fmap then "" 
     else "\nops \n"^print_op_map fmap)^
    (if Symtab_id.is_empty pmap then "" 
     else "\npreds \n"^print_pred_map pmap)

(*
fun print_global_env1 (sn,(lenv,formulas)) =
    print_SIMPLE_ID sn^"\n"^
    print_local_env lenv^"\n"^
    print_L_FORMULAList formulas^"\n"
*)

fun indent x = x

fun print_spec_env1 (senv,an) =
    print_ANNOS an^"\n"^print_spec_env senv

and print_spec_env (basic_env (lenv,formulas)) =
    print_local_env lenv^"\nAxioms:\n"^print_L_FORMULAList formulas^"\n"
  | print_spec_env (translate_env (senv,mor)) =
    "translate\n"^indent (print_spec_env senv)^
    "along \n"^print_morphism mor^"\n"
  | print_spec_env (derive_env (senv,mor)) =
    "derive\n"^indent (print_spec_env senv)^
    "along \n"^print_morphism mor^"\n"
  | print_spec_env (union_env (senv_list)) =
    "union\n"^indent (print_list print_spec_env "\n" senv_list)
  | print_spec_env (extension_env (senv_list)) =
    "extension\n"^indent (print_list print_spec_env1 "\nthen " senv_list)
  | print_spec_env (free_spec_env senv) =
    "free\n"^indent (print_spec_env senv)
  | print_spec_env (cofree_spec_env senv) =
    "cofree\n"^indent (print_spec_env senv)
  | print_spec_env (closed_spec_env senv) =
    "closed\n"^indent (print_spec_env senv)
  | print_spec_env (spec_inst_env (name,senv,mor,senv_list)) =
    "instantiate "^print_SIMPLE_ID name^"\n"^
     indent (print_spec_env senv)^
    " with\n"^
     indent (print_list print_spec_env "\n" senv_list)
  | print_spec_env dummy_spec_env =
    "dummy_env\n"
(*
and print_spec_lenv1 (SPEC_ENV (lenv,senv,_)) =
    print_spec_env senv
*)
    

fun output_spec_lenv h (SPEC_ENV (Sigma,hSigma,senv)) =
    (* (output(h,"Flattened local environment:\n");
     output_sign h Sigma;
     output(h,"Hidden local environment:\n");
     output_sign h hSigma 
    ) *)
    output(h,indent (print_spec_env senv))


fun print_spec_lenv (SPEC_ENV (Sigma,hSigma,senv)) =
    ("Flattened local environment:\n"
     ^print_sign Sigma
     ^"Hidden local environment:\n"
     ^print_sign hSigma 
    )
   
    
fun output_global_entry h (spec_defn_env ((imp,pars,_),sp)) =
    (output(h,"Imports:\n");
     output_spec_lenv h imp;
     output(h,"\nParameters:\n");
     do_print_list h output_spec_lenv "\n" pars;
     output(h,"\nBody:\n");
     output_spec_lenv h sp;
     output(h,"\n")
    )
  | output_global_entry h _ = ()

fun print_global_entry (spec_defn_env ((imp,pars,_),sp)) =
    ("Imports:\n"
     ^print_spec_lenv imp
     ^"\nParameters:\n"
     ^(print_list print_spec_lenv "\n" pars)
     ^"\nBody:\n"
     ^print_spec_lenv sp
     ^"\n")
  | print_global_entry _ = ""

fun output_ext_sig h (Sigma,HSigma,Ax) =
    ((* print "Signature\n"; *)
     output_sign h Sigma;
    (* print "Hidden signature\n";
     output_sign h HSigma; *)
     output(h,print_L_FORMULAList Ax) )
     
fun print_ext_sig (Sigma,HSigma,Ax) =
    (* print "Signature\n"; *)
     print_sign Sigma
    (* print "Hidden signature\n";
     print_sign h HSigma; *)
     ^print_L_FORMULAList Ax
     
         
fun output_f_global_entry h (name,f_spec_defn_env ((imp,pars),sp)) =
    (output(h,"spec "^print_SIMPLE_ID name);
     if null pars then () else output(h," %% [...] %% ");
    (* if imp=empty_ext_signature then () else output(h," given ...");*)
     output(h," =\n");
     (* output(h,"Imports:\n");
     print_ext_sig h imp;
     output(h,"\nParameters:\n");
     do_print_list h print_ext_sig "\n" pars;
     output(h,"\nBody:\n"); *)
     output_ext_sig h sp;
     output(h,"\nend\n")
    )
  | output_f_global_entry h (name,f_view_defn_env(_,_,mor,_)) = 
   (output(h,"%[ view "^print_SIMPLE_ID name^" =\n");
    output(h,print_morphism mor); 
    output(h,"\nend ]%\n")
    )

  | output_f_global_entry h (name,f_arch_spec_defn_env) = 
    (output(h,"%[ arch spec "^print_SIMPLE_ID name);
     output(h,"\nend ]%\n")
    )
  | output_f_global_entry h (name,f_unit_spec_defn_env) = 
    (output(h,"%[ unit spec "^print_SIMPLE_ID name);
     output(h,"\nend ]%\n")
    )


fun print_f_global_entry  (name,f_spec_defn_env ((imp,pars),sp)) =
    ("spec "^print_SIMPLE_ID name
     ^(if null pars then "" else " %% [...] %% ")
    (* if imp=empty_ext_signature then () else output(h," given ...");*)
     ^" =\n"
     ^print_ext_sig sp
     ^"\nend\n"
    )
  | print_f_global_entry  (name,f_view_defn_env(_,_,mor,_)) = 
   ("%[ view "^print_SIMPLE_ID name^" =\n"
    ^print_morphism mor
    ^"\nend ]%\n"
    )

  | print_f_global_entry (name,f_arch_spec_defn_env) = 
    ("%[ arch spec "^print_SIMPLE_ID name
     ^"\nend ]%\n"
    )
  | print_f_global_entry (name,f_unit_spec_defn_env) = 
    ("%[ unit spec "^print_SIMPLE_ID name
     ^"\nend ]%\n"
    )

fun output_global_entry1 h (name,entry) = 
    (output(h,print_SIMPLE_ID name^":\n");
     output_global_entry h entry
    )

fun print_global_entry1 (name,entry) = 
    (print_SIMPLE_ID name^":\n"
     ^print_global_entry entry
    )
  
  

fun print_LIB_ID (url u) = u
  | print_LIB_ID (path_name p) = p
  | print_LIB_ID (pos_LIB_ID (_,lid)) =
    print_LIB_ID lid

fun print_VERSION (version vl) = 
    print_list (fn x=>x) "." vl
  | print_VERSION (pos_VERSION (_,v)) =
    print_VERSION v
  
fun print_LIB_NAME (lib lid) = print_LIB_ID lid
  | print_LIB_NAME (versioned_lib (lid,v)) = 
    print_LIB_ID lid^" version "^print_VERSION v 
  | print_LIB_NAME (pos_LIB_NAME (_,ln)) =
    print_LIB_NAME ln

fun latex_ext_sig h (Sigma,HSigma,Ax) =
    (latex_sign h Sigma;
     latex_sign h HSigma (*;
     output(h,print_L_FORMULAList Ax) !!!*))

fun latex_spec_lenv h (SPEC_ENV (Sigma,_,_)) =
    latex_sign h Sigma

fun latex_global_entry h (spec_defn_env ((imps,pars,_),sp)) =
    (latex_spec_lenv h sp; output(h,"\n"))
  | latex_global_entry h _ = ()

fun latex_f_global_entry h (f_spec_defn_env ((imps,pars),sp)) =
    (latex_ext_sig h sp; output(h,"\n"))
  | latex_f_global_entry h _ = ()

fun get_name (spec_defn (name,_,_,_)) = Some name
(*  | get_name (view_defn (name,_,_,_,_)) = Some name
  | get_name (arch_spec_defn (name,_,_)) = Some name
  | get_name (unit_spec_defn (name,_,_)) = Some name  *)
  | get_name _ = None
  
fun latex_global_entry1 e h lib_item = 
    case get_name lib_item of
    None => ()
    | Some name =>
       case Symtab_sid.lookup (e,name) of
       Some g_entry =>
        (output(h,"{\\Large \\SpecName{"^latex_SIMPLE_ID name^"}:}\n\n");
         latex_global_entry h g_entry
         )
       | None => ()
  
fun latex_f_global_entry1 e h lib_item = 
    case get_name lib_item of
    None => ()
    | Some name =>
       case Symtab_sid.lookup (e,name) of
       Some g_entry =>
        (output(h,"{\\Large \\SpecName{"^latex_SIMPLE_ID name^"}:}\n\n");
         latex_f_global_entry h g_entry
         )
       | None => ()

fun latex_global_env1 h e (AS.lib_defn (name,lib_items,_)) = 
    (output(h,"\\subsection{Library "^print_LIB_NAME name^"}\n\n");
     do_print_list h (latex_global_entry1 e) "\n" lib_items
    )
  | latex_global_env1 h e (AS.pos_LIB_DEFN (_,ld)) =
    latex_global_env1 h e ld
    
fun latex_f_global_env1 h e (AS.lib_defn (name,lib_items,_)) = 
    (output(h,"\\subsection{Library "^print_LIB_NAME name^"}\n\n");
     do_print_list h (latex_f_global_entry1 e) "\n" lib_items
    )
  | latex_f_global_env1 h e (AS.pos_LIB_DEFN (_,ld)) =
    latex_f_global_env1 h e ld


fun output_global_env filename (e,ans) tree = 
    let val h = openOut filename
    in 
    (do_print_list h output_global_entry1 "\n" (Symtab_sid.dest e);
     closeOut h
    )
    end

fun print_global_env  (e,ans) tree = 
    print_list print_global_entry1 "\n" (Symtab_sid.dest e);
   
fun output_f_global_env filename (e,ans) tree = 
    let val h = openOut filename
    in 
    (do_print_list h output_f_global_entry "\n" (Symtab_sid.dest e);
     closeOut h
    )
    end

fun print_f_global_env  (e,ans) tree = 
    print_list print_f_global_entry "\n" (Symtab_sid.dest e);
 
fun latex_global_env  filename (e,ans) tree = 
    let val h = openOut filename
    in 
    (
     output(h,   
       "\n\n\n*** LaTeX version of global environment (reduced information) ***\n\n");
     latex_global_env1 h e tree;
     closeOut h
    )
    end
    
fun latex_f_global_env  filename (e,ans) tree = 
    let val h = openOut filename
    in 
    (
     output(h,   
       "\n\n\n*** LaTeX version of global environment (reduced information) ***\n\n");
     latex_f_global_env1 h e tree;
     closeOut h
    )
    end


end
