(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author:  Bartek Klin                                          	   *)
(* Date: September 2000				 			   *)
(* Purpose of this file: Datatypes of symbols and symbol maps              *)
(*                       + overloading relations                           *)
(*			 						   *)	
(* *********************************************************************** *)


structure Symbols = struct 

local open Utils AS LocalEnv in

exception STAT_EXCEPTION of string;

infix mem;

datatype symbol = 
    SORT_SYMBOL of AS.SORT
  | TOTAL_FUN_SYMBOL of (AS.OP_NAME * (AS.SORT list * AS.SORT))
  | PARTIAL_FUN_SYMBOL of (AS.OP_NAME * (AS.SORT list * AS.SORT))
  | PRED_SYMBOL of (AS.PRED_NAME * (AS.SORT list))
;

datatype raw_symbol = 
    SYMBOL_RAW_SYMBOL of symbol
  | SORT_RAW_SYMBOL of AS.SORT
  | FUN_RAW_SYMBOL of AS.OP_NAME
  | PRED_RAW_SYMBOL of AS.PRED_NAME
  | IMPLICIT_RAW_SYMBOL of AS.ID
;

type symbol_map = (symbol * symbol) list;
type raw_symbol_map = (raw_symbol * raw_symbol) list;

fun symbol_name(sym) = case sym of
    SORT_SYMBOL(sn) => sn
  | TOTAL_FUN_SYMBOL(fnm,_) => fnm
  | PARTIAL_FUN_SYMBOL(fnm,_) => fnm
  | PRED_SYMBOL(pn,_) => pn
;

fun raw_symbol_name(rsym) = case rsym of
    SYMBOL_RAW_SYMBOL(sym) => symbol_name(sym)
  | SORT_RAW_SYMBOL(sn) => sn
  | FUN_RAW_SYMBOL(fnm) => fnm
  | PRED_RAW_SYMBOL(pn) => pn
  | IMPLICIT_RAW_SYMBOL(idn) => idn
;

fun sig_symbols((ssenv,_,fenv,penv):LocalEnv.local_env):symbol list =
  let val senvl = Symtab_id.dest(ssenv)
  val fenvl = Symtab_id.dest(fenv)
  val penvl = Symtab_id.dest(penv)
  val ssyms =
    foldl (fn (ssymsi,(s,_)) => SORT_SYMBOL(s)::ssymsi)
          ([],senvl)
  val fsyms =
    foldl (fn (fsymsi,(on,otl)) =>
      foldl (fn (fsymsii,ot) => case ot of
          total_op_type(sorts(ss),s) =>
            TOTAL_FUN_SYMBOL(on,(ss,s))::fsymsii
        | partial_op_type(sorts(ss),s) =>
            PARTIAL_FUN_SYMBOL(on,(ss,s))::fsymsii)
        (fsymsi,otl))
      (ssyms,fenvl)
  val psyms =
    foldl (fn (psymsi,(pn,ptl)) =>
      foldl (fn (psymsii,pred_type(sorts(pt))) =>
        PRED_SYMBOL(pn,pt)::psymsii)
        (psymsi,ptl))
      (fsyms,penvl)
  in psyms
  end
;

(* Symbol matching relation.*)
(* symbol * raw_symbol -> boolean *)
(* p. 83 *)
fun matches (s,rs) = case rs of
    SYMBOL_RAW_SYMBOL(s1) => s=s1
  | SORT_RAW_SYMBOL(sn) => (case s of
      SORT_SYMBOL(sn1) => sn=sn1
    | _ => false)
  | FUN_RAW_SYMBOL(fnm) => (case s of
      TOTAL_FUN_SYMBOL(fn1,_) => fnm=fn1
    | PARTIAL_FUN_SYMBOL(fn1,_) => fnm=fn1
    | _ => false)
  | PRED_RAW_SYMBOL(pn) => (case s of
      PRED_SYMBOL(pn1,_) => pn=pn1
    | _ => false)
  | IMPLICIT_RAW_SYMBOL(id) => (case s of
      SORT_SYMBOL(sn) => sn=id
    | TOTAL_FUN_SYMBOL(fnm,_) => fnm=id
    | PARTIAL_FUN_SYMBOL(fnm,_) => fnm=id
    | PRED_SYMBOL(pn,_) => pn=id)
;

(*************** Overloading relations ****************)

(* intersection; should be moved to Utils *)
fun intersect(l1,l2) = filter (fn x => x mem l1) l2;

(* LocalEnv.local_env * sort * sort -> bool *)
fun sort_is_leq((ssenv,_,_,_),s1,s2) = 
  s1 mem (if_none (Symtab_id.lookup(ssenv,s2)) [])
;

(* LocalEnv.local_env * sort * sort -> sort list *)
fun common_subsorts((ssenv,_,_,_),s1,s2) = 
  intersect(if_none (Symtab_id.lookup(ssenv,s1)) [],
            if_none (Symtab_id.lookup(ssenv,s2)) [])
;

(* LocalEnv.local_env * sort * sort -> sort list *)
fun common_supersorts((ssenv,_,_,_),s1,s2) = 
  map fst (filter (fn (_,sl) => (s1 mem sl) andalso (s2 mem sl))
                  (Symtab_id.dest(ssenv)))
;

(* LocalEnv.local_env * sort * sort -> bool *)
fun have_common_subsorts(lenv,s1,s2) = 
  not(common_subsorts(lenv,s1,s2)=[])
;

(* LocalEnv.local_env * sort * sort -> bool *)
fun have_common_supersorts((ssenv,_,_,_),s1,s2) = 
  exists' (fn (_,sl) => (s1 mem sl) andalso (s2 mem sl))
          (Symtab_id.dest(ssenv))
;

(* LocalEnv.local_env * sort list * sort list -> bool *)
fun have_common_subseqs(lenv,l1,l2) = 
  forall' (fn (s1,s2) => have_common_subsorts(lenv,s1,s2)) 
          (zip(l1,l2))
;

(* LocalEnv.local_env * symbol * symbol -> bool *)
fun symbols_overload(lenv,sym1,sym2) =
  case (sym1,sym2) of
    (TOTAL_FUN_SYMBOL(fn1,(sseq1,s1)),TOTAL_FUN_SYMBOL(fn2,(sseq2,s2))) =>
       (fn1=fn2) andalso (length sseq1 = length sseq2) andalso
       (have_common_subseqs(lenv,sseq1,sseq2)) andalso
       (have_common_supersorts(lenv,s1,s2))
  | (TOTAL_FUN_SYMBOL(fn1,(sseq1,s1)),PARTIAL_FUN_SYMBOL(fn2,(sseq2,s2))) =>
       (fn1=fn2) andalso (length sseq1 = length sseq2) andalso
       (have_common_subseqs(lenv,sseq1,sseq2)) andalso
       (have_common_supersorts(lenv,s1,s2))
  | (PARTIAL_FUN_SYMBOL(fn1,(sseq1,s1)),TOTAL_FUN_SYMBOL(fn2,(sseq2,s2))) =>
       (fn1=fn2) andalso (length sseq1 = length sseq2) andalso
       (have_common_subseqs(lenv,sseq1,sseq2)) andalso
       (have_common_supersorts(lenv,s1,s2))
  | (PARTIAL_FUN_SYMBOL(fn1,(sseq1,s1)),PARTIAL_FUN_SYMBOL(fn2,(sseq2,s2))) =>
       (fn1=fn2) andalso (length sseq1 = length sseq2) andalso
       (have_common_subseqs(lenv,sseq1,sseq2)) andalso
       (have_common_supersorts(lenv,s1,s2))
  | (PRED_SYMBOL(fn1,sseq1),PRED_SYMBOL(fn2,sseq2)) =>
       (fn1=fn2) andalso (length sseq1 = length sseq2) andalso
       (have_common_subseqs(lenv,sseq1,sseq2))
  | _ => false
;

(* LocalEnv.local_env -> (symbol * symbol list) list *)
fun overloading_relation(lenv) =
  let val sigsyms = sig_symbols lenv in 
    map (fn sym => (sym, filter (fn sym2 => symbols_overload(lenv,sym,sym2)) 
                                sigsyms))
        sigsyms
  end
;  

end

end
