(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author:  Bartek Klin                                          	   *)
(* Date: 19.03.2000				 			   *)
(* Purpose of this file: Static semantics of signature morphisms           *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* This module contains the static analysis of signature morphisms,
   following the CASL semantics July 99, study note S-9,
   to which the page numbers refer.
*)

structure Symmaps_analysis : sig
  val stat_renaming:   (* used: p. 97 *)
    (LocalEnv.sign * AS.ANNO list) * AS.RENAMING * string list ->
    GlobalEnv.morphism * (LocalEnv.sign * AS.ANNO list) * string list;
  val stat_restriction: (* used: p. 99 *)
    (LocalEnv.sign * AS.ANNO list) * (LocalEnv.sign * AS.ANNO list) * AS.RESTRICTION * string list ->
    (LocalEnv.sign * AS.ANNO list) * GlobalEnv.morphism * (LocalEnv.sign * AS.ANNO list) * string list;
  val stat_symb_map_items_star: (* used: p. 113, 116 *)
    Symbols.symbol list * AS.SYMB_MAP_ITEMS list * string list ->
    Symbols.raw_symbol_map * string list;
end

(* Comment on using the above functions: 

  (see p. 97)
  stat_renaming(lenv,ren,errs) = (mor,lenv',errs'), where:
     lenv = source signature of renaming morphism
     ren = renaming, taken from AS tree
     errs = given list of errors
     mor = morphism induced from renaming
     lenv' = target signature of renaming morphism
     errs' = new list of errors.

  (see p. 99)
  stat_restriction(lenv1,lenv2,restr,errs) = (lenv3,mor,lenv4,errs'), where:
     lenv1 = local environment (which cannot be affected by the restriction)
     lenv2 = source signature of the restriction (includes lenv1)
     restr = restriction, taken from AS tree
     errs = given list of errors
     lenv3 = restricted signature before renaming
     mor = renaming morphism (from lenv3 to lenv4)
     lenv4 = restricted signature after renaming
     errs' = new list of errors.

   (see p. 125)
   stat_symb_map_items_star(syms,smis,errs) = (rmap,errs'), where:
     syms = symbols to be renamed (to get it from a source signature,
            use function Stat_symmaps.sig_symbols)
     smis = symb_map_items*, taken from AS tree
     errs = given list of errors
     rmap = computed raw symbol map
     errs' = new list of errors

*)

= struct

open Utils AS Symbols;
infix mem;


fun intersect([],_) = []
  | intersect(h::t,l) = 
     if h mem l then (h::intersect(t,l)) else (intersect(t,l))
;

(* SYM_KIND * AS.SYMBOL -> raw_symbol *)
(* p. 121-122 *)
fun stat_symb(symkind,sym,errs) =
  case symkind of
    implicitk => 
      (case sym of
        symb_id(ident) => (IMPLICIT_RAW_SYMBOL(ident),errs)
      | qual_id(f,op_symb_type(total_op_type(sorts(ss),s))) =>
          (SYMBOL_RAW_SYMBOL(TOTAL_FUN_SYMBOL(f,(ss,s))),errs)
      | qual_id(f,op_symb_type(partial_op_type(sorts(ss),s))) =>
          (SYMBOL_RAW_SYMBOL(PARTIAL_FUN_SYMBOL(f,(ss,s))),errs)
      | qual_id(p,pred_symb_type(pred_type(sorts(ss)))) =>
          (SYMBOL_RAW_SYMBOL(PRED_SYMBOL(p,ss)),errs))
  | sortsk =>
      (case sym of 
        symb_id(s) => (SYMBOL_RAW_SYMBOL(SORT_SYMBOL(s)),errs)
      | _ => raise (STAT_EXCEPTION "")) 
  | opsk =>
      (case sym of
        symb_id(f) => (FUN_RAW_SYMBOL(f),errs)
      | qual_id(f,op_symb_type(total_op_type(sorts(ss),s))) =>
          (SYMBOL_RAW_SYMBOL(TOTAL_FUN_SYMBOL(f,(ss,s))),errs)
      | qual_id(f,op_symb_type(partial_op_type(sorts(ss),s))) =>
          (SYMBOL_RAW_SYMBOL(PARTIAL_FUN_SYMBOL(f,(ss,s))),errs)
      | _ => raise (STAT_EXCEPTION ""))
  | predsk =>
      (case sym of
        symb_id(p) => (PRED_RAW_SYMBOL(p),errs)
      | qual_id(p,pred_symb_type(pred_type(sorts(ss)))) =>
          (SYMBOL_RAW_SYMBOL(PRED_SYMBOL(p,ss)),errs)
      | _ => raise (STAT_EXCEPTION ""))
  | pos_SYMB_KIND(_,k) => stat_symb(k,sym,errs)
;

(* symb_items -> raw_symbol list *)
(* p. 122 *)
fun stat_symb_items(symb_items(symk,syml),errs) =  
  (foldl (fn ((rsli,errsi),sym) =>
    let val (rs,errsi1) = stat_symb(symk,remove_pos_SYMB sym,errsi) in
      ((rs::rsli),errsi1)
    end)
    (([],errs),syml)
  handle (STAT_EXCEPTION s) => ([],("Bad symbol list: "^s)::errs) )
  | stat_symb_items(pos_SYMB_ITEMS(_,si),errs) =
    stat_symb_items(si,errs)
;

(* symb_items list -> raw_symbol list *)
(* p. 122 *)
fun stat_symb_items_raw(symil,errs) = 
  foldl (fn ((rsli,errsi),symi) => 
    let val (rsli1,errsi1) = stat_symb_items(symi,errsi) in
    (remove_dups(rsli@rsli1),errsi1) end)
   (([],errs),symil)
;

(* symbol list * symb_items list -> symbol list *)
(* p. 123 *)
fun stat_symb_items_sym(sl,symil,errs) =
  let val (rsys,errs1) = stat_symb_items_raw(symil,errs) in
    (filter (fn sym => 
       (exists' (fn rsym => Stat_symmaps.matches(sym,rsym)) rsys)) sl,
     errs1)
  end
;

(* sym_kind * symb_or_map -> raw_symbol_map *)
(* p. 123-124 *)
fun stat_symb_or_map(sk,som,errs) = case som of
    symb(sym) => 
      (let val (rsy,errs1) = (stat_symb(sk,remove_pos_SYMB sym,errs))
      in ([(rsy,rsy)],errs1) 
      end
      handle (STAT_EXCEPTION s) => ([],("Bad symbol map"^s)::errs))
  | symb_or_map(symb_map(sym1,sym2)) =>
      (let val (rsy1,errs1) = (stat_symb(sk,remove_pos_SYMB sym1,errs))
      val (rsy2,errs2) = (stat_symb(sk,remove_pos_SYMB sym2,errs1))
      in ([(rsy1,rsy2)],errs2)
      end
      handle (STAT_EXCEPTION s) => ([],("Bad symbol map"^s)::errs))
  | pos_SYMB_OR_MAP(_,som1) => stat_symb_or_map(sk,som1,errs)
;

(* symb_map_items -> raw_symbol_map *)
(* p. 124 *)
fun stat_symb_map_items(symb_map_items(sk,soml),errs) =
  foldl (fn ((rsmi,errsi),som) =>
    let val (rsmi1,errsi1) = stat_symb_or_map(sk,som,errsi) in
      (remove_dups(rsmi@rsmi1),errsi1)
    end)
   (([],errs),soml)
  | stat_symb_map_items(pos_SYMB_MAP_ITEMS(_,si),errs) =
    stat_symb_map_items(si,errs)
;

(* symb_map_items list -> raw_symbol_map *)
(* p. 124 *)
fun stat_symb_map_items_raw(smisl,errs) =
  foldl (fn ((rsli,errsi),smis) =>
    let val (rsli1,errsi1) = stat_symb_map_items(smis,errsi) in
      (remove_dups(rsli@rsli1),errsi1) 
    end)
   (([],errs),smisl)
;

(* symbol list * symb_map_items list -> (symbol list * raw_symbol_map) *)
(* p. 125 *)
fun stat_symb_map_items_plus(syms,smisl,errs) =
  let val (r,errs1) = stat_symb_map_items_raw(smisl,errs) 
  val domr = map (fn (x,y) => x) r in
  if forall' (fn rs => (exists'(fn s => Stat_symmaps.matches(s,rs)) syms)) domr 
  then 
    let val syms1 = 
      filter (fn s => (exists'(fn rs => Stat_symmaps.matches(s,rs)) domr)) syms
    in (syms1,r,errs1)
    end 
  else
    ([],[],"Mapping inconsistent with source signature"::errs1)
  end
;

(* symbol list * symb_map_items list -> raw_symbol_map *)
(* p. 125 *)
fun stat_symb_map_items_star(syms,smisl,errs) =
  let val (_,r,errs1) = stat_symb_map_items_plus(syms,smisl,errs) in 
    (r,errs1)
  end
;

(* LocalEnv.sign * renaming -> GlobalEnv.morphism * LocalEnv.sign *)
(* p. 97 *)
fun stat_renaming((sigma,an),renaming(smisl),errs) =
  let val (syms,r,errs1) = 
    stat_symb_map_items_plus(Stat_symmaps.sig_symbols(sigma),smisl,errs) in
   (let val (sigma1,an1,mor) = Stat_symmaps.induced_from_morphism(r,sigma,an) 
     in (mor,(sigma1,an1),errs1)
   end)
   handle (STAT_EXCEPTION s) => (Stat_symmaps.inclusion_morphism(sigma),(sigma,an),
                            ("Renaming not uniquely defined: "^s)::errs1)
  end 
  | stat_renaming(san,pos_RENAMING(_,ren),errs) =
    stat_renaming(san,ren,errs)
;

(* LocalEnv.sign * LocalEnv.sign * restriction ->
   LocalEnv.sign * GlobalEnv.morphism * LocalEnv.sign *)
(* p. 99 *)
fun stat_restriction((sigma,an),(sigma1,an1),restr,errs) =
  case restr of
    hide(sil) =>
      let val (syms,errs1) = 
        stat_symb_items_sym(Stat_symmaps.sig_symbols(sigma1),sil,errs) in
        let val (sigma_1,an_1,_) = 
          Stat_symmaps.cogenerated_signat(syms,sigma1,an1) in
        if intersect(syms,Stat_symmaps.sig_symbols(sigma))=[] then
          ((sigma_1,an_1),Stat_symmaps.inclusion_morphism(sigma_1),(sigma_1,an_1),errs1)
        else
          ((sigma_1,an_1),Stat_symmaps.inclusion_morphism(sigma_1),(sigma_1,an_1),
           "Hiding must not affect local environment"::errs1)
        end
        handle (STAT_EXCEPTION s) => 
          ((sigma1,an1),Stat_symmaps.inclusion_morphism(sigma1),(sigma1,an1),
           s::errs1) 
      end      
  | reveal(smil) =>
      let val (syms,r,errs1) = 
        stat_symb_map_items_plus(Stat_symmaps.sig_symbols(sigma1),smil,errs)
      val syms1 = syms@(Stat_symmaps.sig_symbols(sigma)) in
        let val (sigma_1,an_1,_) = Stat_symmaps.generated_signat
                                   (syms1,sigma1,an1) in
        if intersect(syms,Stat_symmaps.sig_symbols(sigma))=[] then
            let val (sigma2,an2,mor) = 
                Stat_symmaps.induced_from_morphism(r,sigma_1,an_1) in
              ((sigma_1,an_1),mor,(sigma2,an2),errs1)
            end
            handle (STAT_EXCEPTION s) => 
              ((sigma_1,an_1),Stat_symmaps.inclusion_morphism(sigma_1),(sigma_1,an_1),
               ("Renaming while revealing not uniquely defined:"^s)::errs1) 
        else
          ((sigma_1,an_1),Stat_symmaps.inclusion_morphism(sigma_1),(sigma_1,an_1),
          (*"Revealing must not affect local environment"::*)errs1)
        end
        handle (STAT_EXCEPTION s) =>
          ((sigma1,an1),Stat_symmaps.inclusion_morphism(sigma1),(sigma1,an1),
           s::errs1)
      end
   | pos_RESTRICTION(_,restr1) => 
     stat_restriction((sigma,an),(sigma1,an1),restr1,errs)
;  

end; 
