(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author:  Bartek Klin                                          	   *)
(* Date: 19.03.2000				 			   *)
(* Purpose of this file: Sharing analysis for architectural specifications *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* The static analysis of architectural specifications needs a
   sharing analysis for static correctness of amalgamation and
   model translation.
   The sharing analysis follows the algorithms described in

   ??? Insert reference to MFCS paper here !
 *)

structure SharingAnalysis = struct

exception INTERNAL_ERROR;

local open Utils AS LocalEnv Morphisms Symbols ArchTypes in
infix mem;

(* Translation between two representations of overloading relation.
   To be deleted afterwards. *)
(* ('a * 'a list) list -> ('a * 'a) list *)
fun transl_overloading(symrel) = 
  flat (map (fn (sym,syml) => (map (fn sym2 => (sym,sym2)) syml)) symrel)
; 

fun intersect(l1,l2) =
  filter (fn x => (x mem l1) andalso (x mem l2)) (remove_dups(l1@l2))
;

fun remove_list(l1,[]) = l1
  | remove_list(l1,h::t) = remove h (remove_list(l1,t))
;

fun is_subset([],_) = true
  | is_subset(h::t,l) = h mem l andalso is_subset(t,l)
;

(* ('a * 'a -> bool) -> 'a list -> bool *)
(* For each two undordered pair (the function should be symmetric) *)
fun forall_two func [] = true
  | forall_two func (h::t) = 
      (forall' (fn x => func(h,x)) t) andalso
      (forall_two func t)
;

(* Support functions *)

(* sub_sig -> e_symbol list *)
(* p. 131 *)
local
  open Symbols;
in
fun e_sig_symbols((ssenv,venv,fenv,penv):sub_sig):e_symbol list =
  let val sigsyms = sig_symbols(ssenv,venv,fenv,penv)
  val esyms1 = map (fn x => SYM_ESYMBOL(x)) sigsyms
  val senvl = Symtab_id.dest(ssenv)
  val esyms =
    foldl (fn (esymsi,(s,ss)) =>
      foldl (fn (esymsii,s1) => EMB_ESYMBOL(s1,s)::esymsii)
            (esymsi,ss))
      (esyms1,senvl) in
  esyms
  end
end;

(* sub_sig * sub_sig -> bool *)
fun is_subsig(sig1,sig2) =
  is_subset(e_sig_symbols(sig1),e_sig_symbols(sig2))
;

(* e_symbol * morphism -> e_symbol *)
fun e_symbol_via_morphism(esym,mor) =
  case esym of
    SYM_ESYMBOL(sym) => SYM_ESYMBOL(symbol_via_morphism(mor,sym))
  | EMB_ESYMBOL(s1,s2) =>
      EMB_ESYMBOL(sort_via_morphism mor s1, sort_via_morphism mor s2)
;

(* sub_sig * GlobalEnv.morphism -> e_symbol Symtab_esym.table *)
(* p. 131 *)
fun e_morphism_symmap((sigma:sub_sig),(mor:GlobalEnv.morphism)) =
  foldl (fn (symmapi,esym) =>
    Symtab_esym.update_new((esym,e_symbol_via_morphism(esym,mor)),symmapi))
    (Symtab_esym.empty,e_sig_symbols(sigma))
;

(* tag_sig -> sub_sig *)
(* p. 132 *)
fun untag_sig(ss,_) = ss;

(* unit_tag_sig -> unit_sig *)
(* p. 133 *)
fun untag_unit_sig(tsl,ts) = (map untag_sig tsl,untag_sig ts);

(* comp_tag_sigs -> bool *)
(* p. 133 *)
fun compatible_tag_sigs(tsl:comp_tag_sigs) =
  compatible_sigs(map untag_sig tsl)
;

val merge_nusigs:(named_unit_sigs * named_unit_sigs -> named_unit_sigs) =
  Symtab_sid.merge (fn (_,_) => false);


(* tagging_map -> tagging_map *)
(* p. 132 *)
local
  open Symbols;
in
fun embedding_closure(tm:tagging_map):tagging_map =
  let val tmlist = Symtab_esym.dest(tm) in
  foldl (fn (tm1i,(esym,vall)) => case esym of
      SYM_ESYMBOL(SORT_SYMBOL(s)) =>
        foldl (fn (tm1ii,(un1,esym1)) => case esym1 of
            SYM_ESYMBOL(SORT_SYMBOL(s1)) =>
              Symtab_esym.update((EMB_ESYMBOL(s,s),
                                 (un1,EMB_ESYMBOL(s1,s1))::(
                case Symtab_esym.lookup(tm,EMB_ESYMBOL(s,s)) of
                    None => []
                  | Some(tagl) => tagl)),tm1ii)
          | _ => ((print "err1\n"); raise INTERNAL_ERROR))
          (tm1i,vall)
    | EMB_ESYMBOL(s1,s2) =>
        foldl (fn (tm1ii,(esym1,vall1)) => (case esym1 of
            EMB_ESYMBOL(s2',s3) => if not(s2=s2') then tm1ii else
              foldl (fn (tm1iii,(un1,vesym)) => (case vesym of
                  EMB_ESYMBOL(s11,s21) =>
                    foldl (fn (tm1iv,(un1,vesym1)) => (case vesym1 of
                        EMB_ESYMBOL(s21',s31) =>
                          if not(s21'=s21) then tm1iv else
                          Symtab_esym.update((EMB_ESYMBOL(s1,s3),
                                             (un1,EMB_ESYMBOL(s11,s31))::(
                          case Symtab_esym.lookup(tm,EMB_ESYMBOL(s1,s3)) of
                             None => []
                           | Some(tagl) => tagl)),tm1iv)
                      | _ => ((print "err2\n"); raise INTERNAL_ERROR)))
                      (tm1iii,vall1)
                | _ => ((print "err3\n"); raise INTERNAL_ERROR)))
                (tm1ii,vall)
          | _ => tm1ii))
          (tm1i,tmlist)
    | _ => tm1i)
    (tm,tmlist)
  end
end;

(* JUST TO DEBUG *)
fun print_esymbol(SYM_ESYMBOL(x)) = " "^(Morphisms.print_symbol x)
  | print_esymbol(EMB_ESYMBOL(s1,s2)) = " "^(BasicPrint.print_ID s1)^"<"
                                         ^(BasicPrint.print_ID s2)
;

fun print_tagging_map(m) = 
  let val mlist = Symtab_esym.dest(m) in
    map (fn (esym,rsyml) => 
      ((print ((print_esymbol(esym)^" --> ")));
      (map (fn (un,esym1) => print ((BasicPrint.print_SIMPLE_ID un)^"."^
                                    (print_esymbol esym1)^" "))
           rsyml);
      (print "\n")))
      mlist
  end
;


(* tagging_map list -> tagging_map *)
(* p. 133 *)
fun merge_tagging_maps(tml:tagging_map list):tagging_map =
  let val symlist =
    remove_dups (map (fn (a,b) => a) (appendlists (map Symtab_esym.dest tml)))
  val keylist = map (fn esym =>
    let val vallist = map (fn tm => case Symtab_esym.lookup(tm,esym) of
      None => [] | Some(x) => x) tml
    in (esym,remove_dups(appendlists vallist))
    end) symlist
  in Symtab_esym.make keylist
  end
;

(* comp_tag_sigs -> tag_sig *)
(* p. 133 *)
fun merge_tag_sigs(tsigs:comp_tag_sigs):tag_sig =
 let val sigs = map (fn (a,b) => a) tsigs
 val tms = map (fn (a,b) => b) tsigs
 val sigsum = foldl (fn (sigsumi,sigma) =>
        (*StaticAnalysis.*)merge_lenvs(sigsumi,sigma))
      (LocalEnv.empty_local_env,sigs)
 in (sigsum,embedding_closure(merge_tagging_maps(tms)))
 end
;

(* p. 158 *)
(* local_env * morphism * local_env * tagging_map -> tagging_map *)
fun tagging_map_via_morphism(sigma,mor,sigma1,m) =
  let val smap = e_morphism_symmap(sigma,mor) in
    foldl (fn (m1i,x1) =>
      let val newval = foldl (fn (newvali,x) =>
        if Symtab_esym.lookup(smap,x)=Some(x1) then
          case Symtab_esym.lookup(m,x) of
            Some(mval) => remove_dups(mval@newvali)
          | _ => newvali
        else newvali)
        ([],e_sig_symbols(sigma))
      in (Symtab_esym.update_new((x1,newval),m1i))
      end)
      (Symtab_esym.empty,e_sig_symbols(sigma1))
  end 
;

(* tag_sig list -> bool *)
fun amalg_ensures_symbol_sharing(tsigs) =
  let val sigms = map (fn (sigma,m) => (e_sig_symbols(sigma),m)) tsigs in
  forall_two (fn ((esymsi,mi),(esymsj,mj)) =>
    forall' (fn x =>
      let val mix = case Symtab_esym.lookup(mi,x) of
        None => [] | Some(y) => y
      val mjx = case Symtab_esym.lookup(mj,x) of
        None => [] | Some(y) => y
      in (length(intersect(mix,mjx))>0)
      end)
      (intersect(esymsi,esymsj)))
    sigms
  end
;

(* tag_sig * morphism -> bool *)
fun transl_ensures_symbol_sharing((sigma,m),mor) =
  let val esyms = e_sig_symbols(sigma) in
  forall' (fn x =>
    forall' (fn y =>
      if e_symbol_via_morphism(x,mor)=e_symbol_via_morphism(y,mor)
      then (x=y) orelse 
        (let val mx = case Symtab_esym.lookup(m,x) of
           None => [] | Some(y) => y
         val my = case Symtab_esym.lookup(m,y) of
           None => [] | Some(y) => y
         in (length(intersect(mx,my))>0)
         end)
      else true)
      esyms)
    esyms
  end
;

(* 'a list list -> 'a list list *)
fun all_combinations([]) = [[]]
  | all_combinations(l::t) = 
      let val tls = all_combinations(t) in
        foldl (fn (cmbi,s) => (map (fn x => s::x) tls) @ cmbi)
              ([],l)
      end
; 

(* 'a list * 'b list -> ('a * 'b) list *)
fun all_pairs([],_) = []
  | all_pairs(h::t,l) = 
      (map (fn x => (h,x)) l) @
      (all_pairs(t,l))
;

(* Maximality in preorder: x is quasi-maximal iff forall y>=x: y<=x. *)
(* local_env * sort list * sort -> bool *)
fun is_quasi_maximal_in((ssenv,_,_,_),sl,s) =
  let val ssl = case Symtab_id.lookup(ssenv,s) of None => [] | Some(x) => x
  in 
    forall' (fn s1 => case Symtab_id.lookup(ssenv,s1) of
        None => true
      | Some(sl1) => if s mem sl1 then s1 mem ssl else true)
    sl
  end
;

fun is_quasi_minimal_in((ssenv,_,_,_),sl,s) =
  let val ssl = case Symtab_id.lookup(ssenv,s) of None => [] | Some(x) => x
  in
    forall' (fn s1 => case Symtab_id.lookup(ssenv,s1) of
        None => not(s1 mem ssl)
      | Some(sl1) => if s1 mem ssl then s mem sl1 else true)
    sl
  end
;

(* local_env * sort list -> sort list *)
(* Purpose: from set of (quasi-min(max)imal) sorts take one representant
   from each clique *)
fun representants(_,[]) = []
  | representants(lenv,s::t) =
      let val (ssenv,_,_,_) = lenv
      val sl = case Symtab_id.lookup(ssenv,s) of None => [] | Some(x) => x 
      val t1 = filter (fn s1 => not(s1 mem sl)) t in
        s::(representants(lenv,t1))
      end
;

(* local_env * sort * sort -> sort list *)
fun quasi_minimal_common_supersorts(lenv,s1,s2) =
  let val common = common_supersorts(lenv,s1,s2) 
  val qmcsall = filter (fn s => is_quasi_minimal_in(lenv,common,s)) common
  in representants(lenv,qmcsall)
  end
;

fun quasi_maximal_common_subsorts(lenv,s1,s2) =
  let val common = common_subsorts(lenv,s1,s2)
  val qmcsall = filter (fn s => is_quasi_maximal_in(lenv,common,s)) common
  in representants(lenv,qmcsall)
  end
;


fun quasi_maximal_common_subseqs(lenv,seq1,seq2) =
  let val subss = 
    map (fn (s1,s2) => quasi_maximal_common_subsorts(lenv,s1,s2))
        (zip(seq1,seq2))
  in all_combinations(subss)
  end
;

fun quasi_maximal_common_fun_subprofiles(lenv,(seq1,s1),(seq2,s2)) =
  let val sseqs = quasi_maximal_common_subseqs(lenv,seq1,seq2)
  val ssorts = quasi_minimal_common_supersorts(lenv,s1,s2)
  in all_pairs(sseqs,ssorts)
  end
;

(* sort * sort * tagging_map -> (unit_name * sort * sort) list *)
(* Where is subsorting relation shared by another unit name? *)
fun subsorting_is_shared(s1,s2,m) = 
  let val emsym = EMB_ESYMBOL(s1,s2)
  val shr = case Symtab_esym.lookup(m,emsym) of None=>[] | Some(x) => x
  in
    map (fn (un,emsym1) => 
      case emsym1 of 
        EMB_ESYMBOL(s11,s21) => (un,s11,s21)
      | _ => raise INTERNAL_ERROR)
      shr
  end
;

(* (unit_name * sort * sort) list list -> 
     (unit_name * sort list * sort list) list *)
fun subseq_sharing_combinations([]) = []
  | subseq_sharing_combinations([l]) =
      map (fn (un,s1,s2) => (un,[s1],[s2])) l
  | subseq_sharing_combinations(l::t) =
      let val tls = subseq_sharing_combinations(t) in
        foldl (fn (cmbi,(un,s1,s2)) => 
          let val tls1 = filter (fn (un1,seq1,seq2) => un=un1) tls in
            (map (fn (un1,seq1,seq2) => (un1,s1::seq1,s2::seq2)) tls1)@cmbi
          end)
          ([],l)
      end
;

(* sort list * sort list * tagging_map -> 
   (unit_name * sort list * sort list) list *)
(* As above, but for sort sequences *)
fun subseqing_is_shared(seq1,seq2,m) = 
  let val cands = map subsorting_is_shared 
                      (map (fn (x,y) => (x,y,m)) (zip(seq1,seq2)))
  in subseq_sharing_combinations cands
  end
;

(* fun_profile * fun_profile * tagging_map ->
     (unit_name * fun_profile * fun_profile) list *)
(* As above, but for function profiles *)
fun subprofiling_is_shared((seq1,s1),(seq2,s2),m) =
  let val shseq = subseqing_is_shared(seq1,seq2,m)
  val shsort = subsorting_is_shared(s1,s2,m)
  in
    foldl (fn (cmbi,(un,seq1i,seq2i)) =>
      let val shsort1 = filter (fn (un1,_,_) => un1=un) shsort in
        foldl (fn (cmbii,(un1,s1i,s2i)) =>
          ((un1,(seq1i,s1i),(seq2i,s2i))::cmbii))
          (cmbi,shsort1)
      end)
      ([],shseq)
  end
;

(* e_symbol -> ID *)
fun e_symbol_name(SYM_ESYMBOL(x)) = symbol_name x
  | e_symbol_name(_) = raise INTERNAL_ERROR

(* symbol -> fun_profile *)
fun symbol_fun_profile(TOTAL_FUN_SYMBOL(_,x)) = x
  | symbol_fun_profile(PARTIAL_FUN_SYMBOL(_,x)) = x
  | symbol_fun_profile(_) = raise INTERNAL_ERROR
;

(* symbol -> pred_profile *)
fun symbol_pred_profile(PRED_SYMBOL(_,x)) = x
  | symbol_pred_profile(_) = raise INTERNAL_ERROR
;

(* e_symbol -> fun_profile *)
fun e_symbol_fun_profile(SYM_ESYMBOL(x)) = symbol_fun_profile x
  | e_symbol_fun_profile(_) = raise INTERNAL_ERROR
;

(* e_symbol -> pred_profile *)
fun e_symbol_pred_profile(SYM_ESYMBOL(x)) = symbol_pred_profile x
  | e_symbol_pred_profile(_) = raise INTERNAL_ERROR
;

(* e_symbol * e_symbol * tagging_map -> 
    (unit_name * fun_profile * fun_profile) list *)
(* Are equality of names of the two symbols shared somewhere? *)
fun fun_names_are_shared(esym1,esym2,m) =
  let val shr1 = case Symtab_esym.lookup(m,esym1) of 
                  None => [] | Some(x) => x
  val shr2 = case Symtab_esym.lookup(m,esym2) of 
                  None => [] | Some(x) => x
  in
    foldl (fn (cmbi,(un1,esym3)) =>
      foldl (fn (cmbii,(un2,esym4)) => 
        if (un1=un2) andalso (e_symbol_name(esym3)=e_symbol_name(esym4)) then
          ((un1,e_symbol_fun_profile esym3,
                e_symbol_fun_profile esym4)::cmbii) else cmbii)
        (cmbi,shr2))
      ([],shr1)
  end
;

(* e_symbol * e_symbol * tagging_map -> 
    (unit_name * pred_profile * pred_profile) list *)
(* Are equality of names of the two symbols shared somewhere? *)
fun pred_names_are_shared(esym1,esym2,m) =
  let val shr1 = case Symtab_esym.lookup(m,esym1) of 
                  None => [] | Some(x) => x
  val shr2 = case Symtab_esym.lookup(m,esym2) of 
                  None => [] | Some(x) => x
  in
    foldl (fn (cmbi,(un1,esym3)) =>
      foldl (fn (cmbii,(un2,esym4)) => 
        if (un1=un2) andalso (e_symbol_name(esym3)=e_symbol_name(esym4)) then
          ((un1,e_symbol_pred_profile esym3,
                e_symbol_pred_profile esym4)::cmbii) else cmbii)
        (cmbi,shr2))
      ([],shr1)
  end
;

(* e_symbol * e_symbol * fun_profile * tagging_map -> bool *)
fun fun_overloading_is_shared(esym1,esym2,subprof,m) = 
  let val (fn1,fprof1) = (e_symbol_name esym1, e_symbol_fun_profile esym1)
  val (fn2,fprof2) = (e_symbol_name esym2, e_symbol_fun_profile esym2)
  in
    if fn1=fn2 then
      let val shprof1 = subprofiling_is_shared(subprof,fprof1,m)
      val shprof2 = subprofiling_is_shared(subprof,fprof2,m)
      val shnm = fun_names_are_shared(esym1,esym2,m) 
      in
        exists' (fn (un1',fprof1',fprof2') =>
          exists' (fn (un2',fprof3',fprof4') =>
            (un1'=un2') andalso (fprof4'=fprof1') andalso
            (exists' (fn (un3',fprof5',fprof6') =>
               (un3'=un2') andalso (fprof6'=fprof2') andalso
               (fprof5'=fprof3'))
               shprof2))
            shprof1)
          shnm
      end
    else true  
  end
;

(* e_symbol * e_symbol * pred_profile * tagging_map -> bool *)
fun pred_overloading_is_shared(esym1,esym2,subprof,m) = 
  let val (pn1,pprof1) = (e_symbol_name esym1, e_symbol_pred_profile esym1)
  val (pn2,pprof2) = (e_symbol_name esym2, e_symbol_pred_profile esym2)
  in
    if pn1=pn2 then
      let val shprof1 = subseqing_is_shared(subprof,pprof1,m)
      val shprof2 = subseqing_is_shared(subprof,pprof2,m)
      val shnm = pred_names_are_shared(esym1,esym2,m)   
      in
        exists' (fn (un1',pprof1',pprof2') =>
          exists' (fn (un2',pprof3',pprof4') =>
            (un1'=un2') andalso (pprof4'=pprof1') andalso
            (exists' (fn (un3',pprof5',pprof6') =>
               (un3'=un2') andalso (pprof6'=pprof2') andalso
               (pprof5'=pprof3'))
               shprof2))
            shprof1)
          shnm
      end
    else true  
  end
;

(* Some auxiliary functions involving subsorting *)
(* lenv * sort * sort -> bool *)
fun sort_is_leq((ssenv,_,_,_),s1,s2) = 
  case Symtab_id.lookup(ssenv,s2) of None => false
     | Some(x) => s1 mem x
;

fun seq_is_leq(lenv,seq1,seq2) =
  forall' (fn (s1,s2) => sort_is_leq(lenv,s1,s2)) (zip(seq1,seq2))
;

fun fun_profile_is_leq(lenv,(seq1,s1),(seq2,s2)) =
  (seq_is_leq(lenv,seq1,seq2)) andalso (sort_is_leq(lenv,s2,s1))
;

(* lenv * symbol * symbol * fun_profile -> bool *)
fun sig_gives_fun_overloading(lenv,sym1,sym2,fprof) = 
  let val (fn1,fprof1) = (symbol_name sym1, symbol_fun_profile sym1)
  val (fn2,fprof2) = (symbol_name sym2, symbol_fun_profile sym2)
  val syms = sig_symbols(lenv)
  in
    if fn1=fn2 then
      (sym1 mem syms) andalso (sym2 mem syms) andalso
      (fun_profile_is_leq(lenv,fprof,fprof1)) andalso
      (fun_profile_is_leq(lenv,fprof,fprof2))
    else true
  end
;

(* lenv * symbol * symbol * pred_profile -> bool *)
fun sig_gives_pred_overloading(lenv,sym1,sym2,pprof) = 
  let val (pn1,pprof1) = (symbol_name sym1, symbol_pred_profile sym1)
  val (pn2,pprof2) = (symbol_name sym2, symbol_pred_profile sym2)
  val syms = sig_symbols(lenv)
  in
    if pn1=pn2 then
      (sym1 mem syms) andalso (sym2 mem syms) andalso
      (seq_is_leq(lenv,pprof,pprof1)) andalso
      (seq_is_leq(lenv,pprof,pprof2))
    else true
  end
;

(* tag_sig list -> bool *)
fun amalg_ensures_overloading_sharing(tsigs) = 
  let val (sigs,ms) = (map fst tsigs,map snd tsigs)
  val sigma = merge_lenvlist(sigs)
  val m = merge_tagging_maps(ms)
  val ovrl = transl_overloading(overloading_relation(sigma))
  val fovrl = filter (fn (sym1,_) => 
    case sym1 of TOTAL_FUN_SYMBOL(_) => true
               | PARTIAL_FUN_SYMBOL(_) => true
               | _ => false) ovrl
  val povrl  = filter (fn (sym1,_) => 
    case sym1 of PRED_SYMBOL(_) => true
               | _ => false) ovrl
  in
    (forall' (fn (fsym1,fsym2) => 
      let val fprof1 = symbol_fun_profile(fsym1)
      val fprof2 = symbol_fun_profile(fsym2)
      val subprfs = 
        quasi_maximal_common_fun_subprofiles(sigma,fprof1,fprof2)
      in
        forall' (fn subprof => 
          (exists' (fn sigmai => 
            sig_gives_fun_overloading(sigmai,fsym1,fsym2,subprof))
            sigs) orelse
          (fun_overloading_is_shared(SYM_ESYMBOL(fsym1),
                                     SYM_ESYMBOL(fsym2),
                                     subprof,
                                     embedding_closure(m))))
          subprfs
      end)
      fovrl) andalso
    (forall' (fn (psym1,psym2) => 
      let val pprof1 = symbol_pred_profile(psym1)
      val pprof2 = symbol_pred_profile(psym2)
      val subprfs = 
        quasi_maximal_common_subseqs(sigma,pprof1,pprof2)
      in
        forall' (fn subprof => 
          (exists' (fn sigmai => 
            sig_gives_pred_overloading(sigmai,psym1,psym2,subprof))
            sigs) orelse
          (pred_overloading_is_shared(SYM_ESYMBOL(psym1),
                                      SYM_ESYMBOL(psym2),
                                      subprof,
                                      embedding_closure(m))))
          subprfs
      end)
      povrl)
  end
;

(* CaslEnv.local_env * CaslEnv.morphism * symbol -> symbol list *)
fun morphism_symbol_domain(lenv,mor,sym) = 
  filter (fn sym1 => symbol_via_morphism(mor,sym1)=sym)
         (sig_symbols(lenv))
;

(* CaslEnv.local_env * CaslEnv.morphism * sort -> sort list *)
fun morphism_sort_domain(lenv,mor,s) = 
  let val symdom = morphism_symbol_domain(lenv,mor,SORT_SYMBOL(s))
  in map (fn sym => case sym of
        SORT_SYMBOL(s1) => s1
       | _ => raise INTERNAL_ERROR)
     symdom
  end
;

(* local_env * morphism * sort list -> sort list list *)
fun morphism_sortseq_domain(lenv,mor,seq) = 
  let val sortdoms = map (fn s => morphism_sort_domain(lenv,mor,s)) seq
  in
    all_combinations(sortdoms)
  end
;

(* local_env * morphism * fun_profile -> fun_profile list *)
fun morphism_fun_profile_domain(lenv,mor,(seq,s)) =
  let val seqdom = morphism_sortseq_domain(lenv,mor,seq)
  val sortdom = morphism_sort_domain(lenv,mor,s)
  in
    all_pairs(seqdom,sortdom)
  end
;

(* local_env * morphism * symbol * symbol * fun_profile -> bool *)
fun morphism_gives_fun_overloading(lenv,mor,sym1,sym2,subprof) =
  let val symdom1 = morphism_symbol_domain(lenv,mor,sym1)
  val symdom2 = morphism_symbol_domain(lenv,mor,sym2)
  val subprofdom = morphism_fun_profile_domain(lenv,mor,subprof)
  in
    exists' (fn sym1 =>
      exists' (fn sym2 =>
        let val (fn1,fprof1) = (symbol_name sym1,symbol_fun_profile sym1)
        val (fn2,fprof2) = (symbol_name sym2,symbol_fun_profile sym2)
        in
          (fn1=fn2) andalso 
          (exists' (fn subprof1 =>
             (fun_profile_is_leq(lenv,subprof1,fprof1)) andalso
             (fun_profile_is_leq(lenv,subprof1,fprof2)))
            subprofdom)
        end)
       symdom2)
     symdom1
  end
;

(* local_env * morphism * symbol * symbol * pred_profile -> bool *)
fun morphism_gives_pred_overloading(lenv,mor,sym1,sym2,subprof) =
  let val symdom1 = morphism_symbol_domain(lenv,mor,sym1)
  val symdom2 = morphism_symbol_domain(lenv,mor,sym2)
  val subprofdom = morphism_sortseq_domain(lenv,mor,subprof)
  in
    exists' (fn sym1 =>
      exists' (fn sym2 =>
        let val (pn1,pprof1) = (symbol_name sym1,symbol_pred_profile sym1)
        val (pn2,pprof2) = (symbol_name sym2,symbol_pred_profile sym2)
        in
          (pn1=pn2) andalso 
          (exists' (fn subprof1 =>
             (seq_is_leq(lenv,subprof1,pprof1)) andalso
             (seq_is_leq(lenv,subprof1,pprof2)))
            subprofdom)
        end)
       symdom2)
     symdom1
  end
;  

(* tag_sig * morphism * local_env -> bool *)
fun transl_ensures_overloading_sharing((sig1,m1),mor,sig2) = 
  let val m2 = tagging_map_via_morphism(sig1,mor,sig2,m1)
  val ovrl = transl_overloading(overloading_relation(sig1))
  val fovrl = filter (fn (sym1,_) => 
    case sym1 of TOTAL_FUN_SYMBOL(_) => true
               | PARTIAL_FUN_SYMBOL(_) => true
               | _ => false) ovrl
  val povrl  = filter (fn (sym1,_) => 
    case sym1 of PRED_SYMBOL(_) => true
               | _ => false) ovrl
  in
    (forall' (fn (fsym1,fsym2) => 
      (fsym1=fsym2) orelse  
      let val fprof1 = symbol_fun_profile(fsym1)
      val fprof2 = symbol_fun_profile(fsym2)
      val subprfs = 
        quasi_maximal_common_fun_subprofiles(sig1,fprof1,fprof2)
      in
        forall' (fn subprof =>
          (morphism_gives_fun_overloading(sig1,mor,fsym1,fsym2,subprof))
          orelse
          (fun_overloading_is_shared(SYM_ESYMBOL(fsym1),
                                     SYM_ESYMBOL(fsym2),
                                     subprof,
                                     embedding_closure(m2))))
          subprfs
      end)
      fovrl) andalso
    (forall' (fn (psym1,psym2) => 
      let val pprof1 = symbol_pred_profile(psym1)
      val pprof2 = symbol_pred_profile(psym2)
      val subprfs = 
        quasi_maximal_common_subseqs(sig1,pprof1,pprof2)
      in
        forall' (fn subprof =>
          (morphism_gives_pred_overloading(sig1,mor,psym1,psym2,subprof)) 
          orelse
          (pred_overloading_is_shared(SYM_ESYMBOL(psym1),
                                      SYM_ESYMBOL(psym2),
                                      subprof,
                                      embedding_closure(m2))))
          subprfs
      end)
      povrl)
  end
;

(* comp_tag_sigs -> bool *)
(* p. 134-136 *)
fun ensures_model_compatibility(tsigs) = 
  (amalg_ensures_symbol_sharing(tsigs)) andalso 
  (amalg_ensures_overloading_sharing(tsigs))
;

(* tag_sig * morphism * local_env -> bool *)
(* p. 137-138 *)
fun induces_translations(tsig,mor,sig1) = 
  (transl_ensures_symbol_sharing(tsig,mor)) andalso
  (transl_ensures_overloading_sharing(tsig,mor,sig1))
;
           
end;

end;
