(**************************************************************** *)
(*  							 	  *)
(* Project : CATS						  *)
(* Author: Till Mossakowski, University of Bremen		  *)
(* Date: 28.10.99; Origin: 24.05.97			  	  *)
(* Purpose of this file: Static analysis of basic specifications  *)
(*	 							  *)	
(*								  *)
(**************************************************************** *)

(* This module should follow the semantics of CASL basic specifications
   as described in CoFI study nore S-9.
   However, due to historical reasons, the static analysis of
   basic specifications is done in a slightly different way:
   Given a basic specification, the following data or collected
   (in this order):
   - All the sorts
   - All the subsort relations
   - All the opration symbols
   - All the predicate symbols
   - All the axioms
   For each of these kinds of things, a bunch of functions is
   provided. These traverse the abstract syntax tree and
   collect the relevant things.

   todo:

   A future version (implementing linear visibility) should
   take a signature plus a set of axioms (i.e. a tuple
   of the above things) and carry around this signature
   while traversing the abstract syntax tree.
   This would follow the semantics more closely.

   Forbid renamings of local env
   generated: Sollen neue Subsorten-Injektionen in sort_gen_ax 
     aufgenommen werden?
   Bei Fehler in stat_ana kein Overloadres mehr
   Statische checks
     Sind sorten in type s ::= sort t deklariert?
     totale + partielle Funktionen nicht gleichzeitig
   Generierte Variablennamen auf Disjunktheit mit ID pruefen.
   Axiomnamen fuer induzierte Axiome: Sort def
     Induzierte Namen alphabetisch weit hinten

   Axiom forall x:Array . x=x  fuehrt zu Fehler

Nach: 
 sort s, t
 op o : s
 var o : t

liefert:
. (o) = (o)
Cannot disambiguate:
((op o:s)) = ((op o:s))
(o:t) = (o:t)

. (o:s) = (o:s)
wird akzeptiert, ebenso (var o:s) = (var o:s)


und:
. (op o:s) = (op o:s)

liefert: Variable o declared as t but used as s in: 
forall o : t . (op o:s) = (op o:s)

*)


structure BasicAnalysis :
sig


val basic_analysis : 
     bool -> (* Should output be in html? *)
     AS.BASIC_SPEC * (LocalEnv.local_env * AS.ANNO list) ->
     AS.BASIC_SPEC * (* new AS tree with all formulas mixfix analysed *) 
     LocalEnv.sign * (* complete new signature, including local environment *) 
     LocalEnv.sign * (* new parts of signature, excluding local environment *)
     AS.L_FORMULA list * string list

val prefix_idn : string -> AS.ID list -> AS.ID

val parse_axiom : bool -> (* is it a goal? *) 
                  LocalEnv.sign -> AS.ANNO list -> string -> AS.FORMULA * string list
val parse_term  : LocalEnv.sign -> AS.ANNO list -> string -> AS.TERM * string list
     
end
=
struct

open Utils AS IDOrder LocalEnv 
     CASLParser IsabelleMixfixParser 
     Overload TextForm BasicPrint
      
infix mem \\

fun cons_entry1 entry (key,tab) = cons_entry ((key,entry),tab)

fun cons_entry_list entry_list (tab,key) =
    Symtab_id.update ((key, entry_list @ Symtab_id.lookup_multi (tab, key)), tab);

fun null_entry key = (key,())
fun simpleIdToIds (c,t) = simple_id ([token c],c,None);

fun mk_var  (c,t) = application (op_symb ((simpleIdToIds (c,t)), None),terms [])
fun mk_var1 v = application (op_symb ((simpleIdToIds (v,v)), None),terms [])

type Sortenv = unit Symtab_id.table


(* Sorts *)
fun get_sortItem (sort_decl (sortlist)) = sortlist
| get_sortItem (subsort_decl (sortlist, s)) = s::sortlist
| get_sortItem  (subsort_defn(s,v,s1,phi,annos)) = [s]
| get_sortItem (iso_decl (sortlist))  = sortlist
| get_sortItem (pos_SORT_ITEM (_,si)) =
  get_sortItem si

fun get_sort_Datatype(datatype_decl (s,l,label)) = s
  | get_sort_Datatype(pos_DATATYPE_DECL (_,d)) =
    get_sort_Datatype d

fun get_sortItem_label ((s,l)) = (get_sortItem s)
fun get_sort_sig_items(sort_items (sl,_)) = flat (map get_sortItem_label sl)
  | get_sort_sig_items(datatype_items(dlist,_)) =  (map get_sort_Datatype dlist)
  | get_sort_sig_items(pos_SIG_ITEMS (_,si)) =
    get_sort_sig_items si
  | get_sort_sig_items(x) = []
  
and get_sorts_bit(sig_items(decl)) = get_sort_sig_items(decl)
  | get_sorts_bit(sort_gen(decls,_)) = flat (map get_sort_sig_items decls)
  | get_sorts_bit(sort_cogen(decls,_)) = flat (map get_sort_sig_items decls)
  | get_sorts_bit(free_datatype(decls,_)) =  (map get_sort_Datatype decls)
  | get_sorts_bit(cofree_datatype(decls,_)) =  (map get_sort_Datatype decls)
  | get_sorts_bit(pos_BASIC_ITEMS (_,bit)) =
    get_sorts_bit bit
  | get_sorts_bit(x) = []

and get_sorts(basic_spec bits): SORT list =
	flat(map get_sorts_bit bits)
  | get_sorts(pos_BASIC_SPEC (_,b)) =
    get_sorts b

(* Subsorts *)

fun add_subsorts r (sortenv:Sort_env) (subsortlist:SORT list) ((env:Subsort_env,errs),s) =
	case (Symtab_id.lookup(sortenv,s)) of
  	  None => ((env,(perr r^"Undeclared sort: "^print_SORTS1[s])::errs))
  	  | Some _ => (Symtab_id.update ((s,subsortlist@Symtab_id.lookup_multi(env,s)),env),errs);

fun get_sortlist (((total_construct (c,c1),_)::altlist):L_ALTERNATIVE list):SORT list = get_sortlist altlist
  | get_sortlist (((partial_construct (c,c1),_)::altlist):L_ALTERNATIVE list):SORT list = get_sortlist altlist
  | get_sortlist ((subsort (s),_)::altlist) = s@get_sortlist altlist
  | get_sortlist ((pos_ALTERNATIVE (_,a),x)::altlist) = get_sortlist ((a,x)::altlist)
  | get_sortlist nil = nil
 
 fun get_subsortItem r sortenv (subsortenv, subsort_decl (sortlist, l)) = 
 	add_subsorts r sortenv sortlist (subsortenv,l)
| get_subsortItem r  sortenv ((env,errs),subsort_defn(s,v,s1,phi,_)) = 	
  	(case Symtab_id.lookup (sortenv,s1) of
  	  None => (env,(perr r^"Undeclared sort: "^print_SORTS1 [s1])::errs)
  	  | Some _ => (cons_entry ((s1,s),env),errs))
| get_subsortItem r sortenv (subsortenv,iso_decl(sl)) =
  	foldl (add_subsorts r sortenv sl) (subsortenv,sl)
| get_subsortItem _ sortenv (subsortenv,pos_SORT_ITEM (r,si)) =
        get_subsortItem r sortenv (subsortenv,si)
| get_subsortItem _ sortenv (subsortenv,decls) = subsortenv


fun get_subsort_Datatype sortenv (subsortenv, datatype_decl (s,altlist,label)) = 
      add_subsorts null_region sortenv (get_sortlist altlist) (subsortenv,s)
  | get_subsort_Datatype sortenv (subsortenv, pos_DATATYPE_DECL (_,dd)) =
    get_subsort_Datatype sortenv (subsortenv,dd)

fun get_subsortItem_label sortenv (subsortenv, (s,l)) =
      (get_subsortItem null_region sortenv (subsortenv, s))

fun get_subsort_sig_items sortenv (subsortenv ,sort_items (sl,_)) = 
      foldl (get_subsortItem_label  sortenv) (subsortenv, sl)
  | get_subsort_sig_items sortenv (subsortenv, datatype_items (dlist,_)) = 
      foldl (get_subsort_Datatype  sortenv) (subsortenv, dlist ) 
  | get_subsort_sig_items sortenv (subsortenv, pos_SIG_ITEMS(_,si)) =
    get_subsort_sig_items sortenv (subsortenv, si)
  | get_subsort_sig_items sortenv (subsortenv, _) = subsortenv
  
and get_subsorts_bit sortenv (subsortenv, sig_items(decl)) = 
      get_subsort_sig_items sortenv (subsortenv,decl)
  | get_subsorts_bit sortenv (subsortenv, sort_gen(decls,_)) = 
      foldl (get_subsort_sig_items sortenv) (subsortenv,decls)
  | get_subsorts_bit sortenv (subsortenv, sort_cogen(decls,_)) = 
      foldl (get_subsort_sig_items sortenv) (subsortenv,decls)
  | get_subsorts_bit  sortenv (subsortenv, free_datatype(decls,_)) =  
      foldl (get_subsort_Datatype  sortenv) (subsortenv,decls) 
  | get_subsorts_bit  sortenv (subsortenv, cofree_datatype(decls,_)) =  
      foldl (get_subsort_Datatype  sortenv) (subsortenv,decls) 
  | get_subsorts_bit sortenv (subsortenv, pos_BASIC_ITEMS(_,bi)) =
    get_subsorts_bit sortenv (subsortenv,bi)
  | get_subsorts_bit sortenv (subsortenv,x) = subsortenv

and get_subsorts sortenv (subsortenv,basic_spec bits) =
	foldl (get_subsorts_bit sortenv) (subsortenv,bits)
  | get_subsorts sortenv (subsortenv,pos_BASIC_SPEC(_,b)) =
    get_subsorts sortenv (subsortenv,b)

(* Functions *)

fun check_sort r sortenv s =
	case Symtab_id.lookup (sortenv,s) of
	  None => [perr r^"Undeclared sort: "^print_SORTS1 [s]]
	  | Some _ => nil

fun check_sort_vdecl r sortenv (_,s) = check_sort r sortenv s

fun check_sort_vlist r sortenv vlist =
    flat (map (check_sort_vdecl r sortenv) vlist)

fun check_sorts r (sortenv:Sort_env) (total_op_type (args,res)) = 
	flat (map (check_sort r sortenv) (res::Utils.get_sorts args))
  | check_sorts r sortenv (partial_op_type (args,res)) = 
	flat (map (check_sort r sortenv) (res::Utils.get_sorts args))
  | check_sorts _ sortenv (pos_OP_TYPE (r,t)) =
    check_sorts r sortenv t

fun add_err (fenv,errs) e = (fenv,e::errs)

fun add_funs r (sortenv:Sort_env) ((env:Fun_env,errs),(fnames,ftype)) =
	(foldr (cons_entry1 ftype) (fnames,env),
	 check_sorts r sortenv ftype @ errs);
 
fun get_arg_sort (total_select(selnames,s)) = map (K s) selnames
  | get_arg_sort (partial_select(selnames,s)) = map (K s) selnames
  | get_arg_sort (sort_component(s)) = [s]
  | get_arg_sort (pos_COMPONENTS(_,c)) =
    get_arg_sort c


fun get_selector_list r is_co s ((total_select(selname,s1))::rest) =
  	(selname,total_op_type(sorts[s],s1))::(get_selector_list r is_co s rest)
  | get_selector_list r is_co s ((partial_select(selname,s1))::rest) =
  	(selname,partial_op_type(sorts[s],s1))::(get_selector_list r is_co s rest ) 	
  | get_selector_list r is_co s ((sort_component(s1))::rest) =
    if is_co then raise ERR (perr r^"cofree or cogenerated types must have selectors")
    else get_selector_list r is_co s rest 
  | get_selector_list _ is_co s (pos_COMPONENTS(r,c)::rest) =
    get_selector_list r is_co s (c::rest)
  | get_selector_list _ is_co s nil = nil


fun get_selector_list2 s ((total_select(selnames,s1))::rest) =
     map (fn selname => (selname,total_op_type(sorts[s],s1))) selnames
     @ get_selector_list2 s rest
  | get_selector_list2 s ((partial_select(selnames,s1))::rest) =
     map (fn selname => (selname,partial_op_type(sorts[s],s1))) selnames
     @ get_selector_list2 s rest
  | get_selector_list2 s ((sort_component(s1))::rest) =
    get_selector_list2 s rest 
  | get_selector_list2 s (pos_COMPONENTS(r,c)::rest) =
    get_selector_list2 s (c::rest)
  | get_selector_list2 s nil = nil

fun get_selector_list1 s (total_construct(fname,complist),_) =
    get_selector_list2 s complist 
  | get_selector_list1 s (partial_construct(fname,complist),_) =
    get_selector_list2 s complist 
  | get_selector_list1 s (subsort _,_) = nil 
  | get_selector_list1 s (pos_ALTERNATIVE (_,a),x) =
    get_selector_list1 s (a,x)

fun get_all_selectors s altlist =
    flat (map (get_selector_list1 s) altlist)

fun get_sorts_arg nil = nil
  | get_sorts_arg (arg_decl(vl,s)::args) = map (K s) vl @ get_sorts_arg args
  | get_sorts_arg (pos_ARG_DECL (_,ad)::args) =
    get_sorts_arg (ad::args)
 	
fun get_funlist is_co get_sel s (total_construct(fname,complist),_) =
	(if (not is_co) orelse get_sel
         then [([fname],total_op_type(sorts(flat (map get_arg_sort complist)),s))]
         else nil)
	@
        (if is_co orelse get_sel 
	 then get_selector_list null_region is_co s complist 
	 else nil)
  | get_funlist is_co get_sel s (partial_construct(fname,complist),_) =
	(if (not is_co) orelse get_sel
	 then [([fname],partial_op_type(sorts(flat (map get_arg_sort complist)),s))]
         else nil)
        @
	(if is_co orelse get_sel 
         then get_selector_list null_region is_co s complist 
         else nil)
  | get_funlist is_co get_sel s (subsort _,_) = nil 
  | get_funlist is_co get_sel s (pos_ALTERNATIVE (_,a),x) =
    get_funlist is_co get_sel s (a,x)

fun remove_pos_OP_TYPE (pos_OP_TYPE(_,t)) =
    remove_pos_OP_TYPE t
  | remove_pos_OP_TYPE t = t
 
fun get_funs_op_item r sortenv (funenv,(op_decl(fnames,ftype, attrlist),_)) =
	add_funs r sortenv (funenv,(fnames,remove_pos_OP_TYPE ftype))
  | get_funs_op_item r sortenv (funenv,(op_defn(fname,total_op_head(vars,s),t,_),_)) =
	add_funs r sortenv (funenv,([fname],total_op_type(sorts(get_sorts_arg vars),s)))  
  | get_funs_op_item r sortenv (funenv,(op_defn(fname,partial_op_head(vars,s),t,_),_)) =
	add_funs r sortenv (funenv,([fname],partial_op_type(sorts(get_sorts_arg vars),s)))
  | get_funs_op_item r sortenv (funenv,(op_defn(fname,pos_OP_HEAD(_,oh),t,x),y)) =
    get_funs_op_item r sortenv (funenv,(op_defn(fname,oh,t,x),y))
  | get_funs_op_item _ sortenv (funenv,(pos_OP_ITEM (r,oi),x)) =
    get_funs_op_item r sortenv (funenv,(oi,x))
  

fun mk_inj s s1 = total_op_type (sorts [s1],s)
fun get_injs_alt s (subsort sl,_) = map (mk_inj s) sl 
  | get_injs_alt s _ = nil

fun get_injs_datatype_decl (datatype_decl (s,altlist,l)) =
    flat (map (get_injs_alt s) altlist)
  | get_injs_datatype_decl (pos_DATATYPE_DECL (_,dd)) =
    get_injs_datatype_decl dd
	
fun get_funs_datatype_decl r is_co get_sel sortenv (funenv,datatype_decl(s,altlist, l)) =
       (foldl (add_funs r sortenv) (funenv,flat(map (get_funlist is_co get_sel s) altlist))
        handle ERR e => add_err funenv e)
  | get_funs_datatype_decl _ is_co get_sel sortenv (funenv,pos_DATATYPE_DECL (r,dd)) =
    get_funs_datatype_decl r is_co get_sel sortenv (funenv,dd)


and get_fun_sig_items is_co get_sel sortenv (funenv,op_items(s,_)) =
  	foldl (get_funs_op_item null_region sortenv) (funenv,s)
  | get_fun_sig_items is_co get_sel sortenv (funenv,datatype_items (l,_)) =
        foldl (get_funs_datatype_decl null_region is_co get_sel sortenv) (funenv,l)
  | get_fun_sig_items is_co get_sel sortenv (funenv,pos_SIG_ITEMS(_,si)) =
    get_fun_sig_items is_co get_sel sortenv (funenv,si)
  | get_fun_sig_items is_co get_sel sortenv (funenv,_) = funenv


and get_funs_bit sortenv (funenv,sig_items(l)) =
	foldl (get_fun_sig_items false true sortenv) (funenv,[l])
  | get_funs_bit sortenv (funenv,free_datatype(l,_)) =
        foldl (get_funs_datatype_decl null_region false true sortenv) (funenv,l)
  | get_funs_bit sortenv (funenv,cofree_datatype(l,_)) =
        foldl (get_funs_datatype_decl null_region true true sortenv) (funenv,l)
  | get_funs_bit sortenv (funenv,sort_gen(l,_)) =
	foldl (get_fun_sig_items false true sortenv) (funenv,l)
  | get_funs_bit sortenv (funenv,sort_cogen(l,_)) =
	foldl (get_fun_sig_items true true sortenv) (funenv,l)
  | get_funs_bit sortenv (funenv,pos_BASIC_ITEMS(_,bi)) =
    get_funs_bit sortenv (funenv,bi)
  | get_funs_bit sortenv (funenv,_) = funenv

  	  
and get_funs sortenv (funenv,basic_spec bits) =
	foldl (get_funs_bit sortenv) (funenv,bits)
  | get_funs sortenv (funenv,pos_BASIC_SPEC(_,b)) =
    get_funs sortenv (funenv,b)

(* Predicates *)


fun check_sorts_pred r (sortenv:Sort_env) (pred_type args) = 
	flat (map (check_sort r sortenv) (Utils.get_sorts args))
  | check_sorts_pred _ sortenv (pos_PRED_TYPE (r,t)) =
    check_sorts_pred r sortenv t

fun add_preds (sortenv:Sort_env) ((env:Pred_env,errs),(pnames, ptype)) =
	(foldr (cons_entry1 ptype) (pnames,env),
	 check_sorts_pred null_region sortenv ptype @ errs)

fun remove_pos_PRED_TYPE (pos_PRED_TYPE(_,t)) =
    remove_pos_PRED_TYPE t
  | remove_pos_PRED_TYPE t = t

fun get_preds_pred_item sortenv (predenv,(pred_decl(pnames,ptype),_)) =
	add_preds sortenv (predenv,(pnames,remove_pos_PRED_TYPE ptype))
  | get_preds_pred_item sortenv (predenv,(pred_defn(pnames,pred_head args, formula ,_),_)) =
	add_preds sortenv (predenv,([pnames],pred_type (sorts (get_sorts_arg args))))  
  | get_preds_pred_item sortenv (predenv,(pred_defn(pnames,pos_PRED_HEAD(_,ph),formula,x),y)) =
    get_preds_pred_item sortenv (predenv,(pred_defn(pnames,ph,formula,x),y))
  | get_preds_pred_item sortenv (predenv,(pos_PRED_ITEM(_,pi),x)) =
    get_preds_pred_item sortenv (predenv,(pi,x))
  
and get_preds_sig_items sortenv (predenv,pred_items(pnames,_)) =
	foldl (get_preds_pred_item sortenv) (predenv,pnames)
  | get_preds_sig_items sortenv (predenv,pos_SIG_ITEMS(_,si)) =
    get_preds_sig_items sortenv (predenv,si)
  | get_preds_sig_items sortenv (predenv,x) = predenv
  
and get_preds_bit sortenv (predenv,sig_items(ps)) =
	foldl (get_preds_sig_items sortenv) (predenv,[ps])
  | get_preds_bit sortenv (predenv,sort_gen(ps,_)) =
	foldl (get_preds_sig_items sortenv) (predenv,ps)
  | get_preds_bit sortenv (predenv,sort_cogen(ps,_)) =
	foldl (get_preds_sig_items sortenv) (predenv,ps)
  | get_preds_bit sortenv (predenv,pos_BASIC_ITEMS(_,bi)) =
    get_preds_bit sortenv (predenv,bi)
  | get_preds_bit sortenv (predenv,x) = predenv

  	  
and get_preds sortenv (predenv,basic_spec bits) =
	foldl (get_preds_bit sortenv) (predenv,bits)
  | get_preds sortenv (predenv,pos_BASIC_SPEC(_,b)) =
    get_preds sortenv (predenv,b)

(* Variables *)

local

fun add_var r vtype (vname,(env,errs)) =
  case Symtab_sid.lookup (env,vname) of
  	None => (Symtab_sid.update ((vname,vtype),env),errs)
  	| Some s => 
  	    if SORT_eq(s,vtype) then (Symtab_sid.update ((vname,vtype),env),errs)
  	    else (env,(perr r^"Doubly declared variable: "^print_SIMPLE_ID vname)::errs)

fun add_vars r (sortenv:Sort_env) ((env:Var_env,errs),(vnames,vtype)) =
	foldr (add_var r vtype) (vnames,(env,check_sort null_region sortenv vtype @ errs))

in

fun get_vars_bit r sortenv (varenv,var_items(vnames,_)) =
	foldl (add_vars r sortenv) (varenv,(vnames))
  | get_vars_bit _ sortenv (varenv,pos_BASIC_ITEMS(r,bi)) =
    get_vars_bit r sortenv (varenv,bi)
  | get_vars_bit _ sortenv (varenv,_) = varenv


  	  
and get_vars sortenv (varenv,basic_spec bits) = 
	foldl (get_vars_bit null_region sortenv) (varenv,bits)
  | get_vars sortenv (varenv,pos_BASIC_SPEC(_,b)) =
    get_vars sortenv (varenv,b)

end


(* Axioms *)


fun prefix_id s (simple_id (toks,id,line)) =
    simple_id (token (s^"_")::toks,s^"_"^id,line)
  | prefix_id s (compound_id ((toks,id,line),cids)) =
    compound_id ((token (s^"_")::toks,s^"_"^id,line),cids)

fun combine_2ids (simple_id (toks1,id1,line1),simple_id (toks2,id2,line2)) =
    simple_id (toks1@token "_"::toks2,id1^"_"^id2,line1)
  | combine_2ids (compound_id ((toks1,id1,line1),cids),simple_id (toks2,id2,line2)) =
    compound_id ((toks1@token "_"::toks2,id1^"_"^id2,line1),cids)
  | combine_2ids (simple_id (toks1,id1,line1),compound_id ((toks2,id2,line2),cids)) =
    compound_id ((toks1@token "_"::toks2,id1^"_"^id2,line1),cids)
  | combine_2ids (compound_id ((toks1,id1,line1),cids1),compound_id ((toks2,id2,line2),cids2)) =
    compound_id ((toks1@token "_"::toks2,id1^"_"^id2,line1),cids1@cids2)

fun combine_nids nil = raise ERROR
  | combine_nids (id::ids) = foldl combine_2ids (id,ids)
  
fun prefix_id2 s id1 id2 = prefix_id s (combine_2ids (id1,id2))
fun prefix_idn s ids = prefix_id s (combine_nids ids)

local

fun is_label (label_anno _) = true
  | is_label (pos_ANNO(_,a)) = is_label a
  | is_label _ = false
  
fun get_labels annos = filter is_label annos

fun get_vars (vars,s) = map mk_var  vars

fun get_args (arg_decl(vars,s)) = map mk_var  vars
  | get_args (pos_ARG_DECL(_,ad)) = get_args ad

fun get_varlist(vars:VAR_DECL list) =	flat(map get_vars vars)

fun get_arglist(args:ARG_DECL list) =	flat(map get_args args)

fun argdecl2vardecl (arg_decl(vars,s)) = (vars,s)
  | argdecl2vardecl (pos_ARG_DECL(_,ad)) = 
    argdecl2vardecl ad

fun argdecllist2vardecllist args =  map argdecl2vardecl args 
  
fun get_type (total_select(selnames,s)) = map (K s) selnames
  | get_type (partial_select(selnames,s)) = map (K s) selnames
  | get_type (sort_component(s)) = [s]
  | get_type (pos_COMPONENTS (_,c)) =
    get_type c

 
fun make_vars cnt (s::rest) =
    let val newname = "X"^radixstring (10, "0",cnt)
    in  ([token newname],newname,s)::make_vars (cnt+1) rest
    end
  | make_vars cnt nil = nil
 
fun make_vars' cnt (s::rest) =
    let val newname = "Y"^radixstring (10, "0",cnt)
    in  ([token newname],newname,s)::make_vars' (cnt+1) rest
    end
  | make_vars' cnt nil = nil


fun get_axioms_sel (s,conssymb,is_total,varastermlist,vardecl_list) (Some (s1,selname),v) =
    let val cons_app = application (conssymb, terms varastermlist)
        fun def_cond phi =
          if is_total then phi
          else implication(atom(definedness cons_app),phi)
    in
    Some (quantification (forall,vardecl_list,def_cond(atom(strong_equation(
  		application (op_symb(selname,None),terms [cons_app]),
                v )))),
           [label_anno (prefix_id "ga_selector" selname)])
    end
  | get_axioms_sel p (None,v) = None

fun get_axioms_undef_sel (conssymb as op_symb(consname,_),varastermlist,vardecl_list) (selname,seltype) =
    (quantification (forall,vardecl_list,negation(atom(definedness(
  		application (op_symb(selname,Some seltype),terms [application (conssymb,
  						terms varastermlist)]))))),
           [label_anno (prefix_id2 "ga_selector_undef" selname consname)])

fun make_var_decl (x,v,s) = ([(v,None)],s)
fun make_var_decl' (x,v,s) = ([(v^"'",None)],s)
fun make_varasterm (_,s,ln) = (mk_var  (s,ln))

fun get_selnames (total_select (selnames,s1)) = map (Some o (pair s1)) selnames
  | get_selnames (partial_select (selnames,s1)) = map (Some o (pair s1)) selnames
  | get_selnames (sort_component(s)) = [None]
  | get_selnames (pos_COMPONENTS (_,c)) =
    get_selnames c

fun get_axioms_dt1 s sels is_total fname complist =
  let
        val fsymb = op_symb(fname,None)
  	val fargtypes = flat (map get_type complist)
	val varsortlist = make_vars 0 fargtypes
	val vardecl_list = map make_var_decl varsortlist
	val varastermlist = map make_varasterm varsortlist
        val current_sels = get_selector_list2 s complist
        val other_sels = sels \\ current_sels
  in
  mapfilter (get_axioms_sel (s,fsymb,is_total,varastermlist,vardecl_list))
                (zip (flat (map get_selnames complist),varastermlist))
  @ map (get_axioms_undef_sel (fsymb,varastermlist,vardecl_list)) other_sels
  end
fun get_axioms_dt s sels (total_construct(fname,complist),_) =
  get_axioms_dt1 s sels true fname complist
  | get_axioms_dt s sels (partial_construct(fname,complist),x) =
  get_axioms_dt1 s sels false fname complist
  | get_axioms_dt s sels (subsort s1,_) = 
    flat(map(fn sub =>
       map (get_axioms_undef_sel 
           (op_symb(Overload.INJECTION_NAME,Some (total_op_type(sorts [sub],s))),
            [mk_var ("X",s1)],[([("X",None)],sub)])) sels) s1)
  | get_axioms_dt s sels (pos_ALTERNATIVE (_,a),x) =
    get_axioms_dt s sels (a,x)

 fun get_axioms_datadecl (datatype_decl(s,altlist, l)) = 
     flat(map (get_axioms_dt s (remove_dups (get_all_selectors s altlist))) altlist)
   | get_axioms_datadecl (pos_DATATYPE_DECL(_,dd)) =
     get_axioms_datadecl dd

 fun get_axiom_cons_inj (consname,nil) = nil
   | get_axiom_cons_inj (consname,complist) =
   let
  	val fargtypes = flat (map get_type complist)
	val varsortlist = make_vars 0 fargtypes
	val varsortlist' = make_vars' 0 fargtypes
	val vardecl_list = map make_var_decl varsortlist
	val vardecl_list' = map make_var_decl varsortlist'
	val varastermlist = map make_varasterm varsortlist
	val varastermlist' = map make_varasterm varsortlist'
  in
  [(quantification (forall,vardecl_list@vardecl_list',equivalence(atom(
        strong_equation (application (op_symb(consname,None),terms varastermlist),
                         application (op_symb(consname,None),terms varastermlist') )),
        conjunction (map (atom o strong_equation) (zip (varastermlist,varastermlist')))
    )),
    [label_anno (prefix_id "ga_injective" consname)])]
  end

 fun get_axioms_cons_inj (s,total_construct(fname,complist)) =
     get_axiom_cons_inj (fname,complist)
   | get_axioms_cons_inj (s,partial_construct(fname,complist)) =
     get_axiom_cons_inj (fname,complist)
   | get_axioms_cons_inj (s,subsort s1) =
     nil
   | get_axioms_cons_inj (s,pos_ALTERNATIVE(_,a)) =
     get_axioms_cons_inj (s,a)
 
 fun get_fargtypes (total_construct(fname,complist)):SORT list = flat (map get_type complist)
   | get_fargtypes (partial_construct(fname,complist)) = flat (map get_type complist)
   | get_fargtypes (subsort (s1)) = s1
   | get_fargtypes (pos_ALTERNATIVE(_,a)) =
     get_fargtypes a
 
 fun get_app (total_construct(consname,complist)) termlist =
      application (op_symb(consname,None),terms termlist)
   | get_app (partial_construct(consname,complist)) termlist =
      application (op_symb(consname,None),terms termlist)
   | get_app (subsort s1) [term] = term
   | get_app (subsort _) _ = raise ERROR
   | get_app (pos_ALTERNATIVE (_,a)) termlist =
     get_app a termlist

 fun get_name (total_construct(consname,complist)) = consname
   | get_name (partial_construct(consname,complist)) = consname
   | get_name (subsort ([s1])) = s1
   | get_name (subsort _) = raise ERROR
   | get_name (pos_ALTERNATIVE (_,a)) =
     get_name a

 fun get_axiom_cons_disjoint s alt1 alt2 =
   let
	val fargtypes1 = get_fargtypes alt1	
	val fargtypes2 = get_fargtypes alt2
	val varsortlist1 = make_vars 0 fargtypes1
	val varsortlist2 = make_vars' 0 fargtypes2
	val vardecl_list1 = map make_var_decl varsortlist1
	val vardecl_list2 = map make_var_decl varsortlist2
	val varastermlist1 = map make_varasterm varsortlist1
	val varastermlist2 = map make_varasterm varsortlist2
       
    in
    (quantification (forall,vardecl_list1@vardecl_list2,
         negation(atom(strong_equation(
            get_app alt1 varastermlist1,
            get_app alt2 varastermlist2)))),
     [label_anno (prefix_id2 "ga_disjoint" (get_name alt1) (get_name alt2))])
   end

      
 fun get_axioms_freedt (s,alt,altlist) =
     get_axioms_cons_inj (s,alt)@map (get_axiom_cons_disjoint s alt) altlist
  
 fun get_axioms_freedatadecl1 (s,nil) = nil
   | get_axioms_freedatadecl1 (s,alt::altlist) = 
     get_axioms_freedt (s,alt,altlist) @ get_axioms_freedatadecl1 (s,altlist)
 
 fun mk_subsort s = subsort([s])
  
 fun expand_alt (subsort (slist),_) = map (mk_subsort) slist
   | expand_alt (pos_ALTERNATIVE (r,subsort slist),_) =
     map ((fn x => (pos_ALTERNATIVE(r,x))) o mk_subsort) slist
   | expand_alt (alt,_) = [alt]
   
 fun get_axioms_freedatadecl (datatype_decl(s,altlist, l)) = 
     get_axioms_freedatadecl1 (s,flat(map expand_alt altlist))
   | get_axioms_freedatadecl (pos_DATATYPE_DECL (_,dd)) =
     get_axioms_freedatadecl dd

in

fun get_bin_type (total_op_type (sorts [t1,t2],t3)) =
    if SORT_eq(t1,t2) andalso SORT_eq(t2,t3) then t1
    else t1 (* raise exn!! ???*)
  | get_bin_type (partial_op_type (sorts [t1,t2],t3)) =
    if SORT_eq(t1,t2) andalso SORT_eq(t2,t3) then t1
    else t1 (* raise exn!! ???*)   
  | get_bin_type (pos_OP_TYPE (_,t)) =
    get_bin_type t
  | get_bin_type _ = raise ERROR
 
fun get_axioms_attr tt associative f =
    let val t = get_bin_type tt
        val x = mk_var1 "x"
        val y = mk_var1 "y"
        val z = mk_var1 "z"
    in
    [(quantification (forall,[([("x",None),("y",None),("z",None)],t)],
        atom(strong_equation(application(op_symb(f,None),terms[x,
                                 application(op_symb(f,None),terms[y,z])]),
                             application(op_symb(f,None),terms[application(op_symb(f,None),
                                                                terms[x,y]),
                                                               z]))))
  				,[label_anno (prefix_id "ga_assoc" f)])]
    end			
  | get_axioms_attr tt commutative f =
    let val t = get_bin_type tt
        val x = mk_var1 "x"
        val y = mk_var1 "y"
    in
    [(quantification (forall,[([("x",None),("y",None)],t)],
        atom(strong_equation(application(op_symb(f,None),terms[x,y]),
                             application(op_symb(f,None),terms[y,x]))))
  				,[label_anno (prefix_id "ga_comm" f)])]
    end			
  | get_axioms_attr tt idempotent f =
    let val t = get_bin_type tt
        val x = mk_var1 "x"
    in
    [(quantification (forall,[([("x",None)],t)],
        atom(strong_equation(application(op_symb(f,None),terms[x,x]),
                             x)))
  				,[label_anno (prefix_id "ga_idem" f)])]
    end
  | get_axioms_attr tt (unit_op_attr u) f =
    let val t = get_bin_type tt
        val x = mk_var1 "x"
    in
    [(quantification (forall,[([("x",None)],t)],
        atom(strong_equation(application(op_symb(f,None),terms[x,u]),
                             x)))
  				,[label_anno (prefix_id "ga_right_unit" f)]),
     (quantification (forall,[([("x",None)],t)],
        atom(strong_equation(application(op_symb(f,None),terms[u,x]),
                             x)))
  				,[label_anno (prefix_id "ga_left_unit" f)])]
    end
  | get_axioms_attr tt (pos_OP_ATTR (r,att)) f =
    get_axioms_attr tt att f
  
fun get_axioms_attr1 (op_list,t) attr =
    flat (map (get_axioms_attr t attr) op_list)
fun get_axioms_op_item (op_decl (op_list,t,attrlist),_) : L_FORMULA list =
    flat (map (get_axioms_attr1 (op_list,t)) attrlist)
  | get_axioms_op_item (op_defn (f,total_op_head(vlist,s),t,annos1),annos2) =
	[(quantification (forall,(argdecllist2vardecllist vlist),atom(strong_equation(
  				application(op_symb(f,None),terms (get_varlist (argdecllist2vardecllist vlist))),
                                sorted_term(t,s) ))),
           get_labels (annos1@annos2))]
  | get_axioms_op_item (op_defn (f,partial_op_head(vlist,s),t,annos1),annos2) =
	[(quantification (forall, (argdecllist2vardecllist vlist),atom(strong_equation(
  				application(op_symb(f,None),terms (get_varlist(argdecllist2vardecllist vlist))),
                                sorted_term(t,s) ))),
           get_labels (annos1@annos2))] 
  | get_axioms_op_item (op_defn (f,pos_OP_HEAD(_,oh),t,annos),x) =
    get_axioms_op_item (op_defn (f,oh,t,annos),x)
  | get_axioms_op_item (pos_OP_ITEM(r,oi),x) =
    get_axioms_op_item (oi,x)
 
fun get_axioms_sortitems (subsort_defn (s1,v,s,phi,annos),_) =
    [(pos_FORMULA(get_pos_FORMULA phi,get_brack_FORMULA phi,
         (quantification (forall,[([v],s)],
  	                  equivalence (atom (membership (mk_var v,s1)),phi)))),get_labels annos)]
  | get_axioms_sortitems (pos_SORT_ITEM(r,subsort_defn (s1,v,s,phi,annos)),_) =
    [(pos_FORMULA(r,false,
         (quantification (forall,[([v],s)],
  	                  equivalence (atom (membership (mk_var v,s1)),phi)))),get_labels annos)]
  | get_axioms_sortitems (pos_SORT_ITEM(_,si),x) =
    get_axioms_sortitems (si,x)
  | get_axioms_sortitems _ = []

fun get_axioms_pred_item (pred_defn (p,pred_head(vlist),(phi,l),annos1),annos2) =
	[(quantification (forall, (argdecllist2vardecllist vlist),
	                           equivalence 
	                                 (atom 
	                                     (predication
	                                        (pred_symb(p,None),
	                                              terms (get_varlist(argdecllist2vardecllist vlist)))),
	                                   phi)),get_labels (annos1@annos2))]
   | get_axioms_pred_item (pred_defn (p,pos_PRED_HEAD(_,ph),phi,annos),x) =
     get_axioms_pred_item (pred_defn (p,ph,phi,annos),x)
   | get_axioms_pred_item (pos_PRED_ITEM (_,pi),x) =
     get_axioms_pred_item (pi,x)
   | get_axioms_pred_item _ = []

and get_axioms_sig (sort_items (ls,_)) = flat (map get_axioms_sortitems ls)
  | get_axioms_sig (datatype_items (decl,_)) = flat (map get_axioms_datadecl decl)
  | get_axioms_sig (op_items (decl,_)) = flat (map get_axioms_op_item decl)
  | get_axioms_sig (pred_items (decl,_)) = flat (map get_axioms_pred_item decl)
  | get_axioms_sig (pos_SIG_ITEMS (_,si)) = get_axioms_sig si
  
  
and get_axioms_phi (f,l) = (f,l)

and get_axioms_local_var_axioms1 (vlist, []) = []  
  | get_axioms_local_var_axioms1 (vlist, (phi,l)::phis) =
    (quantification (forall,vlist,phi) ,l)::get_axioms_local_var_axioms1 (vlist, phis)
    
and get_axioms_local_var_axioms r sortenv (ax,errs) (vlist,phis) = 
    (ax @ get_axioms_local_var_axioms1 (vlist,phis),
     errs @ check_sort_vlist r sortenv vlist)

and get_fun_symbs (f,types) =
    map (fn t => op_symb(f, Some t)) types
    
and get_sort_gen is_co decls =
    let val srts = flat (map get_sort_sig_items decls)
        val (funenv,_) = foldl (get_fun_sig_items is_co false Symtab_id.empty) ((Symtab_id.empty,[]),decls)
        val funlist = Symtab_id.dest funenv
        val funs = flat (map get_fun_symbs funlist)
    in if is_co
       then (sort_cogen_ax(srts,funs),[label_anno (prefix_idn "ga_cogenerated" srts)])
       else (sort_gen_ax(srts,funs),[label_anno (prefix_idn "ga_generated" srts)])
    end
  
and get_free_sort_gen is_co decls =
    let val srts = map get_sort_Datatype decls
        val (funenv,_) = 
          foldl (get_funs_datatype_decl null_region is_co false Symtab_id.empty)
                ((Symtab_id.empty,[]),decls)
        val injs = flat (map get_injs_datatype_decl decls)
        val funlist = Symtab_id.dest funenv
        val funlist1 = if null injs then funlist
                       else (Overload.INJECTION_NAME,injs)::funlist
        val funs = flat (map get_fun_symbs funlist1)
    in if is_co
       then (sort_cofree_ax(srts,funs),[label_anno (prefix_idn "ga_cofree" srts)])
       else (sort_gen_ax(srts,funs),[label_anno (prefix_idn "ga_generated" srts)])
    end
    
and get_axioms_bit r sortenv ((ax,errs), (axiom_items (f,_))) =  
    (ax @ map get_axioms_phi f , errs)
  | get_axioms_bit r sortenv ((ax,errs), (local_var_axioms (vars,axs,_))) =  
    get_axioms_local_var_axioms r sortenv (ax,errs) (vars,axs)
  | get_axioms_bit r sortenv ((ax,errs), (sig_items decl)) =    
    (ax @ get_axioms_sig decl , errs)
  | get_axioms_bit r sortenv ((ax,errs), (free_datatype (decls,_))) = 
     (ax @ get_free_sort_gen false decls :: flat (map get_axioms_datadecl decls) 
      @ flat (map get_axioms_freedatadecl decls)  , errs)
  | get_axioms_bit r sortenv ((ax,errs), (cofree_datatype (decls,_))) = 
     (ax @ get_free_sort_gen true decls :: flat (map get_axioms_datadecl decls), errs)
  | get_axioms_bit r sortenv ((ax,errs), (sort_gen (decls,_))) = 
    (ax @ get_sort_gen false decls :: flat (map get_axioms_sig decls)  , errs)
  | get_axioms_bit r sortenv ((ax,errs), (sort_cogen (decls,_))) = 
    (ax @ get_sort_gen true decls :: flat (map get_axioms_sig decls)  , errs)
  | get_axioms_bit _ sortenv ((ax,errs), (pos_BASIC_ITEMS (r,bit))) =
    get_axioms_bit r sortenv ((ax,errs), bit)
  | get_axioms_bit _ sortenv ((ax,errs), _) = (ax,errs) 


and strip_empty_quant (quantification (q,[],phi),l) =
    strip_empty_quant (phi,l)
  | strip_empty_quant (pos_FORMULA (r,b,phi),l) =
    let val (phi',l') = strip_empty_quant (phi,l)
    in (pos_FORMULA (r,b,phi'),l')
    end
  | strip_empty_quant x = x
  
and get_axioms sortenv (basic_spec bits) = 
    let val (phis,errs) = (foldl (get_axioms_bit null_region sortenv) (([],[]),bits))
    in (map strip_empty_quant phis,errs)
    end
  | get_axioms sortenv (pos_BASIC_SPEC (_,b)) =
    get_axioms sortenv b
end


fun mk_var s v = ([v], s)
fun normalize_VAR_DECL (vlist, s) =
    map (mk_var s) vlist

fun normalize_TERM (a:TERM):TERM = 
case a of
   application (opsymb, terms tt) => 
   application (opsymb, terms (map normalize_TERM tt))
 | sorted_term (t, s) => 
   sorted_term (normalize_TERM t, s) 
 | cast (t, s) =>  
   cast (normalize_TERM t, s) 
 | conditional (t1, phi, t2) =>
   conditional (normalize_TERM t1, normalize_FORMULA phi, normalize_TERM t2)
 | pos_TERM (r,b,t) =>
   pos_TERM(r,b,normalize_TERM t)
 | t => t
 
and normalize_ATOM (a:ATOM):ATOM =
case a of
   predication (psymb,ts) => 
   predication (psymb,terms (map normalize_TERM (get_terms ts)))
 | definedness t =>  
   definedness (normalize_TERM t)
 | existl_equation (t1 , t2) =>  
   existl_equation (normalize_TERM t1 , normalize_TERM t2)
 | strong_equation (t1 , t2) =>  
   strong_equation (normalize_TERM t1 , normalize_TERM t2)
 | membership (t, s)  => 
   membership (normalize_TERM t, s)  
 | ttrue => ttrue
 | ffalse => ffalse

and normalize_FORMULA (phi:FORMULA):FORMULA =
case phi of
 quantification (quant ,vlist, f) => 
    let val nf = normalize_FORMULA f
        val nvlist = flat (map normalize_VAR_DECL vlist) 
    in case nf of
       (quantification (quant', vlist', f')) =>
           if quant=quant'
           then quantification (quant, nvlist@vlist',f')
           else quantification (quant, nvlist, nf)
       | _ => quantification (quant, nvlist, nf)
    end   
 | conjunction ff =>  conjunction (map normalize_FORMULA ff)
 | disjunction ff => disjunction (map normalize_FORMULA ff)
 | implication (f1, f2) => implication (normalize_FORMULA f1, normalize_FORMULA f2)
 | equivalence (f1, f2) => equivalence (normalize_FORMULA f1, normalize_FORMULA f2)
 | negation f =>  negation (normalize_FORMULA f)
 | atom a => atom (normalize_ATOM a)
 | sort_gen_ax x => sort_gen_ax x
 | sort_cogen_ax x => sort_cogen_ax x
 | sort_cofree_ax x => sort_cofree_ax x
 | pos_FORMULA (r,b,phi) => pos_FORMULA (r,b,normalize_FORMULA phi)
 | _ => raise ERROR

and normalize_L_FORMULA (phi,l) = (normalize_FORMULA phi,l)


(************************* Static analysis of basic specifications **************************)
(* This function is still quite incorrect: it assumes non-linear visibility *)
(* Moreover, some checks are missing *)

fun basic_analysis (html:bool) 
    (bspec : BASIC_SPEC,
     (Sigma : sign,
      an    : ANNO list))
     : (BASIC_SPEC * sign * sign * L_FORMULA list * string list) =
  let
    (* Collect local environment *)
    val (subsortenv1,varenv1,funenv1,predenv1) = Sigma
    val (sortlist1, _,_,_,_) = env_to_list Sigma
    val sortlist = remove_dups_eq SORT_eq (sortlist1 @ (get_sorts bspec))   
    val sortenv = Symtab_id.make (map null_entry sortlist)
        handle DUP => raise ERR "Internal error: sort list incorrect"
    val (subsortenv',errs1) = 
        get_subsorts sortenv ((Symtab_id.make (identity sortlist),[]),bspec)
        handle DUP => raise ERR "Internal error: reflexive sort list incorrect"
    val (varenv',errs2) = get_vars sortenv ((Symtab_sid.empty,errs1),bspec)
    val (funenv',errs3) = get_funs sortenv ((Symtab_id.empty,errs2),bspec) 
    val (predenv',errs4) = get_preds sortenv ((Symtab_id.empty,errs3),bspec)
    
    (* Merge local env with new declarations *)
    val env = merge_lenvs 
             (Sigma, 
               (* Variables from the local env are not taken,
                  since their scope is limited to the current basic spec *)
              (subsortenv',Symtab_sid.empty,funenv',predenv'))
    val (subsortenv,varenv,funenv,predenv) = env
    val Sigma_frag = (subsortenv',varenv',funenv',predenv')
    val Sigma' = (subsortenv,varenv',funenv,predenv)
    
    (* Mixfix analysis *)
    val (bspec1,errs5) = mixfix_parse html (Sigma',an) bspec
    (* bspec1 is the abstract syntax tree with axioms and terms all parsed *)
    
    (* Get axioms *)
    val (ax,errs6) = get_axioms sortenv bspec1

    (* Overload resolution *)
    val (errs7,resax) = overload_resolution false (Sigma',ax)
    val normax = map normalize_L_FORMULA resax

  in (bspec1,Sigma',Sigma_frag,normax,errs4@errs5@errs6@errs7)	
  end  




fun get_ids (lenv,an) = LocalEnv.get_ids lenv

fun parse_axiom is_goal Sigma an ax =
    (Global.cur_file := "";
     IsabelleMixfixParser.parse_axiom is_goal (Sigma,an) ax)

fun parse_term Sigma an t =
    (Global.cur_file := "";
     IsabelleMixfixParser.parse_term (Sigma,an) t)

end
