(* ------------------------------------------------------------------------ *)
(* options.sml                                                              *)
(* parsing of command line options for the casl standalone tool             *)
(* Pascal Schmidt <pharao90@tzi.de>                                         *)
(* ------------------------------------------------------------------------ *)

(* $Id: options.sml,v 1.1 2004/02/13 11:29:10 7till Exp $ *)

(* todo: 
   warn if option is not known
   Add option for textual env with/without axioms
*)

signature OptionsSig
=
sig

  datatype env_type = casenv | hcasenv | fcasenv
  datatype enc_type = subpcfol | subcfol | cfol | sol | cfol1 | sol1
  datatype out_type = aterm | text | latex | xml

  type env_opt = env_type list * enc_type * out_type list

  datatype sp_type = spec_aterm | spec_xml | spec_gen_aterm | spec_text
  type sp_opt = sp_type list

  datatype parser_opt = yacc | isabelle

  datatype input_type = casl | casfix | gen_aterm | in_xml
  datatype static_type = static | nostatic
  datatype bin_type = bin | nobin
  datatype basic_type = empty | basic
  type input_opt = input_type * static_type * bin_type * basic_type

  datatype latex_type = tex | notex
  datatype envir_type = env | noenv
  datatype graph_type = graph | nograph
  type output_opt = envir_type * latex_type * bin_type * graph_type

  datatype gr_modes = dot | dotps | davinci
  type graph_opt = gr_modes list

  type quiet_opt = bool

  type casl_options = input_opt * parser_opt * output_opt * env_opt
                      * graph_opt * quiet_opt * sp_opt

  val get_options   : string list -> casl_options

  val input_casl    : casl_options -> bool
  val input_static  : casl_options -> bool
  val input_bin     : casl_options -> bool
  val input_basic   : casl_options -> bool
  val input_gen_aterm : casl_options -> bool
  val input_xml     : casl_options -> bool

  val sp_aterm      : casl_options -> bool
  val sp_xml        : casl_options -> bool
  val sp_gen_aterm  : casl_options -> bool
  val sp_text       : casl_options -> bool
  
  val output_env    : casl_options -> bool
  val output_tex    : casl_options -> bool
  val output_bin    : casl_options -> bool
  val output_graph  : casl_options -> bool
  
  val parser_yacc   : casl_options -> bool
  
  val env_casenv    : casl_options -> bool
  val env_hcasenv   : casl_options -> bool
  val env_fcasenv   : casl_options -> bool
  
  val env_subpcfol  : casl_options -> bool
  val env_subcfol   : casl_options -> bool
  val env_cfol      : casl_options -> bool
  val env_sol       : casl_options -> bool
  val env_cfol1     : casl_options -> bool
  val env_sol1      : casl_options -> bool

  val env_aterm     : casl_options -> bool
  val env_text      : casl_options -> bool
  val env_latex     : casl_options -> bool
  val env_xml       : casl_options -> bool

  val graph_dot     : casl_options -> bool
  val graph_ps      : casl_options -> bool
  val graph_davinci : casl_options -> bool

  val quiet         : casl_options -> bool
end

structure Options : OptionsSig
=
struct
open Utils
infix mem
(* ------------------------------------------------------------------------ *)
(* Useful exceptions                                                        *)
(* ------------------------------------------------------------------------ *)

(* Exception Malformed is raised when an option does not match the format   *)
(* key=value,...,value                                                      *)

exception Malformed

(* ------------------------------------------------------------------------ *)
(* Datatype declarations and default values                                 *)
(* ------------------------------------------------------------------------ *)

datatype env_type = casenv | hcasenv | fcasenv
datatype enc_type = subpcfol | subcfol | cfol | sol | cfol1 | sol1
datatype out_type = aterm | text | latex | xml

type env_opt = env_type list * enc_type * out_type list

val env_default = ([casenv],subpcfol,[aterm]):env_opt

datatype sp_type = spec_aterm | spec_xml | spec_gen_aterm | spec_text
type sp_opt = sp_type list

val sp_default = [spec_aterm]

datatype parser_opt = yacc | isabelle

val parser_default = yacc

datatype input_type = casl | casfix | gen_aterm | in_xml
datatype static_type = static | nostatic
datatype bin_type = bin | nobin
datatype basic_type = empty | basic
type input_opt = input_type * static_type * bin_type * basic_type

val input_default = (casl,static,bin,basic):input_opt

datatype latex_type = tex | notex
datatype envir_type = env | noenv
datatype graph_type = graph | nograph
type output_opt = envir_type * latex_type * bin_type * graph_type

val output_default = (noenv,notex,bin,nograph):output_opt

datatype gr_modes = dot | dotps | davinci
type graph_opt = gr_modes list

val graph_default = ([davinci]):graph_opt

type quiet_opt = bool

val quiet_default = false:quiet_opt

type casl_options = input_opt * parser_opt * output_opt * env_opt
                    * graph_opt * quiet_opt * sp_opt

(* ------------------------------------------------------------------------ *)
(* Utility functions                                                        *)
(* ------------------------------------------------------------------------ *)

fun is_eq ch x = x=ch

fun split_at ch st = String.fields (is_eq ch) st

fun is_some (Some x) = true
  | is_some _        = false

fun strip_some (Some x) = x
  | strip_some _        = raise Malformed

fun decide [] x     = x
  | decide (h::t) x = if (is_some h) then
                          decide t (strip_some h)
                        else
                          decide t x

fun gather [] x     = x
  | gather (h::t) x = if (is_some h) then
                        gather t (x @ [strip_some h])
                      else
                        gather t x

(* ------------------------------------------------------------------------ *)
(* Option splitting                                                         *)
(* ------------------------------------------------------------------------ *)

fun split_opt_arg x = split_at (#"=") x

fun split_args x = (map split_opt_arg) x

fun split_arg_list [x,y] = (x,split_at (#",") y)
  | split_arg_list [x]   = (x,[])
  | split_arg_list _     = raise Malformed

fun split_opts x = (map split_arg_list) (split_args x)

fun opt_filter x []             = []
  | opt_filter x ((id,args)::t) = if id=x then
                                    (id,args) :: (opt_filter x t)
                                  else
                                    opt_filter x t

fun has_opt x l = (opt_filter x l)<>[]

fun join_args [] x            = x
  | join_args ((_,args)::t) x = join_args t (x @ args) 

fun get_args x l = join_args (opt_filter x l) []

(* ------------------------------------------------------------------------ *)
(* Option evaluation                                                        *)
(* ------------------------------------------------------------------------ *)

(* ------------------------------------------------------------------------ *)
(* -quiet/-noquiet                                                          *)
(* ------------------------------------------------------------------------ *)

fun get_quiet_opt l = if (has_opt "-quiet" l) then
                        if (has_opt "-noquiet" l) then
                          false:quiet_opt
                        else
                          true:quiet_opt
                      else
                        false:quiet_opt

(* ------------------------------------------------------------------------ *)
(* -env                                                                     *)
(* ------------------------------------------------------------------------ *)

fun flt_env [] = []
  | flt_env ("CasEnv"::t) = if ("noCasEnv" mem t) then
                              flt_env t
                            else
                              "CasEnv" :: (flt_env t)
  | flt_env ("FCasEnv"::t) = if ("noFCasEnv" mem t) then
                               flt_env t
                             else
                               "FCasEnv" :: (flt_env t)
  | flt_env ("HCasEnv"::t) = if ("noHCasEnv" mem t) then
                               flt_env t
                             else
                               "HCasEnv" :: (flt_env t)
  | flt_env ("aterm"::t) = if ("noaterm" mem t) then
                             flt_env t
                           else
                             "aterm" :: (flt_env t)
  | flt_env ("text"::t) = if ("notext" mem t) then
                            flt_env t
                          else
                            "text" :: (flt_env t)
  | flt_env ("latex"::t) = if ("nolatex" mem t) then
                             flt_env t
                           else
                             "latex" :: (flt_env t)
  | flt_env ("xml"::t) = if ("noxml" mem t) then
                           flt_env t
                         else
                           "xml" :: (flt_env t)
  | flt_env (h::t) = h :: (flt_env t)

fun which_enc "SubPCFOL" = Some subpcfol
  | which_enc "SubCFOL"  = Some subcfol
  | which_enc "CFOL"     = Some cfol
  | which_enc "SOL"      = Some sol

  | which_enc "CFOL1"    = Some cfol1
  | which_enc "SOL1"     = Some sol1

  | which_enc _          = None

fun which_env "CasEnv"   = Some casenv
  | which_env "HCasEnv"  = ( print ("Warning: HCasEnv not implemented, " ^
                             "falling back to CasEnv." ) ;
                             Some casenv )
  | which_env "FCasEnv"  = Some fcasenv
  | which_env _          = None

fun which_out "text"     = Some text
  | which_out "aterm"    = Some aterm
  | which_out "latex"    = Some latex
  | which_out "xml"      = Some xml
  | which_out _          = None
 
fun map_enc l = (map which_enc) l

fun map_env l = (map which_env) l

fun map_out l = (map which_out) l

fun get_encoding l = decide (map_enc l) subpcfol

fun get_environment l = gather (map_env l) []

fun get_output l = gather (map_out l) []

fun get_env_opt l = let
                      val x = flt_env ( ["CasEnv","aterm"] @
                                        (get_args "-env" l))
                    in
                      if (has_opt "-env" l) then
                        (get_environment x,get_encoding x,get_output x)
                        :env_opt
                      else
                        env_default 
                    end

(* ------------------------------------------------------------------------ *)
(* -parser                                                                  *)
(* ------------------------------------------------------------------------ *)

fun which_parser "yacc"     = Some yacc
  | which_parser "isabelle" = Some isabelle
  | which_parser _          = None

fun map_parser l = (map which_parser) l

fun get_parser l = decide (map_parser l) isabelle

fun get_parser_opt l = let
                         val x = get_args "-parser" l
                       in
                         if (has_opt "-parser" l) then
                           get_parser x
                         else
                           parser_default
                       end

(* ------------------------------------------------------------------------ *)
(* -graph                                                                   *)
(* ------------------------------------------------------------------------ *)

fun flt_gr [] = []
  | flt_gr ("dot"::t) = if ("nodot" mem t) then
                              flt_gr t
                            else
                              "dot" :: (flt_gr t)
  | flt_gr ("ps"::t) = if ("nops" mem t) then
                               flt_gr t
                             else
                               "ps" :: (flt_gr t)
  | flt_gr ("davinci"::t) = if ("nodavinci" mem t) then
                               flt_gr t
                             else
                               "davinci" :: (flt_gr t)
  | flt_gr (h::t) = h :: (flt_gr t)

fun which_grmode "dot"      = Some dot
  | which_grmode "ps"       = Some dotps
  | which_grmode "davinci"  = Some davinci
  | which_grmode _          = None

fun map_grmode l = (map which_grmode) l

fun get_grmode l = gather (map_grmode l) []

fun get_graph_opt l = let
                        val x = flt_gr (["davinci"] @ (get_args "-graph" l))
                      in
                        if (has_opt "-graph" l) then
                          get_grmode x
                        else
                          graph_default
                      end

(* ------------------------------------------------------------------------ *)
(* -input                                                                   *)
(* ------------------------------------------------------------------------ *)

fun which_input "casl"      = Some casl
  | which_input "aterm"     = Some casfix
  | which_input "gen_aterm" = Some gen_aterm
  | which_input "xml"       = Some in_xml
  | which_input _           = None

fun which_stat "static"   = Some static
  | which_stat "nostatic" = Some nostatic
  | which_stat _          = None

fun which_bin "bin"   = Some bin
  | which_bin "nobin" = Some nobin
  | which_bin _       = None

fun which_basic "empty" = Some empty
  | which_basic "basic" = Some basic
  | which_basic _       = None

fun map_input l = (map which_input) l

fun map_stat l = (map which_stat) l

fun map_bin l = (map which_bin) l

fun map_basic l = (map which_basic) l

fun get_input l = decide (map_input l) casl

fun get_stat l = decide (map_stat l) static

fun get_bin l = decide (map_bin l) bin

fun get_basic l = decide (map_basic l) basic

fun get_input_opt l = let
                         val x = get_args "-input" l
                       in
                         if (has_opt "-input" l) then
                           (get_input x,get_stat x,get_bin x,get_basic x)
                           :input_opt
                         else
                           input_default
                       end

(* ------------------------------------------------------------------------ *)
(* -output                                                                  *)
(* ------------------------------------------------------------------------ *)

fun which_envir "env"   = Some env
  | which_envir "noenv" = Some noenv
  | which_envir _       = None

fun which_tex "tex"   = Some tex
  | which_tex "notex" = Some notex
  | which_tex _       = None

fun which_graph "graph"   = Some graph
  | which_graph "nograph" = Some nograph
  | which_graph _         = None

fun map_envir l = (map which_envir) l

fun map_tex l = (map which_tex) l

fun map_graph l = (map which_graph) l

fun get_envir l = decide (map_envir l) noenv

fun get_tex l = decide (map_tex l) notex

fun get_graph l = decide (map_graph l) nograph

fun get_output_opt l = let
                         val x = get_args "-output" l
                       in
                         if (has_opt "-output" l) then
                           (get_envir x,get_tex x,get_bin x,get_graph x)
                           :output_opt
                         else
                           output_default
                       end

(* ------------------------------------------------------------------------ *)
(* -spec                                                                    *)
(* ------------------------------------------------------------------------ *)

fun sp_env [] = []
  | sp_env ("aterm"::t) = if ("noaterm" mem t) then
                              sp_env t
                            else
                              "aterm" :: (sp_env t)
  | sp_env ("xml"::t) = if ("noxml" mem t) then
                               sp_env t
                             else
                               "xml" :: (sp_env t)
  | sp_env ("gen_aterm"::t) = if ("nogen_aterm" mem t) then
				  sp_env t
                              else
				  "gen_aterm" :: (sp_env t)
  | sp_env ("text"::t) = if ("notext" mem t) then
			     sp_env t
			 else
			     "text"::(sp_env t)
  | sp_env (h::t) = h :: (sp_env t)

fun which_sp "aterm"     = Some spec_aterm
  | which_sp "xml"       = Some spec_xml
  | which_sp "gen_aterm" = Some spec_gen_aterm  
  | which_sp "text"      = Some spec_text
  | which_sp _           = None

fun map_sp l = (map which_sp) l

fun get_sp l = gather (map_sp l) []

fun get_sp_opt l =  let
                      val x = flt_env ( ["aterm"] @
                                        (get_args "-spec" l))
                    in
                      if (has_opt "-spec" l) then
                        (get_sp x):sp_opt
                      else
                        sp_default 
                    end

(* ------------------------------------------------------------------------ *)
(* global interface                                                         *)
(* ------------------------------------------------------------------------ *)

fun get_options x = let
                      val l = split_opts x
                    in
                      (get_input_opt l,get_parser_opt l,get_output_opt l,
                       get_env_opt l,get_graph_opt l,get_quiet_opt l,
                       get_sp_opt l)
                       :casl_options
                    end

(* ------------------------------------------------------------------------ *)
(* Functions for tests                                                      *)
(* ------------------------------------------------------------------------ *)

fun input_casl l = let
                     val ((x,_,_,_),_,_,_,_,_,_) = l
                   in
                    x=casl
                   end

fun input_xml l = let
                     val ((x,_,_,_),_,_,_,_,_,_) = l
                   in
                    x=in_xml
                   end

fun input_static l = let
                       val ((_,x,_,_),_,_,_,_,_,_) = l
                     in
                       x=static
                     end

fun input_gen_aterm l = 
    let
        val ((x,_,_,_),_,_,_,_,_,_) = l
    in
	x=gen_aterm
    end
                     
fun input_bin l = let
                    val ((_,_,x,_),_,_,_,_,_,_) = l
                  in
                    x=bin
                  end

fun input_basic l = let
                      val ((_,_,_,x),_,_,_,_,_,_) = l
                    in
                      x=basic
                    end

fun parser_yacc l = let
                      val (_,x,_,_,_,_,_) = l
                    in
                      x=yacc
                    end

fun output_env l = let
                     val (_,_,(x,_,_,_),_,_,_,_) = l
                   in
                     x=env
                   end

fun output_tex l = let
                     val (_,_,(_,x,_,_),_,_,_,_) = l
                   in
                     x=tex
                   end
                   
fun output_bin l = let
                     val (_,_,(_,_,x,_),_,_,_,_) = l
                   in
                     x=bin
                   end                    

fun output_graph l = let
                       val (_,_,(_,_,_,x),_,_,_,_) = l
                     in
                       x=graph
                     end

fun env_xenv l y = let
                     val (_,_,_,(x,_,_),_,_,_) = l
                   in
                     y mem x
                   end 

fun env_casenv l = env_xenv l casenv

fun env_hcasenv l = env_xenv l hcasenv

fun env_fcasenv l = env_xenv l fcasenv

fun env_xenc l y = let
                     val (_,_,_,(_,x,_),_,_,_) = l
                   in
                     x=y
                   end
                   
fun env_subpcfol l = env_xenc l subpcfol

fun env_subcfol l = env_xenc l subcfol

fun env_cfol l = env_xenc l cfol

fun env_sol l = env_xenc l sol

fun env_cfol1 l = env_xenc l cfol1

fun env_sol1 l = env_xenc l sol1


fun env_xout l y = let
                     val (_,_,_,(_,_,x),_,_,_) = l
                   in
                     y mem x
                   end
                   
fun env_aterm l = env_xout l aterm

fun env_text l = env_xout l text

fun env_latex l = env_xout l latex

fun env_xml l = env_xout l xml

fun graph_x l y = let
                    val (_,_,_,_,x,_,_) = l
                  in
                    y mem x
                  end
                
fun graph_dot l = graph_x l dot

fun graph_ps l = graph_x l dotps

fun graph_davinci l = graph_x l davinci

fun quiet (_,_,_,_,_,x,_) = x

fun sp_aterm (_,_,_,_,_,_,x) = spec_aterm mem x

fun sp_gen_aterm (_,_,_,_,_,_,x) = spec_gen_aterm mem x

fun sp_text (_,_,_,_,_,_,x) = spec_text mem x

fun sp_xml   (_,_,_,_,_,_,x) = spec_xml mem x

end
