(* *********************************************************************** *)
(*									   *)
(* Project: CASL: 							   *)
(* Author: Kolyang, University of Bremen	                 	   *)
(* Date: 21.06.97				 			   *)
(* Purpose of this file: Convert Isabelle's abstract syntax to CASL's AS   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* We use the Isabelle parser for mixfix analysis.
   This module converts an Isabelle abstract syntax tree, which is just
   an Isabelle term, to a CASL abstract syntax tree.
   The conversion function just does a recursion over the abstract syntax,
   with a case analysis according to the possible Isabelle constants.
   Mixfix symbols are represented by dynamically generated Isabelle
   constants.
*)

structure Convert : 
sig
  val convert_Formula : AS.ANNO list -> Term.term -> AS.FORMULA
  val convert_ID : Term.term -> AS.ID
  val convert_OpSymb : Term.term -> AS.OP_SYMB
  val convert_VarDeclList : Term.term -> AS.VAR_DECL list
  val convert_TTerm :  AS.ANNO list -> Term.term -> AS.TERM
  val get_OpSymb : Term.term -> AS.OP_SYMB
  val get_args : Term.term list -> Term.term -> Term.term list
  val get_list_anno : AS.ID -> AS.ANNO list -> (AS.ID * AS.ID) Library.option
end
 =
struct

open Library Term AS CASLScanner ML_interface
infix 9  $;
infixr 5 -->;
infixr --->;
infix aconv;

fun print_unparsed_formula sign (t:term) =
    let val old_depth = get_print_depth() * 2
    in
    (Syntax.show_brackets:=false;
     print_depth 10000;
     let val str = 
         Pretty.string_of((Syntax.pretty_term (#syn (Sign.rep_sg sign)) false t))
     in 
        (print_depth old_depth;str)
     end)
    end;

fun print_unparsed_term sign (t:term) =
    let val old_depth = get_print_depth() * 2
    in
    (Syntax.show_brackets:=false;
     print_depth 10000;
     let val str = 
         Pretty.string_of((Syntax.pretty_term (#syn (Sign.rep_sg sign)) false t))
     in 
        (print_depth old_depth;str)
     end)
    end;

fun convert_list (single,listcons,convertsingle) t =
    case t of
    (Const (C,T) $ N) => 
        if C=single then [convertsingle N]
        else error (C^" is not a constructor")
    | (Const (C,T) $ N1 $ N2) =>
        if C=listcons then convertsingle N1::
                 convert_list (single,listcons,convertsingle) N2
        else error (C^" is not a constructor") 
    | _ => raise ERROR
    
fun constant_OP_SYMB c = application(c, terms [])
fun constant_ID c = constant_OP_SYMB(op_symb(c,None))
fun constant_string c = constant_ID(make_sid c)

 
 (************************************ Lexical syntax ***************************)

fun convert_WORDS (t:term)   = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_DOT_WORDS t      = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_SIGN t           = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_DIGIT t          = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_NNUMBER t        = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_FLOAT t          = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_CHAR t           = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_STRING t         = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_PATH t           = case t of (Free(dd, _)) =>  dd | _ => raise ERROR
and convert_URL t            = case t of (Free(dd, _)) =>  dd | _ => raise ERROR


 (******************************* Context free syntax ***************************)

fun convert_NUMBER t=
    case t of 
        (Const("number1", T) $ N ) => convert_DIGIT N
    |   (Const("number2", T) $ N ) => convert_NNUMBER N
    | _ => raise (ERR "Isa-Conversion of NUMBER")

fun convert_WORDS1 t=
    case t of 
        (Const("word1", T) $ N ) => convert_WORDS N
    |   (Const("word2", T) $ N ) => convert_NUMBER N
    | _ => raise (ERR "Isa-Conversion of WORDS")

fun convert_ANNO t=
     case t of 
       (Const("anno", T) $ N ) => convert_STRING N
      | _ => raise (ERR "Isa-Conversion of ANNO")
fun convert_LIST t =
    case t of 
       (Free(dd, T)) => dd
     | _ => raise (ERR "Isa-Conversion of LIST")

fun convert_BRACKET t =
    case t of 
       (Const ("brackets1",_)) => "["
     | (Const ("brackets2",_)) => "]"
     | (Const ("brackets3",_)) => "{"
     | (Const ("brackets4",_)) => "}"
     | _ => raise (ERR "Isa-Conversion of BRACKET")
       
fun convert_SIGN1 t =
    case t of 
       (Const ("sign",_) $ N1) => convert_SIGN N1
     | (Const ("emptyset",T))  => "{}"
     | (Const ("lessthan",T))  => "<"
     | (Const ("times",T))     => "*"
     | (Const ("question",T))  => "?"
     | (Const ("divides",T))   => "/" 
     | (Const ("arrow",T))   => "->" 
     | (Const ("parrow",T))   => "->?"
     | (Const ("unequal",T))   => "<>"
     | _ => raise (ERR "Isa-Conversion of SIGN")

fun convert_SIGNS t =
    case t of 
       (Const ("signs1",_) $ N1) => convert_SIGN1 N1
     | (Const ("signs2",_) $ N1 $ N2) => convert_SIGN1 N1 ^ convert_BRACKET N2
     | (Const ("signs3",_) $ N1 $ N2 $ N3) => convert_SIGN1 N1 ^ convert_BRACKET N2 ^ convert_SIGNS N3
     | (Const ("signs4",_) $ N1) => convert_BRACKET N1
     | (Const ("signs5",_) $ N1 $ N2) => convert_BRACKET N1 ^ convert_SIGNS N2
     | _ => raise (ERR "Isa-Conversion of SIGNS")

fun convert_SIMPLE_ID (t:term) =
   case t of 
     (Const ("simple_id",T) $ N)   => (convert_WORDS N, None)
      | _ => raise (ERR "Isa-Conversion of SIMPLE_ID")

fun convert_TOKEN_OR_MIXFIX (t:term) =
   case t of 
     (Const ("simple_id",T) $ N)   => ([token (convert_WORDS N)], convert_WORDS N, None)
      | _ => raise (ERR "Isa-Conversion of TOKEN_OR_MIXFIX")

fun convert_TOKEN (t:term) =
    case t of 
       (Const ("token1",T) $ N)    => convert_WORDS N
     | (Const ("token2",T) $ N)    => convert_SIGNS N
     | (Const ("token3",T) $ N)    => convert_DOT_WORDS N
     | (Const ("token4",T) $ N)    => convert_DIGIT N
     | (Const ("char",_) $ N1)     => convert_CHAR N1
     | _ => raise (ERR "Isa-Conversion of TOKEN")

fun convert_PLACE (t:term) =
    case t of 
       (Const ("__",T)) => place
     | _ => raise (ERR "Isa-Conversion of PLACE")

fun conv_tok (Token "__") = place
|   conv_tok (Token s) = token s
|   conv_tok (SignSy s) = token s
|   conv_tok (WordsSy s) = token s
|   conv_tok _ = raise (ERR "Isa-Conversion: conv_tok")

fun join_tokens nil = nil
  | join_tokens [t] = [t]
  | join_tokens (token t1::token t2::rest) =
    join_tokens (token (t1^t2)::rest)
  | join_tokens (t::rest) = t::join_tokens rest
  
fun get_tokenlist id = 
    join_tokens (map conv_tok (rev ( tl ( rev (hd (scan id))))))
    handle _ => [token "Illegal token"]

fun get_OpSymb (Const (id,T)) = op_symb ((simple_id(get_tokenlist id,id,None)), None)
|   get_OpSymb (N1 $ N2) = get_OpSymb N1
|   get_OpSymb _ = raise (ERR "Isa-Conversion: get_OpSymb")

fun get_args l (Const (id,T)) = l
|   get_args l (N1 $ N2) = get_args (N2::l) N1
|   get_args _ _ = raise (ERR "Isa-Conversion: get_args")

fun TOKENOrPLACE2String [] = []
  | TOKENOrPLACE2String (t::ts) =
       case t of 
          (token  N) =>  N::TOKENOrPLACE2String ts
        | place   => "__"::TOKENOrPLACE2String ts
           
and TOrPList2String t = concat(TOKENOrPLACE2String t)

fun TOKEN_ID2String nil= nil
|   TOKEN_ID2String [t]  =
    (case t of 
       simple_id(t1,s,ln) => [s]
     | compound_id((t1,s,ln),idl) => s::TOKEN_ID2String idl)
|   TOKEN_ID2String _ = raise (ERR "Isa-Conversion TOKEN_ID2String")

fun T2String   nil = nil
 | T2String ((([place,token t1],None))::t) = t1^"__"::(T2String t)
 | T2String ((([place,token t1],Some idl))::t) = (t1::(TOKEN_ID2String idl))@(T2String t)
 | T2String (_::t) =  T2String t 

  (************************************ Ids ***************************)

fun convert_TOKEN_ID (t) =
    case t of 
        (Const ("simple_tid",T) $ N) 	    => simple_id ([token(convert_TOKEN N)],
      							 convert_TOKEN N, None)
      | (Const ("compound_tid",T) $ N $ N2) => compound_id (
      							([token (convert_TOKEN N)],convert_TOKEN N, None), 
      							 convert_ID_List N2)
      | _ => raise (ERR "Isa-Conversion of TOKEN_ID")

and convert_PLACE_TOKEN_ID t =
     case t of 
       (Const("token", T) $ N $ N1) =>  
           (case (convert_TOKEN_ID N1) of
             simple_id ([token t1],s1,a1) => ([place,token t1],None)
             | compound_id (([token t1],s1,a1),idl) => ([place,token t1],Some idl)
             | _ => raise (ERR "Isa-Conversion of PLACE_TOKEN_ID / ID"))
      | Const("place", T) $ N   => ([place],None)
      | _ => raise (ERR "Isa-Conversion of PLACE_TOKEN_ID")

and convert_PLACE_TOKEN_IDList t = convert_list ("tokenorplace1","tokenorplace2", convert_PLACE_TOKEN_ID) t

and check_comp ((l,None)::rest) = check_comp rest
  | check_comp nil = nil
  | check_comp ((l,Some comp)::rest) = 
    if check_comp rest = nil then comp
    else raise ERROR

and  Ttoken [] = ""
| Ttoken ((token t1)::ls) =   t1^(Ttoken ls)
| Ttoken (place::ls) =   "__"^(Ttoken ls)

and convert_comps list =
               let val tokenlist = flat (map fst list)
                   val complist = check_comp list
		   val liste =  (Ttoken tokenlist)
           in if complist = nil 
              then simple_id (tokenlist,  liste, None)
              else compound_id ((tokenlist, concat(T2String list),None),complist)
           end

and convert_MIXFIX_ID (t) =
    case t of 
       (Const ("mixfix_id2",T) $ N) =>  
           convert_comps (convert_PLACE_TOKEN_IDList N )
          
    | (Const ("mixfix_id1",T) $ N $ N2) => 
         let val t =  case (convert_TOKEN_ID N) of
                  simple_id ([token t1],s1,a1) => ([token t1],None)
                  | compound_id (([token t1],s1,a1),idl) => ([token t1],Some idl)
                  | _ => raise (ERR "Isa-Conversion of OP_SYMB / ID")
         in
         convert_comps (t::convert_PLACE_TOKEN_IDList N2)
         end
    | (Const (id,T)) => 
           let  val id' = implode (tl (tl (explode id)))
	    in  simple_id (get_tokenlist id',id', None)
	   end  
    | _ => raise (ERR "Isa-Conversion of MIXFIX_ID")

and convert_ID (t):ID =
    case t of 
      (Const ("token_id",T) $ N)  => convert_TOKEN_ID N
    | (Const ("mixfix_id",T) $ N) => (convert_MIXFIX_ID N) 
    | (Const (x,T) $ N) => (writeln (" 1 "^x); raise ERROR)
    | (Const (x,T) $ N $ N1) => (writeln (" 2 "^x); raise ERROR)
    | _ => raise (ERR "Isa-Conversion of ID")

and convert_SIMPLE_ID_List t = convert_list ("_id","_consid", convert_SIMPLE_ID) t

and convert_ID_List (t:term):ID list   =  convert_list ("_cid","_conscid", convert_ID) t

and convert_TOKEN_ID_List t =  convert_list ("_tid","_constid", convert_TOKEN_ID) t

fun convert_ID_Group (t:term) = 
    case t of 
    (Const("one_id",T) $ N) => [convert_ID N]
    | (Const("group",T) $ N) => convert_ID_List N
    | _ => raise (ERR "Isa-Conversion of ID_GROUP")

fun toInt s = case Int.fromString s of
     NONE => (writeln ("Bad version number: "^s); raise ERROR)
     | SOME x => x;
     
fun to_pair s = 
    let val s1 = explode s
        val (v1,v') = take_prefix is_digit s1
        val (v'',v2) = take_suffix is_digit v'
    in [ (implode v1), (implode v2)]
    end;

 (************************************ Annotations ***************************)

fun convert_Anno (t:term) : ANNO = 
  case t of
   (Const ("number_anno",_) $ N1) => number_anno (convert_ID N1)
   | (Const ("floating_anno",_) $ N1 $N2) => floating_anno (convert_ID N1,convert_ID N2)
   | (Const ("string_anno",_) $ N1 $ N2) => string_anno (convert_ID N1,convert_ID N2)
   | (Const ("list_anno",_) $ L $ R $ N1 $ N2 ) => 
      let val LL = convert_SIGNS L
          val RR = convert_SIGNS R
          val bracket_id = simple_id ([token LL,place,token RR],LL^"__"^RR,None)
      in
        list_anno (bracket_id,convert_ID N1,convert_ID N2)
      end
   | (Const ("prec_anno1",_) $ N1 $ N2) => prec_anno  (true,convert_ID_Group N1, convert_ID_Group N2)
   | (Const ("prec_anno2",_) $ N1 $ N2) => prec_anno  (false,convert_ID_Group N1, convert_ID_Group N2)
   | Const ("cons",_) => conservative
   | Const ("def",_) => definitional
   | Const ("implies",_) => implies
   | _ => raise (ERR "Isa-Conversion of ANNO")

fun convert_AnnoList t =  convert_list ("F", "_combform", convert_Anno) t
 
 (************************************ Literal ***************************)

fun mk_app f (t1,t2) = application (f,terms [t1,t2])

fun mk_app1 f (t1,t2) = application (f,terms [t2,t1])

fun convert_number an s =
   case LocalEnv.get_number_an an of
        Some (number_anno f) => 
            foldr1 (mk_app1 (op_symb (f,None))) (map constant_string (rev (explode s)))
        | _ => raise (ERR ("No literal syntax annotation for conversion of "^s))

fun mk_char c = "'" ^ c ^ "'"

fun convert_string an s =
   case LocalEnv.get_string_an an of
        Some (string_anno (empty,str_cons)) => 
            foldl (mk_app1 (op_symb (str_cons,None))) 
                  (constant_ID empty,
                   map (constant_string o mk_char) (tl (rev (tl (explode s)))))
        | _ => raise (ERR ("No literal syntax annotation for conversion of "^s))

val uminus = op_symb (simple_id ([token "-",place], "-__",None),None)

fun convert_float an s =
   case LocalEnv.get_float_an an of
        Some (floating_anno (comma,mk_exp)) => 
         let val (pre,rest1) = take_prefix is_digit (explode s);
             val (post,rest2) = case rest1 of
               "."::r => take_prefix is_digit r 
              | _ => ([],rest1)
             val (exp,sign) = case rest2 of
                "E"::"+"::r => (r,false)
              | "e"::"+"::r => (r,false)
              | "E"::"-"::r => (r,true)
              | "e"::"-"::r => (r,true)
              | "E"::r => (r,false)
              | "e"::r => (r,false)
              |  _ => ([],false)
             val mantissa = if post=[] then convert_number an (implode pre)
                 else mk_app (op_symb (comma,None)) 
                      (convert_number an (implode pre),convert_number an (implode post))
         in
            if exp=[] then mantissa
            else mk_app (op_symb (mk_exp,None)) 
                        (mantissa,if sign then application (uminus,
                                       terms [convert_number an (implode exp)])
                                  else convert_number an (implode exp)) 
         end
        | _ => raise (ERR ("No literal syntax annotation for conversion of "^s))
  
fun convert_Literal (an:ANNO list) (t:term) : TERM = 
  case t of
   (Const ("nnumber",_) $ N1) => convert_number an (convert_NNUMBER N1)
 | (Const ("float",_) $ N1) => convert_float an (convert_FLOAT N1)
 | (Const ("string",_) $ N1) => convert_string an (convert_STRING N1)
 | (Const ("list",_) $ N1) => constant_string (convert_LIST N1)
 | _ => raise (ERR "Isa-Conversion of LITERAL")
  
  



   (************************************ Axiom Items ***************************)


and convert_L_Formula t : L_FORMULA =
  case t of (Const("labeled_formula1", T) $ N $ N1) => (convert_uFormula N1, convert_AnnoList N)
    | (Const("labeled_formula2", T) $ N ) => (convert_uFormula N, [])   
    | _ => raise (ERR "Isa-Conversion of L_FORMULA")

and convert_Formula an t : FORMULA =
    case t of
      (Const("Q", T) $ N)             => quantification(convert_Quantification an N)
      |(Const("formula", T) $ N)      => atom(convert_Atom an N)
      |(Const("parFormula", T) $ N)   => pos_FORMULA(((0,0),(0,0)),true,convert_Formula an N) 
      |(Const("FAnd", T) $ N)         => conjunction (convert_AndFormula an N)
      |(Const("FOr", T) $ N)          => disjunction (convert_OrFormula an N)
      |(Const("FImp", T) $ N)         => implication (convert_ImpFormula an N)
      |(Const("FIf", T) $ N )         => convert_IfFormula an N
      |(Const("FORMULAEquiv", T)$N$N1)=> equivalence(convert_Formula an N, convert_Formula an N1)
      |(Const("FORMULAEquiv2", T)$N$N1)=> equivalence(convert_Formula an N, quantification(convert_Quantification an N1))
      |(Const("FORMULANot", T) $ N)   => negation(convert_Formula an N)
      |(Const("FORMULANot2", T) $ N)   => negation(quantification(convert_Quantification an N))
      |(Const("FORMULAImp", T) $ N $ N1)  =>  implication(convert_Formula an N, convert_Formula an N1)
      |(Const("FORMULAImp1", T) $ N $ N1) =>  implication(convert_Formula an N,  implication(convert_ImpFormula an N1)) 
      |(Const("FORMULAImp2", T) $ N $ N1) =>  implication(convert_Formula an N,quantification(convert_Quantification an N1))
      |(Const("FORMULAAnd", T) $ N $ N1)   =>conjunction [convert_Formula an N, convert_Formula an N1]
      |(Const("FORMULAAnd1", T) $ N $ N1) =>  conjunction((convert_Formula an N):: (convert_AndFormula an N1) )
      |(Const("FORMULAAnd2", T) $ N $ N1) =>  conjunction[convert_Formula an N, quantification(convert_Quantification an N1)] 
      |(Const("FORMULAOr", T) $ N $ N1)  => disjunction [convert_Formula an  N, convert_Formula an N1]
      |(Const("FORMULAOr1", T) $ N $ N1) =>  disjunction((convert_Formula an N)::(convert_OrFormula an N1)) 
      |(Const("FORMULAOr2", T) $ N $ N1) =>  disjunction[convert_Formula an N, quantification(convert_Quantification an N1)] 
      |(Const("FORMULAIf2", T) $ N $ N1) =>  implication (quantification(convert_Quantification an N1),convert_Formula an N) 
      | _ => raise (ERR "Isa-Conversion of FORMULA")

and convert_uFormula t : FORMULA =
   unparsed_formula (print_unparsed_formula CASL98Pre.sign t )
          
 
and  convert_Quantification an t : QUANTIFIER * VAR_DECL list * FORMULA =
          case t of
          (Const("quantall", T) $ N $ N1)  => (forall,convert_VarDeclList N, convert_Formula an N1)
         |(Const("quantex", T) $ N $ N1)    => (exists,convert_VarDeclList N, convert_Formula an N1)
         |(Const("quantex1", T) $ N $ N1)   => (exists_uniquely,convert_VarDeclList N, convert_Formula an N1) 
         | _ => raise (ERR "Isa-Conversion of QUANITIFCATION")

and get_PredSymb (Const (id,T)) = pred_symb ((simple_id(get_tokenlist id,id,None)),None)
|   get_PredSymb (N1 $ N2) = get_PredSymb N1
|   get_PredSymb _ = raise (ERR "Isa-Conversion : get_PredSymb")
         
and convert_Atom an (t:term):ATOM =
   case t of 
          (Const("TRUE", T)) => ttrue
   	| (Const("FALSE", T)) => ffalse 
   	| (Const("predication1", T)$ N)    => predication (convert_PredSymb N, terms [])
   	| (Const("predication2", T)$N$N1)  => predication(convert_PredSymb N, (convert_TTerms an N1)) 
   	| (Const("membership", T) $N)      => membership (convert_Membership an N)
   	| (Const("ATOM", T) $ N)           => definedness (convert_TTerm an N) 
   	| (Const("ATOM4", T) $ N $ N1)     => strong_equation (convert_TTerm an N, convert_TTerm an N1)
   	| (Const("ATOM6", T) $ N $ N1)     => existl_equation (convert_TTerm an N, convert_TTerm an N1)  
       | mixterm  => 
      (let val PS = get_PredSymb mixterm;
           val args = get_args [] mixterm; 
       in
            predication( PS,terms(map (convert_TTerm an) args))
       end)    


and convert_AndFormula an t:FORMULA list=
      case t of 
       (Const("FORMULAAnd", T) $ N $ N1)   => [convert_Formula an N, convert_Formula an N1]
     | (Const("FORMULAAnd1", T) $ N $ N1) =>  ((convert_Formula an N):: (convert_AndFormula an N1) )
     | (Const("FORMULAAnd2", T) $ N $ N1) =>  [convert_Formula an N, quantification(convert_Quantification an N1)] 
     | _ => raise (ERR "Isa-Conversion of AND-FORMULA")

and convert_OrFormula an (t:term):FORMULA list=
    case t of 
      (Const("FORMULAOr", T) $ N $ N1)  =>  [convert_Formula an  N, convert_Formula an N1]
    | (Const("FORMULAOr1", T) $ N $ N1) =>  ((convert_Formula an N)::(convert_OrFormula an N1)) 
    | (Const("FORMULAOr2", T) $ N $ N1) =>  [convert_Formula an N, quantification(convert_Quantification an N1)] 
    | _ => raise (ERR "Isa-Conversion of OR_FORMULA")

and convert_ImpFormula an (t:term) =
    case t of 
      (Const("FORMULAImp", T) $ N $ N1)  =>  (convert_Formula an N, convert_Formula an N1)
    | (Const("FORMULAImp1", T) $ N $ N1) =>  (convert_Formula an N,  implication(convert_ImpFormula an N1)) 
    | (Const("FORMULAImp2", T) $ N $ N1) =>  (convert_Formula an N,quantification(convert_Quantification an N1)) 
    | _ => raise (ERR "Isa-Conversion of IMP-FORMULA")
and convert_IfFormula an (t:term)=
    case t of 
      (Const("FORMULAIf", T) $ N $ N1)  => implication(convert_Formula an N1, convert_Formula an N)
    | (Const("FORMULAIf1", T) $ N $ N1) =>  implication(convert_Formula an N1, convert_IfFormula an N) 
    | _ => raise (ERR "Isa-Conversion of IF-FORMULA")

and convert_L_FormulaList t = convert_list ("F", "_combform", convert_L_Formula) t 

and convert_AxiomList t     = convert_list ("F1", "_combform1",convert_L_Formula) t  


 
   (************************************ Pred Symbs  ***************************)

and convert_PredSymb (t:term):PRED_SYMB=
    case t of 
      (Const ("predsymb1", T) $ N)     => pred_symb (convert_ID N, None)
    | (Const ("predsymb2", T) $ N$ N1) => pred_symb(convert_ID N, Some(convert_PredType N1))
    | _ => raise (ERR "Isa-Conversion of PRED_SYMB")


   (************************************ Op Symbs  ***************************)
    
and convert_OpSymb (t:term):OP_SYMB =
    case t of 
      (Const ("opsymb1", T) $ N)     => op_symb (convert_ID N, None)
    | (Const ("opsymb2", T) $ N $N1) => op_symb(convert_ID N, Some (convert_OpType N1))
    | _ => raise (ERR "Isa-Conversion of OP_SYMB")					
    					
    					
   (************************************ Terms  ***************************)

and get_list_anno f [] = None
  | get_list_anno f (list_anno (g,list_nil,list_cons)::rest) =
    if f=g then Some (list_nil,list_cons)
    else get_list_anno f rest
  | get_list_anno f (_::rest) =
    get_list_anno f rest

and convert_TTerms an (t:term):TERMS = terms (convert_list ("_idT","_consT",convert_TTerm an) t)

and convert_TTerm an (t:term):TERM =
    case t of
      (Const ("literal",T) $ N1 ) 	    => convert_Literal an N1
    | (Const ("cast",T) $ N1 $ N) 	    => cast(convert_TTerm an N1,convert_TOKEN_ID N)
    | (Const ("application1",T) $ N1 $ N2)  => application(convert_OpSymb N1,convert_TTerms an N2)
    | (Const ("sortedTerm",T) $ N1 $ N2)    => sorted_term (convert_TTerm an N1, convert_TOKEN_ID N2)
    | (Const ("parTerm",T) $ N1 ) 	    => pos_TERM(((0,0),(0,0)),true,convert_TTerm an N1)
    | (Const ("idterm",T) $ N1 ) 	    => constant_OP_SYMB(op_symb (convert_ID N1, None))
    | (Const ("varterm",T) $ N1 $ N2) 	    => qual_var (convert_SIMPLE_ID N1, convert_TOKEN_ID N2) 
    | (Const ("qualconstterm",T) $ N1 $ N2) => application(op_symb (convert_ID N1,
    						 Some (convert_OpType N2)), terms [])
    | (Const ("whenElseTerm",T) $ N1 $ N2 $N3) => conditional (convert_TTerm an N1,
 							    convert_Formula an N2,
 							    convert_TTerm an N3)
    | mixterm  => 
      (let val OS = get_OpSymb mixterm
           val f = Utils.get_op_name OS
           val t = Utils.get_op_type OS
           val args = (get_args [] mixterm)
       in
          case get_list_anno f an of
            Some (list_nil,list_cons) =>
              let val ts = Utils.get_terms (convert_TTerms an (hd args)) 
              in
                foldl (mk_app1 (op_symb (list_cons,None))) 
                      (constant_ID list_nil,rev ts)
              end
            | None => 
                application(OS,
                   terms(map (convert_TTerm an) args))
       end)    

and convert_uTTerm (t:term):TERM =
    unparsed_term (print_unparsed_term CASL98Pre.sign t)


and convert_Membership an (t:term):TERM *SORT=
     case t of 
        (Const("TERM7", T) $ N1$N2) 	=> (convert_TTerm an N1, convert_TOKEN_ID N2)
         | _ => raise (ERR "Isa-Conversion of TERM")

and convert_TTermList an t = terms (convert_list ("termlist1","termlist2",convert_TTerm an) t) 


and convert_OpType (t:term):OP_TYPE =
    case t of
       (Const("optype1", T) $ N $ N1) 	=> total_op_type (sorts (convert_SomeSorts N), 
       							  convert_TOKEN_ID N1)
     | (Const("optype2", T) $ N$ N1) 	=> partial_op_type 
                                                         (sorts (convert_SomeSorts N), 
                                                          convert_TOKEN_ID N1)
     | (Const("optype3", T) $ N) 	=>  total_op_type  (sorts [], convert_TOKEN_ID N)
     | (Const("optype4", T) $ N) 	=>  partial_op_type(sorts [], convert_TOKEN_ID N) 
     | _ => raise (ERR "Isa-Conversion of OP_TYPE")     

and convert_PredType (t:term):PRED_TYPE =
     case t of 
        (Const("predtype1", T) $ N) 	=> pred_type (sorts (convert_SomeSorts N))
     | (Const("predtype2", T)) 		=> pred_type (sorts [])  
     | _ => raise (ERR "Isa-Conversion of PRED_TYPE")

and convert_VarDecl (t:term):VAR_DECL=
case t of (Const("_idVar1", T)  $ N1 $ N2) =>(convert_SIMPLE_ID_List N1, convert_TOKEN_ID N2) 
          | _ => raise (ERR "Isa-Conversion of VAR_DECL")

and convert_VarDeclList t= convert_list("idVar2", "_consV", convert_VarDecl) t 

and convert_SomeSorts t = convert_list ("P","product", convert_TOKEN_ID) t 


end
