(***************************************************************** *)
(*								   *)
(* Project: CASL						   *)
(* Author: Till Mossakowski, University of Bremen		   *)
(* Date: 29.12.98						   *)
(* Purpose of this file: Isabelle Parser for CASL 		   *)
(*								   *)
(*								   *)
(***************************************************************** *)

(* This module provides mixfix parsing, using the Isabelle parser.
   In case of ambiguities, Isabelle returns a list of parse trees.
   The ambiguity handler given below selects those parse trees
   that are precedence correct w.r.t. the declared precedence
   and associativity annotations. If the resulting list of 
   parse trees is a singleton, everything is o.k., otherwise,
   there is an error.

   Bugs:  +__(t) is ambiguous, since it may be +(__(t)) as well.

   Error messages should not be output immediately!

   Future work: replace this with a combinator parser.
   (Reason: a+b+c+d+e+f+g is too big for the Isabelle parser,
    since *all* possible parse trees are constructed,
    and precedence is only checked afterwards)
*)

infix mem

structure IsabelleParser : 
sig
  exception AMBIGUOUS of Term.term list
  

  type precedence_order = (AS.ID * AS.ID list) list
  
  (* Isabelle parser, used for mixfix parsing *)
      
  val read : bool -> Syntax.syntax -> Term.typ -> Lexicon.token list 
             -> Term.term
  (* Beware! The AS.ANNO list may not contain AS.pos_ANNO ! *)
  val read_term : AS.region -> (LocalEnv.sign * string list * precedence_order * AS.ANNO list * bool) -> 
                  bool -> Syntax.syntax -> Term.typ -> Lexicon.token list 
                  -> Term.term
  val read_formula : AS.region -> (LocalEnv.sign * string list * precedence_order * AS.ANNO list * bool) -> 
                     bool -> Syntax.syntax -> Term.typ -> Lexicon.token list
                      -> Term.term
  val CASL98 : Syntax.syntax
  val CASL98Pre : Syntax.syntax
  val TTermType : Term.typ
  val TTermsType : Term.typ
  val AtomType : Term.typ
  val MixfixType : Term.typ
  val FormulaType : Term.typ
end
 =
struct

datatype PARSING_ANNO = no_assoc | left_assoc | right_assoc

open Utils AS Term  CASLScanner TextForm ML_interface
infix 9  $;
infixr 5 -->;
infixr --->;
infix aconv;
infix upto;

exception AMBIGUOUS of Term.term list

type precedence_order = (ID * ID list) list

val CASL98Pre = Syntax.set_lexicon (#syn (Sign.rep_sg CASL98Pre.sign)) CASL_lexicon;
val CASL98 = Syntax.set_lexicon (#syn (Sign.rep_sg CASL98.sign)) CASL_lexicon;

val TTermType = Type ("TTerm",[])
val TTermsType = Type ("TTerms",[])
val AtomType = Type ("Atom",[])
val MixfixType  = Type ("MIXFIX_ID",[])
val FormulaType  = Type ("Formula",[])


(*** Ambiguity handler for Isabelle paerser ***)

fun print_spaces html (i:int) = 
     (TextForm.print_text html (implode (map (K " ") (1 upto i))))
fun print_nl html (x:unit) = () (*print ("\n")*)
fun print_brk html (i:int) = print_spaces html i;

fun pretty_print (html:bool) (b:Pretty.T) =
      (Pretty.pprint b (TextForm.print_text html,print_spaces html,print_brk html,
       print_nl html,print_nl html);
       TextForm.print_text html "\n"
      );

     
fun print_ambig2 text html syn (ts:term list) = (*raise AMBIG ts*)
    let val old_depth = get_print_depth() * 2
    in
    (Syntax.show_brackets:=true;
     print_depth 1000;
     TextForm.print_text html text;
     seq (pretty_print html o (Syntax.pretty_term syn false)) (remove_dups ts);
     print_depth old_depth)
    end;

fun is_const c (Const (c1,t)) = c=c1
|   is_const _ _ = false

fun is_free c (Free (c1,t)) = c=c1
|   is_free _ _ = false

fun no_of_diff_args n (t1$t2) (u1$u2) = 
    no_of_diff_args (if t2=u2 then n else n+1) t1 u1
|   no_of_diff_args n t u = if t=u then n else 1000;

fun is_same_app t u = 
    no_of_diff_args 0 t u <=1;
 
fun get_fun (t1 $ t2) = t1
  | get_fun _ = raise (ERR "No function application")

fun get_arg (t1 $ t2) = t2
  | get_arg _ = raise (ERR "No function application")

fun print_ambig_args text html syn (ts:term list) =
   case ts of
     (t1 $ t2)::_ =>
       ( print_ambig_args text html syn (map get_fun ts);
         print_ambig1 text html syn (map get_arg ts)
        )
    | _ => ()
    
and print_ambig1 text html syn nil = (print "No parse tree found!\n";())
|   print_ambig1 text html syn (ts as (Const (c,t)::_):term list) =
     if forall' (is_const c) ts
     then ()
     else print_ambig2 text html syn ts
|   print_ambig1 text html syn (ts as (Free (c,t)::_):term list) =
     if forall' (is_free c) ts
     then ()
     else print_ambig2 text html syn ts
|   print_ambig1 text html syn (ts as ((t as (t1 $ t2))::_):term list) =
     if forall' (is_same_app t) ts
     then print_ambig_args text html syn ts
     else print_ambig2 text html syn ts
|   print_ambig1 _ _ _ _ = raise (ERR "print_ambig1")

fun print_ambig text html syn ts =
   (if length ts = 1 then
    print_ambig2 (text^":\n") html syn ts
    else if length ts <4 
    then print_ambig1 (text^":\n") html syn ts
    else print_ambig1 (text^" (list truncated):\n") html syn (take (4,ts));
    raise AMBIGUOUS ts)
                 
fun assoc_fits_with (f : ID) (lassoc_anno g) = f mem g
  | assoc_fits_with (f : ID) (rassoc_anno g) = f mem g
(*  | assoc_fits_with f (pos_ANNO(_,a)) = 
    assoc_fits_with f a*)
  | assoc_fits_with f _ = false

fun is_mixfix (simple_id(tokenlist,id,_)) = 
    LocalEnv.get_symbol_type tokenlist = LocalEnv.infix_sym
  | is_mixfix _ = raise (ERR "Mixfix / compound id")

fun is_const_type (total_op_type (sorts [],_)) = true
  | is_const_type (partial_op_type (sorts [],_)) = true
  | is_const_type (pos_OP_TYPE (_,t)) =
    is_const_type t
  | is_const_type (_) = false

fun has_arity n (total_op_type (sl,_)) = length (get_sorts sl) = n
  | has_arity n (partial_op_type (sl,_)) = length (get_sorts sl) = n
  | has_arity n (pos_OP_TYPE (_,opt)) =
    has_arity n opt
  
fun is_pc_Term (prec_info as ((subsortenv,varenv,funenv,predenv),vl:string list,prec_ord,an,is_goal)) (t:term) =
    case t of
      (Const ("literal",T) $ N1 ) 	    => true
    | (Const ("cast",T) $ N1 $ N) 	    => is_pc_Term  prec_info N1
    | (Const ("application1",T) $ N1 $ N2)  => 
      is_pc_Terms prec_info N2
(* really useful???
      andalso
      (let val op_symb(id,_) = Convert.convert_OpSymb N1
       in
         case Symtab_id.lookup(funenv,id) of
           Some funts => 
              exists' (has_arity (length (Convert.get_args [] N2))) funts
         | None => false
       end)
*)
    | (Const ("sortedTerm",T) $ N1 $ N2)    => is_pc_Term  prec_info N1
    | (Const ("parTerm",T) $ N1 ) 	    => is_pc_Term  prec_info N1
    | (Const ("idterm",T) $ N1 ) 	    => 
       is_goal orelse
       let val id = Convert.convert_ID N1
           val funts = Symtab_id.lookup(funenv,id)
       in
       (is_some funts andalso (exists' is_const_type (the funts)))
       orelse
       case id of
            simple_id (_,s,l) => is_some (Symtab_sid.lookup (varenv,(s,None)))
                                 orelse (s mem vl)
          | _ => false
       end
    | (Const ("varterm",T) $ N1 $ N2) 	    => true 
    | (Const ("qualconstterm",T) $ N1 $ N2) => true
    | (Const ("whenElseTerm",T) $ N1 $ N2 $N3) => (is_pc_Term  prec_info N1) 
 						  andalso (is_pc_Formula prec_info N2)
 						  andalso (is_pc_Term prec_info N3)
    | mixterm  => 
      (let val fname = get_op_name (Convert.get_OpSymb mixterm)
           val args = Convert.get_args [] mixterm;
       (* Is the mixfix ID part of a %list syntax ? Then there is no precedence to check *)
       in case Convert.get_list_anno fname an of
          Some _ => is_pc_Terms prec_info (hd args)
          | None =>
       (* Get first and last token of the mixfix ID, since they determine
          the interaction with surrounding mixfix IDs. 
          Also get the associativity *)
       let val first_fname = get_op_name (Convert.get_OpSymb (hd args))
           val last_fname = get_op_name (Convert.get_OpSymb (last_elem args))
           val associativity = case find_first (assoc_fits_with fname) an of
                 Some (lassoc_anno _) => left_assoc 
                 | Some (rassoc_anno _) => right_assoc
                 | _ => no_assoc 
       in   not (is_mixfix fname) orelse
            (forall' (is_pc_Term prec_info) args
             andalso ((not (is_mixfix first_fname)) orelse
                      (* for the first token of the mixfix ID, check the precedence ... *)
                      (leq prec_ord fname first_fname  
                      (* ... and, in case of same precedence, also the associativity *) 
                       andalso ( not (leq prec_ord first_fname fname) orelse  (fname=first_fname andalso associativity=left_assoc))))
             andalso ((not (is_mixfix last_fname)) orelse
                      (* for the last token of the mixfix ID, check the precedence ... *)
                      (leq prec_ord fname last_fname 
                      (* ... and, in case of same precedence, also the associativity *) 
                       andalso ( not (leq prec_ord last_fname fname ) orelse (fname=last_fname andalso associativity=right_assoc)))))
       end
       end) 
(*       handle (LIST _) => true   *)

and is_pc_Terms prec_info t =
   case t of
     (Const ("_idT",_) $ N) => is_pc_Term prec_info N
   | (Const ("_consT",_) $ N1 $ N2) => (is_pc_Term prec_info N1) andalso (is_pc_Terms prec_info N2)
   | _ => raise (ERR "Prec: TERMS")

and is_pc_Formula prec_info t =
    case t of
      (Const("Q", T) $ N)             => is_pc_Quantification prec_info N
      |(Const("formula", T) $ N)      => is_pc_Atom prec_info N
      |(Const("parFormula", T) $ N)   => is_pc_Formula prec_info N 
      |(Const("FAnd", T) $ N)         => is_pc_AndFormula prec_info N
      |(Const("FOr", T) $ N)          => is_pc_OrFormula prec_info N
      |(Const("FImp", T) $ N)         => is_pc_ImpFormula prec_info N
      |(Const("FIf", T) $ N )         => is_pc_IfFormula prec_info N
      |(Const("FORMULAEquiv", T)$N$N1)=> (is_pc_Formula prec_info N) andalso (is_pc_Formula prec_info N1)
      |(Const("FORMULAEquiv2", T)$N$N1)=> (is_pc_Formula prec_info N) andalso (is_pc_Quantification prec_info N1)
      |(Const("FORMULANot", T) $ N)   => is_pc_Formula prec_info N
      |(Const("FORMULANot2", T) $ N)   => is_pc_Quantification prec_info N
      |(Const("FORMULAImp", T) $ N $ N1)  =>  (is_pc_Formula prec_info N) andalso (is_pc_Formula prec_info N1)
      |(Const("FORMULAImp1", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_ImpFormula prec_info N1) 
      |(Const("FORMULAImp2", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_Quantification prec_info N1)
      |(Const("FORMULAAnd", T) $ N $ N1)   => (is_pc_Formula prec_info N) andalso (is_pc_Formula prec_info N1)
      |(Const("FORMULAAnd1", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_AndFormula prec_info N1) 
      |(Const("FORMULAAnd2", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_Quantification prec_info N1) 
      |(Const("FORMULAOr", T) $ N $ N1)  => (is_pc_Formula prec_info  N) andalso (is_pc_Formula prec_info N1)
      |(Const("FORMULAOr1", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_OrFormula prec_info N1) 
      |(Const("FORMULAOr2", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_Quantification prec_info N1) 
      |(Const("FORMULAIf2", T) $ N $ N1) =>  (is_pc_Quantification prec_info N1) andalso (is_pc_Formula prec_info N) 
      | _ => raise (ERR "Prec: FORMULA")

and is_pc_Quantification (Sigma,vl,prec_ord,an,is_goal) t  =
          case t of
          (Const(Q, T) $ N $ N1)  => 
            is_pc_Formula (Sigma,map fst(flat(map fst (Convert.convert_VarDeclList N)))@vl,prec_ord,an,is_goal) N1
           | _ => raise (ERR "Prec: QUANITIFACTION")

and is_pc_Atom prec_info (t:term) =
   case t of 
          (Const("TRUE", T)) => true
   	| (Const("FALSE", T)) => true 
   	| (Const("predication1", T)$ N)    => true
   	| (Const("predication2", T)$N$N1)  =>  is_pc_Terms prec_info N1 
   	| (Const("membership", T) $ (Const("TERM7", T1) $ N1$N2))      =>  is_pc_Term  prec_info N1
   	| (Const("ATOM", T) $ N)           =>  is_pc_Term  prec_info N 
   	| (Const("ATOM4", T) $ N $ N1)     => ( is_pc_Term  prec_info N) andalso ( is_pc_Term  prec_info N1)
   	| (Const("ATOM6", T) $ N $ N1)     => ( is_pc_Term  prec_info N) andalso ( is_pc_Term  prec_info N1)  
       | mixterm  => 
      (let val args = Convert.get_args [] mixterm; 
       in
            forall' ( is_pc_Term  prec_info) args
       end)    

and is_pc_AndFormula prec_info t=
      case t of 
       (Const("FORMULAAnd", T) $ N $ N1)   => (is_pc_Formula prec_info N) andalso (is_pc_Formula prec_info N1)
     | (Const("FORMULAAnd1", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_AndFormula prec_info N1) 
     | (Const("FORMULAAnd2", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_Quantification prec_info N1) 
     | _ => raise (ERR "Prec: AND_FORMULA")

and is_pc_OrFormula prec_info (t:term) =
    case t of 
      (Const("FORMULAOr", T) $ N $ N1)  =>  (is_pc_Formula prec_info N) andalso (is_pc_Formula prec_info N1)
    | (Const("FORMULAOr1", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_OrFormula prec_info N1) 
    | (Const("FORMULAOr2", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_Quantification prec_info N1)
    | _ => raise (ERR "Prec: OR_FORMULA")

and is_pc_ImpFormula prec_info (t:term) =
    case t of 
      (Const("FORMULAImp", T) $ N $ N1)  =>  (is_pc_Formula prec_info N) andalso (is_pc_Formula prec_info N1)
    | (Const("FORMULAImp1", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_ImpFormula prec_info N1) 
    | (Const("FORMULAImp2", T) $ N $ N1) =>  (is_pc_Formula prec_info N) andalso (is_pc_Quantification prec_info N1) 
    | _ => raise (ERR "Prec: IMP_FORMULA")

and is_pc_IfFormula prec_info (t:term)=
    case t of 
      (Const("FORMULAIf", T) $ N $ N1)  => (is_pc_Formula prec_info N1) andalso (is_pc_Formula prec_info N)
    | (Const("FORMULAIf1", T) $ N $ N1) =>  (is_pc_Formula prec_info N1) andalso (is_pc_IfFormula prec_info N) 
    | _ => raise (ERR "Prec: IF_FORMULA")

fun get_pc_Term prec_info ts = 
    filter (is_pc_Term prec_info) ts

fun get_pc_Formula prec_info ts = 
    filter (is_pc_Formula prec_info) ts


val ambiguous_text = ("(mixfix analysis): Ambigous input.\n Possible parses");
val incorrect_prec_text = ("(mixfix analysis): No precedence correct parse tree\n"^
                           "(or undeclared identifier). Some incorrect trees")

fun read_term r prec_info html syn t tokens =
   (Syntax.ambiguity_level := 1000;
    let val trees = Syntax.CASL_read_token syn t tokens
    in
    case get_pc_Term prec_info trees of
        nil => print_ambig (perr r^incorrect_prec_text) html syn trees
        | [t] => t
        | ts => print_ambig (perr r^ambiguous_text) html syn ts
    end);

fun read_formula r prec_info html syn t tokens =
   (Syntax.ambiguity_level := 1000;
    let val trees = Syntax.CASL_read_token syn t tokens
    in
    case get_pc_Formula prec_info trees of
        nil => print_ambig (perr r^incorrect_prec_text) html syn trees
        | [t] => t
        | ts => print_ambig (perr r^ambiguous_text) html syn ts 
    end);

fun read html syn t tokens =
   (Syntax.ambiguity_level := 1000;
    case Syntax.CASL_read_token syn t tokens of
        [t] => t
        | ts => print_ambig ambiguous_text html syn ts);

end











