(* *********************************************************************** *)
(*									   *)
(* Project: CATS	                                                   *)
(* Author: Till Mossakowski and Kolyang, University of Bremen              *)
(* Date: 30 06 1999 				 			   *)
(* Purpose of this file: Data structures for signatures/basic specs        *)
(*									   *)
(* *********************************************************************** *)

(* Central datastructure for CASL signatures (local environment).
   We use symbol tables (2-3-trees) for efficiency reasons.
   Some basic functions on local environments are provided.

  todo:
  Im Env in Funktionsliste assoc, comm etc.
  Annotationen effizienter verwalten
    (als Symtab, und z.B. Praezedenzordnung nur einmal berechnen)

*)

structure LocalEnv 		:
sig  

(* Signatures and signature fragments *)

type sort_entry = AS.SORT list	(* list of subsorts of a given sort *)
type var_entry = AS.SORT		(* sort of a given variable *)
type fun_entry = AS.OP_TYPE list	(* list of types of a given fun name *)
type pred_entry = AS.PRED_TYPE list	(* list of types of a given pred name *)

type Sort_env = unit Symtab_id.table
(* List of subsorts of a given sort. If the given sort is not among them,
   this means that only the relations, but not the sort are part of the signature fragment *)
type Subsort_env = sort_entry Symtab_id.table
type Var_env = var_entry Symtab_sid.table
type Fun_env = fun_entry Symtab_id.table
type Pred_env = pred_entry Symtab_id.table

type local_env =  (Subsort_env * Var_env * Fun_env * Pred_env)

val empty_local_env:local_env
val merge_lenvs : local_env * local_env -> local_env

type Sort_list = AS.SORT list
type Subsort_list = (AS.SORT * sort_entry) list
type Var_list = (AS.VAR * var_entry) list
type Fun_list = (AS.OP_NAME * fun_entry) list;
type Pred_list = (AS.PRED_NAME * pred_entry) list

type Subsort_list_multi = (AS.SORT * AS.SORT) list
type Fun_list_multi = (AS.OP_NAME * AS.OP_TYPE) list;
type Pred_list_multi = (AS.PRED_NAME * AS.PRED_TYPE) list

type local_list = (Sort_list * Subsort_list * Var_list * Fun_list * Pred_list)
type local_list_multi = (Sort_list * Subsort_list_multi * Var_list * Fun_list_multi * Pred_list_multi)


val cons_entry: (Symtab_id.key * 'a) * 'a list Symtab_id.table -> 'a list Symtab_id.table
val env_to_list: local_env -> local_list 
val list_to_env: local_list -> local_env 
val env_to_list_multi : local_env -> local_list_multi

datatype symbol_type = ordinary_sym | postfix_sym | prefix_sym | infix_sym | outfix_sym
val get_symbol_type  : AS.token_or_place list -> symbol_type


val isemptyLocalenv : local_env -> bool



val get_number_an : AS.ANNO list -> AS.ANNO Library.option
val get_string_an : AS.ANNO list -> AS.ANNO Library.option
val get_float_an : AS.ANNO list -> AS.ANNO Library.option
val get_list_an : AS.ANNO list -> AS.ANNO Library.option

(* CASL signatures *)

type sign = local_env

val empty_signature : sign

val signature_union : sign * sign -> sign
val signature_union_list : sign list -> sign

val signature_diff : sign * sign -> sign

val get_ids : sign -> AS.ID list
val get_lenv : sign -> local_env

end = 
struct


open Library AS Utils IDOrder;



(******************************************************************)
(*                                                                *)
(*                  Local Environment	                          *)
(*                                                                *)
(******************************************************************)

type sort_entry = SORT list;	(* list of subsorts of a given sort *)
type var_entry = SORT;		(* sort of a given variable *)
type fun_entry = OP_TYPE list;	(* list of types of a given fun name *)
type pred_entry = PRED_TYPE list;	(* list of types of a given pred name *)

type Sort_env = unit Symtab_id.table
type Subsort_env = sort_entry Symtab_id.table
type Var_env = var_entry Symtab_sid.table
type Fun_env = fun_entry Symtab_id.table
type Pred_env = pred_entry Symtab_id.table


type local_env =  
	(Subsort_env * Var_env * Fun_env * Pred_env)

val empty_local_env:local_env =
	 (Symtab_id.empty,Symtab_sid.empty,Symtab_id.empty,Symtab_id.empty)

fun isemptyLocalenv e = 
let val (es,ev,ef,ep) = e
    val esb = Symtab_id.is_empty es 
    val evb = Symtab_sid.is_empty ev
    val efb = Symtab_id.is_empty ef
    val epb = Symtab_id.is_empty ep
 in (esb) andalso (evb) andalso (efb andalso epb)
end;

type Sort_list = SORT list;
type Subsort_list = (SORT * sort_entry) list;
type Var_list = (VAR * var_entry) list;
type Fun_list = (OP_NAME * fun_entry) list;
type Pred_list = (PRED_NAME * pred_entry) list;

type Subsort_list_multi = (AS.SORT * AS.SORT) list
type Fun_list_multi = (AS.OP_NAME * AS.OP_TYPE) list;
type Pred_list_multi = (AS.PRED_NAME * AS.PRED_TYPE) list

type local_list = (Sort_list * Subsort_list * Var_list * Fun_list * Pred_list)
type local_list_multi = (Sort_list * Subsort_list_multi * Var_list * Fun_list_multi * Pred_list_multi)


fun cons_entry ((key, entry:'a), tab:'a list Symtab_id.table) =
  Symtab_id.update ((key, entry :: Symtab_id.lookup_multi (tab, key)), tab);


fun env_to_list ((subsortenv,varenv,funenv,predenv):local_env):local_list =
    let val subsortlist = Symtab_id.dest subsortenv
    in  (map fst subsortlist,
         subsortlist,
	 Symtab_sid.dest varenv,
	 Symtab_id.dest funenv,
	 Symtab_id.dest predenv)
	  
    end

fun env_to_list_multi ((subsortenv,varenv,funenv,predenv):local_env):local_list_multi =
    let val subsortlist = Symtab_id.dest subsortenv
    in  (map fst subsortlist,
         Symtab_id.dest_multi subsortenv,
	 Symtab_sid.dest varenv,
	 Symtab_id.dest_multi funenv,
	 Symtab_id.dest_multi predenv)
	  
    end

	  

datatype symbol_type = ordinary_sym | postfix_sym | prefix_sym | infix_sym | outfix_sym

fun get_symbol_type (nil : token_or_place list) : symbol_type =
    ordinary_sym
  | get_symbol_type ((token _)::rest) =
    if rest = [] orelse not (place mem rest)
    then ordinary_sym
    else (case last rest of
         place => prefix_sym
         | token _ => outfix_sym)
  | get_symbol_type (place::rest) =
    if rest = [] then infix_sym
    else case last rest of
         place => infix_sym
         | token _ => postfix_sym


fun get_ids_local ((l,an)) =
    let val (slist,_,_,flist,plist) = env_to_list l
    in slist @ map fst flist @ map fst plist
    end
    


fun is_number_an (number_anno _) = true
         | is_number_an _ = false
fun get_number_an an = find_first is_number_an an
       
fun is_string_an (string_anno _) = true
         | is_string_an _ = false
fun get_string_an an = find_first is_string_an an

fun is_float_an (floating_anno _) = true
         | is_float_an _ = false
fun get_float_an an = find_first is_float_an an

fun is_list_an (list_anno _) = true
         | is_list_an _ = false
fun get_list_an an = find_first is_list_an an  


(* Constructing local environment out of an arbitrary list *)
     
fun cons_entry_nodups member (tab, (key, x)) =
  let val l = Symtab_id.lookup_multi (tab, key)
  in
    Symtab_id.update ((key, if member(x,l) then l else x :: l), tab)
    end;

fun make_multi_nodups member pairs = foldl (cons_entry_nodups member) (Symtab_id.empty,pairs);

fun merge_list member (list) = 
    make_multi_nodups member (list)

fun flatten1 (x,l) = map (pair x) l
fun flatten l = flat (map flatten1 l)

fun list_to_env (_,subsortlist,varlist,funlist,predlist) =
        (merge_list IDOrder.SORT_member (flatten subsortlist),
          Symtab_sid.make varlist,
          merge_list IDOrder.OP_TYPE_member (flatten funlist),
          merge_list IDOrder.PRED_TYPE_member (flatten predlist))

(* Merging local environments *)

fun merge_tabs member (tab1,tab2) = 
    make_multi_nodups member (Symtab_id.dest_multi tab1 @ Symtab_id.dest_multi tab2)

fun print_list f delim nil = ""
  | print_list f delim (x::nil) = f x
  | print_list f delim (x::xs) = f x^delim^print_list f delim xs

fun print_ID (simple_id (tks,id,no)) = id
  | print_ID (compound_id ((tks,id,no),nil)) = id
  | print_ID (compound_id ((tks,id,no),idlist))
    = id^"["^print_IDs idlist^"]"


and print_IDs nil = ""
  | print_IDs (id::nil) = print_ID id
  | print_IDs (id1::id2::rest) 
     = print_ID id1^","^print_IDs (id2::rest)
     
fun print_sort_entry (s,subs) =
    let val newsubs = remove s subs
    in
      (if null newsubs then ""
         else (print_list print_ID "," newsubs ^" < ")
      )^ print_ID s
    end  

fun write_subs env =
    writeln(print_list (fn x => "  "^print_sort_entry x) ";\n" (Symtab_id.dest env))

fun merge_lenvs (l1,l2:local_env):local_env =
let  val (subsortenv1,varenv1,funenv1,predenv1) = l1
     val (subsortenv2,varenv2,funenv2,predenv2) = l2
     (*val _ = (writeln("Merging") ; write_subs subsortenv1; writeln("and"); write_subs subsortenv2; writeln("to"))*)
     val subsortenv' = merge_tabs SORT_member (subsortenv1,subsortenv2)
     val subsortenv = Symtab_id.make (transitive_closure SORT_eq (Symtab_id.dest subsortenv'))
     (*val _ = write_subs subsortenv*)
     val funenv  = merge_tabs OP_TYPE_member (funenv1,funenv2)
     val predenv = merge_tabs PRED_TYPE_member (predenv1,predenv2)
     val varenv = Symtab_sid.empty
in
     (subsortenv,varenv,funenv,predenv)
end

(* Difference of local environments *)


fun diff_tabs eq member (tab1,tab2) = 
    make_multi_nodups member (gen_rems eq (Symtab_id.dest_multi tab1,Symtab_id.dest_multi tab2))

fun SORTS_eq ((s1,s2),(s1',s2')) =
    SORT_eq (s1,s1') andalso SORT_eq (s2,s2')

fun OP_profile_eq ((f,t),(f',t')) =
    ID_eq(f,f') andalso OP_TYPE_eq(t,t')

fun PRED_profile_eq ((f,t),(f',t')) =
    ID_eq(f,f') andalso PRED_TYPE_eq(t,t')


fun diff_lenvs (l1,l2:local_env):local_env =
let  val (subsortenv1,varenv1,funenv1,predenv1) = l1
     val (subsortenv2,varenv2,funenv2,predenv2) = l2
     val subsortenv' = diff_tabs SORTS_eq SORT_member (subsortenv1,subsortenv2)
     val subsortenv = Symtab_id.make (transitive_closure SORT_eq (Symtab_id.dest subsortenv'))
     val funenv  = diff_tabs OP_profile_eq OP_TYPE_member (funenv1,funenv2)
     val predenv = diff_tabs PRED_profile_eq PRED_TYPE_member (predenv1,predenv2)
     val varenv = Symtab_sid.empty
in
     (subsortenv,varenv,funenv,predenv)
end

val signature_diff = diff_lenvs

(* CASL signatures *)

type sign = local_env

val empty_signature = empty_local_env

val signature_union = merge_lenvs

fun signature_union_list sigs =
    foldl signature_union (empty_signature,sigs)

fun get_ids (l:local_env) =
    let val (slist,_,_,flist,plist) = env_to_list l
    in slist @ map fst flist @ map fst plist
    end

fun get_lenv l = l
end


