(*  Title:      Pure/Thy/thy_read.ML
    ID:         thy_read.ML,v 1.42 1995/12/18 12:02:45 clasohm Exp
    Author:     Carsten Clasohm and Markus Wenzel and Sonia Mahjoub and
                Tobias Nipkow and L C Paulson
    Copyright   1994 TU Muenchen
    
    Cut down and modified for HOL-CASL by Till Mossakowski

Functions for reading CASL files, and storing and retrieving theories,
theorems and the global simplifier set.
Based on encode_isabelle.sml, transform a CASL specification
into an Isabelle/HOL theory and store it under its name.
*)

(*exception TERMS of term list*)

signature READCASLTHY =
sig


   val read_formula   : bool -> casltheory -> string -> term  
   val read_term      : typ -> casltheory -> string -> term list
   val encode_formula : casltheory-> AS.FORMULA-> term
  
   val casltheory_of      : string -> CASLTheory.casltheory
   val casltheory_of_sign : Sign.sg -> CASLTheory.casltheory
   val casltheory_of_thm  : thm -> CASLTheory.casltheory
   val caslthyname_of_sign: Sign.sg -> string


   val store_caslthm      : string * thm -> thm
   val bind_caslthm       : string * thm -> unit
   val caslqed            : string -> unit  
   val path_of: string -> string

end;


structure CASLThyRead: READCASLTHY =
struct

infix $

open AS Global GlobalEnv BasicPrint Subsorts StructEncodeIsabelle 
     ThmDatabase Simplifier CASLTheory CASLScanner;

fun write_type (Type (s,_)) = writeln s
  | write_type _ = writeln "??"

fun get_lambda str =
    case hd (scan str) of
      SignSy "%"::toks => 
        (case take_prefix (fn tok => tok <> Token ".") toks of
           (vars,Token "."::body) => (map str_of_token vars,concat(map str_of_token body))
         | _ => ([],str)
        )
     | _ => ([],str)
    handle _ => ([],str)

fun get_type v (Const _) = None
  | get_type v (Free (w,ty)) = 
    if v=w then Some ty else None
  | get_type _ (Var _) = None
  | get_type _ (Bound _) = None
  | get_type v (Abs (w,_,t)) =
    if v=w then None else get_type v t
  | get_type v (t1 $ t2) =
    case get_type v t1 of
      Some ty => Some ty
    | None => get_type v t2


fun make_lambda vars phi =
    let fun get_var_type v =
        (v,case (get_type v phi) of
            None => dummyT
            | Some ty => ty)
    in list_abs_free (map get_var_type vars, phi)
    end

fun encode_formula (parsethy as (CASLthy{casl_env=SPEC_ENV(Sigma, _, _), thy})) formula =
let  val (errs,[ax1]) = Overload.overload_resolution true (Sigma,[(formula,[])])  
 in
    if errs = [] then
    let val sq = compute_s_question_mark (!use_projections) 
	                                 (LocalEnv.env_to_list Sigma)
	val name = Sign.name_of(sign_of thy)  
(*	val (vars,str1) = get_lambda str *)
	val ax2 = BasicEncode.encode_ax_SubPCFOL_SubCFOL sq ax1
        val ax3 = BasicEncode.encode_ax_SubCFOL_CFOL ax2
        val ax4 = (fst (BasicEncodeIsabelle.encode_ax_CFOL_HOL name thy ax3))
(*      val ax5 = make_lambda vars ax4 *)
        val ax6 = HOLogic.mk_Trueprop ax4
        val (anno_semterm,_) = Sign.infer_types (sign_of thy) 
				(K None) (K None) [] true ([ax6],propT)
    in anno_semterm      
    end
    else
    let fun addnl x = x^"\n"
        val x = print(concat (map addnl errs))
    in raise ERROR
    end
 end

fun read_formula infer_types (parsethy as (CASLthy {casl_env,thy})) str = 
(if isDummyLenv (#casl_env (rep_casltheory parsethy)) then
    hd (Syntax.read (syn_of HOL_CASL.thy) propT str)
 else
 let val ans = []  (* Where to get them ??? *)
     val name = Sign.name_of(sign_of thy)
     val SPEC_ENV (Sigma,_,_) = casl_env
     val sq = compute_s_question_mark (!use_projections) (LocalEnv.env_to_list Sigma)
     val (vars,str1) = get_lambda str
     val (ax,errs1) = BasicAnalysis.parse_axiom true Sigma ans str1
     val (errs2,[ax1]) = Overload.overload_resolution infer_types (Sigma,[(ax,[])])  
     val errs = errs1@errs2
 in
    if errs = [] then
    let val ax2 = BasicEncode.encode_ax_SubPCFOL_SubCFOL sq ax1
        val ax3 = BasicEncode.encode_ax_SubCFOL_CFOL ax2
        val ax4 = (fst (BasicEncodeIsabelle.encode_ax_CFOL_HOL name thy ax3))
        val ax5 = make_lambda vars ax4
        val ax6 = HOLogic.mk_Trueprop ax5
        val (anno_semterm,_) = Sign.infer_types (sign_of thy) 
				(K None) (K None) [] true ([ax6],propT)
    in anno_semterm      
    end
    else
    let fun addnl x = x^"\n"
        val x = print(concat (map addnl errs))
    in raise ERROR
    end
 end)
 
fun strip_arrows (_::rest) (Type ("fun",[_,t])) =
    strip_arrows rest t
  | strip_arrows _ t = t

(* Check it type of encoded t=t fits type required for t 
   and simultaneously abstract wrt. vars *)
fun check_type vars T  (Const (_,Type (_,T1::_)) $ t $ _) =
      if strip_arrows vars T = T1 then Some (make_lambda vars t) else None
  | check_type vars T  (Free (_,Type (_,T1::_)) $ t $ _) =
      if strip_arrows vars T = T1 then Some (make_lambda vars t) else None 
  | check_type _ _ _ = None
                              
fun read_term (T:typ) (parsethy as (CASLthy {casl_env,thy})) str = 
 case T of 
  Type ("bool",[]) => 
    (case read_formula true parsethy str of
     (Const ("Trueprop",_) $ t) => [t])
 | _ =>
 let val ans = []  (* Where to get them ??? *)
     val name = Sign.name_of(sign_of thy)
     val SPEC_ENV (Sigma,_,_) = casl_env
     val sq = compute_s_question_mark (!use_projections) (LocalEnv.env_to_list Sigma)
     val (vars,str1) = get_lambda str
     val (t,errs) = BasicAnalysis.parse_term Sigma ans str1
     val _ = if null errs then () else raise ERROR
     val ts = Overload.overload_resolution_TERM  (Sigma,t)
     (* Encode each possible term t as t=t *)
     val def_ts = map (fn t => (AS.atom (AS.strong_equation (fst t,fst t)),[])) ts 
     val def_ts2 = map (BasicEncode.encode_ax_SubPCFOL_SubCFOL sq) def_ts
     val def_ts3 = map BasicEncode.encode_ax_SubCFOL_CFOL def_ts2
     val def_ts4 = map (fst o (BasicEncodeIsabelle.encode_ax_CFOL_HOL name thy)) def_ts3
     val ts = mapfilter (check_type vars T) def_ts4
     (*val _ = raise TERMS ts*)
     in (if ts = [] then writeln("No correct type found for "^str) else ();
         ts)
 end


(*Get caslthy_info for a loaded theory *)
fun get_caslthy_info tname = Symtab.lookup (!loaded_caslthys, tname);

(*Get theory object for a loaded theory *)
fun casltheory_of name =
  case get_caslthy_info name of Some (CASLThyInfo {theory = t, ...}) => t
                         | _ => if name="HOL" then casltheory_of "HOL_CASL"
                                else error ("Theory " ^ name ^
                                            " not stored by caslloader");

(*Get path where theory's files are located*)
local open OS.Path in
fun path_of tname =
  let val CASLThyInfo {lib_file_name, ...} = the (get_caslthy_info tname)
      val {dir, ...} = splitDirFile lib_file_name
  in lib_file_name end; 
end 
 
(*** Store and retrieve theorems ***)
(*Guess the name of a theory object
  (First the quick way by looking at the stamps; if that doesn't work,
   we search loaded_thys for the first theory which fits.)
*)

fun caslthyname_of_sign sg = Sign.name_of sg;

(*Guess to which theory a signature belongs and return it's thy_info*)
fun caslthyinfo_of_sign sg =
  let val name = caslthyname_of_sign sg;
  in (name, the (get_caslthy_info name)) end;


(*Try to get the theory object corresponding to a given signature*)
fun casltheory_of_sign sg =
  case caslthyinfo_of_sign sg of
    (_, CASLThyInfo {theory =  thy, ...}) => thy



(*Try to get the theory object corresponding to a given theorem*)
val casltheory_of_thm =casltheory_of_sign o #sign o rep_thm;

val store_caslthm = store_thm; 
val bind_caslthm  = bind_thm;   
val caslqed       = qed;
 

end; 
