(* *********************************************************************** *)
(*									   *)
(* Project: HOL-CASL 							   *)
(* Author: Till Mossakowski, University of Bremen		           *)
(* Date: 13.09.2001				 			   *)
(* Purpose of this file: Encoding of CFOL-encoded CASL into Isabelle/HOL   *)
(*			 - the structured level				   *)
(*									   *)
(* *********************************************************************** *)

(* This module lifts the encoding of CASL inso Isabelle/HOL provided
   by basic_encode.sml to the level of structured specifications
   and whole libraries.

  todo:
  Theorie LL, auf Nat basierend, verursacht Fehler beim Laden von Numbers.ML
 
  Bei Erweiterung von parametrisierten Specs, die mit
  formalen Parameter instantiiert werden, auch in Isabelle
  eine Extension erzeugen

  Unions: change the kernel of Isabelle to deal with same name - same thing
  better: using renaming (ISabelle 2002)

  Free: implement inequalities

  Cofree: implement infinite trees (using Isabelles datatypes)

  Annotationen fr das Formel-Parsing weitergeben!

  HOL-Theorien mit in HOL-CASL reinnehmen, z.B. HOLthys.Nat
  Nach Fehler mit HOL-CASL codierung nicht mehr weitermachen!
  HOLisierung von CASL verstecken durch Trennung Objekt-Meta-Theorie:
    eigene konstanten fuer CASL-Junktoren und -Quantoren
    eigene Regeln fuer CASL-Junktoren und -Quantoren beweisen


  The following produces an error because new question mark sorts
  arsing in unions are not included:

library FileSystem

spec FiniteSet  [sort Elem] =
  sort FinSet[Elem]
  op f : Elem -> FinSet[Elem]
end

spec FiniteMap  [sort S][sort T] =
  sort FiniteMap[S,T]
  op  eval:  S * FiniteMap[S,T] ->? T;
and 
     FiniteSet [sort T fit Elem |-> T]
then 
     preds
          __::__->__: FiniteMap[S,T] * FinSet[T]* FinSet[T]
*)

val CASL_Thy = ref(CASLthy{casl_env = GlobalEnv.empty_spec_lenv, thy =HOL.thy});
val caslsimpset = ref(empty_ss:simpset);

exception test of GlobalEnv.global_env

val lib_mod_times = ref(Symtab_str.empty : Time.time Symtab_str.table);
val loaded_libs = ref(GlobalEnv.empty_lib_env);

structure StructEncodeIsabelle :
sig
  datatype caslthy_info =
    CASLThyInfo of {lib_file_name: string,
                    theory: CASLTheory.casltheory}
   val loaded_caslthys    : caslthy_info Symtab.table ref
   val encode_struct_CFOL_HOL : string -> GlobalEnv.global_env ->  AS.LIB_DEFN -> 
                                (string * CASLTheory.casltheory) list
   val encode_L_SPEC : GlobalEnv.global_env 
                     -> AS.SPEC_NAME -> AS.L_SPEC -> CASLTheory.casltheory
   val use_casl        : string -> unit
   val use_casl1       : string -> unit (*CASLTheory.casltheory*)
   val update_casl     : string -> unit
 
   val get_casltheory : string -> CASLTheory.casltheory option

end
= 
struct

open Theory CASLTheory AS LocalEnv GlobalEnv BasicPrint BasicEncode StructEncode BasicEncodeIsabelle;




datatype caslthy_info =
  CASLThyInfo of {lib_file_name: string,
              theory: CASLTheory.casltheory}

val loaded_caslthys  = 
   ref (Symtab.make [("HOL1",
                     CASLThyInfo {lib_file_name = pwd()^"HOL.thy",
                        theory = CASLTheory.CASLthy{casl_env = 
                                     GlobalEnv.empty_spec_lenv,
                                     thy = HOL.thy}
                                 })
                    ]);


fun get_casltheory thn =
    case Symtab.lookup (!loaded_caslthys,thn) of
           Some (CASLThyInfo info) => Some (#theory info) 
         | None => None;
 

fun get_simp_ax (n,thm) =
    case snd (take_prefix  (fn x => x<>".") (explode n)) of
    "."::"g"::"a"::"_"::"g" :: "e" :: "n" :: "e" :: "r":: _ => []
    | "."::"g"::"a"::"_"::"p" :: "r" :: "e" :: "d" :: "_":: _ => []
     (* induction axioms loop, and they are already delt with by CASL_induct_tac *)
(*    | "."::"g"::"a"::"_"::"i" :: "n" :: "j" :: "e" :: "c":: _ => []*)
    | "."::"g"::"a"::"_"::"d" :: "i" :: "s" :: "j" :: "o":: _ => [thm RS (neq_commute RS iffD1),thm]
    | "."::"g"::"a"::"_"::"o" :: "v" :: "e" :: "r" :: "l":: _ => [] 
     (* overload axioms sometimes lead to undesired simplifications *)
    | "."::"g"::"a"::"_"::"s" :: "t" :: "r" :: "i" :: "c"::"t"::"n"::"e"::"s"::"s"::"_"::"("::"p":: _ => [thm]
     (* predicate strictness  axioms are harmless, ... *)
    | "."::"g"::"a"::"_"::"s" :: "t" :: "r" :: "i" :: "c"::"t"::"n"::"e"::"s"::"s"::"_"::"a":: _ => [thm]
     (* strictness and totality axioms are harmless, but ... *)
    | "."::"g"::"a"::"_"::"s" :: "t" :: "r" :: "i" :: "c":: _ => []
     (* ... other strictness axioms loop *)
    | "."::"g"::"a"::"_"::"c" :: "o" :: "m" :: "m" ::  _ => []
     (* ... commutativity axioms loop *)
    | "."::"g"::"a"::"_"::"m" :: "e" :: "m" :: "b" :: "e":: _ => []
     (* membership axioms introduce unwanted existential quantifiers *)
    | "."::"g"::"a"::"_"::_ => [thm]
    | _ => [];

(* ???  _injective as cancellation rule : now achieved through equivalence in generated axioms
  Better way: encode as object implication an then derive equivalence
  Problem: how to get rid off flex-flex equations
*)

fun get_new_simps thy = 
    flat (map get_simp_ax (axioms_of thy))


fun get_thy name =
    case Symtab.lookup (!loaded_caslthys,name) of
       Some (CASLThyInfo info) => #thy (rep_casltheory (#theory info))
     | _ => (writeln ("Theory "^name^" not found"); raise ERROR)


fun get_name (spec_inst_env (name,_,_,_)) = name
  | get_name _ = raise ERROR


fun add_CASL n = "CASL_"^n

fun encode_spec lib_path (*lib_env*) genv (*tree*) (spec_defn(sp_name,_,_,_)) =
    (case Symtab_sid.lookup (genv,sp_name) of
      Utils.Some (spec_defn_env(gen,
           slenv as SPEC_ENV(Sigma1,_, extension_env [(union_env sps,_),
                                                 (basic_env(Sigma,ax),_),
                                          imp_benv] ))) =>
      (let (*fun loopup_sp sp =
           fun mk_name sp =
               let val n = get_name sp
                   val lib_path = lookup_sp sp   (Symtab_str.dest lib_env)
                   val lib_file = #file(OS.Path.splitDirFile lib_path)
               in
                   lib_file^"_"^print_SIMPLE_ID n
               end*)
           fun mk_name sp = add_CASL (print_SIMPLE_ID (get_name sp))
           val refs = map mk_name sps
           val ref_thys = map get_thy refs
(*           val lib_file = #file(OS.Path.splitDirFile lib_path)
           val name1 = lib_file^"_"^(fst sp_name)*)
           val name0 = print_SIMPLE_ID sp_name
           val name1 = add_CASL name0
           val base_thys = if null ref_thys then [HOL_CASL.thy] else ref_thys
           val thy0 = PureThy.begin_theory name1 base_thys
           val thy1 = encode_flat_CFOL_HOL name1 thy0 
                      (Sigma,empty_signature,Utils.remove_dups ax)
           val thy2 = PureThy.end_theory thy1 
           val thy3 = Simplifier.change_simpset_of 
                      (op addsimps)
                      (get_new_simps thy2)
                      thy2
           val casl_thy = (CASLthy{casl_env = slenv, thy = thy3})    
           val _ = loaded_caslthys := 
                    Symtab.update ((name1,CASLThyInfo {lib_file_name=lib_path,
                                                       theory = casl_thy}),
                                   !loaded_caslthys)
   	   val _ = CASL_Thy := casl_thy      
       in (Utils.use_strings 
           ["structure "^ name0 ^"= struct val casl = !CASL_Thy;",
            "val axioms = caslaxioms_of casl;",
            "end;"];
           Some (name1,casl_thy))  
       end
       handle e => 
        (print_exn e;
         writeln ("Could not encode theory "^fst sp_name^" into HOL"); 
         None)  )
    | _ => (writeln ("Cannot find a good entry for "^fst sp_name); 
            None))
  | encode_spec lib_file genv (pos_LIB_ITEM(_,li)) = 
    encode_spec lib_file genv li
  | encode_spec _ _ _ = None

(* Encode a standalone L_SPEC on the fly *)

fun encode_L_SPEC genv sp_name sp =
    let val (slenv,sp',errs) = StructuredAnalysis.structured_analysis 
                                false (empty_signature,genv,sp)
        val x = if null errs then ()
                else raise (error (concat (map (fn x => x^"\n") errs)))
        val slenv1 = eliminate_hiding_slenv sp_name slenv
        val slenv2 = eliminate_translation_slenv sp_name slenv1
        val slenv3 = BasicEncode.encode_slenv_SubPCFOL_Sub_CFOL 
                       true (!use_projections) slenv2
        val slenv4 = eliminate_union_slenv sp_name slenv3
        val  SPEC_ENV(Sigma1,_, extension_env [(union_env sps,_),
                                                 (basic_env(Sigma,ax),_),
                                                 imp_benv] ) = slenv4
        fun mk_name sp = add_CASL (print_SIMPLE_ID (get_name sp))
        val refs = map mk_name sps
        val ref_thys = map get_thy refs
        val name0 = print_SIMPLE_ID sp_name
        val name1 = add_CASL name0
        val base_thys = if null ref_thys then [HOL_CASL.thy] else ref_thys
        val thy0 = PureThy.begin_theory name1 base_thys
        val thy1 = encode_flat_CFOL_HOL name1 thy0 
                   (Sigma,empty_signature,Utils.remove_dups ax)
        val thy2 = PureThy.end_theory thy1 
        val thy3 = Simplifier.change_simpset_of 
                   (op addsimps)
                   (get_new_simps thy2)
                    thy2
        val casl_thy = (CASLthy{casl_env = slenv, thy = thy3})    
    in casl_thy
    end
    handle e => 
     (print_exn e;
      raise (error "Could not encode theory into HOL")) 

(* To encode a library,
   we can rely on linear visibility and encode the specs
   in the order in which they occur in the library *)    

fun get_lib_items (lib_defn(_,lits,_)) = lits
  | get_lib_items (pos_LIB_DEFN (_,ld)) =
    get_lib_items ld

fun encode_struct_CFOL_HOL lib_file Gamma libdefn =
    let (*val _ = writeln "Eliminating hiding"*)
        val Gamma1 = eliminate_hiding Gamma
        (*val _ = writeln "Eliminating renaming"*)
        val Gamma2 = eliminate_translation Gamma1
        (*val _ = writeln "Encoding"*)
        val Gamma3 = encode_SubPCFOL_CFOL (!use_projections) Gamma2
        (*val _ = writeln "Eliminating union"*)
        val Gamma4 = eliminate_union Gamma3
        val lits = get_lib_items libdefn
        val thys = mapfilter (encode_spec lib_file (fst Gamma4)) lits
    in thys
    end 

fun try_to_use file =
    (writeln ("Trying "^file);
    if Utils.exists file then use file else ())

fun get_CASL_LIB_path name =
  if Utils.exists name then name
  else case OS.Process.getEnv "CASL_LIB" of
                  NONE => name
                  | SOME p => p^"/"^name

fun enter_mod_time name =
    let val full_name = get_CASL_LIB_path (LibraryAnalysis.with_casl name)
        val t = OS.FileSys.modTime full_name
    in lib_mod_times := Symtab_str.update ((full_name,t),!lib_mod_times)
    end

fun remove_outdated1 (name,entry) =
    let val full_name = get_CASL_LIB_path (LibraryAnalysis.with_casl name)
        val t_new = OS.FileSys.modTime full_name
    in case Symtab_str.lookup (!lib_mod_times,full_name) of
              Utils.None => None
              | Utils.Some t_old => if Time.>(t_new,t_old) then None
                              else Some (name,entry)
    end
    handle _ => None

fun remove_outdated libenv =
    Symtab_str.make (mapfilter remove_outdated1 (Symtab_str.dest libenv)) 

fun outdate_lib name =
    lib_mod_times := Symtab_str.update((name,Time.zeroTime),!lib_mod_times)

fun loader (name,global_env,libdefn) = 
    (encode_struct_CFOL_HOL name global_env libdefn;
     try_to_use (name^".ML");
     enter_mod_time name)

fun use_casl1 name  =
  let
    val _ = Global.test := true;
    val full_name = LibraryAnalysis.with_casl name
    val simple_name = LibraryAnalysis.without_casl name
    val _ = loaded_libs := remove_outdated(!loaded_libs)
    val (new_lib_env,global_env, libdefn) = 
          LibraryAnalysis.use_casl0 name loader (!loaded_libs)
    val _ = loaded_libs := new_lib_env
    (*val new_casl_thys = encode_struct_CFOL_HOL simple_name global_env libdefn
    val _ = try_to_use (simple_name^".ML") 
    in snd (hd (new_casl_thys)) handle LIST _ => (writeln "Could not encode any theory"; raise ERROR)*)
    in ()
  end

fun use_casl name = (use_casl1 name; ())

fun update_casl name = 
    (outdate_lib name; use_casl name)
end
