(*  Title:      Pure/Syntax/printer.ML
    ID:         $Id: printer.ML,v 1.1 2004/02/13 11:29:20 7till Exp $
    Based on:   Tobias Nipkow and Markus Wenzel, TU Muenchen
    Modified:   Burkhart Wolff, ALU Freiburg

Pretty printing of asts, terms, types and print (ast) translation.
*)

signature PRINTER0 =
sig
  val show_brackets: bool ref
  val show_sorts: bool ref
  val show_types: bool ref
  val show_no_free_types: bool ref
  val show_ext_syntax: bool ref (* bu *)
  val print_mode: string list ref
end;

signature PRINTER =
sig
  include PRINTER0

  type prtab
  val empty_prtab     : prtab
  val extend_prtab    : prtab -> SynExt.xprod list -> prtab
  val update_prtab    : prtab -> SynExt.xprod list -> prtab
  val merge_prtab     : prtab -> prtab -> prtab
  val overwrite_prtab : prtab -> prtab -> prtab
  val infixes_of      : int -> prtab -> (string * string) list


  type prtabs
  val empty_prtabs    : prtabs
  val extend_prtabs   : prtabs -> string -> SynExt.xprod list -> prtabs
  val update_prtabs   : prtabs -> string -> SynExt.xprod list -> prtabs
  val merge_prtabs    : prtabs -> prtabs -> prtabs
  val overwrite_prtabs: prtabs -> prtabs -> prtabs
  val get_tab         : prtabs -> string -> prtab
  val get_anno_sign   : prtabs * string -> (int*int) Library.option

  val term_to_ast: (string -> (bool -> Term.typ -> Term.term list -> Term.term) Library.option)
    -> prtabs -> Term.term -> Ast.ast
  val typ_to_ast: (string -> (bool -> Term.typ -> Term.term list -> Term.term) Library.option)
    -> prtabs -> Term.typ -> Ast.ast
  val sort_to_ast: (string -> (bool -> Term.typ -> Term.term list -> Term.term) Library.option)
    -> prtabs -> Term.sort -> Ast.ast

  val pretty_term_ast: bool -> prtabs
    -> (string -> (Ast.ast list -> Ast.ast) Library.option)
    -> (string -> (string -> string * int) Library.option) -> Ast.ast -> Pretty.T
  val pretty_typ_ast: bool -> prtabs
    -> (string -> (Ast.ast list -> Ast.ast) Library.option)
    -> (string -> (string -> string * int) Library.option) -> Ast.ast -> Pretty.T

end;

(* structure Printer = (* for debugging only *) *) 
structure Printer : PRINTER = 
struct

open Library Term Lexicon Ast SynExt TypeExt SynTrans;

infix 9  $;
infixr 5 -->;
infixr --->;
infix aconv;


(** options for printing **)

val show_types = ref false;
val show_sorts = ref false;
val show_brackets = ref false;
val show_no_free_types = ref false;
val show_ext_syntax = ref false;
val print_mode = ref ([]:string list);



(** misc utils **)

(* apply print (ast) translation function *)

fun apply_trans name a f args =
  (f args handle
    Match => raise Match
  | exn => (writeln ("Error in " ^ name ^ " for " ^ quote a); raise exn));

fun uncurry3 f (x, y, z) = f x y z;


datatype symb =
  Arg of int |
  TypArg of int |
  String of int * string | (* cxl: first component is the logical length *)
  Break of int |
  Block of int * symb list;

fun SString s = String(size s,s);


(* xprods_to_fmts *)


fun xprod_to_fmt (XProd (_, _, "", _)) = None
  | xprod_to_fmt (XProd (_, xsymbs, const, pri)) =
      let	 
        fun cons_str s (String(l', s') :: syms) = String((size s) + l', s ^ s') :: syms
          | cons_str s syms = (SString s) :: syms

	(* set the logical length of the following string to l *)
	fun set_logical_length l ((String(_, s')):: syms)= String(l, s') :: syms
	  | set_logical_length l syms = (* this shouldn't happen (but can with
					 * a pathological grammar) *)
	    (writeln("printer.xprod_to_fmt: logical length without symbol.");
	     (String(l, ""))::syms)

        fun arg (s, p) =
          (if s = "type" then TypArg else Arg)
          (if is_terminal s then max_pri else p);

        fun xsyms_to_syms (Delim s :: xsyms) =
              apfst (cons_str s) (xsyms_to_syms xsyms)
          | xsyms_to_syms (Argument s_p :: xsyms) =
              apfst (cons (arg s_p)) (xsyms_to_syms xsyms)
          | xsyms_to_syms (Space s :: xsyms) =
              apfst (cons_str s) (xsyms_to_syms xsyms)
(* here is where we set the logical length: writing (x)foo as a symbol name
 * gets parsed as [Bg x, En, Delim "foo"], so the following line catches
 * this situation and sets the logical length accordingly [cxl] *)
	  | xsyms_to_syms (Bg i :: En :: xs) =
	      apfst (set_logical_length i) (xsyms_to_syms xs)
          | xsyms_to_syms (Bg i :: xsyms) =
              let
                val (bsyms, xsyms') = xsyms_to_syms xsyms;
                val (syms, xsyms'') = xsyms_to_syms xsyms';
              in
                (Block (i, bsyms) :: syms, xsyms'')
              end
          | xsyms_to_syms (Brk i :: xsyms) =
              apfst (cons (Break i)) (xsyms_to_syms xsyms)
          | xsyms_to_syms (En :: xsyms) = ([], xsyms)
          | xsyms_to_syms [] = ([], []);

        fun nargs (Arg _ :: syms) = nargs syms + 1
          | nargs (TypArg _ :: syms) = nargs syms + 1
          | nargs (String _ :: syms) = nargs syms
          | nargs (Break _ :: syms) = nargs syms
          | nargs (Block (_, bsyms) :: syms) = nargs syms + nargs bsyms
          | nargs [] = 0;
      in
        (case xsyms_to_syms xsymbs of
          (symbs, []) => Some (const, (symbs, nargs symbs, pri))
        | _ => sys_error "xprod_to_fmt: unbalanced blocks")
      end;

(** type prtab **)

type prtab = (symb list * int * int)list Symtab.table;

(* empty, extend, merge prtabs *)

val empty_prtab = Symtab.empty;

(* Durch patchen von 1.1.1.2 entstand:*)
fun extend_prtab tab xprods =
  let
    val fmts = rev (mapfilter xprod_to_fmt xprods);
  in if null fmts then tab
     else Symtab.make_multi (fmts@Symtab.dest_multi tab)   
                                                   	(*prefer new entries*)
  end;

fun merge_prtab tab1 tab2 =
  generic_merge (op =) Symtab.dest Symtab.make tab2 tab1;   

(* There seems to be no difference between extend, merge, overwrite now ...*) 

(* old Version:  (* *)

fun extend_prtab tab xprods =
  Symtab.extend (op =) (tab, xprods_to_fmts xprods)
    handle Symtab.DUPS cs => err_dup_fmts cs;

fun merge_prtabs tab1 tab2 =
  Symtab.merge (op =) (tab1, tab2)
    handle Symtab.DUPS cs => err_dup_fmts cs;
*)

fun update_prtab tab xprods =
  (case mapfilter xprod_to_fmt xprods of
	 (k,xx) :: S => Symtab.update ((k,xx :: (map snd S)), tab)
        | _ => if null xprods then tab 
               else sys_error "illegal copy rule");

fun overwrite_prtab prtab1 prtab2 =
(* more precise: only overwrite those elements in prtabs2,
that occur in both prtab's. *)
    let val con1 = map fst (Symtab.dest prtab1)
        val con2 = map fst (Symtab.dest prtab2)
        fun m (t,x)  = if x mem con2 
			then Symtab.update((x,the (Symtab.lookup(prtab1,x))),t) 
			else t
    in  foldl m (prtab2, con1) end;

type prtabs = (string * prtab) list;

val empty_prtabs = [];

(*find tab for mode*)
fun get_tab prtabs mode =
  if_none (assoc (prtabs, mode)) Symtab.empty;

(*collect tabs for mode hierarchy (default "")*)
fun tabs_of prtabs modes =
  mapfilter (fn mode => assoc (prtabs, mode)) (modes @ [""]);


fun extend_prtabs prtabs mode xprods =
  overwrite (prtabs, (mode, extend_prtab (get_tab prtabs mode) xprods));

fun update_prtabs prtabs mode xprods =
  overwrite (prtabs, (mode, update_prtab (get_tab prtabs mode) xprods));


fun merge_prtabs prtabs1 prtabs2 =
  let
    val modes = distinct (map fst (prtabs1 @ prtabs2));
    fun merge mode = (mode,merge_prtab (get_tab prtabs1 mode)
                                       (get_tab prtabs2 mode));   
  in
    map merge modes
  end;

fun overwrite_prtabs prtabs1 prtabs2 =
(* more precise: only overwrite those elements in prtabs2,
that occur in both prtab's. *)
   let val dom1 = map fst prtabs1;
       val (owv,R)   = partition (fn (x,_)=> exists(fn y=>y=x) dom1) prtabs2 
       fun over (mode,prtab) = (mode,overwrite_prtab (get_tab prtabs1 mode)
                                                     prtab);
   in  (map over owv) @ R end;
   

(*find formats in tab hierarchy*)
fun get_fmts [] _ = []
  | get_fmts (tab :: tabs) a = 
       let val x = Symtab.lookup_multi (tab, a)
       in  if null x then get_fmts tabs a else x end;




fun args_of (Arg _) = 1
   |args_of (Block(_,S)) = foldl(op +) (0,map args_of S)
   |args_of _ = 0;

fun get_anno_sign (prtabs,a) =
    case get_fmts (tabs_of prtabs (! print_mode)) a of
          [([String(k,""),Block(_,S)], _, _)] =>
                Some(k, foldl(op +) (0,map args_of S))
        | _ => None;




(* bu : *)
fun  print_symb d [] = ""
   | print_symb d (x :: S) =
     let fun drop_blanks  [] = [] 
            |drop_blanks (a::r) = if is_blank a then drop_blanks r else a::r
	 fun strip_blanks str = implode(rev(
				   drop_blanks(rev(drop_blanks(explode str)))))
	 val r = case x of
	          Arg _     => "_"
	        | TypArg  _ => "_"
	        | String(_,S) => if d>0 then S else strip_blanks S
	        | Break   _ => if d>1 then "/" else ""
	        | Block(_,S)=> if d>2 then "(" ^ (print_symb d S) ^ ")"
				    else (print_symb d S)
    in    r ^ (print_symb d S) end;

fun infixes_of d tab =
    let fun prin (key,(xsy,_,_)) = (key,print_symb d xsy)
    in  map prin (Symtab.dest_multi tab) end;



(* simple_ast_of *)

fun simple_ast_of (Const (c, _)) = Constant ("",c)
  | simple_ast_of (Free (x, _)) = Variable ("",x)
  | simple_ast_of (Var (xi, _)) = Variable ("",string_of_vname xi)
  | simple_ast_of (t as Const ("_bound", _) $ Free (x, _)) = 
                                          Variable ("_bound",x)
  | simple_ast_of (t as Const ("_free", _) $ Free (x, _)) = 
                                          Variable ("_free",x)  
  | simple_ast_of (t as Const ("_var", _) $ Var (xi, _)) = 
                                          Variable ("_var",string_of_vname xi)  | simple_ast_of (t as _ $ _) =
      let val (f, args) = strip_comb t in
        mk_appl (simple_ast_of f) (map simple_ast_of args)
      end
  | simple_ast_of (Bound i) = Variable ("_bound","B." ^ string_of_int i)
  | simple_ast_of (Abs _) = sys_error "simple_ast_of: Abs";


(** sort_to_ast, typ_to_ast **)

fun ast_of_termT trf tm =
  let
    fun ast_of (t as Const ("_class", _) $ Free (x, _)) = Variable ("_class",x)
      | ast_of (t as Const ("_sort", _) $ Free (x, _)) = Variable ("_sort",x)
      | ast_of (t as Const ("_tfree", _) $ Free (x, _)) = Variable ("_tfree",x)
      | ast_of (t as Const ("_tvar", _) $ Var (xi, _)) = 
                                          Variable ("_tvar",string_of_vname xi)
      | ast_of (Const (a, _)) = trans a []
      | ast_of (t as _ $ _) =
          (case strip_comb t of
            (Const (a, _), args) => trans a args
          | (f, args) => Appl (map ast_of (f :: args)))
      | ast_of t = raise TERM ("ast_of_termT: bad term encoding of type", [t])
    and trans a args =
      (case trf a of
        Some f => ast_of (apply_trans "print translation" a (uncurry3 f)
          (false, dummyT, args))
      | None => raise Match)
          handle Match => mk_appl (Constant ("",a)) (map ast_of args);
  in
    ast_of tm
  end;

fun sort_to_ast trf prtab S = ast_of_termT trf (term_of_sort S);
fun typ_to_ast trf prtab T = ast_of_termT trf (term_of_typ (! show_sorts) T);



(** term_to_ast **)

fun mark_freevars (t $ u) = mark_freevars t $ mark_freevars u
  | mark_freevars (Abs (x, T, t)) = Abs (x, T, mark_freevars t)
  | mark_freevars (t as Free _) = const "_free" $ t
  | mark_freevars (t as Var _) = const "_var" $ t
  | mark_freevars a = a;

fun ast_of_term trf no_freeTs show_types show_sorts prtab tm =
  let
    fun prune_typs (t_seen as (Const _, _)) = t_seen
      | prune_typs (t as Free (x, ty), seen) =
          if ty = dummyT then (t, seen)
          else if no_freeTs orelse t mem seen then (free x, seen)
          else (t, t :: seen)
      | prune_typs (t as Var (xi, ty), seen) =
          if ty = dummyT then (t, seen)
          else if no_freeTs orelse t mem seen then (var xi, seen)
          else (t, t :: seen)
      | prune_typs (t_seen as (Bound _, _)) = t_seen
      | prune_typs (Abs (x, ty, t), seen) =
          let val (t', seen') = prune_typs (t, seen);
          in (Abs (x, ty, t'), seen') end
      | prune_typs (t1 $ t2, seen) =
          let
            val (t1', seen') = prune_typs (t1, seen);
            val (t2', seen'') = prune_typs (t2, seen');
          in
            (t1' $ t2', seen'')
          end;

    fun mk_anno a T args =
        case  get_anno_sign (prtab, a) of
            Some(k,no)=>let val args' = map ast_of args
                        in if length(args) <= no
                           then Anno(k,(Constant ("",a))::args')
                           else Appl(Anno(k,(Constant ("",a))
                                     ::take(k,args')) :: drop (k,args'))
                        end
           |None   => trans a T args

    and ast_of tm =
      (case strip_comb tm of
        (t as Abs (_,_,s), ts) => 
              (case strip_comb s of
                  (Const (a, T), args) => 
                      if is_some(get_anno_sign (prtab, a))
                         andalso !eta_contract
                      then let val _ = (eta_contract:=false)
                               val r = mk_appl (ast_of (abs_tr' t)) 
                                               (map ast_of ts)
                               val _ = (eta_contract:=true)
                           in r end
                      else mk_appl (ast_of (abs_tr' t)) 
                                   (map ast_of ts)
                  | _ =>   mk_appl (ast_of (abs_tr' t)) 
                                   (map ast_of ts)
               )
      | ((c as Const ("_free", _)), Free (x, T) :: ts) =>
          mk_appl (constrain (c $ free x) T) (map ast_of ts)
      | ((c as Const ("_bound", _)), Free (x, T) :: ts) =>
          mk_appl (constrain (c $ free x) T) (map ast_of ts)
      | ((c as Const ("_var", _)), Var (xi, T) :: ts) =>
          mk_appl (constrain (c $ var xi) T) (map ast_of ts)
      | (Const (a, T), args) => (mk_anno a T args)
      | (t, ts) => mk_appl (simple_ast_of t) (map ast_of ts))
 

    and trans a T args =
      (case trf a of
        Some f =>
          ast_of (apply_trans "print translation" a (uncurry3 f)
            (show_sorts, T, args))
      | None => raise Match)
          handle Match => mk_appl (Constant ("",a)) (map ast_of args)

    and constrain t T =
      if show_types andalso T <> dummyT then
        Appl [Constant ("",constrainC), simple_ast_of t,
          ast_of_termT trf (term_of_typ show_sorts T)]
      else simple_ast_of t
  in
    tm
    |> prop_tr'
    |> (if show_types then #1 o prune_typs o rpair [] else I)
    |> mark_freevars
    |> ast_of
  end;

(* term_to_ast *)
fun term_to_ast trf prtabs tm =
  ast_of_term trf (! show_no_free_types) (!show_types orelse !show_sorts) 
    (!show_sorts) prtabs tm;




(** pretty term or typ asts **)

fun chain [Block (_, pr)] = chain pr
  | chain [Arg _] = true
  | chain _  = false;


fun strip_logical_length "" = (None,"")
   |strip_logical_length str= 
         if String.sub(str, 0)= #"(" 
         then let fun cls j= if String.sub(str, j)= #")" 
                             then j else (cls (j+1))
	      val j = cls 1
	      in  case Int.fromString (String.substring(str, 1, j-1)) of
	              NONE =>    (None,str)
		     |SOME len =>(Some len,String.substring(str, j+1, 
                                                  (String.size str)-(j+1)))
	      end handle Subscript => (None,str)
         else (None,str);

fun varname str = case strip_logical_length str of
                  (None,str) => Pretty.str str
                 |(Some len,str) => Pretty.strlen str len



fun pretty prtabs trf tokentrf type_mode curried ast0 p0 =
  let
    val trans = apply_trans "print ast translation";

    fun token_trans c [Variable (_,x)] =
        let val (ss,x) = strip_logical_length x in
            case tokentrf c of
              None => None
            | Some f => Some (if is_none ss then f x else (fst(f x),the ss))
        end
      | token_trans _ _ = None;

    (*default applications: prefix / postfix*)
    val appT =
      if type_mode then tappl_ast_tr'
      else if curried then applC_ast_tr'
      else appl_ast_tr'

    fun synT _ ([], args) = ([], args)
    
      | synT None (Arg p :: symbs, t :: args) =
          let val (Ts, args') = synT None (symbs, args);
          in (astT (t, p) @ Ts, args') end
      | synT (Some (1,p)) (Arg _ :: symbs, t :: args) =
          let val (Ts, args') = synT (Some (0,p))(symbs, args);
          in (astT (t, p) @ Ts, args') end
      | synT (Some (k,p'))(Arg p :: symbs, t :: args) =
          let val (Ts, args') = synT (Some (k-1,p'))(symbs, args);
          in ((map Pretty.set_length_zero (astT (t, p))) @ Ts, args') end
          
      | synT None (TypArg p :: symbs, t :: args) =
          let
            val (Ts, args') = synT None (symbs, args);
          in
            if type_mode then (astT (t, p) @ Ts, args')
            else (pretty prtabs trf tokentrf true curried t p @ Ts, args')
          end
      | synT (Some (1,p)) (TypArg _ :: symbs, t :: args) =
          let
            val (Ts, args') = synT (Some (0,p)) (symbs, args);
          in
            if type_mode then (astT (t, p) @ Ts, args')
            else (pretty prtabs trf tokentrf true curried t p @ Ts, args')
          end
      | synT (Some (k,p')) (TypArg p :: symbs, t :: args) =
          let
            val (Ts, args') = synT (Some (k-1,p')) (symbs, args);
          in
            if type_mode then (astT (t, p) @ Ts, args')
            else ((map Pretty.set_length_zero 
                   (pretty prtabs trf tokentrf true curried t p)) @ Ts, args')
          end
          
      | synT None (String(l, s) :: symbs, args) =
          let val (Ts, args') = synT None (symbs, args);
          in ((Pretty.strlen s l) :: Ts, args') end
 
      | synT X (String(l, s) :: symbs, args) =
          let val (Ts, args') = synT X (symbs, args);
          in ((Pretty.strlen s 0) :: Ts, args') end

      | synT X (Block (i, bsymbs) :: symbs, args) =
          let
            val (bTs, args') = synT X (bsymbs, args);
            val (Ts, args'') = synT X (symbs, args');
          in (Pretty.blk (i, bTs) :: Ts, args'') end
      | synT X (Break i :: symbs, args) =
          let val (Ts, args') = synT X (symbs, args);
          in ((if i < 0 then Pretty.fbrk else Pretty.brk i) :: Ts, args') end
      | synT X (_ :: _, []) = sys_error "synT"

    and parT X (pr, args, p, p': int) = #1 (synT X
          (if p > p' orelse
            (! show_brackets andalso p' <> max_pri andalso not (chain pr))
            then [Block (1, SString "(" :: pr @ [SString ")"])]
            else pr, args))

    and prefixT (_, a, [], _) = [Pretty.str a]
      | prefixT (c, _, args, p) =
          if c = Constant ("","_appl") orelse c = Constant ("","_applC") then
            error "Syntax insufficient for printing prefix applications!"
          else astT (appT (c, args), p)

    and splitT 0 ([x], ys) = (x, ys)
      | splitT 0 (rev_xs, ys) = (Appl (rev rev_xs), ys)
      | splitT n (rev_xs, y :: ys) = splitT (n - 1) (y :: rev_xs, ys)
      | splitT _ _ = sys_error "splitT"

    and combT (tup as (c, a, args, p), f, args_wo_annos) =
      (* PRE c is constant and a its name.
       * Principle: only when mix-fix notation is generated for c, *
       * all annotations occuring in the spine are ommited.
       * Otherwise ast-translations for appl-generation are
       * fired - on the original tree (f,args) !*)
      let
        val nargs = length args_wo_annos;
        fun prnt (pr, n, p') =
            if   nargs = n then 
                  (* Print mixfix, possibly with parantheses *)
                  parT None (pr, args_wo_annos, p, p')
            else if nargs < n orelse type_mode then 
                     (* Print mixfix, possibly with parantheses *)
		     if !show_ext_syntax then 
			let val a' = print_symb 0 pr 
                        (* not optimally implemented: will ommit annos
                           in the appl-spine; bu *)
			in  prefixT (Constant ("",a'),a',args_wo_annos, p) end
		     else prefixT (f, a, args, p)
                 else astT (appT (splitT n ([c], args)), p);
      in(case token_trans a args of     (*try token translation function*)
          Some (s, len) => [Pretty.strlen s len]
        | None =>
            (case (trf a, get_fmts prtabs a) of
                  (* No mixfix available *)
                  (None, [])        => prefixT (f, a, args, p)
                  (* Mixfix available, fire prnp *)
                | (None, (prnp::_)) => prnt prnp
                  (* ast-translation available; will not work with tagging *)
                | (Some f, []) =>
                      (astT (trans a f args, p) 
                       handle Match => prefixT (c, a, args, p))
                  (* ast-translation available; will not work with tagging *)
                | (Some f, (prnp::_)) =>
                      (astT (trans a f args, p) 
                       handle Match => prnt prnp)))
      end

    and combAnnoT k (tup as (c, a, args, p)) =
        (case get_fmts prtabs a of (* lookup template of annotation *)
      	   [] => raise AST("pretty: malformed annotation with no infix",args)
      	  |[(pr, n, _)] =>  
      	    (if length args = n 
      	     then #1 (synT (Some (k-1,p))(pr, args)) 
      	     else raise AST("pretty: malformed annotation \
                            \with too few args",args))
          | _ =>raise AST("pretty:malformed annotation \
                             \with multiple infixes",args))
     

    and astT (c as Constant (_,a), p) = 
              (* decompose for mixfix analysis *)
              combT((c, a, [], p),c,[])
      | astT (Variable (_,x), _) = 
              (* just translate variable names *)
              [varname x]
      | astT (Anno (k, (c as Constant (_,a)) :: (args as _ :: _)), p) = 
              combAnnoT k (c, a, args, p) 
      | astT (ast as Anno (_, _), p) = 
              raise AST("pretty: malformed ast",[ast])
      | astT (ast as Appl (f::args), p) =
	  (case strip_anno_spine ast of
              (* decompose for mixfix analysis *)
              (c as Constant (_,a),args' as _ :: _)=>
                         combT((c,a,args,p),f,args')
              (* leading symbol variable => standard appl construction
                 via ast translation *)
             |(f',args' as _ :: _) => astT (appT (f, args), p)
             |(_,_) => raise AST("pretty: malformed ast 2",[ast]))
      | astT (ast as Appl _, _) = 
             raise AST ("pretty: malformed ast", [ast])
  in
    astT (ast0, p0)
  end;


(* pretty_term_ast *)

fun pretty_term_ast curried prtabs trf tokentrf ast =
  Pretty.blk (0, pretty (tabs_of prtabs (! print_mode))
    trf tokentrf false curried ast 0);


(* pretty_typ_ast *)

fun pretty_typ_ast _ prtabs trf tokentrf ast =
  Pretty.blk (0, pretty (tabs_of prtabs (! print_mode))
    trf tokentrf true false ast 0);


end;
