(* ******************************************************************* *)
(*								       *)
(* Project: CASL: 					       	       *)
(* Author: Till Mossakowski, University of Bremen		       *)
(* Date: November 1999			 			       *)
(* Purpose of this file: Static analysis of libraries		       *)
(*			 					       *)
(*								       *)
(* ******************************************************************* *)

(* Static analysis of libraries.
   This depends on the static analysis of individual LIB-ITEMs
   (provided by struct_ana.sml and arch/arch_analysis.sml).
  
   A library is read in, and the LIB-ITEMs are checked.
   Download-ITEMs are handled explicitly here by recursively calling
   the library analysis for the imported library.
   It is checked whether such imports are circular. 

  todo:
  Better library management (store and read in .xml files,
    perform up-to-date check and parse&check only modified libraries
  Richtige Reihenfolge der Fehlermeldungen! (besonders bei download-items)
  Up-to-date-check fuer Libraries, wo moelgich Rueckgriff auf
    gespeichertes globales Env
  Zerstueckeln der Libraries in einzelne LIB-ITEMs
    (als Text, als Syntax tree, als globales Env,
     + Info zu welcher Libraries das File gehoert
     + Library-Info-File mit Verweis auf einzelne Stuecke
       und importierte Librareis)
    Library-Namen als Directories, in denen der ganze Kram dann liegt
  Download von URLs
  Versionsmanagement
  Globale Registrations-Prozedur

  LibEnv needs all indirectly used specs! (e.g. if ExtBooleanAlgebra
    is imported, also BooleanAlgebra and SigOrder have to be)
*)

structure LibraryAnalysis :
sig

type loader = string * GlobalEnv.global_env * AS.LIB_DEFN -> unit 

(* Main function for reading in a CASL library, wrt. the basic datatypes *)
val use_casl : string -> unit

(* The same function, but giving back the new envs and the abstract syntax tree.
   The library environment can be supplied,
   the loader function can do furhter processing on the file name, e.g.
   load a tactic script.
*)
val use_casl0 : string -> loader -> GlobalEnv.lib_env
         -> GlobalEnv.lib_env * GlobalEnv.global_env * AS.LIB_DEFN

(* The same function, but now a bool flag determindes whether the
   basic datatypes are pre-loaded or the empty lib_env is used.
*)
val use_casl1 : string -> loader -> bool
         -> GlobalEnv.lib_env * GlobalEnv.global_env * AS.LIB_DEFN

(* The same, but with fewer input and output *)
val use_casl2 : string -> AS.LIB_DEFN

val use_casl3 : string -> GlobalEnv.global_env * AS.LIB_DEFN

(* The same function, but always using the empty library environment *)
val parse_and_check1 : string 
          -> GlobalEnv.lib_env * GlobalEnv.global_env * AS.LIB_DEFN

(* Same function, but you can supply a library environment.
   The bool flag determindes whether output should be HTML-conform
   (this is used for the web interface)
*)
val parse_and_check : GlobalEnv.lib_env -> loader -> bool -> string 
          -> GlobalEnv.lib_env * GlobalEnv.global_env * AS.LIB_DEFN

(* parse_and_check1 and parse_and_check without parsing. 
   Supply an AS tree instead *)
val check1 : AS.LIB_DEFN 
          -> GlobalEnv.lib_env * GlobalEnv.global_env * AS.LIB_DEFN
val check :  GlobalEnv.lib_env -> loader -> bool -> AS.LIB_DEFN 
          -> GlobalEnv.lib_env * GlobalEnv.global_env * AS.LIB_DEFN


(* For debugging the mixfix analysis. Gives back (for the first mixfix
   analysis error) the list of possible parses as Isabelle terms
*)
val use_casl_term : string -> Term.term list

val extract_LIB_NAME : AS.LIB_NAME -> string

(* For a filename, remove or add the .casl extension, if not already done *)
val without_casl : string -> string
val with_casl : string -> string

(* Environment of libraries of basic datatypes *)
val basic_lib_env : GlobalEnv.lib_env ref

(* Compute the path of a CASL library *)
val get_casl_lib_path :  GlobalEnv.lib_env -> string -> string

(* Get the libraries referenced by a library *)
val get_referenced_libs : AS.LIB_DEFN -> string list

end
= struct

open Utils AS GlobalEnv StructuredAnalysis ArchitecturalAnalysis
     BasicPrint StructuredPrint TextIO TextForm


type loader = string * GlobalEnv.global_env * AS.LIB_DEFN -> unit 
    
val basic_lib_env = ref empty_lib_env


fun addstarnl  s = s^"\n"
          
fun print_name (spec_defn (n,_,_,_)) = "spec "^print_SIMPLE_ID n
  | print_name (view_defn (n,_,_,_,_)) = "view "^print_SIMPLE_ID n
  | print_name (arch_spec_defn (n,_,_)) = "arch spec "^print_SIMPLE_ID n
  | print_name (unit_spec_defn (n,_,_)) = "unit spec "^print_SIMPLE_ID n
  | print_name (download_items (n,_,_)) = "from "^print_LIB_NAME n
  | print_name (pos_LIB_ITEM (_,li)) = print_name li

fun error_handle html exn (genv,treelist) =
   ( print_text html(
   "*** Internal error 1 !!!!! Please send us your specification!\n");
          if !Global.test then raise exn else ();
          (genv,treelist)
         );

fun print_errs html 
    (loaded_libs,genv1,tree1,errs) treelist 
    = (print_text html (concat (map addstarnl (errs)));
         (loaded_libs,genv1,tree1::treelist)
                              )
fun without_casl name  = 
  let val n = explode name 
  in if (rev (take (5,rev n)))=explode(".casl") 
     then implode(rev (drop (5,rev n))) else name
  end;

fun with_casl name  = without_casl name  ^ ".casl";


(*** Load a LIB_ITEM from a library ***)

fun add_genv html r lib_genv (genv,item_name name) =
   (Symtab_sid.update_new 
      ((name,the (Symtab_sid.lookup (lib_genv,name))),genv)
    handle OPTION => 
               (print_text html (perr r^print_SIMPLE_ID name
                                 ^" not found\n"); genv)
           | DUP => (print_text html 
                      (perr r^print_SIMPLE_ID name^
                       " already exists\n"); genv))
  | add_genv html r lib_genv (genv,item_name_map (name,newname)) =
   (Symtab_sid.update_new 
      ((newname,the (Symtab_sid.lookup (lib_genv,name))),genv)
    handle OPTION => (print_text html (perr r^print_SIMPLE_ID name
                                       ^" not found\n"); genv)
           | DUP => (print_text html (perr r^print_SIMPLE_ID newname
                                       ^" already exists\n"); genv))
  | add_genv html _ lib_genv (genv,pos_ITEM_NAME_OR_MAP(r,inm)) =
    add_genv html r lib_genv (genv,inm)


(*** Load a library ***)

fun get_casl_lib_path loaded_libs name =
    case Symtab_str.lookup (loaded_libs,name) of
    Some _ => name
    | None =>  case OS.Process.getEnv "CASL_LIB" of
                  NONE => name
                  | SOME p => p^"/"^name


fun extract_LIB_ID (url p) = p
  | extract_LIB_ID (path_name p) = p
  | extract_LIB_ID (pos_LIB_ID (r,lid)) =
    extract_LIB_ID lid

fun extract_LIB_NAME (lib libid) =
    extract_LIB_ID libid
  | extract_LIB_NAME (versioned_lib (libid,_)) =
    extract_LIB_ID libid
  | extract_LIB_NAME (pos_LIB_NAME (_,ln)) =
    extract_LIB_NAME ln

fun get_referenced_libs1 (download_items (libname,_,_)) =
    Some (extract_LIB_NAME libname)
  | get_referenced_libs1 _ = None

fun get_referenced_libs (lib_defn(name,treelist,ans)) =
    mapfilter get_referenced_libs1 treelist
  | get_referenced_libs (AS.pos_LIB_DEFN (_,ld)) =
    get_referenced_libs ld

fun use_casl_lib2 loaded_libs loader html r (url p) =  
    use_casl_lib loaded_libs loader html r p (get_casl_lib_path loaded_libs p)
  | use_casl_lib2 loaded_libs loader html r (path_name p) = 
    use_casl_lib loaded_libs loader html r p (get_casl_lib_path loaded_libs p)
  | use_casl_lib2 loaded_libs loader html _ (pos_LIB_ID (r,lid)) =
    use_casl_lib2 loaded_libs loader html r lid


and use_casl_lib1 loaded_libs loader html (lib libid) = 
    use_casl_lib2 loaded_libs loader html null_region libid
  | use_casl_lib1 loaded_libs loader html (versioned_lib (libid,_)) =
    use_casl_lib2 loaded_libs loader html null_region libid
  | use_casl_lib1 loaded_libs loader html (pos_LIB_NAME (_,ln)) =
    use_casl_lib1 loaded_libs loader html ln

(*** Check one spec_defn ***)

and check_lib_item_aux loader html reg
         (loaded_libs,
          Gamma,
          lib_item as (spec_defn _)) =
    let val (Gamma',lib_item',errs) =
            spec_or_view_defn_analysis html (Gamma,pos_LIB_ITEM(reg,lib_item))
    in (loaded_libs,Gamma',lib_item',errs)
    end

(*** Check one view_defn ***)

|  check_lib_item_aux loader html reg
         (loaded_libs,
          Gamma,
          lib_item as (view_defn _)) =
    let val (Gamma',lib_item',errs) =
            spec_or_view_defn_analysis html (Gamma,pos_LIB_ITEM(reg,lib_item))
    in (loaded_libs,Gamma',lib_item',errs)
    end

(*** Check one unit_spec_defn ***)

|  check_lib_item_aux loader html reg
   (loaded_libs,Gamma,tree as (unit_spec_defn (name,usp,ans))) =
   let val (Gamma',(name',usp'),errs) = stat_unit_spec_defn (Gamma,(name,usp),[])
   in (loaded_libs,Gamma',unit_spec_defn(name',usp',ans),errs) 
   end

(*** Check one arch_spec_defn ***)

|  check_lib_item_aux loader html reg
   (loaded_libs,Gamma,tree as (arch_spec_defn (name,asp,ans))) =
   let val (Gamma',(name',asp'),errs) = stat_arch_spec_defn (Gamma,(name,asp),[])
   in (loaded_libs,Gamma',arch_spec_defn(name',asp',ans),errs) 
   end
   
(*** Check one download_items ***)

|  check_lib_item_aux loader html reg
   (loaded_libs,(genv,ans1),tree as (download_items (libname,items,_))) =
   let val (loaded_libs1,(genv1,ans2),_) = use_casl_lib1 loaded_libs loader html libname
       val genv2 = foldl (add_genv html null_region genv1) (genv,items)
   in
     (loaded_libs1,(genv2,ans1@ans2),tree,[])  (* ??? Delete conflicting annos !!! *)
   end

(*** Handle region information ***)

|  check_lib_item_aux loader html _
   (loaded_libs,Gamma,pos_LIB_ITEM(r,li)) =
   let val (loaded_libs',Gamma',li',errs) = 
           check_lib_item_aux loader html r (loaded_libs,Gamma,li)
   in (loaded_libs',Gamma',pos_LIB_ITEM(r,li'),errs)
   end

  
(*** Check one LIB_ITEM ***)


and parse_and_check_lib_item loader html
    ((loaded_libs,genv,treelist),tree) =
     (if !Global.quiet_mode then () else 
        print (print_heading html ("Analyzing "^print_name tree^"...")); 
      print_errs html (check_lib_item_aux loader html null_region
                         (loaded_libs,genv,tree)) treelist 
     )

and check_lib_item loader html ((loaded_libs,genv,treelist),tree) =
     print_errs html 
          (check_lib_item_aux loader html null_region (loaded_libs,genv,tree))
                                   treelist 
 

(*** Parse and type check a LIBRARY ***)


and exn_handler exn html libname loaded_libs =
    if !Global.test then raise exn 
    else case exn of
       (ERR e) => (print_text html ("*** "^e^"\n"); raise ERROR)
       | ERROR => (print_text html ("*** Library "^print_LIB_NAME libname
                                               ^" contained errors\n");
                   raise ERROR)
       | Match => (Global.InternalError "non-exhaustive match"; raise ERROR)
       | x =>  raise x (*print_text html ("*** Library "
                          ^print_LIB_NAME libname^" contained errors\n")*)

and get_ld (pos_LIB_DEFN(_,ld)) = get_ld ld
  | get_ld (lib_defn(name,treelist,ans)) = (name,treelist,ans)

and parse_and_check loaded_libs loader html (thisspec:string) =
  let val (name,treelist,ans) = 
          get_ld (CASLParser.parse html thisspec)
          (*handle _ => raise ERROR *)
      val pans = CASLParser.ParseAnnos ans
          handle _ => raise ERROR
      val (libs',genv,treelist') = 
           foldl (parse_and_check_lib_item loader html)
                 ((loaded_libs,(Symtab_sid.empty,pans),[]),treelist)
           handle exn => 
                  exn_handler exn html empty_lib_name loaded_libs 
  in
    (libs',genv,lib_defn (name, rev treelist',pans))
  end



and parse_and_check1 s = parse_and_check empty_lib_env (fn x => ()) false s
 
and check loaded_libs loader html 
    (lib_defn (libname,treelist,ans)) =
  let val (libs',genv,treelist1) = 
           (foldl (check_lib_item loader html) 
                  ((loaded_libs,(Symtab_sid.empty,ans),[]),
                    treelist))
                handle exn => (exn_handler exn html libname loaded_libs;
                              (loaded_libs,empty_global_env,treelist))
  in (libs',genv,lib_defn (libname,rev treelist1,ans))
  end
 | check loaded_libs loader html (pos_LIB_DEFN(_,ld)) =
   check loaded_libs loader html ld

and check1 s = check empty_lib_env (fn x => ()) false s

and use_casl_lib loaded_libs loader html r name fullname = 
    let val name1 = without_casl name
        val casl_name = with_casl fullname
        val lib_entry = Symtab_str.lookup (loaded_libs,name1)
    in
    case lib_entry of
      Some (genv,tree) => 
       (if (* genv=empty_global_env andalso *) tree=empty_lib_defn
        then print_text html 
               ("***"^perr r^"cyclic library reference "^name1^"\n")
        else ();      
        (loaded_libs,genv,tree)
       )
     | None => 
       (let 
          val err_txt = perr r
          val old_file = !Global.cur_file
          val spec = (if !Global.quiet_mode then () 
                      else print_text html 
                              ("\nReading "^casl_name^"\n");
                      Global.cur_file := with_casl name; (*#file(OS.Path.splitDirFile casl_name);*)
                      Utils.read_without_cr (casl_name)
                      handle _ => (print_text html 
                                   (err_txt^casl_name^" not found\n"); "") )
          val temp_libs = Symtab_str.update
              ((name1,(empty_global_env,empty_lib_defn)),loaded_libs)
          val (libs',genv,trees) = 
              fst (parse_and_check temp_libs loader html spec,
                   if !Global.quiet_mode then () else print_text html "\n")
          val _ = loader (without_casl fullname,genv,trees)
          val _ = Global.cur_file := old_file 
          in
            (Symtab_str.update ((name1,(genv,trees)),libs'),genv,trees)
          end)
    end

and use_casl0 name loader lib_env =
    use_casl_lib lib_env loader false null_region name name
        
and use_casl1 name loader use_basic_lib = 
    use_casl0 name loader
        (if use_basic_lib then (!basic_lib_env)
           else empty_lib_env)
    
val trivial_loader = (fn x => ())
    
fun use_casl  s = (use_casl1 s trivial_loader true; ()) 

fun use_casl_term s = (Global.test:=true; 
                       use_casl s; 
                       []) handle IsabelleParser.AMBIGUOUS ts => ts

fun use_casl2 name =
    let val (a,b,c) = use_casl1 name trivial_loader true
    in c
    end

fun use_casl3 name =
    let val (a,b,c) = use_casl1 name trivial_loader true
    in (b,c)
    end


end;
