(* *********************************************************************** *)
(*                                                                         *)
(* Project: CATS                                                           *)
(* Author: Pascal Schmidt, University of Bremen (Student)                  *)
(* Date: Fr, 27 Aug 1999                                                   *)
(* Purpose of this file: conversion of AS trees to LaTeX                   *)
(*                                                                         *)
(* *********************************************************************** *)

(* $Id: mklatex.sml,v 1.1 2004/02/13 11:29:08 7till Exp $ *)

(* The conversion function traverses the abstract syntax tree and outputs
   LaTeX as required for the casl.sty package, see CoFI study note C-1.

   Uses pprinter.sml for pretty printing of terms and formulas
*)

signature Makellos
=
sig
  val mkTeX      : (bool * GlobalEnv.global_env * AS.LIB_DEFN) -> string
  val mkLaTeX    : AS.LIB_ITEM -> string

  val testFun    : GlobalEnv.global_env * AS.LIB_DEFN -> string

  val frame      : string -> string;
end

structure Makellos : Makellos
=
struct

open Utils AS GlobalEnv

exception Unsupported
exception LaTeX_DisplayAnnotation

(****************************************************************************)

fun cur_vers () = "$Id: mklatex.sml,v 1.1 2004/02/13 11:29:08 7till Exp $"

val cur_spec = ref ""
val cur_genv = ref (empty_global_env)
val cur_syn  = ref (IsabelleParser.CASL98,[]:PPL.precedence,
                    ([],[],[],[],[],[],[],[]):PPL.literals)
val cur_mode = ref false
val cur_maxl = ref 30

fun get_ref (ref x) = x


fun init []     = []
  | init [h]    = []
  | init (h::t) = h :: (init t)

fun uni_space sp st = if ((String.size st) < sp) then
                      (uni_space sp (st^" ")) else st

fun mkComment co st = (uni_space 30 co) ^ "%%[Makellos 0.1]%% " ^ st

fun uni_sp sp [s,t] = if ((String.size s) < sp) then
                      (uni_sp sp [(s^" "),t]) else [s,t]
  | uni_sp sp [s]   = [s]  
  | uni_sp sp []    = [""]      
  | uni_sp sp (s::t) = if ((String.size s) < sp) then
                       (uni_sp sp (s^" "::t)) else (s::t)

fun is_eq ch x = if x=ch then true else false

fun split_at ch st = String.fields (is_eq ch) st 

fun rejoin ch [s,t] = s ^ (str ch) ^ t
  | rejoin ch [s]   = s
  | rejoin ch []    = ""
  | rejoin ch (s::t) = s ^ (str ch) ^ (rejoin ch t)

fun relink []     = ""
  | relink [h]    = h
  | relink (h::t) = h ^ "\n" ^ (relink t)

fun f_len [s,t]  = String.size s
  | f_len [s]    = 0
  | f_len []     = 0
  | f_len (s::t) = String.size s

fun find_len [] = 0
  | find_len (h::t) = if (find_len t)>(f_len h) then
                        (find_len t)
                      else
                         (f_len h)

fun form_amp x = ( relink o (map (rejoin #"&")) o
                   (map (uni_sp (find_len  
                    ( ( (map (split_at #"&")) o (split_at #"\n") ) x )
                   ) ) ) o
                   (map (split_at #"&")) o (split_at #"\n") ) x

fun chkL [] = []
  | chkL ((#"/")::(#"/")::t) = (#"/") :: (chkL t)
  | chkL (h::t) = if (h=(#"_")) then ((String.explode "\\_") @ (chkL t))
                  else
                  if (h=(#"^")) then ((String.explode "\\symbol{94}") @ (chkL t))
                  else
(*                  if (h=(#"~")) then ((String.explode "\\symbol{126}") @ (chkL t))
                  else *)
                  if (h=(#"{")) then ((String.explode "\\{") @ (chkL t))
                  else
                  if (h=(#"}")) then ((String.explode "\\}") @ (chkL t))
                  else
                  if (h=(#"#")) then ((String.explode "\\#") @ (chkL t))
                  else
                  if (h=(#"\n")) then (chkL t)
                  else
                  (h :: (chkL t))

fun chkL2 [] = []
  | chkL2 ("_"::("_"::t)) = (explode "\\_\\_") @ (chkL2 t)
  | chkL2 (h::t)          = h :: (chkL2 t)

fun check2 x = (implode o chkL2 o explode) x

fun check x = (String.implode o chkL o String.explode) x 

fun wstr [] = []
  | wstr ((#" ")::t) = (wstr t)
  | wstr (h::t)      = h :: (wstr t)

fun wstrip s = (String.implode o wstr o String.explode) s

fun cclist [] = ""
  | cclist [h] = h
  | cclist (h::t) = h ^ ", " ^ (cclist t)

fun cclist2 x = String.concat x

fun cchelp long st ((h::t):string list) = 
                             if long<(!cur_maxl) then
                               (cchelp (long+(String.size h))
                                (st ^ ", " ^ h) t)
                             else
                               st
  | cchelp long st [] = st

fun cchelp2 long st ((h::t):string list) = 
                             if long<75 then
                               (cchelp2 (long+(String.size h))
                                (st ^ h) t)
                             else
                               st
  | cchelp2 long st [] = st

fun cchelpr long st ((h::t):string list) = 
                              if long<(!cur_maxl) then
                                (cchelpr (long+(String.size h))
                                 (st ^ ", " ^ h) t)
                              else
                                (h::t)
  | cchelpr long st [] = []

fun cchelpr2 long st ((h::t):string list) = 
                              if long<75 then
                                (cchelpr2 (long+(String.size h))
                                 (st ^ h) t)
                              else
                                (h::t)
  | cchelpr2 long st [] = []

and chks s = (String.implode o chksy o String.explode) s

and chksy ( (#",")::((#" ")::t) ) = t
  | chksy x                       = x 

fun ccpart l = chks(cchelp 0 "" l)

fun ccpart2 l = chks(cchelp2 0 "" l)

fun ccpartr l = cchelpr 0 "" l

fun ccpartr2 l = cchelpr2 0 "" l

fun endfilt ("\\I{}"::(""::
    ("\\end{Items}                   %%[Makellos 0.1]%% basic-spec"::
    (""::t))))
    = ["","\\end{Items}                   %%[Makellos 0.1]%% basic-spec",
       ""] @ (endfilt t)
  | endfilt (h::t) = h :: (endfilt t)
  | endfilt []     = []

fun efilt x = (relink o endfilt o (split_at (#"\n"))) x

(****************************************************************************)


(****************************************************************************)
(*                                                                          *)
(*                              LaTeX generator                             *)
(*                                                                          *)
(****************************************************************************)

(* ids **********************************************************************)



(* get_stropt converts a string option to string, None becomes the       *)
(*  empty string                                                         *)

fun get_stropt (Some s) = s
  | get_stropt None     = ""



(* pl_filt marks places __ in a string by replacing them with ASCII      *)
(*  character #255. This may be insecure if this is a legal input        *)
(*  character in CASL.                                                   *)

fun pl_filt ("_"::("_"::t)) = "\255" :: (pl_filt t)
  | pl_filt (x::t)          = (x::(pl_filt t))
  | pl_filt []              = []



(* beautify tries to set off non-math symbols in applications when no    *)
(*  display annotation was given                                         *)

fun beautify2 [] = []
  | beautify2 (""::(x::t)) = if x<>"" then
                               if ((Char.isAlpha o hd o String.explode) x) then
                                 "" :: (beautify2 (("~"^x)::t))
                               else
                                 "" :: (beautify2 (x::t))
                             else
                               "" :: (beautify2 (x::t))
  | beautify2 (x::(""::t)) = if x<>"" then
                               if ((Char.isAlpha o hd o String.explode) x)
                                  orelse ((hd o String.explode) x)=(#"~") then
                                 (x^"~") :: (beautify2 (""::t))
                               else
                                 x :: (beautify2 (""::t))
                             else
                               x :: (beautify2 (""::t))
  | beautify2 (h::t) = h :: (beautify2 t)                               



(* mk_nice? adds some space between adjacent place symbols               *)

fun mk_nice [] = []
  | mk_nice ("\255"::("\255"::t)) = ["\255","~"] @ (mk_nice ("\255" :: t))
  | mk_nice (h::t) = h :: (mk_nice t)

fun mknice s = (implode o mk_nice o explode) s

fun mknice2 [] = []
  | mknice2 (""::(""::t)) = ["","~"] @ (mknice2 (""::t))
  | mknice2 (h::t) = h :: (mknice2 t)

fun mk_nice3 [] = []
  | mk_nice3 ("_"::("_"::("_"::("_"::t)))) = ["_","_","~"] @
                                             (mk_nice3 (["_","_"] @ t))
  | mk_nice3 ("_"::("_"::(" "::("_"::("_"::t))))) = ["_","_","~"] @
                                                    (mk_nice3 (["_","_"] @ t))
  | mk_nice3 (h::t) = h :: (mk_nice3 t)
  
fun mknice3 s = (implode o mk_nice3 o explode) s



(* do_place inserts empty strings into a list of strings, for string     *)
(*  lists generated by split_at (#"\255"), used in list_conv. This way,  *)
(*  places are converted to empty strings in listconv below.             *)

fun do_place []           = []
  | do_place [h]          = [h]
  | do_place (""::t)      = "" :: (do_place t)
  | do_place (h::(""::t)) = [h,""] @ (do_place t)
  | do_place (h::t)       = [h] @ [""] @ (do_place t)



(* listconv takes a latex string from a display anno, i.e. "__\\div__"   *)
(*  and converts it to a string list suitable for SYMB_APPL, in this     *)
(*  example case ["","\\div",""], places are converted to empty strings  *)

fun listconv s = (do_place o (split_at (#"\255")) o mknice 
                  o implode o pl_filt o ev_filt o explode) s



(* str_tom and strtom are used to convert a TOKEN_OR_MIXFIX to LaTeX     *)

and str_tom (token (t)) = (check t)
  | str_tom (place)     = ""

and strtom l = beautify2 ((map str_tom) l)


(* da_filt filters an ID list for display annotations. It takes the      *)
(*  full string of the id, which is needed for LaTeXing op_decls and     *)
(*  the like. Display annotations for LaTeX are taken "as is".           *)

and da_filt (simple_id (_,tok,_),ann) = (tok,get_stropt ann)
  | da_filt (compound_id ( (_,tok,_),l ),ann) = (tok,get_stropt ann)



(* da_filt2 filters an ID list for display annotations. It takes the     *)
(*  token_or_place list, which is needed for LaTeXing op and pred        *)
(*  applications for mixfix. Display annotations for LaTeX are converted *)
(*  to the string list format needed for SYMB_APPL.                      *)

and da_filt2 (simple_id (tok,_,_),ann) = (tok,listconv(get_stropt ann))
  | da_filt2 (compound_id ( (tok,_,_),l ),ann) = (tok,listconv(get_stropt ann))


(* ev_filt functions are used to filter out LaTeX annotation from a      *)
(*  display annotation.                                                  *)

and ev_filt3 x ("<"::t) = x
  | ev_filt3 x ("%"::t) = x
  | ev_filt3 x (h::t)   = ev_filt3 (x @ [h]) t
  | ev_filt3 x []       = raise LaTeX_DisplayAnnotation

and ev_filt2 ("<"::("l"::("a"::("t"::("e"::("x"::(">"::t))))))) = 
                 (ev_filt3 [] t)
  | ev_filt2 (h::t) = ev_filt2 t
  | ev_filt2 _ = []

and ev_filt ("%"::("!"::t)) = (ev_filt2 t)
  | ev_filt _               = []

fun eval_filt (x,str) = (x,(implode o ev_filt o explode) str)



(* da_list returns a list of (id string * latex string) pairs from the   *)
(*  current global environment                                           *)

fun da_list () = (map eval_filt)
                 ((map da_filt) (get_ids (get_ref cur_genv)
                                         (get_ref cur_spec,None)))
                 handle ERROR => (map eval_filt) ((map da_filt) [])
                 


(* da_list2 returns a list of (token or mixfix list * string list) pairs *)
(*  from the current global environment                                  *)

fun da_list2 () = (map da_filt2) (get_ids (get_ref cur_genv)
                                          (get_ref cur_spec,None))
                  handle ERROR => (map da_filt2) []



(* subst_id is used to convert an ID string to LaTeX, taking into        *)
(*  account any display annotions from the environment                   *)

fun subst_id s [] = (check (mknice3 s))
  | subst_id s ((i,l)::t) = if s=i then
                              if l="" then
                                (check (mknice3 s))
                              else
                                (check2 (mknice3 l))
                            else
                              subst_id s t



(* subst_id2 is used to convert a token_or_mixfix list to a LaTeX       *)
(*  string list for use in SYMB_APPL, taking into account any display   *)
(*  annotations from the environment                                    *)

fun subst_id2 l [] = mknice2 (strtom l)
  | subst_id2 l ((i,s)::t) = if l=i then
                               if s=[""] then
                                 mknice2 (strtom l)
                               else
                                 s
                             else
                               subst_id2 l t



(* subst is the interface to subst_id                                   *)

fun subst s = subst_id s (da_list())



(* subst2 is the interace to subst_id2                                  *)

fun subst2 l = subst_id2 l (da_list2())



fun TP_STRING []     = ""
  | TP_STRING (h::t) = (TP_WORD h) ^ (TP_STRING t)
  
and TP_WORD (place)     = "__"
  | TP_WORD (token (s)) = s

fun L_SIMPLE_ID ((x,a):SIMPLE_ID) = subst x

fun L_TOM ((l,s,n):TOKEN_OR_MIXFIX) = subst (TP_STRING l)

fun L_ID1 (simple_id (x)) = (L_TOM x)
  | L_ID1 (compound_id (x,l)) = (L_TOM x) ^ "[" ^ (L_ID1_LIST l) ^ "]"
  
and L_ID1_LIST [] = ""
  | L_ID1_LIST (h::t) = (L_ID1 h) ^ (L_ID1_LIST t)

fun L_ID ((x,a):ID) = (L_ID1 x)

(****************************************************************************)

(* primitives ***************************************************************)

fun L_SORT ((x):SORT) = (L_ID x)

fun L_OP_NAME ((x):OP_NAME) = (L_ID x)

fun L_PRED_NAME ((x):PRED_NAME) = (L_ID x)

fun L_VAR ((x):VAR) = (L_SIMPLE_ID x)

fun L_SPEC_NAME ((x):SPEC_NAME) = (L_SIMPLE_ID x)

fun L_ARCH_SPEC_NAME ((x):ARCH_SPEC_NAME) = (L_SIMPLE_ID x)

fun L_UNIT_NAME ((x):UNIT_NAME) = (L_SIMPLE_ID x)

fun L_VIEW_NAME ((x):VIEW_NAME) = (L_SIMPLE_ID x)

fun L_ITEM_NAME ((x):ITEM_NAME) = (L_SIMPLE_ID x)

fun L_URL ((x):URL) = (check x)

fun L_PATH_NAME ((x):PATH_NAME) = (check x)

(****************************************************************************)

fun L_LIB_ITEM (spec_defn (n,g,s))    = (L_SPEC_DEFN (n,g,s))
  | L_LIB_ITEM (view_defn (n,g,t,l))  = (L_VIEW_DEFN (n,g,t,l))
  | L_LIB_ITEM (download_items (n,l)) = (L_DOWNLOAD (n,l))
  | L_LIB_ITEM (lib_name (l))         = ( print ("Warning: lib_name encountered"^
                                          ", ignoring.\n") ; "" )
  | L_LIB_ITEM x                      = ( print ("Warning: architectural spec"^
                                                "ification encountered, igno"^
                                                "ring.\n") ; "" )

and L_DOWNLOAD (n,l) = "\\From~ " ^ (L_LIB_NAME n) ^ " ~\\Get\\\\\n"
                       ^ (L_INOM_LIST l)
   
and L_LIB_NAME (lib (x))                       = (L_LIB_ID x)
  | L_LIB_NAME (versioned_lib (x,(version y))) = (L_LIB_ID x) ^ " ~\\Version~ "
                                                 ^ (L_VERLIST y)

and L_LIB_ID (url (x))       = (L_URL x)
  | L_LIB_ID (path_name (x)) = (L_PATH_NAME x)
 
and L_VERLIST []     = ""
  | L_VERLIST [h]    = h
  | L_VERLIST (h::t) = h ^ "." ^ (L_VERLIST t)

and L_INOM_LIST x = ( cur_maxl := 60;
                      (INOM_FORM o (map L_INOM)) x )

and INOM_FORM [] = ""
  | INOM_FORM l  = if (length (ccpartr l))=0 then
                   "\\hspace*{25pt} " ^ (cclist l) ^ "\\\\\n\n"
                 else
                   "\\hspace*{25pt} " ^ (ccpart l) ^ ",\\\\\n" ^
                   (INOM_FORM (ccpartr l)) 

and L_INOM (item_name (i))       = (L_ITEM_NAM i)
  | L_INOM (item_name_map (i,j)) = (L_ITEM_NAM i) ^ " ~\\mapsto~ " ^
                                   (L_ITEM_NAM j)

and L_ITEM_NAM x = "\\SpecName{" ^ (L_ITEM_NAME x) ^ "}"

and getname ((x,n):SIMPLE_ID) = x 
  
and L_SPEC_DEFN (n,g,s) = (cur_spec := (getname n) ;
                           cur_syn := PPL.get_syn (!cur_genv) (!cur_spec);
                           if get_ref(cur_mode) then
                             ()
                           else
                             print ("LaTeXing spec "^(getname n)^"...\n") 
                          ; "\\begin{SpecDefn}{" ^
                          (L_SPEC_NAME n) ^
			  "}" ^
                          (L_GENERICITY (g," =")) ^
			  (L_SPEC s) ^
			  "\\I\\End\n" ^
			  "\\end{SpecDefn}\n\n\n")

and L_VIEW_DEFN (n,g,t,l) = (cur_spec := (getname n);
                             if get_ref(cur_mode) then
                               ()
                             else
                               print ("LaTeXing view "^(getname n)^"...\n")
                            ; "\\begin{SpecDefn}[\\View]{" ^
                            (L_VIEW_NAME n) ^
                            "}" ^
                            (L_GENERICITY (g,":")) ^
                            (L_VIEW_TYPE t) ^
                            "\\I{=}~\n\n" ^
                            (L_VIEW_SYMB l) ^
                            "\\I\\End\n" ^
                            "\\end{SpecDefn}\n")

and L_VIEW_TYPE (view_type (x,y)) = (L_SPEC x) ^ "\\I\\To~\n\n" ^ (L_SPEC y)

and L_VIEW_SYMB (l) = 
  "\\( \\[\n" ^
  form_amp (L_SYMB_MAP_ITEMSL2 l) ^
  "\\] \\)\n\n"

and L_BASIC_SPEC (basic_spec (l)) =
  if length(l)>0 then
    (mkComment "\\begin{Items}" "basic-spec") ^ "\n\n" ^
    (L_BASIC_ITEMS_LIST l) ^
    (mkComment "\\end{Items}" "basic-spec") ^ "\n\n"
  else
    "\\Group\\EndGroup"

and L_BASIC_ITEMS (sig_items (s)) = (L_SIG_ITEMS s)
  | L_BASIC_ITEMS (free_datatype (l)) = if length(l)>1 then
                                          (L_FREE_DATA l) ^ "\\I{}\n\n"
                                        else
                                          (L_FREE_DATA l) ^ "\n"
  | L_BASIC_ITEMS (sort_gen (l)) = (L_GEN_GRP l)
  | L_BASIC_ITEMS (var_items (l)) = if length(l)>1 then
                                      (L_VAR_DEF l) ^ "\\I{}\n\n"
                                    else
                                      (L_VAR_DEF l) ^ "\n"
  | L_BASIC_ITEMS (local_var_axioms (v,l))= (L_LOCAL_AXIOMS v l)
  | L_BASIC_ITEMS (axiom_items (l)) = (L_AXIOMS l) 
  | L_BASIC_ITEMS (anno (a)) = L_PARSE_ANNO a

and L_SIG_ITEMS (sort_items (l)) = if length(l)>1 then
                                     (L_SORT_DEF l) ^ "\\I{}\n\n"
                                   else
                                     (L_SORT_DEF l) ^ "\n"
  | L_SIG_ITEMS (op_items (l)) = if length(l)>1 then
                                   (L_OP_DEF l) ^ "\\I{}\n\n"
                                 else
                                   (L_OP_DEF l) ^ "\n"
  | L_SIG_ITEMS (pred_items (l)) = if length(l)>1 then
                                     (L_PRED_DEF l) ^ "\\I{}\n\n"
                                   else
                                     (L_PRED_DEF l) ^ "\n"
  | L_SIG_ITEMS (datatype_items (l)) = if length(l)>1 then
                                         (L_DATA_DEF l) ^ "\\I{}\n\n"
                                       else
                                         (L_DATA_DEF l) ^ "\n"

and L_BASIC_ITEMS_LIST [] = ""
  | L_BASIC_ITEMS_LIST (h::t) = (L_BASIC_ITEMS h) ^
                                (L_BASIC_ITEMS_LIST t)

and L_SPEC (basic (x)) = (L_BASIC_SPEC x)
  | L_SPEC (extension (l)) = (L_THEN l)
  | L_SPEC (union_spec (l)) = (L_UNION l)
  | L_SPEC (free_spec (s)) = (L_FREE s)
  | L_SPEC (closed_spec (s)) = (L_CLOSED s)
  | L_SPEC (translation (s,r)) = (L_TRANSL s r)
  | L_SPEC (reduction (s,r)) = (L_REDUCT s r)
  | L_SPEC (local_spec (s,t)) = (L_LOCAL s t) 
  | L_SPEC (spec_inst (n,l)) = (L_SPECINST n l)


(* genericity ***************************************************************)

and plen (params (l1)) (imports (l2)) = length(l1) + length(l2)

and L_GENERICITY ((genericity (p,i)),s) = if (plen p i)=0 then
                                        s ^ "\n\n"
                                      else
                                        (L_PARAMS p) ^ (L_IMPORTS i) ^
                                        "\\I{" ^ (wstrip s) ^ "}~\n\n"

and L_PARAMS (params (l)) = if length(l)>0 then
                              "\n" ^
                              (L_PARLIST l)
                            else 
                              ""

and L_PARLIST [] = ""
  | L_PARLIST (h::t) = (L_SPECP h) ^ (L_PARLIST t)

and L_SPECP (spec_inst (x,[])) = "\\I{}\n[\\SpecName{" ^ (L_SPEC_NAME x) ^
                                 "}]\n"
  | L_SPECP (spec_inst (x,[fit_spec (spec_inst (y,[]),[])])) =
                                 "\\I{}\n[\\SpecName{" ^ (L_SPEC_NAME x) ^
                                 "} [\\SpecName{" ^ (L_SPEC_NAME y) ^
                                 "}]]\n\n"
  | L_SPECP (basic (basic_spec (x))) = if (simplesp x) then
                                 "\\I{}\n[" ^
                                 (L_SPECs (x))
                                 ^ "]\n"
                               else
                                 (mkComment "\\begin{Items}" "param spec") ^
                                 "\n" ^ "\\I{[}\n\n" ^
                                 (L_SPEC (basic (basic_spec (x)))) ^  
                                 (mkComment "\\end{Items}" "param spec") ^
                                 "\n" ^ "\\I{]}\n\n"
  | L_SPECP x = (mkComment "\\begin{Items}" "param spec") ^ "\n" ^
                "\\I{[}\n\n" ^
                (L_SPEC x) ^
                (mkComment "\\end{Items}" "param spec") ^ "\n" ^
                "\\I{]}\n\n"

and simplesp ([free_datatype (l)])              = (length(l)=1)
  | simplesp ([var_items (l)])                  = (length(l)=1)
  | simplesp ([axiom_items (l)])                = (length(l)=1)
  | simplesp ([sig_items (sort_items (l))])     = (length(l)=1)
  | simplesp ([sig_items (op_items (l))])       = (length(l)=1)
  | simplesp ([sig_items (pred_items (l))])     = (length(l)=1)
  | simplesp ([sig_items (datatype_items (l))]) = (length(l)=1)
  | simplesp x                                  = false

and L_SPECs ([free_datatype (l)]) =
      "\\Free~\\Type~ \\( \\[" ^ (L_DATA l) ^ "\\] \\)"
  | L_SPECs ([sig_items (datatype_items (l))]) =
      "\\Type~ \\( \\[" ^ (L_DATA l) ^ "\\] \\)"
  | L_SPECs ([axiom_items (l)]) =
      "\\Axiom~ \\( \\[" ^ 
      (PPL.pretty_axioms (!cur_syn) (!cur_genv) (!cur_spec) l)
       ^ "\\] \\)"
  | L_SPECs ([var_items ([(l,s,a)])]) =
      if (length l)=1 then
        "\\Var~ \\( \\[" ^ (L_VAR_LIST2 ([(l,s,a)])) ^ "\\] \\)"
      else
        "\\Vars~ \\( \\[" ^ (L_VAR_LIST2 ([(l,s,a)])) ^ "\\] \\)"
  | L_SPECs ([sig_items (sort_items (l))]) =
      if (singlesort l) then
        "\\Sort~ \\( \\[" ^ (L_L_SORT_ITEM_LIST l) ^ "\\] \\)"
      else
        "\\Sorts~ \\( \\[" ^ (L_L_SORT_ITEM_LIST l) ^ "\\] \\)"
  | L_SPECs ([sig_items (op_items (l))]) =
      "\\Op~ \\( \\[" ^ (L_L_OP_ITEM_LIST l) ^ "\\] \\)"
  | L_SPECs ([sig_items (pred_items (l))]) =
      "\\Pred~ \\( \\[" ^ (L_L_PRED_ITEM_LIST l) ^ "\\] \\)"
  | L_SPECs x = raise Unsupported

and singlesort ([( sort_decl (l), a)]) = (length(l)=1)
  | singlesort x                       = true

and L_IMPORTS (imports (l)) = if length(l)>0 then
                                "\n" ^
                                "\\I\\Given\n" ^
                                (L_IMPLIST l)
                              else
                                ""

and L_IMPLIST [] = ""
  | L_IMPLIST (h::t) = (L_SPEC h) ^ (L_IMPLIST t)

(****************************************************************************)

(* spec inst ****************************************************************)

and L_SPECINST n l = if length(l)=0 then
                       "\\SpecName{" ^ (L_SPEC_NAME n) ^ "}\n\n"
                     else
                       "\\SpecName{" ^ (L_SPEC_NAME n) ^ "} " ^
                       (L_FIT_ARGL l) ^ "\n"
                       
and L_FIT_ARGL [] = ""
  | L_FIT_ARGL (h::t) = (L_FIT_ARG h) ^ (L_FIT_ARGL t)

and L_SPECX (spec_inst (x,[])) = "[\\SpecName{" ^ (L_SPEC_NAME x)
                                ^ "}]\n\n"
  | L_SPECX (spec_inst (x,[fit_spec (spec_inst (y,[]),[])])) =
                                 "[\\SpecName{" ^ (L_SPEC_NAME x) ^
                                 "} [\\SpecName{" ^ (L_SPEC_NAME y) ^
                                 "}]]\n\n"
  | L_SPECX (basic (basic_spec (x))) = if (simplesp x) then
                  "[" ^ (L_SPECs (x)) ^ "]\n\n"
                else
                  (mkComment "\\begin{Items}" "fit spec") ^ "\n" ^
                  "\\I{[}\n\n" ^
                  (L_SPEC (basic(basic_spec (x)))) ^
                  (mkComment "\\end{Items}" "fit spec") ^ "\n" ^
                  "\\I{]}\n\n"
  | L_SPECX x = (mkComment "\\begin{Items}" "fit spec") ^ "\n" ^
                "\\I{[}\n\n" ^
                (L_SPEC x) ^
                (mkComment "\\end{Items}" "fit spec") ^ "\n" ^
                "\\I{]}\n\n"

and L_FIT_ARG (fit_spec (s,l)) = if length(l)=0 then
                                   (L_SPECX s)
                                 else
                                   "\\begin{Items}\n" ^
                                   "\\I{[}\n\n" ^
                                   (L_SPEC s) ^ "\\I\\Fit\n" ^
                                   "\\( \\[" ^
                                   (L_SYMB_MAP_ITEMSL l) ^
                                   "\\] \\)" ^
                                   "\\end{Items}\n" ^
                                   "\\I{]}\n\n" 
  | L_FIT_ARG (fit_view (n,l)) = if length(l)=0 then
                                   "[\\View~ " ^ (L_VIEW_N n) ^ "]\n\n"
                                 else
                                   "\\begin{Items}\n" ^
                                   "\\I{[}\n\n" ^ "\\View~ " ^
                                   (L_VIEW_N n) ^ "\n\n\\I\\Fit\n" ^
                                   (L_FIT_ARGL l) ^
                                   "\\end{Items}\n" ^
                                   "\\I{]}\n\n"

and L_VIEW_N x = "\\SpecName{" ^ (L_VIEW_NAME x) ^ "}"

(****************************************************************************)

(* spec brackets ************************************************************)

and L_SPECB b s = if b then
                    (mkComment "\\begin{Items}" "spec brackets") ^ "\n" ^
                    "\\I\\Group\n\n" ^
                    (L_SPEC s) ^
                    "\\I\\EndGroup~\n" ^
                    (mkComment "\\end{Items}" "spec brackets") ^ "\n\n"
                  else
                    (L_SPEC s)

(****************************************************************************)

(* spec extension ***********************************************************)

and L_THENB (extension (x)) = true
  | L_THENB x               = false

and L_THEN [] = ""
  | L_THEN [h] = (L_SPECB (L_THENB h) h)
  | L_THEN (h::t) = (L_SPECB (L_THENB h) h) ^ "\\I\\Then~\n\n" ^ (L_THEN t)

(****************************************************************************)

(* spec union ***************************************************************)

and L_UNIONB (extension (x))  = true
  | L_UNIONB (union_spec (x)) = true
  | L_UNIONB x                = false

and L_UNION [] = ""
  | L_UNION [h] = (L_SPECB (L_UNIONB h) h)
  | L_UNION (h::t) = (L_SPECB (L_UNIONB h) h) ^ "\\I\\And~\n\n" ^ (L_UNION t)

(****************************************************************************)

(* local spec ***************************************************************)

and L_LOCALB (extension (x))    = true
  | L_LOCALB (union_spec (x))   = true
  | L_LOCALB (local_spec (x,y)) = true
  | L_LOCALB x                  = false

and L_LOCAL s1 s2 = "\\I\\Local~\n\n" ^ (L_SPECB (L_LOCALB s1) s1) ^
                    "\\I\\Within~\n\n" ^ (L_SPECB (L_LOCALB s2) s2)

(****************************************************************************)

(* free spec ****************************************************************)

and L_FREE s =  (mkComment "\\begin{Items}" "free-spec") ^ "\n" ^
                "\\I\\Free\n" ^
                "\\I\\Group\n\n" ^
                (L_SPEC s) ^
                "\\I\\EndGroup\n" ^
                (mkComment "\\end{Items}" "free-spec") ^ "\n\n"

(****************************************************************************)

(* closed spec **************************************************************)

and L_CLOSED s =  (mkComment "\\begin{Items}" "closed-spec") ^ "\n" ^
                  "\\I\\Closed\n" ^
                  "\\I\\Group\n\n" ^
                  (L_SPEC s) ^
                  "\\I\\EndGroup\n" ^
                  (mkComment "\\end{Items}" "closed-spec") ^ "\n\n"

(****************************************************************************)

(* translation spec *********************************************************)

and L_TRANSB (extension (x))     = true
  | L_TRANSB (union_spec (x))    = true
  | L_TRANSB (local_spec (x,y))  = true
  | L_TRANSB (translation (x,y)) = true
  | L_TRANSB (reduction (x,y))   = true
  | L_TRANSB x                   = false

and L_TRANSL s r = (L_SPECB (L_TRANSB s) s) ^ "\\I\\With~\n\n" ^
                   (L_RENAMING r)

(****************************************************************************)

(* reduction spec ***********************************************************)

and L_REDUB (extension (x))     = true
  | L_REDUB (union_spec (x))    = true
  | L_REDUB (local_spec (x,y))  = true
  | L_REDUB (translation (x,y)) = true
  | L_REDUB (reduction (x,y))   = true
  | L_REDUB x                   = false

and L_REDUCT s r = (L_SPECB (L_REDUB s) s) ^ (L_RESTRICTION r)

and L_RESTRICTION (hide (l)) = "\\I\\Hide~\n\n" ^ 
                               "\\( \\[\n" ^
                               (L_SYMB_ITEMSL l) ^
                               "\\] \\)\n\n"
  | L_RESTRICTION (reveal (l)) = "\\I\\Reveal~\n\n" ^
                                 (L_RENAMING (renaming l))

(****************************************************************************)

(* renaming *****************************************************************)

and L_RENAMING (renaming (l)) = 
  "\\( \\[\n" ^
  form_amp (L_SYMB_MAP_ITEMSL l) ^
  "\\] \\)\n\n"

and L_SYMB_ITEMSL [] = ""
  | L_SYMB_ITEMSL [h] = (L_SYMB_ITEMS h) ^ "\\\\\n"
  | L_SYMB_ITEMSL (h::t) = (L_SYMB_ITEMS h) ^ ",\\\\\n" ^
                           (L_SYMB_ITEMSL t)
                           
and L_SYMB_ITEMS (symb_items (k,l)) = "   " ^ (L_SYMB_KIND2 k l) ^ "& " ^
                                      (L_SYMB_LIST l)
                                      
and L_SYMB_KIND2 implicitk l = ""
  | L_SYMB_KIND2 sortsk    l = if length(l)=1 then "\\Sort~ " else "\\Sorts~ "
  | L_SYMB_KIND2 opsk      l = if length(l)=1 then "\\Op~ " else "\\Ops~ "
  | L_SYMB_KIND2 predsk    l = if length(l)=1 then "\\Pred~ " else "\\Preds~ "

and L_SYMB_LIST [] = ""
  | L_SYMB_LIST [h] = (L_SYMB h)
  | L_SYMB_LIST (h::t) = (L_SYMB h) ^ ", " ^ (L_SYMB_LIST t)

and L_SYMB_MAP_ITEMSL [] = ""
  | L_SYMB_MAP_ITEMSL [h] = (L_SYMB_MAP_ITEMS h) ^ "\\\\\n"
  | L_SYMB_MAP_ITEMSL (h::t) = (L_SYMB_MAP_ITEMS h) ^ ",\\\\\n"
                               ^ (L_SYMB_MAP_ITEMSL t)

and L_SYMB_MAP_ITEMSL2 [] = ""
  | L_SYMB_MAP_ITEMSL2 [h] = (L_SYMB_MAP_ITEMS2 h) ^ "\\\\\n"
  | L_SYMB_MAP_ITEMSL2 (h::t) = (L_SYMB_MAP_ITEMS2 h) ^ ",\\\\\n"
                                ^ (L_SYMB_MAP_ITEMSL2 t)

(****************************************************************************)

(* symb-map-items ***********************************************************)

and L_SYMB_MAP_ITEMS (symb_map_items (k,l)) = 
  if length(l)<6 then
    "   " ^ (L_SYMB_KIND k l) ^
    "& " ^ (L_SYMB_OR_MAPL l)
  else
    "   " ^ (L_SYMB_KIND k l) ^
    "& " ^ (L_SYMBS l)

and L_SYMB_MAP_ITEMS2 (symb_map_items (k,l)) =
  "   " ^ (L_SYMB_KIND k l) ^ "& " ^ (L_SYMB_OR_MAPL l)

and L_SYMBS l = ( cur_maxl := 30;
                  (SYMB_FORM o (map L_SYMB_OR_MAP2)) l )

and SYMB_FORM [] = ""
  | SYMB_FORM l  = if (length (ccpartr l))=0 then
                     (cclist l)
                   else
                     (ccpart l) ^ ",\\\\\n   & " ^
                     (SYMB_FORM (ccpartr l)) 

and L_SYMB_KIND implicitk l = ""
  | L_SYMB_KIND sortsk    l = if length(l)=1 then "\\Sort~ " else "\\Sorts~ "
  | L_SYMB_KIND opsk      l = if length(l)=1 then "\\Op~ " else "\\Ops~ "
  | L_SYMB_KIND predsk    l = if length(l)=1 then "\\Pred~ " else "\\Preds~ "

and L_SYMB_OR_MAPL [] = ""
  | L_SYMB_OR_MAPL [h] = (L_SYMB_OR_MAP h)
  | L_SYMB_OR_MAPL (h::t) = (L_SYMB_OR_MAP h) ^ ",\\\\\n   & " ^
                            (L_SYMB_OR_MAPL t)

and L_SYMB_OR_MAP (symb (s)) = (L_SYMB s)
  | L_SYMB_OR_MAP (symb_or_map (s)) = (L_SYMB_MAP s)

and L_SYMB_OR_MAP2 (symb (s)) = (L_SYMB s)
  | L_SYMB_OR_MAP2 (symb_or_map (s)) = (L_SYMB_MAP2 s)
  
and L_SYMB_MAP (symb_map (s,t)) = (L_SYMB s) ^ " & ~\\mapsto~ " ^ (L_SYMB t)

and L_SYMB_MAP2 (symb_map (s,t)) = (L_SYMB s) ^ " ~\\mapsto~ " ^ (L_SYMB t)

and L_SYMB (symb_id (i)) = (L_ID i)
  | L_SYMB (qual_id (i,t)) = (L_ID i)

(****************************************************************************)

(* var/vars *****************************************************************)

and V_NAME (l,s,x) = (V_NAME_HELP l)
and V_NAME_HELP [] = ""
  | V_NAME_HELP [h] = (L_VAR h)
  | V_NAME_HELP (h::t) = (L_VAR h) ^ ", " ^ (V_NAME_HELP t)
and V_SORT (l,s,x) = (L_SORT s)

and L_VAR_LIST [] = ""
  | L_VAR_LIST [h] = "   " ^ (V_NAME h) ^ " & ~:~ " ^ (V_SORT h) ^ "\\\\\n"
  | L_VAR_LIST (h::t) = "   " ^ (V_NAME h) ^ " & ~:~ " ^ (V_SORT h) ^
                        ";\\\\\n" ^ (L_VAR_LIST t)

and L_VAR_LIST2 [] = ""
  | L_VAR_LIST2 [h] = "   " ^ (V_NAME h) ^ " : " ^ (V_SORT h) ^ "\\\\\n"
  | L_VAR_LIST2 (h::t) = "   " ^ (V_NAME h) ^ " : " ^ (V_SORT h) ^
                        ";\\\\\n" ^ (L_VAR_LIST t)

and V_NAMES (l,s,x) = l

and L_VAR_DEF x = if length(x)=1 then
                    if length(V_NAMES (hd x))=1 then
                      "\\I\\Var\n" ^
                      "\\( \\[\n" ^
                      (L_VAR_LIST2 x) ^
                      "\\] \\)\n"
                    else
                      "\\I\\Vars\n" ^
                      "\\( \\[\n" ^
                      (L_VAR_LIST2 x) ^
                      "\\] \\)\n"
                  else
                    "\\I\\Vars\n" ^
                    "\\( \\[\n" ^
                    form_amp (L_VAR_LIST x) ^
                    "\\] \\)\n"
                    
(****************************************************************************)

(* sort/sorts ***************************************************************)
	
and S_ONE ((sort_decl (i),l):L_SORT_ITEM) =
    if length(i)=1 then true else false
  | S_ONE ((subsort_decl (i,s),l):L_SORT_ITEM) =
    if length(i)=1 then true else false
  | S_ONE ((subsort_defn (s,v,t,f),l):L_SORT_ITEM) =
    true
  | S_ONE ((iso_decl (i),l):L_SORT_ITEM) =
    if length(i)=1 then true else false

and S_SINGLE x = if length(x)>1 then false else (S_ONE (hd x))	
	
and L_SORT_DEF x = if (S_SINGLE x) then
                     "\\I\\Sort\n" ^
                     "\\( \\[ \n" ^
                     (L_L_SORT_ITEM_LIST x) ^
                     "\\] \\)\n"
                   else
                     "\\I\\Sorts\n" ^
                     "\\( \\[\n" ^
                     (L_L_SORT_ITEM_LIST x) ^
                     "\\] \\)\n"

and L_L_SORT_ITEM_LIST [] = ""
  | L_L_SORT_ITEM_LIST (h::t) = (L_L_SORT_ITEM h) ^ "\n" ^
                                (L_L_SORT_ITEM_LIST t)
                                
and L_L_SORT_ITEM ((i,l):L_SORT_ITEM) = (L_SORT_ITEM i)

and L_SORT_DECLS [] = ""
  | L_SORT_DECLS [h] = (L_SORT h) ^ ";\\\\"
  | L_SORT_DECLS (h::t) = (L_SORT h) ^ ", " ^ (L_SORT_DECLS t)

and L_ISO_DECLS [] = ""
  | L_ISO_DECLS [h] = (L_SORT h) ^ ";\\\\"
  | L_ISO_DECLS (h::t) = (L_SORT h) ^ " = " ^ (L_ISO_DECLS t)

and L_SUB_HELP [] = ""
  | L_SUB_HELP [h] = (L_SORT h)
  | L_SUB_HELP (h::t) = (L_SORT h) ^ ", " ^ (L_SUB_HELP t)
  
and L_SUB_DECL l s = (L_SUB_HELP l) ^ " < " ^ (L_SORT s) ^ ";\\\\"

and L_SUB_DEFN s v t f = (L_SORT s) ^ " = \\{ " ^ (L_VAR v) ^
                         " : " ^ (L_SORT t) ^ " \\. \\[" ^
(*                         (laststrp (L_FFORM f)) ^  *)
    (PPL.pretty_subsort_defn (!cur_syn) (!cur_genv) (!cur_spec) f) ^
                         "\\}; \\]\\\\"

and laststrp s = (implode o last_strp o explode) s

and last_strp [] = []
  | last_strp ["\\","\\"] = []
  | last_strp (h::t)      = h :: (last_strp t)

and L_SORT_ITEM (sort_decl (l)) = "   " ^ (L_SORT_DECLS l)
  | L_SORT_ITEM (subsort_decl (l,s)) = "   " ^ (L_SUB_DECL l s)
  | L_SORT_ITEM (subsort_defn (s,v,t,f)) = "   " ^ (L_SUB_DEFN s v t f)
  | L_SORT_ITEM (iso_decl (l)) = "   " ^ (L_ISO_DECLS l)
	
(****************************************************************************)	

(* op/ops *******************************************************************)

and O_ONE ((op_decl (l,t,a),lb,an):L_OP_ITEM) = if length(l)=1 then true
                                                else false
  | O_ONE ((op_defn (n,h,t),lb,an):L_OP_ITEM) = true

and O_SINGLE x = if length(x)=1 then
                   if (O_ONE (hd x)) then true else false
                 else false

and L_L_OP_ITEM_LIST [] = ""
  | L_L_OP_ITEM_LIST (h::t) = (L_L_OP_ITEM h) ^ (L_L_OP_ITEM_LIST t)
  
and L_L_OP_ITEM ((i,l,a):L_OP_ITEM) = (L_OP_ITEM i)

and L_OP_ITEM (op_decl (l,t,a)) = (L_OPS (l,t,a))
  | L_OP_ITEM (op_defn (n,h,t)) = (L_OPZ (n,h,t))

(*
and L_OPN [] = ""
  | L_OPN [h] = "   " ^ (L_OP_NAME h) ^ " & ~:~ & "
  | L_OPN (h::t) = "   " ^ (L_OP_NAME h) ^ " ,~ & \\\\\n" ^ (L_OPN t)
*)

and L_OPN l = ( cur_maxl := 30;
                (OP_FORM o (map L_OP_NAME)) l )

and OP_FORM [] = ""
  | OP_FORM l  = if (length (ccpartr l))=0 then
                   "   " ^ (cclist l) ^ " & ~:~ & "
                 else
                   "   " ^ (ccpart l) ^ " ,~ & \\\\\n" ^
                   (OP_FORM (ccpartr l)) 

and L_OP_ATTR (associative) = "\\Assoc"
  | L_OP_ATTR (commutative) = "\\Comm"
  | L_OP_ATTR (idempotent)  = "\\Idem"
  | L_OP_ATTR (unit_op_attr (t)) = "\\Unit~ \\[ " ^
                  (PPL.pretty_unit (!cur_syn) (!cur_genv) (!cur_spec) t)
                                   ^ " \\]"

and L_OP_ATTR_LIST [] = ""
  | L_OP_ATTR_LIST [h] = (L_OP_ATTR h)
  | L_OP_ATTR_LIST (h::t) = (L_OP_ATTR h) ^ ", " ^ (L_OP_ATTR_LIST t)

and L_OPA x = if length(x)=0 then
                ";\\\\\n"
              else
                ",\\\\\n" ^
                "   &     & " ^ (L_OP_ATTR_LIST x) ^ ";\\\\\n"

and L_SORTSS [] = ""
  | L_SORTSS [h] = (L_SORT h)
  | L_SORTSS (h::t) = (L_SORT h) ^ " \\* " ^ (L_SORTSS t)

and L_SORTS (sorts (x)) = (L_SORTSS x)

and slength (sorts (x)) = length x

and L_OPT (total_op_type (l,s)) = if slength(l)>0 then
                                    (L_SORTS l) ^
                                    " \\tfun " ^
                                    (L_SORT s)
                                  else
                                    (L_SORT s)
  | L_OPT (partial_op_type (l,s)) = if slength(l)>0 then
                                      (L_SORTS l) ^
                                      " \\pfun " ^
                                      (L_SORT s)
                                    else
                                      (L_SORT s)

and L_OPS (l,t,a) = (L_OPN l) ^ (L_OPT t) ^ (L_OPA a)

and L_OPZ (n,h,t) = "   " ^ (L_OP_NAME n) ^ (L_OP_HEAD h) ^ " = \\[ " ^
                   (PPL.pretty_op_defn (!cur_syn) (!cur_genv) (!cur_spec) t)
                    ^ "\\] ;\\\\\n"
                    
and L_OP_HEAD (total_op_head (l,s)) = if length(l)=0 then
                                        " & ~:~ & " ^ (L_SORT s)
                                      else
                                        " ( " ^ (L_ARG_DECLS l) ^
                                        " ) & ~:~ & " ^
                                        (L_SORT s)
  | L_OP_HEAD (partial_op_head (l,s)) = if length(l)=0 then
                                          " & ~:?~ & " ^ (L_SORT s)
                                        else
                                          " ( " ^ (L_ARG_DECLS l) ^
                                          " ) & ~:?~ & "
                                          ^ (L_SORT s)
                                        
and L_ARG_DECLS [] = ""
  | L_ARG_DECLS [h] = (L_ARG_DECL h)
  | L_ARG_DECLS (h::t) = (L_ARG_DECL h) ^ "; " ^ (L_ARG_DECLS t)
  
and L_ARG_DECL (arg_decl (l,s,a)) = (L_V_LIST l) ^ " : " ^ (L_SORT s)

and L_V_LIST [] = ""
  | L_V_LIST [h] = (L_VAR h)
  | L_V_LIST (h::t) = (L_VAR h) ^ ", " ^ (L_V_LIST t)
  
and L_OP_DEF x = if (O_SINGLE x) then
                   "\\I\\Op\n" ^
                   "\\( \\[\n" ^
                   form_amp (L_L_OP_ITEM_LIST x) ^
                   "\\] \\)\n"
                 else
                   "\\I\\Ops\n" ^
                   "\\( \\[\n" ^
                   form_amp (L_L_OP_ITEM_LIST x) ^
                   "\\] \\)\n"

(****************************************************************************)

(* pred/preds ***************************************************************)

and P_ONE ((pred_decl (l,t),lb,an):L_PRED_ITEM) = if length(l)=1 then true
                                                  else false
  | P_ONE ((pred_defn (n,h,f),lb,an):L_PRED_ITEM) = true

and P_SINGLE x = if length(x)=1 then
                   if (P_ONE (hd x)) then true else false
                 else false

and L_L_PRED_ITEM_LIST [] = ""
  | L_L_PRED_ITEM_LIST (h::t) = (L_L_PRED_ITEM h) ^ (L_L_PRED_ITEM_LIST t)
  
and L_L_PRED_ITEM ((i,l,a):L_PRED_ITEM) = (L_PRED_ITEM i)

and L_PREDN l = ( cur_maxl := 30;
                  (PRED_FORM o (map L_PRED_NAME)) l )

and PRED_FORM [] = ""
  | PRED_FORM l  = if (length (ccpartr l))=0 then
                     "   " ^ (cclist l) ^ " & ~:~ "
                   else
                     "   " ^ (ccpart l) ^ " ,~ & \\\\\n" ^
                     (PRED_FORM (ccpartr l)) 

and L_PRED_ITEM (pred_decl (l,t)) = (* "   " ^ (L_PRED_NAME_LIST l) ^ " & ~:~ " ^
                                    (L_PRED_TYPE t) ^ ";\\\\\n" *)
                                    (L_PREDN l) ^ (L_PRED_TYPE t) ^ ";\\\\\n"
  | L_PRED_ITEM (pred_defn (n,h,f)) = "   " ^ (L_PRED_NAME n) ^
                                      (L_PRED_HEAD h) ^ " & ~\\iff~ \\[ " ^
         (PPL.pretty_pred_defn (!cur_syn) (!cur_genv) (!cur_spec) f)
                                      ^ "; \\]\\\\\n"

and L_PRED_HEAD (pred_head (x)) = if length(x)=0 then
                                    ""
                                  else
                                    "( " ^ (L_ARG_DECLS x) ^ " )"

(*                                    
and L_PRED_NAME_LIST [] = ""
  | L_PRED_NAME_LIST [h] = (L_PRED_NAME h)
  | L_PRED_NAME_LIST (h::t) = (L_PRED_NAME h) ^ " ~,~ & \\\\\n   " ^
                              (L_PRED_NAME_LIST t)
*)

and lsorts (sorts (l)) = (length l)
                              
and L_PRED_TYPE (pred_type (s)) =
  if lsorts(s)=0 then
    "()"
  else
    (L_SORTS s)
                                    
and L_PRED_DEF x = if (P_SINGLE x) then
                     "\\I\\Pred\n" ^
                     "\\( \\[\n" ^
                     form_amp (L_L_PRED_ITEM_LIST x) ^
                     "\\] \\)\n"
                   else
                     "\\I\\Preds\n" ^
                     "\\( \\[\n" ^
                     form_amp (L_L_PRED_ITEM_LIST x) ^
                     "\\] \\)\n"
                     
(****************************************************************************)

(* free type/types **********************************************************)

and L_FREE_DATA x = if length(x)=1 then
                      "\\I\\Free \\Type~\n" ^
                      "\\( \\[\n" ^
                      form_amp (L_DATA x) ^
                      "\\] \\)\n"
                    else
                      "\\I\\Free \\Types~\n" ^
                      "\\( \\[\n" ^
                      form_amp (L_DATA x) ^
                      "\\] \\)\n"

(****************************************************************************)

(* generated group **********************************************************)

and L_GEN_GRP x = "\\I\\Generated\n" ^
                  "\\I{~~~\\Group}\n" ^
                  "\\begin{Items}\n\n" ^
                  (L_SI_LIST x) ^
                  "\\end{Items}\n" ^
                  "\\EndGroup;\n\n"

and L_SI_LIST [] = ""
  | L_SI_LIST (h::t) = (L_SIG_ITEMS h) ^ (L_SI_LIST t)

(****************************************************************************)

(* local axiom/axioms *******************************************************)

and L_LOCAL_AXIOMS v l = (L_VAR_DEF v) ^
           (PPL.pretty_local_axioms (!cur_syn) (!cur_genv) (!cur_spec) l) ^ "\\I{}\n\n"

(*
and L_LAXLIST [] = ""
  | L_LAXLIST (h::t) = "\\I\\. \\(\\[ " ^ (L_AXFORM h) ^ " \\]\\)\n" ^
                       (L_LAXLIST t)

and L_AXFORM x = form_amp (l_axform (LL_L_FORMULA x))

and L_FFORM  x = form_amp (l_axform (LL_FORMULA x))

and l_axform [] = ""
  | l_axform l  = if (length (ccpartr2 l))=0 then
                   "& " ^ (cclist2 l) ^ " \\\\"
                 else
                   "& " ^ (ccpart2 l) ^ " \\\\\n" ^
                   (l_axform (ccpartr2 l)) 
*)

(****************************************************************************)

(* axiom/axioms *************************************************************)

and L_AXIOMS l = if length(l)=1 then
                   "\\I\\Axiom\n" ^
                   "\\( \\[\n" ^
                   (* (L_AXLIST l) ^ *)
                   (PPL.pretty_axioms (!cur_syn) (!cur_genv) (!cur_spec) l) ^
                   "\\] \\)\n\n"
                 else
                   "\\I\\Axioms\n" ^
                   "\\( \\[\n" ^
                   (* (L_AXLIST l) ^ *)
                   (PPL.pretty_axioms (!cur_syn) (!cur_genv) (!cur_spec) l) ^
                   "\\] \\)\n" ^
                   "\\I{}\n\n"  

(*
and L_AXLIST [] = ""
  | L_AXLIST (h::t) = "   " ^ (L_AXFORM h) ^ ";\\\\\n" ^ (L_AXLIST t)
*)

(****************************************************************************)

(* formula ******************************************************************)

(*
and strip_255 [] = []
  | strip_255 ("\255"::t) = strip_255 t
  | strip_255 (h::t)      = h :: (strip_255 t)

and strip255 s = (implode o strip_255 o explode) s

and L_L_FORMULA (f,l) = strip255 (L_FORMU f)

and L_FORMUL f = strip255 (L_FORMU f)

and L_FORMU (atom (x)) = (L_ATOM x)
  | L_FORMU (conjunction (l)) = (L_CONJ l) 
  | L_FORMU (disjunction (l)) = (L_DISJ l)
  | L_FORMU (implication (a,b)) = (L_IMPL a b)
  | L_FORMU (equivalence (a,b)) = (L_EQUI a b)
  | L_FORMU (quantification (q,l,f)) = (L_QUAN q l f)
  | L_FORMU (negation (x)) = (L_NEGA x)
  | L_FORMU (unparsed_formula (s)) = "\\verb'" ^ (check s) ^ "'"
  | L_FORMU (sort_gen_ax (l,m)) =
         ( print ("Error: sort_gen_ax encountered in LaTeX generation.\n" ^
                  "Please send a bug report.\n" );
           raise Unsupported )

and LL_L_FORMULA (f,l) = (LL_FORMULA f)
and LL_FORMULA f = split_at (#"\255") (L_FORMU f)

and L_BRACK co x = if co then
                     "( " ^ (L_FORMU x) ^ " )"
                   else
                     (L_FORMU x)


and L_CONJ_B (negation (x)) = false
  | L_CONJ_B (atom (x))     = false
  | L_CONJ_B x              = true 

and L_CONJ_S x = (L_BRACK (L_CONJ_B x) x)

and L_CONJ [] = ""
  | L_CONJ [h] = (L_CONJ_S h)
  | L_CONJ (h::t) = (L_CONJ_S h) ^ " \255~\\land~\255 " ^ (L_CONJ t)

and L_DISJ_B (negation (x)) = false
  | L_DISJ_B (atom (x))     = false
  | L_DISJ_B x              = true

and L_DISJ_S x = (L_BRACK (L_DISJ_B x) x)
                   
and L_DISJ [] = ""
  | L_DISJ [h] = (L_DISJ_S h)
  | L_DISJ (h::t) = (L_DISJ_S h) ^ " \255~\\lor~\255 " ^ (L_DISJ t)
  

and L_IMPL_B (negation (x)) = false
  | L_IMPL_B (disjunction (x)) = false
  | L_IMPL_B (conjunction (x)) = false
  | L_IMPL_B (implication (x)) = false
  | L_IMPL_B (atom (x))        = false
  | L_IMPL_B x                 = true

and L_IMPL_C (negation (x)) = false
  | L_IMPL_C (disjunction (x)) = false
  | L_IMPL_C (conjunction (x)) = false
  | L_IMPL_C (implication (x)) = true
  | L_IMPL_C (atom (x))        = false
  | L_IMPL_C x                 = true

and L_IMPL_L x = (L_BRACK (L_IMPL_C x) x)

and L_IMPL_R x = (L_BRACK (L_IMPL_B x) x)

and L_IMPL a b = (L_IMPL_L a) ^ " \255~\\imp~\255 " ^ (L_IMPL_R b)


and L_EQUI_B (negation (x)) = false
  | L_EQUI_B (conjunction (x)) = false
  | L_EQUI_B (disjunction (x)) = false
  | L_EQUI_B (atom (x))        = false
  | L_EQUI_B x                 = true
  
and L_EQUI_S x = (L_BRACK (L_EQUI_B x) x)

and L_EQUI a b = (L_EQUI_S a) ^ " \255~\\iff~\255 " ^ (L_EQUI_S b)


and L_QUAN_B (negation (x)) = false
  | L_QUAN_B (conjunction (x)) = false
  | L_QUAN_B (disjunction (x)) = false
  | L_QUAN_B (implication (x,y)) = false
  | L_QUAN_B (equivalence (x,y)) = false
  | L_QUAN_B (atom (x))          = false
  | L_QUAN_B x                   = true

and L_QUAN_S x = (L_BRACK (L_QUAN_B x) x) 

and L_QUAN q l f = (L_QUANTIFIER q) ^ (L_VAR_DECLS l) ^ " \255~\\.~ " ^
                   (L_QUAN_S f)

and L_QUANTIFIER (forall) = "\\forall~ "
  | L_QUANTIFIER (exists) = "\\exists~ "
  | L_QUANTIFIER (exists_uniquely) = "\\exists!~ "

and L_VAR_DECLS [] = ""
  | L_VAR_DECLS [h] = (L_VAR_DECL h)
  | L_VAR_DECLS (h::t) = (L_VAR_DECL h) ^ "; " ^ (L_VAR_DECLS t)
  
and L_VAR_DECL (l,s,a) = (L_V_LIST l) ^ " : " ^ (L_SORT s)


and L_NEGA_B (atom (x)) = false
  | L_NEGA_B x          = true

and L_NEGA x = "\255\\neg~ " ^ (L_BRACK (L_NEGA_B x) x)

(****************************************************************************)

(* atoms ********************************************************************)

and L_ATOM (ttrue) = "\255\\TRUE\255"
  | L_ATOM (ffalse) = "\255\\FALSE\255"
  | L_ATOM (predication (s,t)) = (LL_PRED_APPL s t)
  | L_ATOM (definedness (t)) = "\255\\DEF~ " ^ (LL_TERM t)
  | L_ATOM (existl_equation (a,b)) = (LL_TERM a) ^ " \255~\\eeq~\255 " ^ 
                                     (LL_TERM b)
  | L_ATOM (strong_equation (a,b)) = (LL_TERM a) ^ " \255~=~\255 " ^
                                     (LL_TERM b)
  | L_ATOM (membership (t,s)) = (LL_TERM t) ^ " \255~\\in~\255 " ^ (L_SORT s)

(****************************************************************************)

(* predicate application ****************************************************)

and ID_LIST (id1,a) = (ID1_LIST id1)

and ID1_LIST (simple_id (t)) = (TOM_LIST t)
  | ID1_LIST (compound_id (t,i)) = (TOM_LIST t) @ ["["] @ (ID1L_LIST i) @
                                   ["]"] 

and ID1L_LIST [] = []
  | ID1L_LIST (h::t) = (ID1_LIST h) @ (ID1L_LIST t)

and TOM_LIST (l,s,a) = (* (TOP_LIST l) *) subst2 l

(*
and TOP_LIST [] = []
  | TOP_LIST (h::t) = (TOP_STR h) :: (TOP_LIST t)
  
and TOP_STR (token (s)) = (check (toks s))
  | TOP_STR (place)     = ""
*)

and PRED_APPL (pred_symb (n,t)) (terms (l)) =
  (SYMB_APPL (ID_LIST n) l)

and LL_PRED_APPL (pred_symb (n,t)) (terms (l)) =
  (LL_SYMB_APPL (ID_LIST n) l)

and SYMB_APPL [] []          = ""
  | SYMB_APPL [] l           = "( " ^ (TERM_LIST l) ^ " )"
  | SYMB_APPL (""::x) (t::l) = " " ^ (L_TERMC t) ^ " " ^ (SYMB_APPL x l)
  | SYMB_APPL (h::t) l       = h ^ (SYMB_APPL t l) 

and LL_SYMB_APPL [] []          = ""
  | LL_SYMB_APPL [] l           = "\255( " ^ (LL_TERM_LIST l) ^ " )\255"
  | LL_SYMB_APPL (""::x) (t::l) = " \255" ^ (LL_TERMC t) ^ " \255" ^
                                  (LL_SYMB_APPL x l)
  | LL_SYMB_APPL (h::t) l       = h ^ "\255" ^ (LL_SYMB_APPL t l)

and TERM_LIST [] = ""
  | TERM_LIST [h] = (L_TERM h)
  | TERM_LIST (h::t) = (L_TERM h) ^ ", " ^ (TERM_LIST t)

and LL_TERM_LIST [] = ""
  | LL_TERM_LIST [h] = (LL_TERM h)
  | LL_TERM_LIST (h::t) = (LL_TERM h) ^ "\255,\255 " ^ (LL_TERM_LIST t)

(****************************************************************************)

(* terms ********************************************************************)

and L_TERM (var_or_const (x)) = (L_VAR x)
  | L_TERM (qual_var (x,s)) = "( \\Var~ " ^ (L_VAR x) ^ " ~:~ " ^ (L_SORT s)
                              ^ " )"
  | L_TERM (application (s,t)) = (OP_APPL s t)
  | L_TERM (sorted_term (t,s)) = (L_TERM t) ^ " ~:~ " ^ (L_SORT s)
  | L_TERM (cast (t,s)) = (L_TERM t) ^ " ~\\AS~ " ^ (L_SORT s)
  | L_TERM (conditional (t1,f,t2)) = (L_TERM t1) ^ " ~\\WHEN~ " ^
                                     (L_FORMUL f) ^ " ~\\ELSE~ " ^
                                     (L_TERM t2)
  | L_TERM (unparsed_term (s)) = (check s)

and LL_TERM (var_or_const (x)) = (L_VAR x)
  | LL_TERM (qual_var (x,s)) = "\255( \\Var~\255 " ^ (L_VAR x) ^ " \255~:~\255 " ^
                               (L_SORT s) ^ " )\255"
  | LL_TERM (application (s,t)) = (LL_OP_APPL s t)
  | LL_TERM (sorted_term (t,s)) = (LL_TERM t) ^ " \255~:~\255 " ^ (L_SORT s)
  | LL_TERM (cast (t,s)) = (LL_TERM t) ^ " \255~\\AS~\255 " ^ (L_SORT s)
  | LL_TERM (conditional (t1,f,t2)) = (LL_TERM t1) ^ " \255~\\WHEN~\255 " ^
                                     (L_FORMU f) ^ " \255~\\ELSE~\255 " ^
                                     (LL_TERM t2)
  | LL_TERM (unparsed_term (s)) = (check s)
  
and OP_APPL (op_symb (n,t)) (terms (l)) =
  (SYMB_APPL (ID_LIST n) l)

and LL_OP_APPL (op_symb (n,t)) (terms (l)) =
  (LL_SYMB_APPL (ID_LIST n) l)

and IS_CONST n l = (length(n)=1) andalso (length(l)=0)

and L_TERMC (var_or_const (x)) = (L_VAR x)
  | L_TERMC (qual_var (x,s)) = "( \\Var~ " ^ (L_VAR x) ^ " ~:~ " ^ (L_SORT s)
                              ^ " )"
  | L_TERMC (application (s,t)) = (OP_APPLC s t)
  | L_TERMC (sorted_term (t,s)) = "( " ^ (L_TERM t) ^ " ~:~ " ^ (L_SORT s)
                                  ^ " )"
  | L_TERMC (cast (t,s)) = "( " ^ (L_TERM t) ^ " ~\\AS~ " ^ (L_SORT s) ^ " )"
  | L_TERMC (conditional (t1,f,t2)) = "( " ^ (L_TERM t1) ^ " ~\\WHEN~ " ^
                                     (L_FORMUL f) ^ " ~\\ELSE~ " ^
                                     (L_TERM t2) ^ " )"
  | L_TERMC (unparsed_term (s)) = (check s)

and LL_TERMC (var_or_const (x)) = (L_VAR x)
  | LL_TERMC (qual_var (x,s)) = "\255( \\Var~ " ^ (L_VAR x) ^ " \255~:~\255 "
                                ^ (L_SORT s) ^ " )\255"
  | LL_TERMC (application (s,t)) = (LL_OP_APPLC s t)
  | LL_TERMC (sorted_term (t,s)) = "\255( " ^ (L_TERM t) ^ " \255~:~\255 " ^ (L_SORT s)
                                  ^ " )\255"
  | LL_TERMC (cast (t,s)) = "\255( " ^ (L_TERM t) ^ " \255~\\AS~\255 " ^ (L_SORT s) ^ " )\255"
  | LL_TERMC (conditional (t1,f,t2)) = "\255( " ^ (LL_TERM t1) ^ " \255~\\WHEN~\255 " ^
                                     (L_FORMU f) ^ " \255~\\ELSE~\255 " ^
                                     (LL_TERM t2) ^ " )\255"
  | LL_TERMC (unparsed_term (s)) = (check s)

and OP_APPLC (op_symb (n,t)) (terms (l)) =
  if (IS_CONST (ID_LIST n) l) then
    (SYMB_APPL (ID_LIST n) l)
  else
    "( " ^ (SYMB_APPL (ID_LIST n) l) ^ " )"

and LL_OP_APPLC (op_symb (n,t)) (terms (l)) =
  if (IS_CONST (ID_LIST n) l) then
    (LL_SYMB_APPL (ID_LIST n) l)
  else
    "\255( " ^ (LL_SYMB_APPL (ID_LIST n) l) ^ " )\255"
*)

(****************************************************************************)

(* type/types ***************************************************************)

and L_DATA_DEF x = if length(x)=1 then
                     "\\I\\Type\n" ^
                     "\\( \\[\n" ^
                     form_amp (L_DATA x) ^
                     "\\] \\)\n"
                   else
                     "\\I\\Types\n" ^
                     "\\( \\[\n" ^
                     form_amp (L_DATA x) ^
                     "\\] \\)\n"

and L_DATA [] = ""
  | L_DATA (h::t) = "   " ^ (L_DATATYPE_DECL h) ^ "\\\\\n" ^
                    (L_DATA t)
                    
and L_DATATYPE_DECL (datatype_decl (s,l,a)) = (L_SORT s) ^ " & ~::=~ " ^
                                              (L_ALT_LIST l) ^ ";"
                                              
and L_ALT_LIST [] = ""
  | L_ALT_LIST [h] = (L_ALTERNATIVE h)
  | L_ALT_LIST (h::t) = (L_ALTERNATIVE h) ^ " ~\\|~ " ^ (L_ALT_LIST t)

and L_ALTERNATIVE (total_construct (n,l,a)) =
      (L_OP_NAME n) ^ (L_COM_LIST l)
  | L_ALTERNATIVE (partial_construct (n,l,a)) =
      (L_OP_NAME n) ^ (L_COM_LIST l) ^ "?"
  | L_ALTERNATIVE (subsort (l,a)) =
      if length(l)=1 then
        "\\Sort~ " ^ (L_S_LIST l)
      else
        "\\Sorts~ " ^ (L_S_LIST l)
        
and L_S_LIST [] = ""
  | L_S_LIST [h] = (L_SORT h)
  | L_S_LIST (h::t) = (L_SORT h) ^ ", " ^ (L_S_LIST t)

and L_COM_LIST [] = ""
  | L_COM_LIST  x = "(" ^ (L_COMP_LIST x) ^ ")"
  
and L_COMP_LIST [] = ""
  | L_COMP_LIST [h] = (L_COMPONENTS h)
  | L_COMP_LIST (h::t) = (L_COMPONENTS h) ^ "; " ^ (L_COMP_LIST t)
  
and L_COMPONENTS ((total_select (l,s)),a) =
      (L_ON_LIST l) ^ " : " ^ (L_SORT s)
  | L_COMPONENTS ((partial_select (l,s)),a) =
      (L_ON_LIST l) ^ " :? " ^ (L_SORT s)
  | L_COMPONENTS ((sort_component (s)),a) =
      (L_SORT s)
      
and L_ON_LIST [] = ""
  | L_ON_LIST [h] = (L_OP_NAME h)
  | L_ON_LIST (h::t) = (L_OP_NAME h) ^ ", " ^ (L_ON_LIST t) 

(****************************************************************************)

(* parsing annotations ******************************************************)

and L_PARSE_ANNO (prec_anno (a,b,c))   = L_PREC_ANNO (a,b,c)
  | L_PARSE_ANNO (number_anno (i))     = L_NUMB_ANNO (i)
  | L_PARSE_ANNO (floating_anno (i,j)) = L_FLOA_ANNO (i,j)
  | L_PARSE_ANNO (string_anno (i,j))   = L_STRI_ANNO (i,j)
  | L_PARSE_ANNO (list_anno (a,b,c))   = L_LIST_ANNO (a,b,c)
  | L_PARSE_ANNO (lassoc_anno (i))     = L_LASS_ANNO (i)
  | L_PARSE_ANNO (rassoc_anno (i))     = L_RASS_ANNO (i)
  | L_PARSE_ANNO _                     = ""

and L_PREC_ANNO (a,b,c) = "\\I{{\\bf \\%prec}}\n" ^
                          "\\( \\[\n" ^
                          (PPL.pretty_parse_anno (!cur_syn) (!cur_genv)
                          (!cur_spec) (prec_anno (a,b,c))) ^
                          "\\] \\)\n"

and L_NUMB_ANNO (i)     = "\\I{{\\bf \\%number}}\n" ^
                          "\\( \\[\n" ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 i) ^
                          "\\] \\)\n"

and L_FLOA_ANNO (i,j)   = "\\I{{\\bf \\%floating}}\n" ^
                          "\\( \\[\n" ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 i) ^
                          ", " ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 j) ^
                          "\\] \\)\n"

and L_STRI_ANNO (i,j)   = "\\I{{\\bf \\%string}}\n" ^
                          "\\( \\[\n" ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 i) ^
                          ", " ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 j) ^
                          "\\] \\)\n"

and L_LIST_ANNO (a,b,c) = "\\I{{\\bf \\%list}}\n" ^
                          "\\( \\[\n" ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 a) ^
                          ", " ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 b) ^
                          ", " ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 c) ^
                          "\\] \\)\n"

and L_LASS_ANNO (i)     = "\\I{{\\bf \\%left~assoc}}\n" ^
                          "\\( \\[\n" ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 i) ^
                          "\\] \\)\n"

and L_RASS_ANNO (i)     = "\\I{{\\bf \\%right~assoc}}\n" ^
                          "\\( \\[\n" ^
                          (PPL.pretty_id (!cur_syn) (!cur_genv)
                          (!cur_spec) 80 i) ^
                          "\\] \\)\n"

(****************************************************************************)
				    
(****************************************************************************)
(*                                                                          *)
(*                             User functions                               *)
(*                                                                          *)
(****************************************************************************)

fun frame x = "\\documentclass{article}\n" ^
              "\\usepackage{casl}\n\n" ^
              "\\begin{document}\n\n" ^
              x ^ "\n\n" ^
              "\\end{document}"

fun mkLaTeX ((x):LIB_ITEM) = efilt (L_LIB_ITEM x)

fun mkTeX (q,g,(lib_defn (n,l,_,_,_,_))) = ( cur_mode := q ;
                      if get_ref(cur_mode) then
                        ()
                      else
                        print ("\nMakellos: "^cur_vers()^"\n") ;
                      (efilt o frame) ( cur_genv := g ;
                      String.concat ( (map mkLaTeX) l ) )
                    )

fun testFun (g,(lib_defn (n,l,_,_,_,_))) = (efilt o frame) ( cur_genv := g ;
                    String.concat ( (map mkLaTeX) l ) )

fun tri_fst (a,b,c) = a

end

(****************************************************************************)
