(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author: Till Mossakowski, University of Bremen			   *)
(* Date: 02.03.2000				 			   *)
(* Purpose of this file: Computation of the module graph		   *)
(*			 						   *)
(*									   *)
(* *********************************************************************** *)

(* Take a global environment and traverse it in order to find dependencies
   among named specifications. That is, look for spec_inst in the global
   environment. From this, compute a graph. Mark the nodes as internal
   or external, depending on whether the referenced named specification
   occurs in the same library or not. Mark the edges according to their
   nature (an edge may express that a named specification is be referenced 
   in the body, parameter and/or import of another named specification,
   or it may express that there is a view between two named specifications).

   For the graphs, two output formats are supported, suitable for the tools
   daVinci and dot.

   todo:
   Grosser Graph aus allen Libraries zusammen
     nur die aktuelle Library wird dargestellt, alle anderen
     versteckt, aber Verbindungen zwischen Libraries sichtbar
     Sphaerische Projektion des Graphs
     Anklicken fuehrt zu weiteren Infos/Aktionen (siehe KIV/Inka)
*)
   
structure ModuleGraph:
sig

datatype node_type = internal of string | external of string
datatype edge_type = body_edge | param_edge | import_edge | view_edge of string

type graph = (node_type * (edge_type * string) list) list

val module_graph : GlobalEnv.global_entry Symtab_sid.table
          -> AS.LIB_DEFN
             -> graph
val module_graph_davinci : GlobalEnv.global_entry Symtab_sid.table
          -> AS.LIB_DEFN -> string
val module_graph_dot : GlobalEnv.global_entry Symtab_sid.table
          -> AS.LIB_DEFN -> string
end

=

struct

open Utils AS GlobalEnv BasicPrint;

infix mem;

val EdgeID = ref(0)
val NodeName = ref(0)

fun getEdgeID () = (EdgeID := !EdgeID+1; "_EDGE"^Int.toString (!EdgeID));
fun getNodeName () = (NodeName := !NodeName+1; "N"^Int.toString (!NodeName));

datatype node_type = internal of string | external of string
datatype edge_type = body_edge | param_edge | import_edge | view_edge of string

type graph = (node_type * (edge_type * string) list) list

fun get_refs_arg env (fit_spec (sp,_)) = get_refs env sp
  | get_refs_arg env (fit_view (v,args)) =
    (case (Symtab_sid.lookup (env,v)) of
       Some (view_defn_env (_,_,_,target)) => get_refs_slenv env target
       | _ => [])
    @ get_refs_args env args
  | get_refs_arg env (pos_FIT_ARG(_,fa)) = get_refs_arg env fa
  
and get_refs_args env args = flat (map (get_refs_arg env) args)

and get_refs_senv env (basic_env x) = []
  | get_refs_senv env (translate_env (senv,_)) = get_refs_senv env senv
  | get_refs_senv env (derive_env (senv,_)) = get_refs_senv env senv
  | get_refs_senv env (union_env  senvs) = flat (map (get_refs_senv env) senvs)
  | get_refs_senv env (extension_env senvs) = flat (map ((get_refs_senv env) o fst) senvs)
  | get_refs_senv env (free_spec_env senv) = get_refs_senv env senv
  | get_refs_senv env (cofree_spec_env senv) = get_refs_senv env senv
  | get_refs_senv env (closed_spec_env senv) = get_refs_senv env senv
  | get_refs_senv env (spec_inst_env (name, body, mor, args)) =
     print_SIMPLE_ID name :: get_refs_senv env body @ flat(map (get_refs_senv env) args)
      (* ??? What is with fitting views ??? *)
  | get_refs_senv env (dummy_spec_env) = []

and get_refs_slenv env (SPEC_ENV(_,_,senv)) =
    get_refs_senv env senv
    
and get_refs env ((basic _),_) = []
  | get_refs env ((translation (sp,_,_)),_) = get_refs env sp
  | get_refs env ((reduction (sp,_,_)),_) = get_refs env sp
  | get_refs env ((union_spec sps),_) = flat (map (get_refs env o fst) sps)
  | get_refs env ((extension sps),_) = flat (map (get_refs env o fst) sps)
  | get_refs env ((free_spec (sp,_)),_) = get_refs env sp
  | get_refs env ((cofree_spec (sp,_)),_) = get_refs env sp
  | get_refs env ((local_spec (sp1,_,sp2,_)),_) = get_refs env sp1 @ get_refs env sp2
  | get_refs env ((closed_spec (sp,_)),_) = get_refs env sp
  | get_refs env ((spec_inst  (name,args)),_) = print_SIMPLE_ID name :: get_refs_args env args
  | get_refs env (pos_SPEC(_,_,sp),x) = get_refs env (sp,x)
  
fun add_edge n1 et n2 = (n2,(et,n1))

fun get_spec_name (spec_inst(name,_)) = name
  | get_spec_name (pos_SPEC(_,_,sp)) =
    get_spec_name sp
  | get_spec_name _ = (getNodeName (),None)

fun mg_lib_item env ((inodes,enodes,edges),spec_defn (name,gen, sp,_)) =
    let val (plist,ilist) = get_genericity gen
        val body_refs = get_refs env sp
        val par_refs = flat (map (get_refs env) plist)
        val imp_refs = flat (map (get_refs env) ilist)
        val name_str = print_SIMPLE_ID name
    in
      (name_str::inodes,body_refs @ par_refs @ imp_refs @ enodes,
         map (add_edge name_str body_edge) body_refs
       @ map (add_edge name_str param_edge) par_refs
       @ map (add_edge name_str import_edge) imp_refs
       @ edges )
    end
  | mg_lib_item env ((inodes,enodes,edges),view_defn (name,gen, 
               vt,
               vmap,_)) =
    let val (plist,ilist) = get_genericity gen
        val ((s,_),(t,_)) = get_VIEW_TYPE vt
        val source = get_spec_name s
        val target = get_spec_name t
        val source_str = print_SIMPLE_ID source
        val target_str = print_SIMPLE_ID target
        val view_str = print_SIMPLE_ID name
    in
      (inodes,source_str::target_str::enodes,
       add_edge target_str (view_edge view_str) source_str :: edges)
    end
  | mg_lib_item env (g,pos_LIB_ITEM(_,li)) =
    mg_lib_item env (g,li)
  | mg_lib_item env ((inodes,enodes,edges),_) = (inodes,enodes,edges)

fun get_string (external n) = n
  | get_string (internal n) = n

fun get_lessthan node (n1,n2) =
    if n1=node
    then Some n2
    else None
    
fun get_lessthans edges node =
    (node,remove_dups (mapfilter (get_lessthan (get_string node)) edges))
    
fun make_relation nodes edges =
    map (get_lessthans edges) nodes
  
fun remove_external_dups (inodes,enodes) =
    let val enodes' = filter_out (fn x => x mem inodes) enodes
        val enodes'' = remove_dups enodes'
    in
      (map internal inodes @ map external enodes'')
    end
    
fun module_graph env (lib_defn (_,libitemlist,_)) =
    let val (inodes,enodes,edges) = foldl (mg_lib_item env) (([],[],[]),libitemlist)
    in (make_relation (remove_external_dups (inodes,enodes)) edges)
    end

fun edge_attribute body_edge = ""
  | edge_attribute param_edge = "a(\"EDGEPATTERN\",\"dotted\")"
  | edge_attribute import_edge = "a(\"EDGEPATTERN\",\"dashed\")"
  | edge_attribute (view_edge name) = "a(\"EDGEPATTERN\",\"double\")"
    
fun davinci_edge n1 (e,n2) =
    "l(\"" ^ getEdgeID() ^ "_" ^ n1 ^ "->" ^ n2 ^ "\",e(\"\",[" ^ 
    edge_attribute e ^ "],r(\"" ^ n2 ^ "\")))"

  
fun get_box_attribute (external n) = ",a(\"BORDER\",\"none\")"
  | get_box_attribute (internal n) = ""
  
fun davinci_node (n,lessthans) = 
    "l(\"" ^ get_string n ^ "\",n(\"\",[a(\"OBJECT\",\"" ^ get_string n ^ "\")"
    ^get_box_attribute n^"],["
    ^print_list (davinci_edge (get_string n)) "," lessthans ^ "]))"

    
fun davinci relation =
    "["^print_list  davinci_node "," relation^"]"


fun module_graph_davinci env tree = davinci (module_graph env tree)


fun edge_attribute_dot body_edge = ""
  | edge_attribute_dot param_edge = " [style=dotted]"
  | edge_attribute_dot import_edge = " [style=dotted]"
  | edge_attribute_dot (view_edge name) = " [style=bold]"
 
fun dot_edge n1 (e,n2) =
    n1 ^ " -> " ^ n2 ^ edge_attribute_dot e
    
fun get_box_attribute_dot (external n) = " [shape=box]"
  | get_box_attribute_dot (internal n) = ""

fun dot_node (n,lessthans) =
    get_string n^get_box_attribute_dot n^
    (if (lessthans = []) then ""
     else ";\n  "^print_list (dot_edge (get_string n)) ";\n  " lessthans)
    
fun dot relation =
    "digraph G {\n    size = \"8,6\"\n  "^print_list dot_node ";\n  " relation^";\n}"
fun  module_graph_dot env tree = dot (module_graph env tree)
end
