(* *********************************************************************** *)
(*									   *)
(* Project: CATS 							   *)
(* Author: Till Mossakowski, University of Bremen			   *)
(* Date: 16.05.97				 			   *)
(* Purpose of this file: Overload resolution				   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* This module does type checking and overload resolution, following
   the semantics, see stud note S-9, chapter 4 on subsorting (and see also the
   same chapter in the CASL summary). The most difficult part is to
   implement the check whether there is an expansion of a formula
   to a fully qualified formula that is unique up to equivalence
   (see the cited references for exact definitions of this).
   To realize this part, we use the algorithm MinExp that is
   described in CoFI study note T-2 and in
   @INCOLLECTION(Overtheo,
     author	= {T.\ Mossakowski and Kolyang and B.\ {Krieg-Br{\"u}ckner} },
     title	= {Static semantic analysis and theorem proving for {CASL}},
     booktitle	= {Recent trends in algebraic development techniques.	
		  Proc.\ 12th International Workshop},
     editor	= {F. {Parisi Presicce}},
     series	= {Lecture Notes in Computer Science},
     volume	= {1376},
     pages	= {333--348},
     publisher	= {Springer},
     year	= {1998}

   todo:
   adapt MinExp, since it currently is based on an old version
     of the overloading relations. 
   Genauere Typfehler (wie in SML)
   Undeklarierte Sorten und Operationsnamen melden, statt wie jetzt
     einen Typcheck-Fehler auszugeben
   Minimales profile ausgeben
*)   

structure Overload :
sig
val overload_resolution : 
    bool -> LocalEnv.local_env * AS.L_FORMULA list -> string list * AS.L_FORMULA list 
val overload_resolution_TERM :
    LocalEnv.local_env * AS.TERM -> (AS.TERM * AS.SORT) list 
val INJECTION_NAME : AS.OP_NAME
val make_inj : AS.SORT * AS.SORT -> AS.OP_SYMB
end

= struct

open Utils AS LocalEnv BasicPrint Subsorts IDOrder 

val INJECTION_NAME = simple_id ([token "inj$"],"inj$", None);

fun make_inj (s1,s) = op_symb (INJECTION_NAME,Some (total_op_type (sorts [s1],s)))

type added_env = 
	OP_TYPE list list Symtab_id.table   (* fun symb ==> list of connected components wrt <=F *)
	* PRED_TYPE list list Symtab_id.table  (* pred symb ==> list of connected components wrt <=P *)
type local_env1 = 
	  local_env * added_env


exception NOT_IMPLEMENTED
exception AMBIGOUS of (PRED_SYMB *TERMS) list
exception UNDECLARED_VAR_OR_CONST_SYMB of ID
exception UNDECLARED_OP_SYMB of OP_SYMB
exception UNDECLARED_PRED_SYMB of PRED_SYMB
exception VAR_TYPE_MISMATCH of (SIMPLE_ID * SORT * SORT)
exception WRONG_ARG_NO of SORT list
exception TYPE_MISMATCH_TERM of OP_SYMB
exception TYPE_MISMATCH_COND of (TERM * TERM)
exception TYPE_MISMATCH_PRED of PRED_SYMB
exception WRONG_FUN_ARITY of  (OP_SYMB * TERM list)
exception WRONG_PRED_ARITY of (PRED_SYMB * TERM list)
exception UNDECLARED_SORT of SORT
exception CANNOT_HAPPEN
exception TRACE of ((TERM * SORT) list list * (TERM * SORT) list list * (PRED_SYMB* TERMS) list list)
exception LENGTH of int;
exception FOLLOW_UP;
(*val is_goal = false*)

  
fun lookup_sort (s,srts) =  case Symtab_id.lookup (srts,s) of
	None => (raise (UNDECLARED_SORT s))
	| Some s1 => s1

	              

(* Compare (lists of) sorts by looking up in the sort environment.
   Note that to each sort in the environment is associated the list
   of its subsorts *)  

 
local
fun leq_aux srts (s1,s2) = SORT_member (s1,lookup_sort (s2,srts))
in
fun leq (srts,vars,funs,preds) (s1,s2) = leq_aux srts (s1,s2)
end

fun leq1 (env,env1) = leq env

fun leq_list (env:local_env) (sl1:SORT list,sl2: SORT list) 
   = (length sl1 = length sl2) andalso forall1 (map (leq env) (zip (sl1,sl2)))

fun leq_list1 ((env,env1):local_env1) = leq_list env



(* Lookup a variable *)

fun lookup_var env (simple_id v1) vars vtype:(TERM * SORT) list =
    let val v = token_or_mifix2simple_id v1 in
	(case Symtab_sid.lookup(vars, v) of
	None => nil 
	| Some s => 
	   (case vtype of 
	   None => [(qual_var (v,s),s)]
	   |Some t => (if leq1 env (s,get_res t)
	              then  [(sorted_term(qual_var (v,s),get_res t),get_res t)]
	              else raise (VAR_TYPE_MISMATCH (v,s,get_res t)))
	   )) 
    end 
|  lookup_var env (compound_id v) vars vtype = nil	   

  
	   
(* w->s leqF w'->s' iff w <= w' and s,s' have a common supersort *)

(* old code with  find_first which has been rationalised

fun has_common_upper_bound ((srts,vars,funs,preds):local_env) (s1:SORT,s2:SORT):bool =
	let
		fun is_common_upper_bound(s,subsorts) = 
			SORT_member(s1,subsorts) andalso SORT_member(s2,subsorts)
	in
	case Symtab_id.find_first is_common_upper_bound srts of
	  None => false
	  | Some x => true
	end

*)

(*fun has_common_upper_bound ((srts,vars,funs,preds):local_env) (s1:SORT,s2:SORT):bool =
	let
		fun is_common_upper_bound(s,subsorts) = 
			SORT_member(s1,subsorts) and SORT_member(s2,subsorts)
	in
	case Symtab_id.find_first is_common_upper_bound srts of
	  None => false
	  | Some x => true
	end *)

fun leqF (env:local_env) (type1:OP_TYPE,type2:OP_TYPE):bool =
	leq_list env (get_args type1,get_args type2)  (*andalso
	has_common_upper_bound env (get_res type1,get_res type2) *)

(* pred(w) leqP pred(w') iff w <= w'  *)

fun leqP (env:local_env) (t1:PRED_TYPE,
			  t2:PRED_TYPE):bool =
	leq_list env (get_sorts (get_pred_type t1),get_sorts (get_pred_type t2))


(**************************************************************************************)
(*		Functions for minimizing types	 				      *)
(**************************************************************************************)
	
(* Insert type1 in a list of types while keeping the types in the list minimal.
   If some type in the list is comparable to type1, the minimal of both
   replaces the old one, otherwise type1 is just added to the list *) 	

fun minimize(type1,nil,leq_type) = [type1]
  | minimize(type1,type2::resttypes,leq_type) =
	if leq_type (type2,type1) then type2::resttypes
	else if leq_type (type1,type2) then type1::resttypes
	     else type2::minimize(type1,resttypes,leq_type)

(* Compute the list of minimal types wrt leq_type that satisfy fits_type.
   types_so_far is an accumulator which should be set to [] at the beginning *)

fun lookup_min nil (types_so_far,leq_type,fits_type) = types_so_far
  | lookup_min (type1::env) (types_so_far,leq_type,fits_type) =
     (if fits_type type1 
      then minimize (type1,types_so_far,leq_type)
      else [])
     @ lookup_min env (types_so_far,leq_type,fits_type)



(**************************************************************************************)
(*	  		  MinExp for terms					      *)
(**************************************************************************************)

(* Comparison of functional types. Needed to find the minimal types.
   Note that this is different from leqF, which is needed for
   checking if expansions are equivalent! *)	

fun leq_fun_types (env:local_env1) (pf1:OP_TYPE,pf2:OP_TYPE)
	    = leq_list1 env (get_args pf1,get_args pf2)
	      andalso leq1 env (get_res pf1,get_res pf2)
	      
(* Check if a list of sorts (of actual parameters) fits with a functional type *)

fun fits_fun_type env actualsorts (total_op_type (formalsorts,ressort))
   		= leq_list1 env (actualsorts, get_sorts formalsorts) 
  | fits_fun_type env actualsorts (partial_op_type (formalsorts,ressort))
   	  	= leq_list1 env (actualsorts, get_sorts formalsorts) 
  | fits_fun_type env actualsorts (pos_OP_TYPE (_,t)) =
    fits_fun_type env actualsorts t

(* Compute the set of minimal types for some OP_SYMB wrt to some list of argument sorts *)

fun min_types_term (env:local_env1) 
		      ((fname,Some (ftype)),actualsorts) =
	let
    		val ((srts,vars,funs,preds),env1) = env
    		val formalsorts = get_args ftype
    		val ressort = get_res ftype
    	in
	case Symtab_id.lookup(funs,fname) of
	None => (raise UNDECLARED_OP_SYMB(op_symb(fname,Some ftype)))
	| Some ts => (if not (OP_TYPE_member(ftype,ts))
	              then raise UNDECLARED_OP_SYMB(op_symb(fname,Some ftype))
	              else
	              if leq_list1 env (actualsorts,formalsorts)
		      then [ftype]
		      else [])
	end
  | min_types_term (env:local_env1) ((fname,None),actualsorts) =
    	let
    		val ((srts,vars,funs,preds),env1) = env
    		val ftypes = Symtab_id.lookup_multi(funs,fname)
    	in
    	if null ftypes then raise UNDECLARED_OP_SYMB(op_symb(fname,None))
        else lookup_min ftypes ([],leq_fun_types env,fits_fun_type env actualsorts)
    	end
    
(* Check if the arity of a OP_SYMB fits the number of argument terms *)

fun fits_arity_term (Some (total_op_type (sl,s)),trms) = 
    (length (get_sorts sl) = length trms)
  | fits_arity_term (Some (partial_op_type (sl,s)),trms) =
    (length (get_sorts sl)  = length trms)
  | fits_arity_term (Some (pos_OP_TYPE (_,t)),trms) = 
    fits_arity_term (Some t,trms)
  | fits_arity_term (None,_) = true	

(* Generate an application of a constant to the empty list of arguments *)  
fun make_constant fname ftype =
	let
	  val s = get_res ftype
	in
	  (application (op_symb(fname,Some ftype),terms []),s)
	end
	
(**************************************************************************************)
(*		Equivalent terms and predications 				      *)
(**************************************************************************************)

fun in_relation member nil (a1,a2) = false
  | in_relation member (x::cs) (a1,a2) =
  	 (member(a1,x) andalso member(a2,x)) orelse in_relation member cs (a1,a2)

fun in_relation1 member None (a1,a2) = raise CANNOT_HAPPEN
  | in_relation1 member (Some rel) (a1,a2) = in_relation member rel (a1,a2)

(* Check is two types are equivalent by looking if they
   are in the same equivalence class *)
      	 
fun equivalent_ptype (env,(fenv,penv)) (simple_id([token "_D"], "_D",None), pred_type(sorts [s1]),pred_type(sorts[s2])) =
         leq env (s1,s2) orelse leq env (s2,s1)  	
  | equivalent_ptype (env,(fenv,penv)) (simple_id([token"_seq"],"_seq",None), pred_type(sorts [s1,s2]),pred_type(sorts[s3,s4])) =
         leq env (s1,s3) orelse leq env (s3,s1)  	
  | equivalent_ptype (env,(fenv,penv)) (simple_id([token"_eeq"],"_eeq",None), pred_type(sorts [s1,s2]),pred_type(sorts[s3,s4])) =
         leq env (s1,s3) orelse leq env (s3,s1)  	
  | equivalent_ptype (env,(fenv,penv)) (simple_id([token"_in_"],"_in_",None), pred_type(sorts [s1]),pred_type(sorts[s2])) =
         leq env (s1,s2) orelse leq env (s2,s1)  	
  | equivalent_ptype (env,(fenv,penv)) (pname,ptype1,ptype2) =
  	 in_relation1 PRED_TYPE_member (Symtab_id.lookup(penv,pname)) (ptype1,ptype2)

fun equivalent_ftype (env,(fenv,penv)) (fname,ftype1,ftype2) =
	in_relation1 OP_TYPE_member (Symtab_id.lookup(fenv,fname)) (ftype1,ftype2)

(* Strip off all injections from a term *)

fun strip_embeddings (application (op_symb (simple_id ([token "inj$"],"inj$", None),ftpye), terms [t])) =
	strip_embeddings t
  | strip_embeddings (application (pos_OP_SYMB (_,op_sym),ts)) =
    strip_embeddings (application (op_sym,ts))
  | strip_embeddings (application (op_sym,pos_TERMS (_,ts)))
    = strip_embeddings (application (op_sym,ts))
  | strip_embeddings (application (f,tl)) =
  	application (f,terms (map strip_embeddings (get_terms tl)))
  | strip_embeddings (var_or_const x) = var_or_const x
  | strip_embeddings (qual_var x) = qual_var x
  | strip_embeddings (sorted_term (t,s)) =
  	sorted_term (strip_embeddings t,s)
  | strip_embeddings (cast (t,s)) =
  	cast (strip_embeddings t,s)
  | strip_embeddings (conditional (t1,phi,t2)) =
        conditional(strip_embeddings t1,phi,strip_embeddings t2)
  | strip_embeddings (pos_TERM (_,_,t)) =
    strip_embeddings t
  | strip_embeddings _ = raise (ERR "Overload: strip_embedddings")
          
fun equivalent_term1 (env:local_env1) (qual_var x,qual_var y) = x=y
  | equivalent_term1 (env:local_env1) (var_or_const x,var_or_const y) = x=y
  | equivalent_term1 env (application (op_symb (fname1,ftype1), terms t1),
		          application (op_symb (fname2,ftype2), terms t2)) =
	OP_eq(fname1,fname2) andalso
	equivalent_ftype env (fname1,the ftype1,the ftype2) andalso
	equivalent_terms env (t1,t2)
  | equivalent_term1 env (cast (t1,s1), cast(t2,s2)) =
  	SORT_eq(s1,s2) andalso equivalent_term env (t1,t2)
  | equivalent_term1 env (sorted_term (t1,s1), sorted_term(t2,s2)) =
  	SORT_eq(s1,s2) andalso equivalent_term env (t1,t2)
  | equivalent_term1 env (conditional (t1,phi1,t1'), conditional (t2,phi2,t2')) =
  	equivalent_term env (t1,t2) andalso equivalent_term env (t1',t2')
  | equivalent_term1 env (_,_) = false

(* Check if two terms are equivalent.
   This is done by first stripping off all injections, 
   and then checking a more literal equivalence with equivalent_term1 *)
  
and equivalent_term (env:local_env1) (t1,t2) =
	equivalent_term1 env (strip_embeddings t1,strip_embeddings t2)

(* Check if two lists of terms are element-wise equivalent *)

and equivalent_terms (env:local_env1) (t1,t2) =
	forall1 (map (equivalent_term env) (zip(t1,t2)))
	
(* Check if two predicate symbols are equivalent *)

fun equivalent_pred (env:local_env1) (pred_symb(pname1,Some ptype1),
				      pred_symb(pname2,Some ptype2)) =
	(PRED_eq(pname1,pname2) andalso
	equivalent_ptype env (pname1,ptype1,ptype2))
  | equivalent_pred env (_,_) = false
  	
(* Check if two predications are equivalent *)

fun equivalent (env:local_env1) (p1,t1) (p2,t2) =
	equivalent_pred env (p1,p2) andalso
	equivalent_terms env (get_terms t1,get_terms t2)

(* Check if all members of a list of predications are equivalent *)
	
fun all_equivalent (env:local_env1) nil = true
  | all_equivalent (env:local_env1) [p] = true
  | all_equivalent (env:local_env1) (p1::p2::ps:(PRED_SYMB *TERMS)list) : bool  =
	forall1 (map (equivalent env p1) (p2::ps))


(*******************************************************)
(*           Axuiliary functions                       *)
(*******************************************************)

(* Add a variable to the variable environment. Overwrite any existing declaration
   of the variable, because within nested quantifications with the same
   variable, only the innermost declaration is visible *)
   
fun add_var_env ((srts,vars,funs,preds),env1) (v,s) =
		 ((srts,Symtab_sid.update((v,s),vars),funs,preds),env1)

fun add_var_decl_env env var_decl_list =
	let
		fun add2 s (env,v) = add_var_env env (v,s)
		fun add1 (env,(vlist,s)) = foldl (add2 s) (env,vlist)
	in 
	foldl add1 (env,var_decl_list)
	end

(* Formulas are resolved inductively, by keeping track of the quantified variables *)

fun err_propagate f (errs,phi) = (errs,f phi)
fun err_propagate_pair f ((errs1,phi1),(errs2,phi2)) = (errs1@errs2,f (phi1,phi2))
fun err_propagate_list f l = 
	(appendlists (map fst l),f (map snd l))

fun conv_pred (pred_symb (simple_id([token"_seq"],"_seq",None),t),terms [t1,t2]) = strong_equation (t1,t2)
|   conv_pred (pred_symb (simple_id([token"_eeq"],"_eeq",None),t),terms [t1,t2]) = existl_equation (t1,t2)
|   conv_pred (pred_symb (simple_id([token"_in"],"_in",None),Some(pred_type(sorts [s]))),terms [t1]) = membership (t1,s)
|   conv_pred (pred_symb (simple_id([token"_D"],"_D",None),t),terms [t1]) = definedness (t1)
|   conv_pred (p,t) = predication (p,t)

fun print_ambig l = print_list print_ATOM "\n" (map conv_pred l)
fun print_farity x = ""
fun print_parity x = ""
	
fun add_blank s = s^" "


fun inj (t,s1,s) =
    if SORT_eq(s1,s) then t
    else application (op_symb (INJECTION_NAME,Some (total_op_type (sorts [s1],s))),terms [t]) 

fun add_injections1 ((t,s1),s) =
    inj (t,s1,s)

fun add_injections stermlist fname ftype =
    (application (op_symb(fname,Some ftype), 
	               terms (map add_injections1 (zip (stermlist,(get_args ftype))))),
     get_res ftype)

fun add_injections_cond phi [t1,t2] s =
    (conditional (add_injections1 (t1,s),phi,add_injections1 (t2,s)),s)
  | add_injections_cond _ _ _ = raise (ERR "Overload: add_injections_cond")

fun all_sorts srts (simple_id v1) =
    let val v = token_or_mifix2simple_id v1 
        val sort_list = map fst (Symtab_id.dest srts)
        fun mk_var s = (qual_var (v,s),s)
    in
        map mk_var sort_list
    end
  | all_sorts _ _ = raise (ERR "Overlaod: all_sorts")

(****************************************************************************************)
(* This is the function MinExp described in the study note T-2, with terms as arguments *)
(****************************************************************************************)

(* For constant application check both constant and variable case, since
   they are indistinguishable at the conrete syntax level *)
fun MinExpTerm (is_goal:bool) (env:local_env1) (application (op_sym,ts)): (TERM * SORT) list = 
    (case (get_terms ts) of
      [] =>
    	let
           val ((srts,vars,funs,preds),env1) = env
           val const_or_var = get_op_name op_sym
           val cvtype = get_op_type op_sym
           val const_res= map (make_constant const_or_var) 
           	          (min_types_term env ((const_or_var,cvtype),[])) handle UNDECLARED_OP_SYMB x => []  
           val var_res = lookup_var env const_or_var vars cvtype
           val res = const_res@var_res
    	in
    	if is_goal then
            if const_res = nil then all_sorts srts const_or_var
            else const_res
        else
            if res = nil then
               raise UNDECLARED_VAR_OR_CONST_SYMB const_or_var
            else res
	end 
    | (t::trms) =>
    let
	val fname = get_op_name op_sym
        val ftype = get_op_type op_sym
        fun MinExp1 stermlist = 
	let
		val actualsorts = map get_sort stermlist 
		val types = min_types_term env ((fname,ftype),actualsorts)
	in 	map (add_injections stermlist fname) types
	end
    in  if fits_arity_term (ftype,(t::trms))
        then 
         (case appendlists (map MinExp1 (permute (map (MinExpTerm is_goal env) (t::trms)))) of
           nil => (raise TYPE_MISMATCH_TERM op_sym)
           | m::m1 => m::m1)
        else raise WRONG_FUN_ARITY (op_sym,(t::trms)) 
    end)
  | MinExpTerm (is_goal:bool) (env:local_env1) (sorted_term (t,s)): (TERM * SORT) list = 
  	let
  		val tslist = MinExpTerm is_goal env t
  		fun filter_sort (s,nil) = nil
  		  | filter_sort (s,(t,s1)::rest) = 
  		  	if leq1 env (s1,s) then (inj(t,s1,s),s) :: filter_sort (s,rest)
  		  	else filter_sort (s,rest)
  	in
  		filter_sort (s,tslist)
  	end
  | MinExpTerm (is_goal:bool) (env:local_env1) (cast (t,s)): (TERM * SORT) list = 
  	let
  		val tslist = MinExpTerm is_goal env t
  		fun filter_subsort (s,nil) = nil
  		  | filter_subsort (s,(t,s1)::rest) = 
  		  	if leq1 env (s,s1) then (cast (t,s),s):: filter_subsort (s,rest)
  		  	else filter_subsort (s,rest)
  	in
  		filter_subsort (s,tslist)
  	end
  | MinExpTerm (is_goal:bool) (env:local_env1) (conditional (t1,phi,t2)): (TERM * SORT) list = 
  let
        val ((srts,vars,funs,preds),env1) = env
        val phi_res = ores_formula1 is_goal env phi
        fun fits_type (env:local_env1) ([s1,s2]:SORT list) (s:SORT) = 
            leq1 env (s1,s) andalso leq1 env (s2,s) 
          | fits_type _ _ _ = raise (ERR "Overload: fits_type")
	fun MinExp2 stermlist = 
	let
	     val actualsorts = map get_sort stermlist 
	     val valid_sorts = lookup_min (map fst (Symtab_id.dest srts)) ([],leq1 env,fits_type env actualsorts)
	in map (add_injections_cond phi_res stermlist) valid_sorts
	end
  in
        (case appendlists (map MinExp2 (permute (map (MinExpTerm is_goal env) ([t1,t2])))) of
          nil => (raise TYPE_MISMATCH_COND (t1,t2))
          | m::m1 => m::m1) 
  end
  | MinExpTerm (is_goal:bool) (env:local_env1) (unparsed_term t): (TERM * SORT) list = 
    raise FOLLOW_UP 
  | MinExpTerm (is_goal:bool) (env:local_env1) (pos_TERM (r,b,t)): (TERM * SORT) list = 
    let val termsortlist = MinExpTerm is_goal env t
        fun add_region (t,s) = (pos_TERM (r,b,t),s)
    in map add_region termsortlist
    end
  | MinExpTerm _ _ _ = raise (ERR "Overload: MinExpTerm")

(**************************************************************************************)
(*	  		  MinExp for predications				      *)
(**************************************************************************************)


(* Compute the set of minimal types for some PRED_SYMB wrt to some list of argument sorts *)
	
and min_types (env:local_env1) ((pname,Some ptype),actualsorts) =
    let
   	val ((srts,vars,funs,preds),env1) = env
        val  formalsorts = get_sorts (get_pred_type ptype)
   	fun wrong_arg_no actualsorts = raise (WRONG_ARG_NO actualsorts)
   	fun eq_type s = pred_type (sorts [s,s])
     in
    case pname of
    (simple_id ([token "_D"],"_D",_)) => (if length actualsorts=1 
             then [pred_type (sorts actualsorts)] 
             else wrong_arg_no actualsorts)
    |  (simple_id ([token "_eeq"],"_eeq",_)) => (if length actualsorts=2 
                 then 
                 map eq_type 
              (get_common_upper_bounds (its_an_env srts) (hd(actualsorts),hd(tl(actualsorts))))
    		 else wrong_arg_no actualsorts)
    |  (simple_id ([token "_seq"],"_seq",_)) => (if length actualsorts=2 
                 then 
                 map eq_type 
                     (get_common_upper_bounds (its_an_env srts) (hd(actualsorts),hd(tl(actualsorts))))
    		 else wrong_arg_no actualsorts)
    |  (simple_id ([token "_in_"],"_in_",_)) => (if length actualsorts=1
                 then if leq1 env (hd formalsorts,hd actualsorts)
                      then [pred_type (sorts actualsorts)] 
                      else []
                 else wrong_arg_no actualsorts)
    | x =>
    	(case Symtab_id.lookup(preds,pname) of
    	 None => raise UNDECLARED_PRED_SYMB (pred_symb (pname,Some ptype))
    	 | Some ts => (
    	        if not (PRED_TYPE_member (ptype,ts)) then
    	        raise UNDECLARED_PRED_SYMB (pred_symb (pname,Some ptype))
    	        else
		if leq_list1 env (actualsorts,formalsorts)
		then [pred_type (sorts formalsorts)]
		else []) )
    end
  | min_types (env:local_env1) ((pname,None),actualsorts) =
    let
   	val ((srts,vars,funs,preds),env1) = env
        fun leq_type (t1,t2) = 
          leq_list1 env (get_sorts (get_pred_type t1),get_sorts (get_pred_type t2)) 
   	fun fits_type ptype =  leq_type (pred_type (sorts actualsorts),ptype) 
   	val ptypes = Symtab_id.lookup_multi(preds,pname)
    in
  	if null ptypes then raise UNDECLARED_PRED_SYMB (pred_symb (pname,None))
  	else lookup_min ptypes ([],leq_type,fits_type)
    end

(* Check if the arity of a PRED_SYMB fits the number of argument terms *)
    
and fits_arity (Some t,trms) = 
    (length (get_sorts (get_pred_type t)) = length trms)
  | fits_arity (None,_) = true	
 
(* This is the function MinExp described in the study note T-2, with predications as arguments *)

and MinExp (is_goal:bool) (env:local_env1) (psymb,ts): (PRED_SYMB * TERMS) list =
let
	val pname = get_pred_name psymb
        val ptype = get_pred_type_PRED_SYMB psymb
        val r_p = get_region_PRED_SYMB psymb
        val trms = get_terms ts
        val r_t = get_region_TERMS ts
        fun add_injections1 nil nil = nil
	  | add_injections1 ((t,s1)::tl) (s::sl) =
		(if SORT_eq(s,s1) then t
		 else application (op_symb (INJECTION_NAME,Some (total_op_type (sorts [s1],s))),terms [t]) ) 
						::add_injections1 tl sl
	  | add_injections1 _ _ = nil

	fun add_injections stermlist t =
        let val sl = get_sorts (get_pred_type t)
            val new_psymb = pred_symb(pname,Some (pred_type (sorts (sl))))
            val new_ts = terms (add_injections1 stermlist sl)
            val new_psymb1 =  case r_p of
                   None => new_psymb
                 | Some r1 => pos_PRED_SYMB(r1,new_psymb)
          val new_ts1 =  case r_t of
                   None => new_ts
                 | Some r1 => pos_TERMS(r1,new_ts)
        in (new_psymb1,new_ts1)
        end

	fun MinExp1 stermlist = 
	let
		val sortlist = map get_sort stermlist 
		val types = min_types env ((pname,ptype),sortlist)
	in 	map (add_injections stermlist) types
	end
in  if fits_arity (ptype,trms)
    then appendlists (map MinExp1 (permute (map (MinExpTerm is_goal env) trms)))
(*    then raise (TRACE (map (MinExpTerm  is_goal env) trms,permute (map (MinExpTerm env) trms),map MinExp1 (permute (map (MinExpTerm is_goal env) trms)) )) *)
    else raise WRONG_PRED_ARITY (psymb,trms) 
end


(**************************************************************************************)
(*	  		  Overload resolution					      *)
(**************************************************************************************)

(* To resolve a predication, invoke MinExp and then check if
   all minimal expansions are indeed equivalent *)
   
and ores_predication (is_goal:bool) (env:local_env1) (psymb,t) =
 	let
 		val m = MinExp is_goal env (psymb,t)
 	in
 	case m of 
 	  nil => (raise TYPE_MISMATCH_PRED psymb)
 	  | (m1::mrest) => (if is_goal orelse all_equivalent env m 
                            then find_min_inj m
                            else raise (AMBIGOUS m) )
	end

and sum l = foldl (op +) (0,l)

and count_inj_term (application (op_symb (simple_id ([token "inj$"],"inj$", None),ftpye), ts)) =
    count_inj_terms ts + 1
  | count_inj_term (application (pos_OP_SYMB (_,op_sym),ts)) =
    count_inj_term (application (op_sym,ts))
  | count_inj_term (application (_,ts)) = count_inj_terms ts 
  | count_inj_term (sorted_term (t,_)) = count_inj_term t
  | count_inj_term (cast (t,_)) = count_inj_term t
  | count_inj_term (conditional (t1,phi,t2)) =
    count_inj_term t1 + count_inj_formula phi + count_inj_term t2
  | count_inj_term (pos_TERM (_,_,t)) =
    count_inj_term t
  | count_inj_term _ = 0

and count_inj_terms (terms tt) = sum (map count_inj_term tt)
  | count_inj_terms (pos_TERMS (_,ts)) =
    count_inj_terms ts

and count_inj_predication (_,ts) = count_inj_terms ts

and count_inj_atom (predication p) = count_inj_predication p
  | count_inj_atom(definedness t) = count_inj_term t
  | count_inj_atom(existl_equation (t1,t2)) =
    count_inj_term t1 + count_inj_term t2
  | count_inj_atom(strong_equation  (t1,t2)) =
    count_inj_term t1 + count_inj_term t2
  | count_inj_atom(membership (t,_)) =
    count_inj_term t
  | count_inj_atom ttrue = 0
  | count_inj_atom ffalse = 0

and count_inj_formula(quantification (_,_,phi)) =
    count_inj_formula phi
  | count_inj_formula(pred_quantification (_,_,phi)) =
    count_inj_formula phi
  | count_inj_formula(conjunction phis) =
    sum (map count_inj_formula phis)
  | count_inj_formula(disjunction phis) =
    sum (map count_inj_formula phis)
  | count_inj_formula(implication (phi1,phi2)) =
    count_inj_formula phi1 + count_inj_formula phi2
  | count_inj_formula(equivalence  (phi1,phi2)) =
    count_inj_formula phi1 + count_inj_formula phi2
  | count_inj_formula(negation phi) =
    count_inj_formula phi
  | count_inj_formula(atom a) =
    count_inj_atom a
  | count_inj_formula(pos_FORMULA (_,_,phi)) =
    count_inj_formula phi
  | count_inj_formula _ = 0

and find_min_inj1 nil _ p = p
  | find_min_inj1 (p::rest) count_so_far so_far =
    let val pcount = count_inj_predication p
    in
    if pcount < count_so_far
    then find_min_inj1 rest pcount p
    else find_min_inj1 rest count_so_far so_far
    end

and find_min_inj (p::rest) = find_min_inj1 rest (count_inj_predication p) p 
  | find_min_inj nil = raise ERROR

and ores_atom (is_goal:bool) env ttrue = ttrue
  | ores_atom (is_goal:bool) env ffalse = ffalse
  | ores_atom (is_goal:bool) env (predication (psymb,t)) = predication (ores_predication is_goal env (psymb,t))
  | ores_atom (is_goal:bool) env (definedness t) = 
  	let
  	val (psymb,ts) = ores_predication is_goal env (pred_symb(make_sid "_D",Some(pred_type(sorts[make_sid " "]))),terms [t]) (* was before sorts[" "] *)
        val u = hd(get_terms ts)
  	in (definedness u)
  	end
  | ores_atom (is_goal:bool) env (existl_equation (t1,t2)) =  
  	let
  	val (psymb,ts) = ores_predication is_goal env (pred_symb(make_sid "_eeq",Some(pred_type(sorts[make_sid "",make_sid ""]))),terms[t1,t2]) (* was before sorts ["",""] *)
        val (u1,u2) = case get_terms ts of
            [v1,v2] => (v1,v2)
            | _ => raise (ERR "Overload: _eeq")        
  	in (existl_equation (u1,u2))
  	end
  | ores_atom (is_goal:bool) env (strong_equation (t1,t2)) =  
  	let
  	val (psymb,ts) = ores_predication is_goal env (pred_symb(make_sid "_seq",Some(pred_type(sorts[make_sid "",make_sid ""]))),terms[t1,t2]) (* was before sorts ["",""] *)
        val (u1,u2) = case get_terms ts of
            [v1,v2] => (v1,v2)
            | _ => raise (ERR "Overload: _seq")   
  	in (strong_equation (u1,u2))
  	end
  | ores_atom (is_goal:bool) env (membership (t,s)) =  
  	let
  	val (psymb,ts) = ores_predication is_goal env (pred_symb(make_sid "_in_",Some(pred_type(sorts[s]))),terms [t])
        val u = case get_terms ts of
            [v] => v
           | _ => raise (ERR "Overload: membership")
  	in (membership (u,s))
  	end
 
	
and ores_formula1 (is_goal:bool) (env:local_env1) (quantification (q,varlist,phi)) =
         quantification (q,varlist,ores_formula1 is_goal (add_var_decl_env env varlist) phi)
  | ores_formula1 is_goal env (conjunction flist) = 
  	 conjunction ( (map (ores_formula1 is_goal env)) flist)
  | ores_formula1 is_goal env (disjunction flist) = 
  	 disjunction ( (map (ores_formula1 is_goal env)) flist)
  | ores_formula1 is_goal env (implication (f1,f2)) = 
  	 implication (ores_formula1 is_goal env f1,ores_formula1 is_goal env f2)
  | ores_formula1 is_goal env (equivalence (f1,f2)) = 
  	 equivalence (ores_formula1 is_goal env f1,ores_formula1 is_goal env f2)
  | ores_formula1 is_goal env (negation f) = negation (ores_formula1 is_goal env f)
  | ores_formula1 is_goal env (unparsed_formula f) = unparsed_formula f
  | ores_formula1 is_goal env (atom a) = 
  	atom (ores_atom is_goal env a)
  | ores_formula1 is_goal env (sort_gen_ax x) = sort_gen_ax x
  | ores_formula1 is_goal env (sort_cogen_ax x) = sort_cogen_ax x
  | ores_formula1 is_goal env (sort_cofree_ax x) = sort_cofree_ax x
  | ores_formula1 is_goal env (pos_FORMULA (r,b,x)) =
    pos_FORMULA(r,b,ores_formula1 is_goal env x)

and ores_formula (is_goal:bool) (env:local_env1) (phi:FORMULA) =
    ([],ores_formula1 is_goal env phi)
    handle 	NOT_IMPLEMENTED  => (["Warning: feature not supported yet: "^print_FORMULA phi],phi)
    |	ZIP_ERROR => (["Internal error: zip"],phi)
    |	(AMBIGOUS l) => (["Cannot disambiguate:\n"^(print_ambig l)],phi)
    |	(UNDECLARED_SORT s) => (["Undeclared sort: "^(print_SORTS1 [s])^" in: \n"^print_FORMULA phi],phi)
    |	(UNDECLARED_VAR_OR_CONST_SYMB id) =>  (["Undeclared variable or constant: "^ (print_SORTS1 [id])^" in: \n"^print_FORMULA phi],phi)
    |	(UNDECLARED_OP_SYMB (op_symb(id,t))) =>  (["Undeclared function symbol: "^(print_OP_SYMB (op_symb(id,t)))^" in: \n"^print_FORMULA phi],phi)
    |	(UNDECLARED_PRED_SYMB (pred_symb(id,t))) => (["Undeclared predicate symbol: "^(print_PRED_SYMB (pred_symb(id,t)))^" in: \n"^print_FORMULA phi],phi)
    |	(VAR_TYPE_MISMATCH (v,s,s1)) => (["Variable "^(print_SIMPLE_ID v)^" declared as "^(print_SORTS1 [s])^" but used as "^(print_SORTS1 [s1])^" in: \n"^print_FORMULA phi],phi)
   (*	 |	(WRONG_ARG_NO sortlist) =>  (["Wrong number of arguments: "^(concat (map add_blank sortlist))^" in: \n"^print_FORMULA phi],phi) *)
    |	(TYPE_MISMATCH_TERM sym) =>  (["Type mismatch for "^print_OP_SYMB sym^" in: "^print_FORMULA phi],phi)
    |	(TYPE_MISMATCH_COND (t1,t2)) =>  (["Type mismatch between "^print_TERM t1^" and "^print_TERM t2^" in: "^print_FORMULA phi],phi)
    |	(TYPE_MISMATCH_PRED sym) => (["Type mismatch for "^print_PRED_SYMB sym^" in:  "^print_FORMULA phi],phi)
    |	(WRONG_FUN_ARITY x) => (["Wrong function arity: "^(print_farity x)],phi)
    |	(WRONG_PRED_ARITY x) =>  (["Wrong predicate arity: "^(print_parity x)],phi) 
    |      (FOLLOW_UP) => ([],phi)
(*    |	_ => (["Internal error occured when resolving formula\n"^print_FORMULA phi],phi); *)


fun add_pos p s = perr p^s

fun ores_formula_label (is_goal:bool) (env:local_env1) (f, l) = 
   (map (add_pos (get_pos_FORMULA f)) (fst (ores_formula (is_goal:bool) env f)), 
    (snd (ores_formula (is_goal:bool) env f), l))

(* Compute the connected compenents of a graph which is given
   by a list of nodes and a boolean function indicating whether
   there is an egde between two nodes.
   For each node, the algorithm splits the connected components
   which have been computed so far into two categories:
   those which are connected to the node and those which are not.
   The former are all linked with @ in order to form a new connected
   component, the latter are left untouched. *)
     			 
fun compute_conn_components (edges:'a*'a->bool) (nodes:'a list):'a list list =
  let
    fun is_connected(node,nil) = false
      | is_connected(node,n::c) = 
          edges(node,n) orelse edges(n,node) orelse is_connected(node,c)
    fun split_components(node,nil,acc_comp_of_node,acc_other_comps) = 
    	(node::acc_comp_of_node)::acc_other_comps
      | split_components(node,current_comp::other_comps,acc_comp_of_node,acc_other_comps) =
        if is_connected(node,current_comp)
        then split_components(node,other_comps,current_comp@acc_comp_of_node,acc_other_comps)
        else split_components(node,other_comps,acc_comp_of_node,current_comp::acc_other_comps)
    fun add_node (node:'a,components:'a list list):'a list list =
        split_components(node,components,nil,nil)
  in
  foldr add_node (nodes,[])
  end 

(* Compute the equivalence classes of the equivalence closures of leqF and leqP resp.
   and store them in a table indexed by function and predicate names, resp.
   This is needed when checking if terms or predications are equivalent, since
   this equivalence is defined in terms of  the equivalence closures of leqF and leqP resp. *)


     			 
fun get_conn_components (env:local_env) : local_env1 =
	let
		val (srts,vars,funs,preds) = env
	in
		(env,(Symtab_id.map (compute_conn_components (leqF env)) funs ,
		      Symtab_id.map (compute_conn_components (leqP env)) preds) )
	end


fun quantify axioms (srts,vars,funs,preds) =
let
fun mk_var_decl (v,s) = ([v],s);
fun quantify1 nil (ax,l) = (ax,l)
  | quantify1 vars (sort_gen_ax x,l) = (sort_gen_ax x,l)
  | quantify1 vars (ax,l) =
	(quantification(forall,map mk_var_decl vars,ax),l) 
in
	map (quantify1 (Symtab_sid.dest vars)) axioms
end

fun overload_resolution (is_goal:bool) (env:local_env,axioms:L_FORMULA list):(string list * L_FORMULA list) =
  err_propagate_list (fn x=>x) (map (ores_formula_label is_goal (get_conn_components env)) 
           (if is_goal then axioms else quantify axioms env))
  handle (UNDECLARED_SORT s) => (["Undeclared sort: "^ (print_SORTS1 [s])],[])

fun overload_resolution_TERM  (env:local_env,t:TERM) =
  MinExpTerm true (get_conn_components env) t

end;
