(* *********************************************************************** *)
(*									   *)
(* Project : CATS 							   *)
(* Author: Till Mossakowski                         			   *)
(* Date: 15.08.2001    				 			   *)
(* Purpose of this file: Encodings at the structured level     		   *)
(*			 						   *)
(*									   *)
(* *********************************************************************** *)

(*
   The encoding works in the following steps:
   1. Elimination of translation (by performing the renaming)
      Renamed Named specs are flattened!
   2. Elimination of hiding by shifting it outside (using pushouts)
   3. Normalization of union and extension
      (needed for Isabelle, which can unite and extend just named specs)

   The steps have to be performed in this order (but one can stop anywhere).

  todo:
  Instead of flattening renamed specs, copy them
  Pushout renamings in hide-elimination
*)

structure StructEncode :
sig
val eliminate_translation : GlobalEnv.global_env  -> GlobalEnv.global_env
val eliminate_translation_slenv : 
               AS.SPEC_NAME -> GlobalEnv.spec_lenv -> GlobalEnv.spec_lenv
val eliminate_union : GlobalEnv.global_env  -> GlobalEnv.global_env
val eliminate_union_slenv : 
               AS.SPEC_NAME -> GlobalEnv.spec_lenv -> GlobalEnv.spec_lenv
val eliminate_hiding : GlobalEnv.global_env  -> GlobalEnv.global_env
val eliminate_hiding_slenv : 
               AS.SPEC_NAME -> GlobalEnv.spec_lenv -> GlobalEnv.spec_lenv
end

= struct

infix mem
open AS Utils LocalEnv GlobalEnv BasicPrint

fun write_err sp_name msg =
    writeln ("*** Error in translation of "^print_SIMPLE_ID sp_name^"\n"^msg)

fun write_warn sp_name msg =
    writeln ("*** Warning in translation of "^print_SIMPLE_ID sp_name^"\n"^msg)

(*********** Elimination of translation *****************)
(* It is assumed that the pushing hiding outside already has been done ! *)

fun rename_sign [] s = s
  | rename_sign (m::mors) s = rename_sign mors (Rename.rename_sign m s)

fun rename_ax1 (s,m) = Rename.rename_axs m s

fun eliminate_translation2 sp_name mors (senv,an) =
    (eliminate_translation1 sp_name mors senv,an)

and eliminate_translation1 sp_name mors (basic_env(Sigma,ax)) = 
    basic_env(rename_sign mors Sigma,
              Utils.foldl rename_ax1 (ax,mors))
  | eliminate_translation1 sp_name mors (translate_env(senv,mor)) =
    eliminate_translation1 sp_name (mor::mors) senv 
  | eliminate_translation1 sp_name mors (derive_env(senv,mor)) = 
    (write_err sp_name "Translation-elimination does not work for hidings."; raise ERROR)
  | eliminate_translation1 sp_name mors (union_env senvs) = 
    union_env (map (eliminate_translation1 sp_name mors) senvs) 
  | eliminate_translation1 sp_name mors (extension_env senvs) =
    extension_env (map (eliminate_translation2 sp_name mors) senvs)
  | eliminate_translation1 sp_name mors (free_spec_env senv) =
    free_spec_env (eliminate_translation1 sp_name mors senv) 
  | eliminate_translation1 sp_name mors (cofree_spec_env senv) =
    cofree_spec_env (eliminate_translation1 sp_name mors senv) 
  | eliminate_translation1 sp_name mors (closed_spec_env senv) = 
    closed_spec_env (eliminate_translation1 sp_name mors senv)
  | eliminate_translation1 sp_name mors (senv1 as (spec_inst_env (name,senv,mor,senvs))) =
    (* spec_inst_env (name,  
                   eliminate_translation1 sp_name (mor::mors) senv, 
                   empty_morphism,
                   map (eliminate_translation1 sp_name mors) senvs)*)
    if null mors andalso eq_mor(mor,empty_morphism)
    then senv1  (* do not create a new instantiation *)
    else (* if there is a translation, we cannot re-use the spec at all.
            Instead, we just include its renamed flattened version.
            Better would be to create a renamed copy following the structure of the spec,
            cf. Meseguer's TCS paper *)
    eliminate_translation1 sp_name mors (basic_env (Flatten.flatten_senv senv1))
  | eliminate_translation1 sp_name mors dummy_spec_env = dummy_spec_env


fun eliminate_translation_senv sp_name senv = eliminate_translation1 sp_name [] senv

fun eliminate_translation_slenv sp_name (SPEC_ENV(Sigma1,Sigma2,senv)) =
    SPEC_ENV(Sigma1,Sigma2,eliminate_translation_senv sp_name senv)


fun elinimate_translation_gen_env sp_name (slenv,slenvs,Sigma) =
    (eliminate_translation_slenv sp_name slenv,
     map (eliminate_translation_slenv sp_name) slenvs,
     Sigma)

fun elinimate_translation_ge (sp_name,spec_defn_env (gen_env,slenv)) =
    spec_defn_env(elinimate_translation_gen_env sp_name gen_env,
                  eliminate_translation_slenv sp_name slenv)
  | elinimate_translation_ge (sp_name,view_defn_env (gen_env,slenv1,mor,slenv2)) =
    view_defn_env (elinimate_translation_gen_env sp_name gen_env,
                   eliminate_translation_slenv sp_name slenv1,
                   mor,
                   eliminate_translation_slenv sp_name slenv2)
  | elinimate_translation_ge (sp_name,_) = 
    (write_err sp_name "Translation of architectural and unit specs not defined"; raise ERROR)

fun eliminate_translation (genv,ans) =
    (Symtab_sid.keymap elinimate_translation_ge genv,ans)

(*************************************)
(* Elimination of hiding             *)
(*************************************)

(*
Transformations-Funktion zur Eliminierung von Hiding
   basic_env(sig,ax) => (sig,ax,empty_signature)
   translate_env(senv,sigma): rekursiv anwenden, ergibt (senv',Sigma,Sigma_h)
                              sigma' gibt den Symbolen in sigma(Sigma) geschnitten Sigma_h neue Namen
                         Ergebnis ist dann (translate_env(senv',sigma'),sigma(Sigma),sigma'(Sigma_h))
   derive_env(senv,sigma): rekursiv anwenden, ergibt (senv',Sigma,Sigma_h)
                           Sigma1=domain(sigma)
                  Ergebnis dann (senv',Sigma1,Sigma_1 vereinigt (Sigma ohne Sigma_1))
   union_env: rekursiv anwenden, ergibt (senv_i,Sigma_i,Sigma_h_i)
              sigma_i gibt den Symbolen in Sigma_h_i geschnitten (Vereinigung der Sigma_i) neue Namen
              Ergebnis: (union_env([ translate_env(senv_i,sigma_i) (i=1..n)],Vereinigung der Sigma_i,
                          Vereinigung der sigma_i(Sigma_h_i))
   extension_env: wie union (?)
   free_spec_env: Fehler
   closed_spec_env: rekursiv aufrufen
   spec_inst_env: SPEC_NAME bleibt, erstes spec_env + morphismus wie bei translate_env behandeln
                  dann dies mit den anderen spec_envs in union_env tun und rekursiv aufrufen.
   dummy_spec_env: Fehler
*)

fun contains_hiding (basic_env (Sigma,ax)) = false
  | contains_hiding (translate_env (senv,sigma)) =
    contains_hiding senv
  | contains_hiding (derive_env (senv,sigma)) = true
  | contains_hiding (union_env senvs) =
    Utils.exists' contains_hiding senvs
  | contains_hiding (extension_env senvs) =
    Utils.exists' contains_hiding (map fst senvs)
  | contains_hiding (free_spec_env senv) =
    contains_hiding senv
  | contains_hiding (cofree_spec_env senv) =
    contains_hiding senv
  | contains_hiding (closed_spec_env senv) =
    contains_hiding senv
  | contains_hiding (spec_inst_env (name,body,sigma,args)) =
    contains_hiding body orelse
    Utils.exists' contains_hiding args
  | contains_hiding dummy_spec_env = false 

fun eliminate_hiding2 sp_name names (senv,an) =
    let val (senv',Sigma,HSigma) = eliminate_hiding1 sp_name names senv
    in ((senv',an),Sigma,HSigma)
    end

and eliminate_hiding1 sp_name names (basic_env(Sigma,ax)) = 
    (basic_env(Sigma,ax),Sigma,empty_signature)
  | eliminate_hiding1 sp_name names (translate_env(senv,sigma)) =
    let val (senv',Sigma,HSigma) = eliminate_hiding1 sp_name names senv
        val sigma_Sigma = Rename.rename_sign sigma Sigma
        val sigma' = sigma
        val sigma'_HSigma = Rename.rename_sign sigma HSigma
    in (translate_env (senv',sigma'),sigma_Sigma,sigma'_HSigma)
    end 
  | eliminate_hiding1 sp_name names (derive_env(senv,mor)) = 
    let val (senv',Sigma,HSigma) = eliminate_hiding1 sp_name names senv
    in (senv',Sigma,HSigma)
    end
  | eliminate_hiding1 sp_name names (union_env senvs) = 
    let val senvs' = map (eliminate_hiding1 sp_name names) senvs
        val Sigma = signature_union_list (map #2 senvs')
        val HSigma = signature_union_list (map #3 senvs')
        val senv' = union_env (map #1 senvs')
    in (senv',Sigma,HSigma)
    end
  | eliminate_hiding1 sp_name names (extension_env senvs) =
    let val senvs' = map (eliminate_hiding2 sp_name names) senvs
        val Sigma = signature_union_list (map #2 senvs')
        val HSigma = signature_union_list (map #3 senvs')
        val senv' = extension_env (map #1 senvs')
    in (senv',Sigma,HSigma)
    end
  | eliminate_hiding1 sp_name names (free_spec_env senv) =
    if contains_hiding senv 
    then (write_err sp_name "Cannot eliminate hiding within free"; raise ERROR)
    else
    let val (senv',Sigma,HSigma) = eliminate_hiding1 sp_name names senv
    in (free_spec_env senv',Sigma,HSigma)
    end
  | eliminate_hiding1 sp_name names (cofree_spec_env senv) =
    if contains_hiding senv 
    then (write_err sp_name "Cannot eliminate hiding within cofree"; raise ERROR)
    else
    let val (senv',Sigma,HSigma) = eliminate_hiding1 sp_name names senv
    in (cofree_spec_env senv',Sigma,HSigma)
    end
  | eliminate_hiding1 sp_name names (closed_spec_env senv) = 
    let val (senv',Sigma,HSigma) = eliminate_hiding1 sp_name names senv
    in (closed_spec_env senv',Sigma,HSigma)
    end
  | eliminate_hiding1 sp_name names (spec_inst_env (name,senv,mor,senvs)) =
    let val (senv',Sigma,HSigma) = eliminate_hiding1 sp_name names senv
        val senvs' = map (#1 o eliminate_hiding1 sp_name names) senvs
    in (spec_inst_env (name,senv',mor,senvs'),empty_signature,empty_signature)
    end
  | eliminate_hiding1 sp_name names dummy_spec_env = (dummy_spec_env,empty_signature,empty_signature)


fun eliminate_hiding_senv sp_name senv = eliminate_hiding1 sp_name [] senv

(* Needs to be changed to Simga',Sigma_h' later !!!*)

fun eliminate_hiding_slenv sp_name (SPEC_ENV(Sigma,Sigma_h,senv)) =
    let val (senv',Sigma',Sigma_h') = eliminate_hiding_senv sp_name senv
    in SPEC_ENV(Sigma,Sigma_h,senv')
    end

fun elinimate_hiding_gen_env sp_name (slenv,slenvs,Sigma) =
    (eliminate_hiding_slenv sp_name slenv,
     map (eliminate_hiding_slenv sp_name) slenvs,
     Sigma)

fun elinimate_hiding_ge (sp_name,spec_defn_env (gen_env,slenv)) =
    spec_defn_env(elinimate_hiding_gen_env sp_name gen_env,
                  eliminate_hiding_slenv sp_name slenv)
  | elinimate_hiding_ge (sp_name,view_defn_env (gen_env,slenv1,mor,slenv2)) =
    view_defn_env (elinimate_hiding_gen_env sp_name gen_env,
                   eliminate_hiding_slenv sp_name slenv1,
                   mor,
                   eliminate_hiding_slenv sp_name slenv2)
  | elinimate_hiding_ge (sp_name,_) = 
    (write_err sp_name "Translation of architectural and unit specs not defined"; raise ERROR)

fun eliminate_hiding (genv,ans) =
    (Symtab_sid.keymap elinimate_hiding_ge genv,ans)



(* Eleminate unions and extensions by
   rearranging an env without translate and derive
   such that all spec_insts are moved to the front
   (this is possible since these are self-contained)
   and such that the rest is flattenend to one basic spec
   (except the %implied parts).
   The resulting form is:
   extension (union (all the spec-insts) basic-spec implied-basic-spec)
   where the actual parameters of the spec-insts
   are transformed to the same format.
*)

fun get_sig (extension_env [_,(basic_env(Sigma,ax),_),_]) =
    Sigma
fun get_ax (extension_env [_,(basic_env(Sigma,ax),_),_]) =
    ax
fun get_implied_sig (extension_env [_,_,(basic_env(Sigma,ax),_)]) =
    Sigma
fun get_implied_ax (extension_env [_,_,(basic_env(Sigma,ax),_)]) =
    ax
fun get_sp_list (extension_env [(union_env envs,_),_,_]) =
    envs

fun contains_imp nil = false
  | contains_imp (implies::_) = true
  | contains_imp (pos_ANNO (_,an)::rest) = contains_imp (an::rest) 
  | contains_imp (_::rest) = contains_imp rest

fun is_imp (_,ans) = contains_imp ans

fun eliminate_union_senv sp_name (basic_env (Sigma,ax)) = 
    extension_env [(union_env [],[]),
                   (basic_env(Sigma,ax),[]),
                   (basic_env(empty_signature,[]),[implies])]
  | eliminate_union_senv sp_name (translate_env (senv,sigma)) =
    (write_err sp_name "Union elimination does not work for translations"; raise ERROR)
  | eliminate_union_senv sp_name (derive_env (senv,sigma)) =
    (write_err sp_name "Union elimination does not work for hidings"; raise ERROR)
  | eliminate_union_senv sp_name (union_env senvs) =
    let val norm_envs = map (eliminate_union_senv sp_name) senvs
        val sigs = map get_sig norm_envs
        val axs = map get_ax norm_envs
        val imp_sigs = map get_implied_sig norm_envs
        val imp_axs = map get_implied_ax norm_envs
        val sps = flat (map get_sp_list norm_envs)
        val benv = basic_env(signature_union_list sigs,
                             flat axs)
        val imp_benv = basic_env(signature_union_list imp_sigs,
                             flat imp_axs)
    in extension_env [(union_env sps,[]),
                      (benv,[]),
                      (imp_benv,[implies])]
    end
  | eliminate_union_senv sp_name (extension_env senvs) =
    let val specified_envs = map (eliminate_union_senv sp_name o fst) (filter_out is_imp senvs)
        val imp_envs = map (eliminate_union_senv sp_name o fst) (filter is_imp senvs)
        val sigs = map get_sig specified_envs
        val axs = map get_ax specified_envs
        val imp_sigs = map get_implied_sig (specified_envs@imp_envs)
                       @map get_sig imp_envs
        val imp_axs = map get_implied_ax (specified_envs@imp_envs)
                       @map get_ax imp_envs
        val sps = flat (map get_sp_list specified_envs)

        val imp_sps = flat (map get_sp_list imp_envs)
        val _ = if null imp_sps then () 
                else (write_err sp_name "Implied parts may not contain named specifications"; raise ERROR)
        val benv = basic_env(signature_union_list sigs,
                             flat axs)
        val imp_benv = basic_env(signature_union_list imp_sigs,
                             flat imp_axs)
    in extension_env [(union_env sps,[]),
                      (benv,[]),
                      (imp_benv,[implies])]
    end
  | eliminate_union_senv sp_name (free_spec_env senv) =
    (write_warn sp_name "free {sp} will be reduced to sp";
     eliminate_union_senv sp_name senv)
  | eliminate_union_senv sp_name (cofree_spec_env senv) =
    (write_warn sp_name "cofree {sp} will be reduced to sp";
     eliminate_union_senv sp_name senv)
  | eliminate_union_senv sp_name (closed_spec_env senv) =
    eliminate_union_senv sp_name senv
  | eliminate_union_senv sp_name (spec_inst_env (name,body,sigma,args)) =
    let val extension_env [(union_env sps,_),benv,imp_benv] =
            eliminate_union_senv sp_name (union_env args)
    in extension_env [(union_env (spec_inst_env (name,body,empty_morphism,[])::sps),[]),
                      benv,
                      imp_benv]
    end

fun eliminate_union_slenv sp_name (SPEC_ENV(Sigma,HSigma,senv)) =
    SPEC_ENV(Sigma,HSigma,eliminate_union_senv sp_name senv)

fun elinimate_union_gen_env sp_name (slenv,slenvs,Sigma) =
    (eliminate_union_slenv sp_name slenv,
     map (eliminate_union_slenv sp_name) slenvs,
     Sigma)

fun elinimate_union_ge (sp_name,spec_defn_env (gen_env,slenv)) =
    spec_defn_env(elinimate_union_gen_env sp_name gen_env,
                  eliminate_union_slenv sp_name slenv)
  | elinimate_union_ge (sp_name,view_defn_env (gen_env,slenv1,mor,slenv2)) =
    view_defn_env (elinimate_union_gen_env sp_name gen_env,
                   eliminate_union_slenv sp_name slenv1,
                   mor,
                   eliminate_union_slenv sp_name slenv2)
  | elinimate_union_ge (sp_name,_) = 
    (write_err sp_name "Translation of architectural and unit specs not defined"; raise ERROR)

fun eliminate_union (genv,ans) =
    (Symtab_sid.keymap elinimate_union_ge genv,ans)
end
