(* ----------------------------------------------------------------------- *)
(* xml_dtd.sml                            Pascal Schmidt <pharao90@tzi.de> *)
(* ----------------------------------------------------------------------- *)
(* Internal representation of the CASL DTD                                 *)
(* ----------------------------------------------------------------------- *)

(* $Id: xml_dtd.sml,v 1.1 2004/02/13 11:29:21 7till Exp $ *)

structure xml_dtd
=
struct

  open xml_types

  val current_ver = "$Id: xml_dtd.sml,v 1.1 2004/02/13 11:29:21 7till Exp $"

  (* --------------------------------------------------------------------- *)
  (* Exceptions and types                                                  *)
  (* --------------------------------------------------------------------- *)


  exception EmptyList

  datatype elem_case = single of string
                     | optional of string
                     | star_list of string
                     | plus_list of string
                     | verbatim
                     | empty

  type xml_case = string * elem_case list

  fun is_emp (_,[empty]) = true
    | is_emp _           = false

  fun get_tag (s,_) = s

  (* --------------------------------------------------------------------- *)
  (* DTD as list                                                           *)
  (* --------------------------------------------------------------------- *)

  val dtd_elems = [ ("casl",[single "lib-defn"]),
                    ("casl",[single "casenv"]),
                    ("casl",[single "fcasenv"]),
                    ("casl",[single "library"]),
                    ("casl",[single "lib-item"]),
                    
                    ("region",[single "srcpos",single "srcpos"]),
                    
                    ("srcpos",[single "int",single "int"]),
                    
                    ("int",[verbatim]),
                    
                    ("library",[single "lib-name",star_list "anno", single
                                "internal-items",single "external-items"]),
                    
                    ("internal-items",[star_list "lib-item-path"]),
                    
                    ("external-items",[star_list "lib-item-path"]),
                    
                    ("lib-item-path",[single "lib-item",single "path"]),
                    
                    ("lib-defn",[single "lib-name",star_list "lib-item",
                                 star_list "anno"]),
                    ("lib-defn",[single "lib-name",star_list "lib-item",
                                 star_list "anno",optional "casenv"]),
                    ("lib-defn",[single "lib-name",star_list "lib-item",
                                 star_list "anno",optional "fcasenv"]),
                    ("lib-defn",[single "pos-lib-defn"]),
                    
                    ("pos-lib-defn",[single "region",single "lib-defn"]),
                                 
                    ("lib-name",[single "lib-id"]),
                    ("lib-name",[single "lib-version"]),
                    ("lib-name",[single "pos-lib-name"]),
                    
                    ("pos-lib-name",[single "region",single "lib-name"]),
                    
                    ("lib-id",[single "direct-link"]),
                    ("lib-id",[single "indirect-link"]),
                    ("lib-id",[single "pos-lib-id"]),
                    
                    ("pos-lib-id",[single "region",single "lib-id"]),
                    
                    ("lib-version",[single "lib-id",single "version"]),
                    ("lib-version",[single "pos-lib-version"]),
                    
                    ("pos-lib-version",[single "region",single "lib-version"]),

                    ("direct-link",[single "url"]),
                    
                    ("indirect-link",[single "path"]),
                    
                    ("url",[verbatim]),
                    
                    ("path",[verbatim]),
                    
                    ("version",[plus_list "number"]),
                    
                    ("number",[verbatim]),
                    
                    ("lib-item",[single "spec-defn"]),
                    ("lib-item",[single "view-defn"]),
                    ("lib-item",[single "arch-spec-defn"]),
                    ("lib-item",[single "unit-spec-defn"]),
                    ("lib-item",[single "download-items"]),
                    ("lib-item",[single "pos-lib-item"]),
                    
                    ("pos-lib-item",[single "region",single "lib-item"]),
                    
                    ("spec-defn",[single "spec-name",single "genericity",
                                  single "anno-spec",star_list "anno"]),
                                  
                    ("spec-name",[single "simple-id"]),
                    
                    ("genericity",[single "params",single "imports"]),
                    ("genericity",[single "pos-genericity"]),
                    
                    ("pos-genericity",[single "region",single "genericity"]),
                    
                    ("anno-spec",[single "spec",star_list "anno"]),
                    
                    ("params",[star_list "anno-spec"]),
                    ("params",[single "pos-params"]),
                    
                    ("pos-params",[single "region",single "params"]),
                    
                    ("imports",[star_list "anno-spec"]),
                    ("imports",[single "pos-imports"]),
                    
                    ("pos-imports",[single "region",single "imports"]),
                    
                    ("spec",[single "basic"]),
                    ("spec",[single "translation"]),
                    ("spec",[single "reduction"]),
                    ("spec",[single "union"]),
                    ("spec",[single "extension"]),
                    ("spec",[single "free-spec"]),
                    ("spec",[single "cofree-spec"]),
                    ("spec",[single "local-spec"]),
                    ("spec",[single "closed-spec"]),
                    ("spec",[single "spec-inst"]),
                    ("spec",[single "pos-spec"]),
                    
                    ("pos-spec",[single "region",single "bool-data",
                                 single "spec"]),
                    
                    ("basic",[single "basic-spec"]),
                    
                    ("translation",[single "anno-spec",single "renaming",
                                    star_list "anno"]),
                    
                    ("reduction",[single "anno-spec",single "restriction",
                                  star_list "anno"]),
                                  
                    ("union",[plus_list "anno-anno-spec"]),
                    
                    ("extension",[plus_list "anno-anno-spec"]),
                    
                    ("anno-anno-spec",[single "anno-spec",star_list "anno"]),
                    
                    ("free-spec",[single "anno-spec",star_list "anno"]),

                    ("cofree-spec",[single "anno-spec",star_list "anno"]),
                    
                    ("local-spec",[single "anno-spec",star_list "anno",
                                   single "anno-spec",star_list "anno"]),
                    
                    ("closed-spec",[single "anno-spec",star_list "anno"]),
                    
                    ("spec-inst",[single "spec-name",star_list "fit-arg"]),
                    
                    ("renaming",[plus_list "symb-map-items"]),
                    ("renaming",[single "pos-renaming"]),
                    
                    ("pos-renaming",[single "region",single "renaming"]),
                    
                    ("symb-map-items",[single "symb-kind",plus_list
                                       "symb-or-map"]),
                    ("symb-map-items",[single "pos-symb-map-items"]),
                    
                    ("pos-symb-map-items",[single "region",single
                                           "symb-map-items"]),
                    
                    ("symb-kind",[single "implicit"]),
                    ("symb-kind",[single "sort-kind"]),
                    ("symb-kind",[single "op-kind"]),
                    ("symb-kind",[single "pred-kind"]),
                    ("symb-kind",[single "pos-symb-kind"]),
                    
                    ("pos-symb-kind",[single "region",single "symb-kind"]),
                    
                    ("implicit",[empty]),
                    
                    ("sort-kind",[empty]),
                    
                    ("op-kind",[empty]),
                    
                    ("pred-kind",[empty]),
                    
                    ("symb-or-map",[single "symb"]),
                    ("symb-or-map",[single "symb-map"]),
                    ("symb-or-map",[single "pos-symb-or-map"]),
                    
                    ("pos-symb-or-map",[single "region",single "symb-or-map"]),
                    
                    ("symb",[single "id"]),
                    ("symb",[single "qual-id"]),
                    ("symb",[single "pos-symb"]),
                    
                    ("pos-symb",[single "region",single "symb"]),
                    
                    ("qual-id",[single "id",single "type"]),
                    
                    ("type",[single "op-type"]),
                    ("type",[single "pred-type"]),
                    ("type",[single "pos-type"]),
                    
                    ("pos-type",[single "region",single "type"]),
                    
                    ("symb-map",[single "symb",single "symb"]),
                    ("symb-map",[single "pos-symb-map"]),
                    
                    ("pos-symb-map",[single "region",single "symb-map"]),
                    
                    ("restriction",[single "hide"]),
                    ("restriction",[single "reveal"]),
                    ("restriction",[single "pos-restriction"]),
                    
                    ("pos-restriction",[single "region",single "restriction"]),
                    
                    ("hide",[plus_list "symb-items"]),
                    
                    ("symb-items",[single "symb-kind",plus_list "symb"]),
                    ("symb-items",[single "pos-symb-items"]),
                    
                    ("pos-symb-items",[single "region",single "symb-items"]),
                    
                    ("reveal",[plus_list "symb-map-items"]),
                    
                    ("fit-arg",[single "fit-spec"]),
                    ("fit-arg",[single "fit-view"]),
                    ("fit-arg",[single "pos-fit-arg"]),
                    
                    ("pos-fit-arg",[single "region",single "fit-arg"]),
                    
                    ("fit-spec",[single "anno-spec",star_list
                                 "symb-map-items"]),
                    
                    ("fit-view",[single "view-name",star_list "fit-arg"]),
                    
                    ("view-defn",[single "view-name",single "genericity",
                                 single "view-type",star_list "symb-map-items",
                                 star_list "anno"]),
                                 
                    ("view-name",[single "simple-id"]),
                    
                    ("view-type",[single "anno-spec",single "anno-spec"]),
                    ("view-type",[single "pos-view-type"]),
                    
                    ("pos-view-type",[single "region",single "view-type"]),
                    
                    ("basic-spec",[star_list "basic-items"]),
                    ("basic-spec",[single "pos-basic-spec"]),
                    
                    ("pos-basic-spec",[single "region",single "basic-spec"]),
                    
                    ("basic-items",[single "sig-items"]),
                    ("basic-items",[single "free-datatype"]),
                    ("basic-items",[single "cofree-datatype"]),
                    ("basic-items",[single "sort-gen"]),
                    ("basic-items",[single "sort-cogen"]),
                    ("basic-items",[single "var-items"]),
                    ("basic-items",[single "local-var-axioms"]),
                    ("basic-items",[single "axiom-items"]),
                    ("basic-items",[single "pos-basic-items"]),
                    
                    ("pos-basic-items",[single "region",single "basic-items"]),
                    
                    ("sig-items",[single "sort-items"]),
                    ("sig-items",[single "op-items"]),
                    ("sig-items",[single "pred-items"]),
                    ("sig-items",[single "datatype-items"]),
                    ("sig-items",[single "pos-sig-items"]),
                    
                    ("pos-sig-items",[single "region",single "sig-items"]),
                    
                    ("sort-items",[plus_list "anno-sort-item",star_list
                                   "anno"]),
                    
                    ("anno-sort-item",[single "sort-item",star_list "anno"]),
                    
                    ("sort-item",[single "sort-decl"]),
                    ("sort-item",[single "subsort-decl"]),
                    ("sort-item",[single "subsort-defn"]),
                    ("sort-item",[single "iso-decl"]),
                    ("sort-item",[single "pos-sort-item"]),
                    
                    ("pos-sort-item",[single "region",single "sort-item"]),
                    
                    ("sort-decl",[plus_list "sort"]),
                    
                    ("sort",[single "id"]),
                    
                    ("sorts",[star_list "sort"]),
                    ("sorts",[single "pos-sorts"]),
                    
                    ("pos-sorts",[single "region",single "sorts"]),
                    
                    ("subsort-decl",[single "sorts",single "sort"]),
                    
                    ("subsort-defn",[single "sort",single "var",single
                                     "sort",single "formula",
                                     star_list "anno"]),
                    
                    ("iso-decl",[plus_list "sort"]),
                    
                    ("op-items",[plus_list "anno-op-item",star_list "anno"]),
                    
                    ("anno-op-item",[single "op-item",star_list "anno"]),
                    
                    ("op-item",[single "op-decl"]),
                    ("op-item",[single "op-defn"]),
                    ("op-item",[single "pos-op-item"]),
                    
                    ("pos-op-item",[single "region",single "op-item"]),
                    
                    ("op-decl",[plus_list "op-name",single "op-type",
                                star_list "op-attr"]),
                    
                    ("op-defn",[single "op-name",single "op-head", single
                                "term",star_list "anno"]),
                    
                    ("op-name",[single "id"]),
                    
                    ("op-type",[single "total-op-type"]),
                    ("op-type",[single "partial-op-type"]),
                    ("op-type",[single "pos-op-type"]),
                    
                    ("pos-op-type",[single "region",single "op-type"]),
                    
                    ("total-op-type",[single "sorts",single "sort"]),
                    
                    ("partial-op-type",[single "sorts",single "sort"]),
                    
                    ("op-head",[single "total-op-head"]),
                    ("op-head",[single "partial-op-head"]),
                    ("op-head",[single "pos-op-head"]),
                    
                    ("pos-op-head",[single "region",single "op-head"]),
                    
                    ("total-op-head",[star_list "arg-decl",single "sort"]),
                    
                    ("partial-op-head",[star_list "arg-decl",single "sort"]),
                    
                    ("op-symb",[single "op-name"]),
                    ("op-symb",[single "qual-op-name"]),
                    ("op-symb",[single "pos-op-symb"]),
                    
                    ("pos-op-symb",[single "region",single "op-symb"]),
                    
                    ("qual-op-name",[single "op-name",single "op-type"]),
                    
                    ("op-attr",[single "associative"]),
                    ("op-attr",[single "commutative"]),
                    ("op-attr",[single "idempotent"]),
                    ("op-attr",[single "unit-op-attr"]),
                    ("op-attr",[single "pos-op-attr"]),
                    
                    ("pos-op-attr",[single "region",single "op-attr"]),
                    
                    ("associative",[empty]),
                    
                    ("commutative",[empty]),
                    
                    ("idempotent",[empty]),
                    
                    ("unit-op-attr",[single "term"]),
                    
                    ("arg-decl",[plus_list "var",single "sort"]),
                    ("arg-decl",[single "pos-arg-decl"]),
                    
                    ("pos-arg-decl",[single "region",single "arg-decl"]),
                    
                    ("pred-items",[plus_list "anno-pred-item",star_list
                                   "anno"]),
                    
                    ("anno-pred-item",[single "pred-item",star_list "anno"]),
                    
                    ("pred-item",[single "pred-decl"]),
                    ("pred-item",[single "pred-defn"]),
                    ("pred-item",[single "pos-pred-item"]),
                    
                    ("pos-pred-item",[single "region",single "pred-item"]),
                    
                    ("pred-decl",[plus_list "pred-name",single "pred-type"]),
                    
                    ("pred-defn",[single "pred-name",single "pred-head",
                                  single "anno-formula",star_list "anno"]),
                                  
                    ("pred-name",[single "id"]),
                    
                    ("pred-type",[single "sorts"]),
                    ("pred-type",[single "pos-pred-type"]),
                    
                    ("pos-pred-type",[single "region",single "pred-type"]),
                    
                    ("pred-head",[star_list "arg-decl"]),
                    ("pred-head",[single "pos-pred-head"]),
                    
                    ("pos-pred-head",[single "region",single "pred-head"]),
                    
                    ("pred-symb",[single "pred-name"]),
                    ("pred-symb",[single "qual-pred-name"]),
                    ("pred-symb",[single "pos-pred-symb"]),
                    
                    ("pos-pred-symb",[single "region",single "pred-symb"]),
                    
                    ("qual-pred-name",[single "pred-name",single "pred-type"]),
                    
                    ("datatype-items",[plus_list "datatype-decl",star_list
                                       "anno"]),
                    
                    ("datatype-decl",[single "sort",plus_list
                                      "anno-alternative",star_list "anno"]),
                    ("datatype-decl",[single "pos-datatype-decl"]),
                    
                    ("pos-datatype-decl",[single "region", single
                                          "datatype-decl"]),
                    
                    ("anno-alternative",[single "alternative",star_list
                                         "anno"]),
                    
                    ("alternative",[single "total-construct"]),
                    ("alternative",[single "partial-construct"]),
                    ("alternative",[single "subsorts"]),
                    ("alternative",[single "pos-alternative"]),
                    
                    ("pos-alternative",[single "region",single "alternative"]),
                    
                    ("total-construct",[single "op-name",star_list
                                        "components"]),
                    
                    ("partial-construct",[single "op-name",star_list
                                          "components"]),
                    
                    ("subsorts",[plus_list "sort"]),
                    
                    ("components",[single "total-select"]),
                    ("components",[single "partial-select"]),
                    ("components",[single "sort"]),
                    ("components",[single "pos-components"]),
                    
                    ("pos-components",[single "region",single "components"]),
                    
                    ("total-select",[plus_list "op-name",single "sort"]),
                    
                    ("partial-select",[plus_list "op-name",single "sort"]),
                    
                    ("free-datatype",[plus_list "datatype-decl",star_list
                                      "anno"]),

                    ("cofree-datatype",[plus_list "datatype-decl",star_list
                                        "anno"]),

                    ("sort-gen",[plus_list "sig-items",star_list "anno"]),
                    
                    ("sort-cogen",[plus_list "sig-items",star_list "anno"]),
                    
                    ("var-items",[plus_list "var-decl",star_list "anno"]),
                    
                    ("var-decl",[plus_list "var",single "sort"]),
                    
                    ("var",[single "simple-id"]),
                    
                    ("local-var-axioms",[plus_list "var-decl",plus_list
                                         "anno-formula",star_list "anno"]),
                    
                    ("axiom-items",[plus_list "anno-formula",star_list
                                    "anno"]),
                                    
                    ("anno-formula",[single "formula",star_list "anno"]),
                    
                    ("formula",[single "quantification"]),
                    ("formula",[single "conjunction"]),
                    ("formula",[single "disjunction"]),
                    ("formula",[single "implication"]),
                    ("formula",[single "equivalence"]),
                    ("formula",[single "negation"]),
                    ("formula",[single "sort-gen-ax"]),
                    ("formula",[single "sort-cogen-ax"]),
                    ("formula",[single "sort-cofree-ax"]),
                    ("formula",[single "unparsed"]),
                    ("formula",[single "true"]),
                    ("formula",[single "false"]),
                    ("formula",[single "predication"]),
                    ("formula",[single "definedness"]),
                    ("formula",[single "existl-equation"]),
                    ("formula",[single "strong-equation"]),
                    ("formula",[single "membership"]),
                    ("formula",[single "pos-formula"]),
                    
                    ("pos-formula",[single "region",single "bool-data",
                                    single "formula"]),
                    
                    ("quantification",[single "quantifier",plus_list
                                       "var-decl",single "formula"]),
                    
                    ("pred-quantification",[single "quantifier",star_list
                                            "var-pred-type",single "formula"]),
                                       
                    ("var-pred-type",[single "var",single "pred-type"]),

                    ("quantifier",[single "forall"]),
                    ("quantifier",[single "exists"]),
                    ("quantifier",[single "exists-uniquely"]),
                    ("quantifier",[single "pos-quantifier"]),
                    
                    ("pos-quantifier",[single "region",single "quantifier"]),
                    
                    ("forall",[empty]),
                    ("exists",[empty]),
                    ("exists-uniquely",[empty]),
                    
                    ("conjunction",[plus_list "formula"]),
                    
                    ("disjunction",[plus_list "formula"]),
                    
                    ("implication",[single "formula",single "formula"]),
                    
                    ("equivalence",[single "formula",single "formula"]),
                    
                    ("negation",[single "formula"]),
                    
                    ("sort-gen-ax",[star_list "sort",star_list "op-symb"]),
                    
                    ("sort-cogen-ax",[star_list "sort",star_list "op-symb"]),
                    
                    ("sort-cofree-ax",[star_list "sort",star_list "op-symb"]),
                    
                    ("unparsed",[verbatim]),
                    
                    ("predication",[single "pred-symb",single "terms"]),
                    
                    ("definedness",[single "term"]),
                    
                    ("existl-equation",[single "term",single "term"]),
                    
                    ("strong-equation",[single "term",single "term"]),
                    
                    ("membership",[single "term",single "sort"]),
                    
                    ("terms",[star_list "term"]),
                    ("terms",[single "pos_terms"]),
                    
                    ("pos-terms",[single "region",single "terms"]),
                    
                    ("term",[single "simple-id"]),
                    ("term",[single "qual-var"]),
                    ("term",[single "application"]),
                    ("term",[single "sorted-term"]),
                    ("term",[single "cast"]),
                    ("term",[single "conditional"]),
                    ("term",[single "unparsed"]),
                    ("term",[single "pos-term"]),
                    
                    ("pos-term",[single "region",single "bool-data",
                                 single "term"]),
                    
                    ("qual-var",[single "var",single "sort"]),

                    ("application",[single "op-symb",single "terms"]),

                    ("sorted-term",[single "term",single "sort"]),

                    ("cast",[single "term",single "sort"]),
                    
                    ("conditional",[single "term",single "formula",single
                                    "term"]),
                    
                    ("id",[single "token-id"]),
                    ("id",[single "mixfix-id"]),
                    
                    ("token-id",[single "token"]),
                    ("token-id",[single "comp-token-id"]),
                    
                    ("comp-token-id",[single "token",plus_list "id"]),
                    
                    ("token",[single "words"]),
                    ("token",[single "signs"]),
                    ("token",[single "dot-words"]),
                    
                    ("mixfix-id",[single "token-places"]),
                    ("mixfix-id",[single "comp-mixfix-id"]),
                    
                    ("token-places",[plus_list "token-or-place"]),
                    
                    ("comp-mixfix-id",[single "token-places",plus_list "id"]),
                    
                    ("token-or-place",[single "token"]),
                    ("token-or-place",[single "place"]),
                    
                    ("place",[empty]),
                    
                    ("simple-id",[single "words"]),
                    
                    ("words",[verbatim]),
                    
                    ("signs",[verbatim]),
                    
                    ("dot-words",[verbatim]),
                    
                    ("arch-spec-defn",[single "arch-spec-name",single
                                       "anno-arch-spec",star_list "anno"]),
                    
                    ("arch-spec-name",[single "simple-id"]),
                    
                    ("anno-arch-spec",[single "arch-spec",star_list "anno"]),
                    
                    ("arch-spec",[single "basic-arch-spec"]),
                    ("arch-spec",[single "arch-spec-name"]),
                    ("arch-spec",[single "pos-arch-spec"]),
                    
                    ("pos-arch-spec",[single "region",single "arch-spec"]),
                    
                    ("basic-arch-spec",[plus_list "anno-unit-decl-defn",
                                        single "result-unit",star_list "anno"]),
                    
                    ("anno-unit-decl-defn",[single "unit-decl-defn",star_list
                                            "anno"]),
                    
                    ("unit-decl-defn",[single "unit-decl"]),
                    ("unit-decl-defn",[single "unit-defn"]),
                    ("unit-decl-defn",[single "pos-unit-decl-defn"]),
                    
                    ("pos-unit-decl-defn",[single "region",single
                                           "unit-decl-defn"]),
                    
                    ("unit-decl",[single "unit-name",single "unit-spec",
                                  single "unit-imports"]),
                    ("unit-decl",[single "pos-unit-decl"]),
                    
                    ("pos-unit-decl",[single "region",single "unit-decl"]),
                    
                    ("unit-name",[single "simple-id"]),
                    
                    ("unit-imports",[star_list "unit-term"]),
                    ("unit-imports",[single "pos-unit-imports"]),
                    
                    ("pos-unit-imports",[single "region",single
                                         "unit-imports"]),
                    
                    ("unit-defn",[single "unit-name",single
                                  "unit-expression"]),
                    ("unit-defn",[single "pos-unit-defn"]),
                    
                    ("pos-unit-defn",[single "region",single "unit-defn"]),
                    
                    ("unit-spec-defn",[single "spec-name",single "unit-spec",
                                       star_list "anno"]),
                    
                    ("anno-unit-spec",[single "unit-spec",star_list "anno"]),
                    
                    ("unit-spec",[single "unit-type"]),
                    ("unit-spec",[single "spec-name"]),
                    ("unit-spec",[single "anno-arch-spec"]),
                    ("unit-spec",[single "closed"]),
                    ("unit-spec",[single "pos-unit-spec"]),
                    
                    ("pos-unit-spec",[single "region",single "unit-spec"]),
                    
                    ("closed",[single "unit-spec"]),
                    
                    ("unit-type",[single "anno-specs",single "anno-spec"]),
                    ("unit-type",[single "pos-unit-type"]),
                    
                    ("pos-unit-type",[single "region",single "unit-type"]),
                    
                    ("anno-specs",[star_list "anno-spec"]),
                    
                    ("result-unit",[single "unit-expression",star_list
                                    "anno"]),
                    ("result-unit",[single "pos-result-unit"]),
                    
                    ("pos-result-unit",[single "region",single "result-unit"]),
                    
                    ("unit-expression",[star_list "unit-binding",single
                                        "unit-term"]),
                    ("unit-expression",[single "pos-unit-expression"]),
                    
                    ("pos-unit-expression",[single "region", single 
                                            "unit-expression"]),
                    
                    ("unit-binding",[single "unit-name",single "unit-spec"]),
                    ("unit-binding",[single "pos-unit-binding"]),
                    
                    ("pos-unit-binding",[single "region",single
                                         "unit-binding"]),
                    
                    ("unit-term",[single "unit-translation"]),
                    ("unit-term",[single "unit-reduction"]),
                    ("unit-term",[single "amalgamation"]),
                    ("unit-term",[single "local-unit"]),
                    ("unit-term",[single "unit-appl"]),
                    ("unit-term",[single "pos-unit-term"]),
                    
                    ("pos-unit-term",[single "region",single "bool-data",
				      single "unit-term"]),
                    
                    ("unit-translation",[single "unit-term",single
                                         "renaming"]),
                    
                    ("unit-reduction",[single "unit-term",single
                                       "restriction"]),
                    
                    ("amalgamation",[plus_list "unit-term"]),
                    
                    ("local-unit",[plus_list "unit-defn",single "unit-term"]),
                    
                    ("unit-appl",[single "unit-name",star_list
                                  "fit-arg-unit"]),
                    
                    ("fit-arg-unit",[single "unit-term",star_list
                                     "symb-map-items"]),
                    ("fit-arg-unit",[single "pos-fit-arg-unit"]),
                    
                    ("pos-fit-arg-unit",[single "region",single
                                         "fit-arg-unit"]),
                    
                    ("download-items",[single "lib-name",plus_list
                                       "item-name-or-map",star_list "anno"]),
                    
                    ("item-name-or-map",[single "item-name"]),
                    ("item-name-or-map",[single "item-name-map"]),
                    ("item-name-or-map",[single "pos-item-name-or-map"]),
                    
                    ("pos-item-name-or-map",[single "region",single
                                             "item-name-or-map"]),
                    
                    ("item-name",[single "simple-id"]),
                    
                    ("item-name-map",[single "item-name",single "item-name"]),
                    
                    ("anno",[single "comment-anno"]),
                    ("anno",[single "comment-line-anno"]),
                    ("anno",[single "label-anno"]),
                    ("anno",[single "unparsed-anno"]),
                    ("anno",[single "number-anno"]),
                    ("anno",[single "floating-anno"]),
                    ("anno",[single "string-anno"]),
                    ("anno",[single "list-anno"]),
                    ("anno",[single "display-anno"]),
                    ("anno",[single "prec-anno"]),
                    ("anno",[single "lassoc-anno"]),
                    ("anno",[single "rassoc-anno"]),
                    ("anno",[single "conservative-anno"]),
                    ("anno",[single "definitional-anno"]),
                    ("anno",[single "implies-anno"]),
                    ("anno",[single "pos-anno"]),
                    
                    ("pos-anno",[single "region",single "anno"]),
                    
                    ("comment-anno",[verbatim]),
                    
                    ("comment-line-anno",[verbatim]),
                    
                    ("unparsed-anno",[verbatim]),
                    
                    ("label-anno",[single "id"]),
                    
                    ("number-anno",[single "id"]),
                    
                    ("floating-anno",[single "id",single "id"]),
                    
                    ("string-anno",[single "id",single "id"]),
                    
                    ("list-anno",[single "id",single "id",single "id"]),
                    
                    ("display-anno",[single "id",single
                                     "display-anno-content"]),
                    
                    ("display-anno-content",[verbatim]),
                    
                    ("prec-anno",[single "bool-data",single "id-list",
                                  single "id-list"]),
                    
                    ("bool-data",[single "true"]),
                    ("bool-data",[single "false"]),
                    
                    ("true",[empty]),
                    
                    ("false",[empty]),
                    
                    ("id-list",[star_list "id"]),
                    
                    ("lassoc-anno",[plus_list "id"]),
                    
                    ("rassoc-anno",[plus_list "id"]),
                    
                    ("conservative-anno",[empty]),
                    
                    ("definitional-anno",[empty]),
                    
                    ("implies-anno",[empty]),
                    
                    ("casenv",[star_list "sid-global-entry",star_list "anno"]),
                    
                    ("sid-global-entry",[single "simple-id",single
                                         "global-entry"]),
                                         
                    ("global-entry",[single "spec-defn-env"]),
                    ("global-entry",[single "view-defn-env"]),
                    ("global-entry",[single "arch-spec-defn-env"]),
                    ("global-entry",[single "unit-spec-defn-env"]),
                    
                    ("spec-defn-env",[single "genericity-env",single
                                      "spec-lenv"]),
                    
                    ("view-defn-env",[single "genericity-env",single
                                      "spec-lenv",single "morphism",single
                                      "spec-lenv"]),
                                      
                    ("arch-spec-defn-env",[single "arch-sig"]),
                    
                    ("unit-spec-defn-env",[single "unit-sig"]),
                    
                    ("genericity-env",[single "spec-lenv",star_list
                                       "spec-lenv",single "sign"]),
                    
                    ("spec-lenv",[single "sign",single "sign",single
                                  "spec-env"]),
                    
                    ("spec-env",[single "basic-env"]),
                    ("spec-env",[single "translate-env"]),
                    ("spec-env",[single "derive-env"]),
                    ("spec-env",[single "union-env"]),
                    ("spec-env",[single "extension-env"]),
                    ("spec-env",[single "free-spec-env"]),
                    ("spec-env",[single "cofree-spec-env"]),
                    ("spec-env",[single "closed-spec-env"]),
                    ("spec-env",[single "spec-inst-env"]),
                    ("spec-env",[single "dummy-spec-env"]),
                    
                    ("basic-env",[single "sign",star_list "anno-formula"]),
                    
                    ("translate-env",[single "spec-env",single "morphism"]),
                    
                    ("derive-env",[single "spec-env",single "morphism"]),
                    
                    ("union-env",[star_list "spec-env"]),
                    
                    ("extension-env",[star_list "spec-env"]),
                    
                    ("free-spec-env",[single "spec-env"]),
                    
                    ("cofree-spec-env",[single "spec-env"]),
                    
                    ("closed-spec-env",[single "spec-env"]),
                    
                    ("spec-inst-env",[single "spec-name",single "spec-env",
                                      single "morphism",star_list "spec-env"]),
                    
                    ("dummy-spec-env",[empty]),
                    
                    ("morphism",[single "sort-map",single "fun-map",single
                                 "pred-map"]),
                    
                    ("sort-map",[star_list "id-sort"]),
                    
                    ("id-sort",[single "id",single "sort"]),
                    
                    ("fun-map",[star_list "id-op-type-name"]),
                    
                    ("op-type-name",[single "op-type",single "op-name",
                                     single "bool-data"]),
                    
                    ("id-op-type-name",[single "id",star_list "op-type-name"]),
                    
                    ("pred-map",[star_list "id-pred-type-name"]),
                    
                    ("pred-type-name",[single "pred-type",single "pred-name"]),
                    
                    ("id-pred-type-name",[single "id",star_list
                                          "pred-type-name"]),
                                          
                    ("ext-signature",[single "sign",single "sign",star_list
                                      "anno-formula"]),
                    
                    ("fcasenv",[star_list "sid-f-global-entry",star_list
                                "anno"]),
                    
                    ("sid-f-global-entry",[single "simple-id",single
                                           "f-global-entry"]),
                    
                    ("f-global-entry",[single "f-spec-defn-env"]),
                    ("f-global-entry",[single "f-view-defn-env"]),
                    ("f-global-entry",[single "f-arch-spec-defn-env"]),
                    ("f-global-entry",[single "f-unit-spec-defn-env"]),
                    
                    ("f-spec-defn-env",[single "f-genericity-env",single
                                        "ext-signature"]),
                    
                    ("f-view-defn-env",[single "f-genericity-env",single
                                        "ext-signature",single "morphism",
                                        single "ext-signature"]),
                    
                    ("f-arch-spec-defn-env",[empty]),
                    
                    ("f-unit-spec-defn-env",[empty]),
                    
                    ("f-genericity-env",[single "ext-signature",single
                                         "ext-signatures"]),
                    
                    ("ext-signatures",[star_list "ext-signature"]),
                    
                    ("local-env",[single "subsort-env",single "var-env",
                                  single "fun-env",single "pred-env"]),
                    
                    ("sort-env",[star_list "id"]),
                    
                    ("subsort-env",[star_list "id-sorts"]),
                    
                    ("id-sorts",[single "id",star_list "sort"]),
                    
                    ("var-env",[star_list "sid-sort"]),
                    
                    ("sid-sort",[single "simple-id",single "sort"]),
                    
                    ("fun-env",[star_list "id-op-types"]),
                    
                    ("id-op-types",[single "id", star_list "op-type"]),
                    
                    ("pred-env",[star_list "id-pred-types"]),
                    
                    ("id-pred-types",[single "id",star_list "pred-type"]),
                    
                    ("sign",[single "local-env"]),
                    
                    ("arch-sig",[single "named-unit-sigs",single "unit-sig"]),
                    
                    ("unit-sig",[single "comp-sub-sigs",single "sub-sig"]),
                    
                    ("comp-sub-sigs",[star_list "sub_sig"]),
                    
                    ("named-unit-sigs",[star_list "sid-unit-tag-sig"]),
                    
                    ("sid-unit-tag-sig",[single "simple-id",single
                                         "unit-tag-sig"]),
                    
                    ("unit-tag-sig",[single "comp-tag-sigs",single "tag-sig"]),
                    
                    ("comp-tag-sigs",[star_list "tag-sig"]),
                    
                    ("tag-sig",[single "sub-sig",single "tagging-map"]),
                    
                    ("tagging-map",[star_list "e-ref-sym"]),
                    
                    ("e-ref-sym",[single "e-symbol",star_list "ref-sym"]),
                    
                    ("ref-sym",[single "unit-name",single "e-symbol"]),
                    
                    ("e-symbol",[single "sym-esymbol"]),
                    ("e-symbol",[single "emb-esymbol"]),
                    
                    ("sym-esymbol",[single "symbol"]),
                    
                    ("emb-esymbol",[single "sort",single "sort"]),
                    
                    ("sub-sig",[single "local-env"]),
                    
                    ("symbol",[single "sort-symbol"]),
                    ("symbol",[single "total-fun-symbol"]),
                    ("symbol",[single "partial-fun-symbol"]),
                    ("symbol",[single "pred-symbol"]),
                    
                    ("sort-symbol",[single "sort"]),
                    
                    ("total-fun-symbol",[single "op-name",single "sorts",
                                         single "sort"]),
                    
                    ("partial-fun-symbol",[single "op-name",single "sorts",
                                           single "sort"]),
                    
                    ("pred-symbol",[single "pred-name",star_list "sort"])
                  ]


  val dtd_empty_elems = List.map get_tag (List.filter is_emp dtd_elems)

  (* --------------------------------------------------------------------- *)
  (* DTD as table                                                          *)
  (* --------------------------------------------------------------------- *)


  structure symtab_str     = TableFun(type key = string
                                      val ord = xml_share.str_ord);

  val dtd_elems_table = symtab_str.make_multi dtd_elems


  (* --------------------------------------------------------------------- *)
  (* DTD text generation                                                   *)
  (* --------------------------------------------------------------------- *)

  
  fun tail []     = []
    | tail (h::t) = t

  fun get_tail l = (map tail) l

  fun get_first []          = raise EmptyList
    | get_first ([]::l)     = get_first l
    | get_first ((s::l)::m) = s

  fun last_elem l = (List.length(List.concat (get_tail l)))=0

  fun splitequal_help a [] x = x
    | splitequal_help a ([]::t) x = splitequal_help a t x
    | splitequal_help a ((u::v)::t) (x,y) = if (a=u) then
                                    splitequal_help a t ((u::v)::x,y)
                                  else
                                    splitequal_help a t (x,(u::v)::y)

  (* splitequal sorts a list of list into a pair of list of list with all
     lists with the same first element in the left list *)
  fun splitequal [] = ([],[])
    | splitequal ([]::t) = splitequal t
    | splitequal ((a::b)::t) = splitequal_help a ((a::b)::t) ([],[])

  fun allequal l = let
                     val (x,y) = splitequal l
                   in
                     y=[]
                   end

  fun strfy [] = ""
    | strfy l = if (allequal l) then
                  if (last_elem l) then
                    get_first l
                  else
                    get_first l ^ ", " ^ (strfy (get_tail l))
                else
                  let
                    val (eq,neq) = splitequal l
                    val is_eq_last = last_elem eq
                    val is_neq_last = last_elem neq
                    val eq_para = fn x => if is_eq_last then x else "("^x^")"
                    val neq_para = fn x => if is_neq_last then x else "("^x^")"
                  in
                    "("
                    ^ (eq_para (strfy eq)) ^ " | " ^ (neq_para (strfy neq)) ^ 
                    ")"
                  end

  fun mkcont [] = ""
    | mkcont l  = strfy l

  fun is_empty [["EMPTY"]] = true
    | is_empty _           = false
                      
  fun mkdtd tg cont = if (is_empty cont) then
                        "<!ELEMENT "^tg^" " ^ (mkcont cont) ^ ">\n"
                      else
                        "<!ELEMENT "^tg^" (" ^ (mkcont cont) ^ ")>\n"

  fun gather_from sample [] x          = (x,[])
    | gather_from sample ((s,c)::t) x  = if (s=sample) then
                                           gather_from sample t ((s,c)::x)
                                         else
                                           (x,(s,c)::t)

  fun gather_from_start []         = raise EmptyList
    | gather_from_start ((s,c)::t) = gather_from s t [(s,c)]

  fun getfirst []         = raise EmptyList
    | getfirst ((s,c)::t) = s
  
  fun snd (a,b) = b

  fun cutfirst l = (map snd) l

  fun makestring (single x)    = x
    | makestring (optional x)  = x^"?"
    | makestring (star_list x) = x^"*"
    | makestring (plus_list x) = x^"+"
    | makestring (verbatim)    = "#PCDATA"
    | makestring (empty)       = "EMPTY"

  fun mkstring l = (map makestring) l

  fun mstring l = (map mkstring) l

  fun gather [] = ""
    | gather l  = let
                    val (resl,resr) = gather_from_start l
                  in
                    (mkdtd (getfirst resl) (mstring (cutfirst resl)))
                    ^ (gather resr)
                  end

  val dtd_text = "<?xml version=\"1.0\" encoding=\"ISO-8859-1\" ?>\n"
                 ^ "<!-- " ^ current_ver ^ " -->\n"
                 ^ gather dtd_elems

end
