(* ----------------------------------------------------------------------- *)
(* xml_to_as.sml                          Pascal Schmidt <pharao90@tzi.de> *)
(* ----------------------------------------------------------------------- *)
(* XML tree to abstract syntax conversion                                  *)
(* ----------------------------------------------------------------------- *)

(* $Id: xml_to_as.sml,v 1.1 2004/02/13 11:29:21 7till Exp $ *)

structure xml_to_as
=
struct

  local open xml_types GlobalEnv LocalEnv AS Symbols ArchTypes in

  exception UnexpectedTag
  exception UnexpectedTagList
  exception ListError
  exception OptionError

  (* --------------------------------------------------------------------- *)
  (* helper functions                                                      *)
  (* --------------------------------------------------------------------- *)

  fun unescape ("&"::("l"::("t"::(";"::t)))) = "<" :: (unescape t)
    | unescape ("&"::("g"::("t"::(";"::t)))) = ">" :: (unescape t)
    | unescape ("&"::("a"::("m"::("p"::(";"::t))))) = "&" :: (unescape t)
    | unescape ("&"::("q"::("u"::("o"::("t"::(";"::t)))))) = "\"" ::
                                                             (unescape t)
    | unescape ("&"::("a"::("p"::("o"::("s"::(";"::t)))))) = "'" ::
                                                             (unescape t)
    | unescape ("&"::("#"::("x"::("2"::("5"::(";"::t)))))) = "%" ::
                                                             (unescape t)
    | unescape (h::t) = h :: (unescape t)
    | unescape [] = []
    
  fun unesc x = (Utils.implode o unescape o Utils.explode) x

  fun all_anno_or_lib_item [] = true
    | all_anno_or_lib_item ((tag ("anno",_,_))::t) = all_anno_or_lib_item t
    | all_anno_or_lib_item ((tag ("lib-item",_,_))::t) = all_anno_or_lib_item t
    | all_anno_or_lib_item (h::t) = false

  fun last []      = raise ListError
    | last (h::[]) = h
    | last (h::t)  = last t
  
  fun init []      = raise ListError
    | init (h::[]) = []
    | init (h::t)  = h :: (init t)
    
  fun get_some_die (NONE) = raise OptionError
    | get_some_die (SOME x) = x

  fun split_fst x [] = []
    | split_fst x ((tag (a,b,c))::t) = if (a=x) then
                                         (tag (a,b,c)) :: (split_fst x t)
                                       else
                                         []
    | split_fst x ((empty_tag (a,b))::t) = if (a=x) then
                                             (empty_tag (a,b))::(split_fst x t)
                                           else
                                             []
    | split_fst _ _ = raise ListError
  
  fun split_snd x [] = []
    | split_snd x ((tag (a,b,c))::t) = if (a=x) then
                                         split_snd x t
                                       else
                                         (tag (a,b,c)) :: t
    | split_snd x ((empty_tag (a,b))::t) = if (a=x) then
                                             split_snd x t
                                           else
                                             (empty_tag (a,b)) :: t
    | split_snd _ _ = raise ListError                                       

  fun get_tag (tag (a,_,_)) = a
    | get_tag (empty_tag (a,_)) = a
    | get_tag x = ( print (xml2string x) ; raise UnexpectedTag )

  fun split_first [] = []
    | split_first l = split_fst (get_tag (hd l)) l

  fun split_second [] = []
    | split_second l = split_snd (get_tag (hd l)) l

  fun single_out [x] = x
    | single_out _   = raise ListError

  fun strip_sorts (sorts x) = x
    | strip_sorts (pos_SORTS (r,x)) = strip_sorts x

  fun concat_pcdata [] = ""
    | concat_pcdata ((pcdata x)::t) = (unesc x) ^ (concat_pcdata t)
    | concat_pcdata _ = raise UnexpectedTag

  (* --------------------------------------------------------------------- *)
  (* library definitions                                                   *)
  (* --------------------------------------------------------------------- *)
  
  fun a_lib_defn (tag ("casl",[],[tag ("lib-defn",[],(n::a))])) =
                     if (all_anno_or_lib_item a) then
                       (lib_defn (a_lib_name n,
                        (map a_lib_item) (split_first a),
                        (map a_anno) a),empty_global_env)
                     else
                       let
                         val litems = split_first a
                         val aitems = (split_first o split_second) a
                         val gitems = (split_second o split_second) a
                       in
                         (lib_defn (a_lib_name n,(map a_lib_item) litems,
                          (map a_anno) aitems),a_global_env 
                          (single_out gitems))
                       end
    | a_lib_defn (tag ("lib-defn",[],x)) = a_lib_defn (tag ("casl",[],[
                                           tag ("lib-defn",[],x)]))
    | a_lib_defn x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_lib_name (tag ("lib-name",[],[x])) = a_lib_name1 x
    | a_lib_name x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_lib_name1 (tag ("lib-id",[],[x])) = lib (a_lib_id x)
    | a_lib_name1 (tag ("lib-version",[],[tag ("lib-id",[],[i]),v])) =
                       versioned_lib (a_lib_id i,a_version v)
    | a_lib_name1 (tag ("pos-lib-name",[],[r,x])) = pos_LIB_NAME
                                    (a_region r,a_lib_name x)
    | a_lib_name1 x = ( print (xml2string x) ; raise UnexpectedTag )  

  and a_lib_id (tag ("direct-link",[],[x])) = url (a_url x)
    | a_lib_id (tag ("indirect-link",[],[x])) = path_name (a_path x)
    | a_lib_id (tag ("pos-lib-id",[],[r,x])) = pos_LIB_ID (a_region r,
                                                           a_lib_id x)
    | a_lib_id x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_url (tag ("url",[],l)) = concat_pcdata l
    | a_url x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_srcpos (tag ("srcpos",[],[x,y])) = (a_int x,a_int y)
    | a_srcpos x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_region (tag ("region",[],[x,y])) = (a_srcpos x,a_srcpos y)
    | a_region x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_int (tag ("int",[],l)) = (get_some_die o Int.fromString o
                                  concat_pcdata) l
    | a_int x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_path (tag ("path",[],l)) = concat_pcdata l
    | a_path x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_version (tag ("version",[],x)) = version ((map a_number) x)
    | a_version (tag ("pos-version",[],[r,x])) = pos_VERSION (a_region r,
                                                              a_version x)
    | a_version x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_number (tag ("number",[],l)) = concat_pcdata l
    | a_number x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_lib_item (tag ("lib-item",[],[x])) = a_lib_item1 x
    | a_lib_item x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_lib_item1 (tag ("spec-defn",[],(n::(g::(s::a))))) =
                   spec_defn (a_spec_name n,a_genericity g,a_l_spec s,
                              (map a_anno) a)
    | a_lib_item1 (tag ("view-defn",[],(n::(g::(t::l))))) =
                   view_defn (a_view_name n,a_genericity g,a_view_type t,
                              (map a_symb_map_items) (split_first l),
                              (map a_anno) (split_second l))
    | a_lib_item1 (tag ("arch-spec-defn",[],(n::(s::a)))) =
                   arch_spec_defn (a_arch_spec_name n,a_l_arch_spec s,
                                   (map a_anno) a)
    | a_lib_item1 (tag ("unit-spec-defn",[],(n::(s::a)))) =
                   unit_spec_defn (a_spec_name n,a_unit_spec s,
                                   (map a_anno) a)
    | a_lib_item1 (tag ("download-items",[],(n::l))) =
                   download_items (a_lib_name n,(map a_item_name_or_map)
                                   (split_first l),(map a_anno)
                                   (split_second l))
    | a_lib_item1 (tag ("pos-lib-item",[],[r,x])) =
                   pos_LIB_ITEM (a_region r,a_lib_item x)
    | a_lib_item1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_item_name_or_map (tag ("item-name-or-map",[],[x])) = a_item_nom x
    | a_item_name_or_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_item_nom (tag ("item-name",[],x)) = item_name (a_item_name
                                            (tag ("item-name",[],x)))
    | a_item_nom (tag ("item-name-map",[],x)) = a_item_name_map
                                            (tag ("item-name-map",[],x))
    | a_item_nom (tag ("pos-item-name-or-map",[],[r,x])) =
                        pos_ITEM_NAME_OR_MAP (a_region r,a_item_name_or_map x)
    | a_item_nom x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_item_name (tag ("item-name",[],[x])) = (a_simple_id x)
    | a_item_name x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_item_name_map (tag ("item-name-map",[],[x,y])) = item_name_map
                            (a_item_name x,a_item_name y)
    | a_item_name_map x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* structured specifications                                             *)
  (* --------------------------------------------------------------------- *)

  and a_spec_name (tag ("spec-name",[],[x])) = a_simple_id x
    | a_spec_name x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_simple_id (tag ("simple-id",[],[x])) = (a_words x,Utils.None)
    | a_simple_id x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_words (tag ("words",[],l)) = concat_pcdata l
    | a_words x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_genericity (tag ("genericity",[],[x,y])) = genericity (a_params x,
                                                               a_imports y)
    | a_genericity (tag ("pos-genericity",[],[r,x])) = pos_GENERICITY
                                            (a_region r,a_genericity x)
    | a_genericity x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_params (tag ("params",[],l)) = params ((map a_l_spec) l)
    | a_params (tag ("pos-params",[],[r,x])) = pos_PARAMS (a_region r,
                                                           a_params x)
    | a_params x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_imports (tag ("imports",[],l)) = imports ((map a_l_spec) l)
    | a_imports (tag ("pos-imports",[],[r,x])) = pos_IMPORTS (a_region r,
                                                              a_imports x)
    | a_imports x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_l_spec (tag ("anno-spec",[],(s::a))) = (a_spec s,(map a_anno) a)
    | a_l_spec x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_spec (tag ("spec",[],[x])) = a_spec1 x
    | a_spec x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_spec1 (tag ("basic",[],[x])) = basic (a_basic_spec x)
    | a_spec1 (tag ("translation",[],(s::(r::a)))) =
               translation (a_l_spec s,a_renaming r,(map a_anno) a)
    | a_spec1 (tag ("reduction",[],(s::(r::a)))) =
               reduction (a_l_spec s,a_restriction r,(map a_anno) a)
    | a_spec1 (tag ("union",[],x)) = union_spec (a_union x)
    | a_spec1 (tag ("extension",[],x)) = extension (a_union x)
    | a_spec1 (tag ("free-spec",[],(s::a))) =
               free_spec (a_l_spec s,(map a_anno) a)
    | a_spec1 (tag ("cofree-spec",[],(s::a))) =
               cofree_spec (a_l_spec s,(map a_anno) a)
    | a_spec1 (tag ("closed-spec",[],(s::a))) =
               closed_spec (a_l_spec s,(map a_anno) a)
    | a_spec1 (tag ("local-spec",[],(s::a))) =
               local_spec (a_l_spec s,(map a_anno) (split_first a),
                           a_l_spec ((single_out o split_first o split_second)
                                     a),
                           (map a_anno) ((split_second o split_second) a))
    | a_spec1 (tag ("spec-inst",[],(n::a))) =
               spec_inst (a_spec_name n,(map a_fit_arg) a)
    | a_spec1 (tag ("pos-spec",[],[r,b,x])) =
               pos_SPEC (a_region r,a_bool_data b,a_spec x)
    | a_spec1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_union x = (map a_union_one) x
  
  and a_union_one (tag ("anno-anno-spec",[],(s::a))) = (a_l_spec s,
                                                        (map a_anno) a)
    | a_union_one x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_renaming (tag ("renaming",[],l)) = renaming ((map a_symb_map_items) l)
    | a_renaming (tag ("pos-renaming",[],[r,x])) = pos_RENAMING (a_region r,
                                                   a_renaming x)
    | a_renaming x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb_map_items (tag ("symb-map-items",[],(k::s))) =
                   symb_map_items (a_symb_kind k,(map a_symb_or_map) s)
    | a_symb_map_items (tag ("pos-symb-map-items",[],[r,x])) =
                   pos_SYMB_MAP_ITEMS (a_region r,a_symb_map_items x)
    | a_symb_map_items x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb_kind (tag ("symb-kind",[],[x])) = a_symb_kind1 x
    | a_symb_kind x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb_kind1 (empty_tag ("implicit",[])) = implicitk
    | a_symb_kind1 (empty_tag ("sort-kind",[])) = sortsk
    | a_symb_kind1 (empty_tag ("op-kind",[])) = opsk
    | a_symb_kind1 (empty_tag ("pred-kind",[])) = predsk
    | a_symb_kind1 (tag ("pos-symb-kind",[],[r,x])) = pos_SYMB_KIND (a_region r,
                                                      a_symb_kind x)
    | a_symb_kind1 x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_symb_or_map (tag ("symb-or-map",[],[x])) = a_symb_o_map x
    | a_symb_or_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb_o_map (tag ("symb",[],[x])) = symb (a_symb x)
    | a_symb_o_map (tag ("symb-map",[],x)) = symb_or_map (a_symb_map
                                             (tag ("symb-map",[],x)))
    | a_symb_o_map (tag ("pos-symb-or-map",[],[r,x])) =
                           pos_SYMB_OR_MAP (a_region r,a_symb_or_map x)
    | a_symb_o_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb_map (tag ("symb-map",[],[tag ("symb",[],[a]),tag ("symb",[],[b])
                  ])) = symb_map (a_symb a,a_symb b)
    | a_symb_map (tag ("pos-symb-map",[],[r,x])) = pos_SYMB_MAP (a_region r,
                                                   a_symb_map x)
    | a_symb_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb (tag ("id",[],x)) = symb_id (a_id (tag ("id",[],x)))
    | a_symb (tag ("qual-id",[],[x,y])) = qual_id (a_id x,a_type y)
    | a_symb (tag ("pos-symb",[],[r,x])) = pos_SYMB (a_region r,a_symb x)
    | a_symb x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_type (tag ("type",[],[tag ("op-type",[],x)])) =
              op_symb_type (a_op_type (tag ("op-type",[],x)))
    | a_type (tag ("type",[],[tag ("pred-type",[],x)])) =
              pred_symb_type (a_pred_type (tag ("pred-type",[],x)))
    | a_type (tag ("pos-type",[],[r,x])) = pos_TYPE (a_region r,a_type x)
    | a_type x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_restriction (tag ("restriction",[],[x])) = a_restriction1 x
    | a_restriction x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_restriction1 (tag ("hide",[],l)) = hide ((map a_symb_items) l)
    | a_restriction1 (tag ("reveal",[],l)) = reveal ((map a_symb_map_items) l)
    | a_restriction1 (tag ("pos-restriction",[],[r,x])) = pos_RESTRICTION
                                     (a_region r,a_restriction x)
    | a_restriction1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb_list (tag ("symb",[],[x])) = a_symb x
    | a_symb_list x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symb_items (tag ("symb-items",[],(k::s))) = symb_items
                                   (a_symb_kind k,(map a_symb_list) s)
    | a_symb_items (tag ("pos-symb-items",[],[r,x])) = pos_SYMB_ITEMS
                                   (a_region r,a_symb_items x)
    | a_symb_items x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_fit_arg (tag ("fit-arg",[],[x])) = a_fit_arg1 x
    | a_fit_arg x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_fit_arg1 (tag ("fit-spec",[],(s::l))) =
                  fit_spec (a_l_spec s,(map a_symb_map_items) l)
    | a_fit_arg1 (tag ("fit-view",[],(n::l))) =
                  fit_view (a_view_name n,(map a_fit_arg) l)
    | a_fit_arg1 (tag ("pos-fit-arg",[],[r,x])) =
                  pos_FIT_ARG (a_region r,a_fit_arg x)
    | a_fit_arg1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_view_name (tag ("view-name",[],[x])) = a_simple_id x
    | a_view_name x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_view_type (tag ("view-type",[],[x,y])) = view_type (a_l_spec x,
                                                            a_l_spec y)
    | a_view_type (tag ("pos-view-type",[],[r,x])) = pos_VIEW_TYPE
                                  (a_region r,a_view_type x)
    | a_view_type x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* annotations                                                           *)
  (* --------------------------------------------------------------------- *)

  and a_anno (tag ("anno",[],[x])) = a_anno1 x
    | a_anno x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_anno1 (tag ("comment-anno",[],l)) = comment (concat_pcdata l)
    | a_anno1 (tag ("comment-line-anno",[],l)) =
                                            comment_line (concat_pcdata l)
    | a_anno1 (tag ("label-anno",[],[x])) = label_anno (a_id x)
    | a_anno1 (tag ("unparsed-anno",[],l)) = unparsed_anno (concat_pcdata l)
    | a_anno1 (tag ("number-anno",[],[x])) = number_anno (a_id x)
    | a_anno1 (tag ("floating-anno",[],[x,y])) = floating_anno (a_id x,a_id y)
    | a_anno1 (tag ("string-anno",[],[x,y])) = string_anno (a_id x,a_id y)
    | a_anno1 (tag ("list-anno",[],[x,y,z])) = list_anno (a_id x,a_id y,a_id z)
    | a_anno1 (tag ("display-anno",[],[x,y])) = display_anno (a_id x,
                                                              a_disp_anno y)
    | a_anno1 (tag ("prec-anno",[],[x,y,z])) = prec_anno (a_bool_data x,
                                               a_id_list y,a_id_list z)
    | a_anno1 (tag ("lassoc-anno",[],l)) = lassoc_anno ((map a_id) l)
    | a_anno1 (tag ("rassoc-anno",[],l)) = rassoc_anno ((map a_id) l)
    | a_anno1 (empty_tag ("conservative-anno",[])) = conservative
    | a_anno1 (empty_tag ("definitional-anno",[])) = definitional
    | a_anno1 (empty_tag ("mono-anno",[])) = mono
    | a_anno1 (empty_tag ("implies-anno",[])) = implies
    | a_anno1 (tag ("pos-anno",[],[r,x])) = pos_ANNO (a_region r,a_anno x)
    | a_anno1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_bool_data (tag ("bool-data",[],[x])) = a_bool_data1 x
    | a_bool_data x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_bool_data1 (empty_tag ("true",[])) = true
    | a_bool_data1 (empty_tag ("false",[])) = false
    | a_bool_data1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_disp_anno (tag ("display-anno-content",[],l)) = concat_pcdata l
    | a_disp_anno x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_id_list (tag ("id-list",[],l)) = (map a_id) l
    | a_id_list x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* basic specifications                                                  *)
  (* --------------------------------------------------------------------- *)

  and a_basic_spec (tag ("basic-spec",[],l)) =
                    basic_spec ((map a_basic_items) l)
    | a_basic_spec (tag ("pos-basic-spec",[],[r,x])) =
                    pos_BASIC_SPEC (a_region r,a_basic_spec x)
    | a_basic_spec x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_basic_items (tag ("basic-items",[],[x])) = a_basic_items1 x
    | a_basic_items x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_basic_items1 (tag ("sig-items",[],x)) = sig_items (a_sig_items
                                                (tag ("sig-items",[],x)))
    | a_basic_items1 (tag ("free-datatype",[],l)) =
                      free_datatype ((map a_datatype_decl) (split_first l),
                                     (map a_anno) (split_second l))
    | a_basic_items1 (tag ("cofree-datatype",[],l)) =
                      cofree_datatype ((map a_datatype_decl) (split_first l),
                                     (map a_anno) (split_second l))
    | a_basic_items1 (tag ("sort-gen",[],l)) =
                      sort_gen ((map a_sig_items) (split_first l),
                                (map a_anno) (split_second l))
    | a_basic_items1 (tag ("sort-cogen",[],l)) =
                      sort_cogen ((map a_sig_items) (split_first l),
                                (map a_anno) (split_second l))
    | a_basic_items1 (tag ("var-items",[],l)) =
                      var_items ((map a_var_decl) (split_first l),
                                 (map a_anno) (split_second l))
    | a_basic_items1 (tag ("local-var-axioms",[],l)) =
                      local_var_axioms ((map a_var_decl) (split_first l),
                      (map a_l_formula) (split_first (split_second l)),
                      (map a_anno) (split_second (split_second l)))
    | a_basic_items1 (tag ("axiom-items",[],l)) =
                      axiom_items ((map a_l_formula) (split_first l),
                                   (map a_anno) (split_second l))
    | a_basic_items1 (tag ("pos-basic-items",[],[r,x])) =
                      pos_BASIC_ITEMS (a_region r,a_basic_items x)
    | a_basic_items1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_sig_items (tag ("sig-items",[],[x])) = (a_sig_items1 x):SIG_ITEMS
    | a_sig_items x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_sig_items1 (tag ("sort-items",[],l)) =
                    sort_items ((map a_l_sort_item) (split_first l),
                                (map a_anno) (split_second l))
    | a_sig_items1 (tag ("op-items",[],l)) =
                    op_items ((map a_l_op_item) (split_first l),
                              (map a_anno) (split_second l))
    | a_sig_items1 (tag ("pred-items",[],l)) =
                    pred_items ((map a_l_pred_item) (split_first l),
                                (map a_anno) (split_second l))
    | a_sig_items1 (tag ("datatype-items",[],l)) =
                    datatype_items ((map a_datatype_decl) (split_first l),
                                    (map a_anno) (split_second l))
    | a_sig_items1 (tag ("pos-sig-items",[],[r,x])) =
                    pos_SIG_ITEMS (a_region r,a_sig_items x)
    | a_sig_items1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_l_sort_item (tag ("anno-sort-item",[],(x::y))) = (a_sort_item x,
                                                         (map a_anno) y)
    | a_l_sort_item x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_l_op_item (tag ("anno-op-item",[],(x::y))) = (a_op_item x,
                                                      (map a_anno) y)
    | a_l_op_item x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_l_pred_item (tag ("anno-pred-item",[],(x::y))) = (a_pred_item x,
                                                          (map a_anno) y)
    | a_l_pred_item x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_sort_item (tag ("sort-item",[],[x])) = a_sort_item1 x
    | a_sort_item x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_sort_item1 (tag ("sort-decl",[],l)) = sort_decl ((map a_sort) l)
    | a_sort_item1 (tag ("subsort-decl",[],[x,y])) =
                    subsort_decl ((strip_sorts o a_sorts) x,a_sort y)
    | a_sort_item1 (tag ("subsort-defn",[],(s::(v::(t::(f::a)))))) =
                    subsort_defn (a_sort s,a_var v,a_sort t,a_formula f,
                                  (map a_anno) a)
    | a_sort_item1 (tag ("iso-decl",[],l)) = iso_decl ((map a_sort) l)
    | a_sort_item1 (tag ("pos-sort-item",[],[r,x])) =
                    pos_SORT_ITEM (a_region r,a_sort_item x)
    | a_sort_item1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_sort (tag ("sort",[],[x])) = a_id x
    | a_sort x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_sorts (tag ("sorts",[],l)) = sorts ((map a_sort) l)
    | a_sorts (tag ("pos-sorts",[],[r,x])) = pos_SORTS (a_region r,a_sorts x)
    | a_sorts x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_item (tag ("op-item",[],[x])) = a_op_item1 x
    | a_op_item x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_item1 (tag ("op-decl",[],l)) =
                  op_decl ((map a_op_name) (split_first l),
                           a_op_type ((single_out o split_first o split_second
                                      ) l),
                           (map a_op_attr) ((split_second o split_second) l))
    | a_op_item1 (tag ("op-defn",[],(n::(h::(t::a))))) =
                  op_defn (a_op_name n,a_op_head h,a_term t,(map a_anno) a)
    | a_op_item1 (tag ("pos-op-item",[],[r,x])) =
                  pos_OP_ITEM (a_region r,a_op_item x)
    | a_op_item1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_name (tag ("op-name",[],[x])) = a_id x
    | a_op_name x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_type (tag ("op-type",[],[x])) = a_op_type1 x
    | a_op_type x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_type1 (tag ("total-op-type",[],[x,y])) =
                  total_op_type (a_sorts x,a_sort y)
    | a_op_type1 (tag ("partial-op-type",[],[x,y])) =
                  partial_op_type (a_sorts x,a_sort y)
    | a_op_type1 (tag ("pos-op-type",[],[r,x])) =
                  pos_OP_TYPE (a_region r,a_op_type x)
    | a_op_type1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_head (tag ("op-head",[],[x])) = a_op_head1 x
    | a_op_head x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_head1 (tag ("total-op-head",[],l)) =
                  total_op_head ((map a_arg_decl) (split_first l),
                                 a_sort ((single_out o split_second) l))
    | a_op_head1 (tag ("partial-op-head",[],l)) =
                  partial_op_head ((map a_arg_decl) (split_first l),
                                   a_sort ((single_out o split_second) l))
    | a_op_head1 (tag ("pos-op-head",[],[r,x])) =
                  pos_OP_HEAD (a_region r,a_op_head x)
    | a_op_head1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_arg_decl (tag ("arg-decl",[],[x])) = arg_decl (a_var_decl x)
    | a_arg_decl (tag ("pos-arg-decl",[],[r,x])) = pos_ARG_DECL (a_region r,
                                                   a_arg_decl x)
    | a_arg_decl x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_attr (tag ("op-attr",[],[x])) = a_op_attr1 x
    | a_op_attr x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_attr1 (empty_tag ("associative",[])) = associative
    | a_op_attr1 (empty_tag ("commutative",[])) = commutative
    | a_op_attr1 (empty_tag ("idempotent",[])) = idempotent
    | a_op_attr1 (tag ("unit-op-attr",[],[x])) = unit_op_attr (a_term x)
    | a_op_attr1 (tag ("pos-op-attr",[],[r,x])) = pos_OP_ATTR (a_region r,
                                                  a_op_attr x)
    | a_op_attr1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_symb (tag ("op-symb",[],[x])) = a_op_symb1 x
    | a_op_symb x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_op_symb1 (tag ("op-name",[],x)) = op_symb (a_op_name
                                          (tag ("op-name",[],x)),Utils.None)
    | a_op_symb1 (tag ("qual-op-name",[],[x,y])) = op_symb (a_op_name x,
                                                   Utils.Some (a_op_type y))
    | a_op_symb1 (tag ("pos-op-symb",[],[r,x])) = pos_OP_SYMB (a_region r,
                                                  a_op_symb x)
    | a_op_symb1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_item (tag ("pred-item",[],[x])) = a_pred_item1 x
    | a_pred_item x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_item1 (tag ("pred-decl",[],l)) =
                    pred_decl ((map a_pred_name) (init l),a_pred_type (last l))
    | a_pred_item1 (tag ("pred-defn",[],(n::(h::(f::a))))) =
                    pred_defn (a_pred_name n,a_pred_head h,a_l_formula f,
                               (map a_anno) a)
    | a_pred_item1 (tag ("pos-pred-item",[],[r,x])) =
                    pos_PRED_ITEM (a_region r,a_pred_item x)
    | a_pred_item1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_type (tag ("pred-type",[],[x])) = pred_type (a_sorts x)
    | a_pred_type (tag ("pos-pred-type",[],[r,x])) = pos_PRED_TYPE
                                (a_region r,a_pred_type x)
    | a_pred_type x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_head (tag ("pred-head",[],l)) = pred_head ((map a_arg_decl) l)
    | a_pred_head (tag ("pos-pred-head",[],[r,x])) = pos_PRED_HEAD
                                (a_region r,a_pred_head x)
    | a_pred_head x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_symb (tag ("pred-symb",[],[x])) = a_pred_symb1 x
    | a_pred_symb x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_symb1 (tag ("pred-name",[],x)) = pred_symb (a_pred_name
                                              (tag ("pred-name",[],x)),
                                              Utils.None)
    | a_pred_symb1 (tag ("qual-pred-name",[],[x,y])) = pred_symb
                                  (a_pred_name x,Utils.Some (a_pred_type y))
    | a_pred_symb1 (tag ("pos-pred-symb",[],[r,x])) =
                    pos_PRED_SYMB (a_region r,a_pred_symb x)
    | a_pred_symb1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_name (tag ("pred-name",[],[x])) = a_id x
    | a_pred_name x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_datatype_decl (tag ("datatype-decl",[],(s::l))) =
                       datatype_decl (a_sort s,
                                      (map a_l_alternative) (split_first l),
                                      (map a_anno) (split_second l))
    | a_datatype_decl (tag ("pos-datatype-decl",[],[r,x])) =
                       pos_DATATYPE_DECL (a_region r,a_datatype_decl x)
    | a_datatype_decl x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_l_alternative (tag ("anno-alternative",[],(a::l))) =
                       (a_alternative a,(map a_anno) l)
    | a_l_alternative x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_alternative (tag ("alternative",[],[x])) = a_alternative1 x
    | a_alternative x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_alternative1 (tag ("total-construct",[],(n::l))) =
                      total_construct (a_op_name n,(map a_components) l)
    | a_alternative1 (tag ("partial-construct",[],(n::l))) =
                      partial_construct (a_op_name n,(map a_components) l)
    | a_alternative1 (tag ("subsorts",[],l)) =
                      subsort ((map a_sort) l)
    | a_alternative1 (tag ("pos-alternative",[],[r,x])) =
                      pos_ALTERNATIVE (a_region r,a_alternative x)
    | a_alternative1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_components (tag ("components",[],[x])) = a_components1 x
    | a_components x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_components1 (tag ("total-select",[],l)) =
                     total_select ((map a_op_name) (init l),a_sort (last l))
    | a_components1 (tag ("partial-select",[],l)) =
                     partial_select ((map a_op_name) (init l),a_sort (last l))
    | a_components1 (tag ("sort",[],x)) =
                     sort_component (a_sort (tag ("sort",[],x)))
    | a_components1 (tag ("pos-components",[],[r,x])) =
                     pos_COMPONENTS (a_region r,a_components x)
    | a_components1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_var_decl (tag ("var-decl",[],l)) =
                  ((map a_var) (init l),a_sort (last l))
    | a_var_decl x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_var (tag ("var",[],[x])) = a_simple_id x
    | a_var x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_l_formula (tag ("anno-formula",[],(f::a))) =
                   (a_formula f,(map a_anno) a)
    | a_l_formula x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_formula (tag ("formula",[],[x])) = a_formula1 x
    | a_formula x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_formula1 (tag ("quantification",[],(q::l))) =
                  quantification (a_quantifier q,(map a_var_decl) (init l),
                                  a_formula (last l))
    | a_formula1 (tag ("pred-quantification",[],(q::l))) =
                  pred_quantification (a_quantifier q,(map a_var_pred_type)
                                       (init l),a_formula (last l))
    | a_formula1 (tag ("conjunction",[],l)) =
                  conjunction ((map a_formula) l)
    | a_formula1 (tag ("disjunction",[],l)) =
                  disjunction ((map a_formula) l)
    | a_formula1 (tag ("equivalence",[],[x,y])) =
                  equivalence (a_formula x,a_formula y)
    | a_formula1 (tag ("implication",[],[x,y])) =
                  implication (a_formula x,a_formula y)
    | a_formula1 (tag ("negation",[],[x])) =
                  negation (a_formula x)
    | a_formula1 (tag ("sort-gen-ax",[],l)) =
                  sort_gen_ax ((map a_sort) (split_first l),
                               (map a_op_symb) (split_second l))
    | a_formula1 (tag ("sort-cogen-ax",[],l)) =
                  sort_cogen_ax ((map a_sort) (split_first l),
                               (map a_op_symb) (split_second l))
    | a_formula1 (tag ("sort-cofree-ax",[],l)) =
                  sort_cofree_ax ((map a_sort) (split_first l),
                               (map a_op_symb) (split_second l))
    | a_formula1 (tag ("unparsed",[],l)) =
                  unparsed_formula (concat_pcdata l)
    | a_formula1 (tag ("pos-formula",[],[r,b,x])) =
                  pos_FORMULA (a_region r,a_bool_data b,a_formula x)
    | a_formula1 (empty_tag ("true",[])) =
                  atom (ttrue)
    | a_formula1 (empty_tag ("false",[])) =
                  atom (ffalse)
    | a_formula1 (tag ("predication",[],[x,y])) =
                  atom (predication (a_pred_symb x,a_terms y))
    | a_formula1 (tag ("existl-equation",[],[x,y])) =
                  atom (existl_equation (a_term x,a_term y))
    | a_formula1 (tag ("strong-equation",[],[x,y])) =
                  atom (strong_equation (a_term x,a_term y))
    | a_formula1 (tag ("definedness",[],[x])) =
                  atom (definedness (a_term x))
    | a_formula1 (tag ("membership",[],[x,y])) =
                  atom (membership (a_term x,a_sort y))
    | a_formula1 x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_var_pred_type (tag ("var-pred-type",[],[x,y])) =
                       (a_id x,a_pred_type y)
    | a_var_pred_type x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_quantifier (tag ("quantifier",[],[x])) = a_quantifier1 x
    | a_quantifier x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_quantifier1 (empty_tag ("forall",[])) = forall
    | a_quantifier1 (empty_tag ("exists",[])) = exists
    | a_quantifier1 (empty_tag ("exists-uniquely",[])) = exists_uniquely
    | a_quantifier1 (tag ("pos-quantifier",[],[r,x])) =
                     pos_QUANTIFIER (a_region r,a_quantifier x)
    | a_quantifier1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_terms (tag ("terms",[],l)) = terms ((map a_term) l)
    | a_terms x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_term (tag ("term",[],[x])) = a_term1 x
    | a_term x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_term1 (tag ("simple-id",[],x)) =
               var_or_const (a_simple_id (tag ("simple-id",[],x)))
    | a_term1 (tag ("qual-var",[],[x,y])) =
               qual_var (a_var x,a_sort y)
    | a_term1 (tag ("application",[],[x,y])) =
               application (a_op_symb x,a_terms y)
    | a_term1 (tag ("sorted-term",[],[x,y])) =
               sorted_term (a_term x,a_sort y)
    | a_term1 (tag ("cast",[],[x,y])) =
               cast (a_term x,a_sort y)
    | a_term1 (tag ("conditional",[],[x,y,z])) =
               conditional (a_term x,a_formula y,a_term z)
    | a_term1 (tag ("unparsed",[],l)) =
               unparsed_term (concat_pcdata l)
    | a_term1 (tag ("pos-term",[],[r,b,x])) =
               pos_TERM (a_region r,a_bool_data b,a_term x)
    | a_term1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  (* --------------------------------------------------------------------- *)
  (* identifiers                                                           *)
  (* --------------------------------------------------------------------- *)

  and a_id (tag ("id",[],[x])) = a_id1 x
    | a_id x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_id1 (tag ("token-id",[],[tag ("comp-token-id",[],l)])) =
             a_ctid l
    | a_id1 (tag ("mixfix-id",[],[tag ("comp-mixfix-id",[],l)])) =
             a_cmid l
    | a_id1 (tag ("token-id",[],[x])) =
             simple_id ([a_tokop x],"",Utils.None)
    | a_id1 (tag ("mixfix-id",[],[l])) =
             simple_id (a_token_places l,"",Utils.None)
    | a_id1 x = ( print (xml2string x) ; raise UnexpectedTag )
            
  and a_tokop (empty_tag ("place",[])) = place
    | a_tokop (tag ("token",[],[x])) = token (a_tok x)
    | a_tokop x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_tok (tag ("words",[],l)) = concat_pcdata l
    | a_tok (tag ("dot-words",[],l)) = concat_pcdata l
    | a_tok (tag ("signs",[],l)) = concat_pcdata l
    | a_tok x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_token_places (tag ("token-places",[],l)) = (map a_token_or_place) l
    | a_token_places x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_token_or_place (tag ("token-or-place",[],[x])) = a_tokop x
    | a_token_or_place x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_ctid (t::l) = compound_id (([a_tokop t],"",Utils.None),(map a_id) l)
    | a_ctid _ = raise UnexpectedTagList

  and a_cmid (t::l) = compound_id ((a_token_places t,"",Utils.None),
                                   (map a_id) l)
    | a_cmid _ = raise UnexpectedTagList

  (* --------------------------------------------------------------------- *)
  (* architectural specifications                                          *)
  (* --------------------------------------------------------------------- *)

  and a_arch_spec_name (tag ("arch-spec-name",[],[x])) = a_simple_id x
    | a_arch_spec_name x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_l_arch_spec (tag ("anno-arch-spec",[],(s::l))) =
                     (a_arch_spec s,(map a_anno) l)
    | a_l_arch_spec x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_arch_spec (tag ("arch-spec",[],[x])) = a_arch_spec1 x
    | a_arch_spec x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_arch_spec1 (tag ("basic-arch-spec",[],l)) =
                    basic_arch_spec((map a_anno_unit_decl_defn)
                    (split_first l),
                    a_result_unit ((single_out o split_first o split_second)
                                   l),
                    (map a_anno) ((split_second o split_second) l))
    | a_arch_spec1 (tag ("arch-spec-name",[],x)) =
                    named_arch_spec (a_arch_spec_name
                    (tag ("arch-spec-name",[],x)))
    | a_arch_spec1 (tag ("pos-arch-spec",[],[r,x])) =
                    pos_ARCH_SPEC (a_region r,a_arch_spec x)
    | a_arch_spec1 x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_anno_unit_decl_defn (tag ("anno-unit-decl-defn",[],(u::l))) =
                   (a_unit_decl_defn u,(map a_anno) l)
    | a_anno_unit_decl_defn x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_decl_defn (tag ("unit-decl-defn",[],[x])) = a_unit_decl_defn1 x
    | a_unit_decl_defn x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_decl_defn1 (tag ("unit-decl",[],x)) =
                  unit_decl_case (a_unit_decl (tag ("unit-decl",[],x)))
    | a_unit_decl_defn1 (tag ("unit-defn",[],x)) =
                  unit_defn_case (a_unit_defn (tag ("unit-defn",[],x)))
    | a_unit_decl_defn1 (tag ("pos-unit-decl-defn",[],[r,x])) =
                  pos_UNIT_DECL_DEFN (a_region r,a_unit_decl_defn x)
    | a_unit_decl_defn1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_decl (tag ("unit-decl",[],[x,y,z])) =
                   unit_decl (a_unit_name x,a_unit_spec y,a_unit_imports z)
    | a_unit_decl (tag ("pos-unit-decl",[],[r,x])) =
                   pos_UNIT_DECL (a_region r,a_unit_decl x)
    | a_unit_decl x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_defn (tag ("unit-defn",[],[x,y])) =
                   unit_defn (a_unit_name x,a_unit_expression y)
    | a_unit_defn (tag ("pos-unit-defn",[],[r,x])) =
                   pos_UNIT_DEFN (a_region r,a_unit_defn x)
    | a_unit_defn x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_imports (tag ("unit-imports",[],l)) =
                      unit_imports ((map a_unit_term) l)
    | a_unit_imports (tag ("pos-unit-imports",[],[r,x])) =
                      pos_UNIT_IMPORTS (a_region r,a_unit_imports x)
    | a_unit_imports x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_l_unit_spec (tag ("anno-unit-spec",[],(s::l))) =
                     (a_unit_spec s,(map a_anno) l)
    | a_l_unit_spec x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_spec (tag ("unit-spec",[],[x])) = a_unit_spec1 x
    | a_unit_spec x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_spec1 (tag ("unit-type",[],x)) =
                    unit_type_case (a_unit_type (tag ("unit-type",[],x)))
    | a_unit_spec1 (tag ("spec-name",[],x)) =
                    spec_name_case (a_spec_name (tag ("spec-name",[],x)))
    | a_unit_spec1 (tag ("anno-arch-spec",[],x)) =
                    arch_spec_case (a_l_arch_spec (tag ("anno-arch-spec",[],
                                    x)))
    | a_unit_spec1 (tag ("closed",[],[x])) =
                    closed (a_unit_spec x)
    | a_unit_spec1 (tag ("pos-unit-spec",[],[r,x])) =
                    pos_UNIT_SPEC (a_region r,a_unit_spec x)
    | a_unit_spec1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_type (tag ("unit-type",[],[x,y])) =
                   unit_type (a_anno_specs x,a_l_spec y)
    | a_unit_type (tag ("pos-unit-type",[],[r,x])) =
                   pos_UNIT_TYPE (a_region r,a_unit_type x)
    | a_unit_type x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_anno_specs (tag ("anno-specs",[],l)) = (map a_l_spec) l
    | a_anno_specs x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_result_unit (tag ("result-unit",[],(e::l))) =
                     result_unit (a_unit_expression e,(map a_anno) l)
    | a_result_unit (tag ("pos-result-unit",[],[r,x])) =
                     pos_RESULT_UNIT (a_region r,a_result_unit x)
    | a_result_unit x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_expression (tag ("unit-expression",[],l)) =
                         unit_expression ((map a_unit_binding) (init l),
                                          a_unit_term (last l))
    | a_unit_expression (tag ("pos-unit-expression",[],[r,x])) =
                         pos_UNIT_EXPRESSION (a_region r,a_unit_expression x)
    | a_unit_expression x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_binding (tag ("unit-binding",[],[x,y])) =
                      unit_binding (a_unit_name x,a_unit_spec y)
    | a_unit_binding (tag ("pos-unit-binding",[],[r,x])) =
                      pos_UNIT_BINDING (a_region r,a_unit_binding x)
    | a_unit_binding x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_term (tag ("unit-term",[],[x])) = a_unit_term1 x
    | a_unit_term x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_term1 (tag ("unit-translation",[],[x,y])) =
                    unit_translation (a_unit_term x,a_renaming y)
    | a_unit_term1 (tag ("unit-reduction",[],[x,y])) =
                    unit_reduction (a_unit_term x,a_restriction y)
    | a_unit_term1 (tag ("amalgamation",[],l)) =
                    amalgamation ((map a_unit_term) l)
    | a_unit_term1 (tag ("local-unit",[],l)) =
                    local_unit ((map a_unit_defn) (init l),
                                a_unit_term (last l))
    | a_unit_term1 (tag ("unit-appl",[],(n::l))) =
                    unit_appl (a_unit_name n,(map a_fit_arg_unit) l)
    | a_unit_term1 (tag ("pos-unit-term",[],[r,b,x])) =
                    pos_UNIT_TERM (a_region r,a_bool_data b,a_unit_term x)
    | a_unit_term1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_fit_arg_unit (tag ("fit-arg-unit",[],(t::l))) =
                      fit_arg_unit (a_unit_term t,(map a_symb_map_items) l)
    | a_fit_arg_unit (tag ("pos-fit-arg-unit",[],[r,x])) =
                      pos_FIT_ARG_UNIT (a_region r,a_fit_arg_unit x)
    | a_fit_arg_unit x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_name (tag ("unit-name",[],[x])) = a_simple_id x
    | a_unit_name x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* casenv                                                                *)
  (* --------------------------------------------------------------------- *)

  and a_global_env (tag ("casl",[],[x])) = a_global_env1 x
    | a_global_env (tag ("casenv",[],x)) = a_global_env1 (tag ("casenv",[],x))
    | a_global_env x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_global_env1 (tag ("casenv",[],l)) =
                     (Symtab_sid.make (a_genv (split_first l)),
                      (map a_anno) (split_second l))
    | a_global_env1 x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_genv x = (map a_genv1) x
  
  and a_genv1 (tag ("sid-global-entry",[],[x,y])) =
               (a_simple_id x,a_global_entry y)
    | a_genv1 x = ( print (xml2string x) ; raise UnexpectedTag )
               
  and a_global_entry (tag ("global-entry",[],[x])) = a_global_entry1 x
    | a_global_entry x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_global_entry1 (tag ("spec-defn-env",[],[x,y])) =
                       spec_defn_env (a_genericity_env x,a_spec_lenv y)
    | a_global_entry1 (tag ("view-defn-env",[],[x,y,z,a])) =
                       view_defn_env (a_genericity_env x,a_spec_lenv y,
                                      a_morphism z,a_spec_lenv a)
    | a_global_entry1 (tag ("arch-spec-defn-env",[],[x])) =
                       arch_spec_defn_env (a_arch_sig x)
    | a_global_entry1 (tag ("unit-spec-defn-env",[],[x])) =
                       unit_spec_defn_env (a_unit_sig x)
    | a_global_entry1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_genericity_env (tag ("genericity-env",[],(s::l))) =
                        (a_spec_lenv s,(map a_spec_lenv) (init l),
                         a_sign (last l))
    | a_genericity_env x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_spec_lenv (tag ("spec-lenv",[],[x,y,z])) =
                   SPEC_ENV (a_sign x,a_sign y,a_spec_env z)
                   
    | a_spec_lenv x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_spec_env (tag ("spec-env",[],[x])) = a_spec_env1 x
    | a_spec_env x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and appempty x = (x,[])
    
  and a_spec_env1 (tag ("basic-env",[],(s::l))) =
                   basic_env (a_sign s,(map a_l_formula) l)
    | a_spec_env1 (tag ("translate-env",[],[x,y])) =
                   translate_env (a_spec_env x,a_morphism y)
    | a_spec_env1 (tag ("derive-env",[],[x,y])) =
                   derive_env (a_spec_env x,a_morphism y)
    | a_spec_env1 (tag ("union-env",[],l)) =
                   union_env ((map a_spec_env) l)
    | a_spec_env1 (tag ("extension-env",[],l)) =
                   extension_env ((map (appempty o a_spec_env)) l)
    | a_spec_env1 (tag ("free-spec-env",[],[x])) =
                   free_spec_env (a_spec_env x)
    | a_spec_env1 (tag ("cofree-spec-env",[],[x])) =
                   cofree_spec_env (a_spec_env x)
    | a_spec_env1 (tag ("closed-spec-env",[],[x])) =
                   closed_spec_env (a_spec_env x)
    | a_spec_env1 (tag ("spec-inst-env",[],(x::(y::(z::l))))) =
                   spec_inst_env (a_spec_name x,a_spec_env y,a_morphism z,
                                  (map a_spec_env) l)
    | a_spec_env1 (empty_tag ("dummy-spec-env",[])) = dummy_spec_env
    | a_spec_env1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_morphism (tag ("morphism",[],[x,y,z])) =
                  (a_sort_map x,a_fun_map y,a_pred_map z)
    | a_morphism x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_sort_map (tag ("sort-map",[],x)) = Symtab_id.make (a_smap x)
    | a_sort_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_smap x = (map a_smap1) x
  
  and a_smap1 (tag ("id-sort",[],[x,y])) = (a_id x,a_sort y)
    | a_smap1 x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_fun_map (tag ("fun-map",[],x)) = Symtab_id.make (a_fmap x)
    | a_fun_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_fmap x = (map a_fmap1) x
  
  and a_fmap1 (tag ("id-op-type-name",[],(i::l))) =
                          (a_id i,(map a_fmap2) l)
    | a_fmap1 x = ( print (xml2string x) ; raise UnexpectedTag )
                          
  and a_fmap2 (tag ("op-type-name",[],[x,y,z])) =
                          (a_op_type x,a_op_name y,a_bool_data z)
    | a_fmap2 x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_pred_map (tag ("pred-map",[],x)) = Symtab_id.make (a_pmap x)
    | a_pred_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pmap x = (map a_pmap1) x
  
  and a_pmap1 (tag ("id-pred-type-name",[],(i::l))) =
                          (a_id i,(map a_pmap2) l)
    | a_pmap1 x = ( print (xml2string x) ; raise UnexpectedTag )
                          
  and a_pmap2 (tag ("pred-type-name",[],[x,y])) =
                          (a_pred_type x,a_pred_name y)
    | a_pmap2 x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_ext_signature (tag ("ext-signature",[],(x::(y::l)))) =
                       (a_sign x,a_sign y,(map a_l_formula) l)
    | a_ext_signature x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_sign (tag ("sign",[],[x])) = a_local_env x
    | a_sign x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_local_env (tag ("local-env",[],[a,b,c,d])) =
                   (a_subsort_env a,a_var_env b,a_fun_env c,a_pred_env d)
    | a_local_env x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_subsort_env (tag ("subsort-env",[],x)) = Symtab_id.make (a_senv x)
    | a_subsort_env x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_senv x = (map a_senv1) x
  
  and a_senv1 (tag ("id-sorts",[],(i::l))) = (a_id i,(map a_sort) l)
    | a_senv1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_var_env (tag ("var-env",[],x)) = Symtab_sid.make (a_venv x)
    | a_var_env x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_venv x = (map a_venv1) x
  
  and a_venv1 (tag ("sid-sort",[],[x,y])) = (a_simple_id x,a_sort y)
    | a_venv1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_fun_env (tag ("fun-env",[],x)) = Symtab_id.make (a_fenv x)
    | a_fun_env x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_fenv x = (map a_fenv1) x
  
  and a_fenv1 (tag ("id-op-types",[],(i::l))) = (a_id i,(map a_op_type) l)
    | a_fenv1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_pred_env (tag ("pred-env",[],x)) = Symtab_id.make (a_penv x)
    | a_pred_env x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_penv x = (map a_penv1) x
  
  and a_penv1 (tag ("id-pred-types",[],(i::l))) = (a_id i,(map a_pred_type) l)
    | a_penv1 x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* types from symmaps/symmaps_types.ml                                   *)
  (* --------------------------------------------------------------------- *)

  and a_symbol (tag ("symbol",[],[x])) = a_symbol1 x
    | a_symbol x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_symbol1 (tag ("sort-symbol",[],[x])) = SORT_SYMBOL (a_sort x)
    | a_symbol1 (tag ("total-fun-symbol",[],[x,y,z])) =
                 TOTAL_FUN_SYMBOL (a_op_name x,(strip_sorts (a_sorts y),
                                   a_sort z))
    | a_symbol1 (tag ("partial-fun-symbol",[],[x,y,z])) =
                 PARTIAL_FUN_SYMBOL (a_op_name x,(strip_sorts (a_sorts y),
                                     a_sort z))
    | a_symbol1 (tag ("pred-symbol",[],(n::l))) =
                 PRED_SYMBOL (a_pred_name n,(map a_sort) l)
    | a_symbol1 x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* types from arch/arch_types.ml                                         *)
  (* --------------------------------------------------------------------- *)

  and a_arch_sig (tag ("arch-sig",[],[x,y])) =
                  (a_named_unit_sigs x,a_unit_sig y)
    | a_arch_sig x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_sig (tag ("unit-sig",[],[x,y])) =
                  (a_comp_sub_sigs x,a_sub_sig y)
    | a_unit_sig x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_comp_sub_sigs (tag ("comp-sub-sigs",[],l)) =
                       (map a_sub_sig) l
    | a_comp_sub_sigs x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_named_unit_sigs (tag ("named-unit-sigs",[],x)) =
                         Symtab_sid.make (a_nsub x)
    | a_named_unit_sigs x = ( print (xml2string x) ; raise UnexpectedTag )
                         
  and a_nsub x = (map a_nsub1) x
  
  and a_nsub1 (tag ("sig-unit-tag-sigs",[],[x,y])) =
               (a_simple_id x,a_unit_tag_sig y)
    | a_nsub1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_unit_tag_sig (tag ("unit-tag-sig",[],[x,y])) =
                      (a_comp_tag_sigs x,a_tag_sig y)
    | a_unit_tag_sig x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_comp_tag_sigs (tag ("comp-tag-sigs",[],l)) =
                       (map a_tag_sig) l
    | a_comp_tag_sigs x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_tag_sig (tag ("tag-sig",[],[x,y])) =
                 (a_sub_sig x,a_tagging_map y)
    | a_tag_sig x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_tagging_map (tag ("tagging-map",[],x)) = Symtab_esym.make (a_tmap x)
    | a_tagging_map x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_tmap x = (map a_tmap1) x
  
  and a_tmap1 (tag ("e-ref-sym",[],(s::l))) =
               (a_e_symbol s,(map a_ref_sym) l)
    | a_tmap1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_ref_sym (tag ("ref-sym",[],[x,y])) =
                 (a_unit_name x,a_e_symbol y)
    | a_ref_sym x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_e_symbol (tag ("e-symbol",[],[x])) = a_e_symbol1 x
    | a_e_symbol x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_e_symbol1 (tag ("sym-esymbol",[],[x])) = SYM_ESYMBOL (a_symbol x)
    | a_e_symbol1 (tag ("emb-esymbol",[],[x,y])) = EMB_ESYMBOL (a_sort x,
                                                                a_sort y)
    | a_e_symbol1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_sub_sig (tag ("sub-sig",[],[x])) = a_local_env x
    | a_sub_sig x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* fcasenv                                                               *)
  (* --------------------------------------------------------------------- *)

  and a_f_global_env (tag ("casl",[],[x])) = a_f_global_env1 x
    | a_f_global_env (tag ("fcasenv",[],x)) = a_f_global_env1 
                                              (tag ("fcasenv",[],x))
    | a_f_global_env x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_f_global_env1 (tag ("fcasenv",[],l)) =
                       (Symtab_sid.make (a_f_genv (split_first l)),
                        (map a_anno) (split_second l))
    | a_f_global_env1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_f_genv x = (map a_f_genv1) x
  
  and a_f_genv1 (tag ("sid-f-global-entry",[],[x,y])) =
                 (a_simple_id x,a_f_global_entry y)
    | a_f_genv1 x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_f_global_entry (tag ("f-global-entry",[],[x])) = a_f_global_entry1 x
    | a_f_global_entry x = ( print (xml2string x) ; raise UnexpectedTag )
    
  and a_f_global_entry1 (tag ("f-spec-defn-env",[],[x,y])) =
                         f_spec_defn_env (a_f_genericity_env x,
                                          a_ext_signature y)
    | a_f_global_entry1 (tag ("f-view-defn-env",[],[a,b,c,d])) =
                         f_view_defn_env (a_f_genericity_env a,
                                          a_ext_signature b,
                                          a_morphism c,
                                          a_ext_signature d)
    | a_f_global_entry1 (empty_tag ("f-arch-spec-defn-env",[])) =
                         f_arch_spec_defn_env
    | a_f_global_entry1 (empty_tag ("f-unit-spec-defn-env",[])) =
                         f_unit_spec_defn_env
    | a_f_global_entry1 x = ( print (xml2string x) ; raise UnexpectedTag )

  and a_f_genericity_env (tag ("f-genericity-env",[],[x,y])) =
                          (a_ext_signature x,a_ext_signatures y)
    | a_f_genericity_env x = ( print (xml2string x) ; raise UnexpectedTag )
  
  and a_ext_signatures (tag ("ext-signatures",[],l)) =
                        (map a_ext_signature) l
    | a_ext_signatures x = ( print (xml2string x) ; raise UnexpectedTag )

  (* --------------------------------------------------------------------- *)
  (* end of code                                                           *)
  (* --------------------------------------------------------------------- *)

  end (* local open ... in *)

end
