{- Graph defines the Graph class, which defines the basic things a
   graph must do.  Peculiarities:
   (1) Graphs are directed with labelled nodes and
       arcs.  These nodes and arcs have types. 
   (2) The nodes and arcs are identified by values of type Node and Arc.
       These values are essentially strings.  The strings are provided by
       the user; there is no mechanism for generating new unique strings.
       (This is because this is easy in the applications I have in mind.)
   (3) A necessary feature of these graphs is that it is supposed to
       be easy generate copies, both on the same system and on others.
 -}
module Graph(
   Graph(..), -- the Graph class
   -- Instances are parameterised on 
   -- nodeLabel, nodeTypeLabel, arcLabel, arcTypeLabel.

   -- Nodes, Arc, NodeTypes, Arc
   Node, Arc, NodeType, ArcType,
   -- These are all instances of AtomString.StringClass (and so Read & Show).
   -- This means that they are essentially strings; the different types
   -- are just there to add a little abstraction.
   -- They are also all instances of Eq and Ord.  However there
   -- is no guarantee that the ordering will be the same as for the
   -- corresponding strings.

   -- They are also instances of Typeable.

   -- Updates
   Update(..),
   -- datatype encoding update to shared graph
   -- Like instances of Graph, parameterised on 
   -- nodeLabel, nodeTypeLabel, arcLabel, arcTypeLabel.
   -- Derives Read and Show.

   CannedGraph(..),
   -- contains complete immutable contents of a Graph at some time
   -- Like instances of Graph, parameterised on
   -- nodeLabel, nodeTypeLabel, arcLabel, arcTypeLabel.
   -- Derives Read and Show.

   GraphConnection,
   GraphConnectionData(..),
   -- A GraphConnection contains the information generated by one
   -- instance of Graph, which can be used to construct another,
   -- including a CannedGraph.
   -- Like instances of Graph, parameterised on 
   -- nodeLabel, nodeTypeLabel, arcLabel, arcTypeLabel.

   -- Utility functions.
   newEmptyGraph, 
   -- :: Graph graph =>
   --    IO (graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel)

   ) where

import Control.Concurrent.MVar

import AtomString
import Computation(done)
import Dynamics
import NewNames

class Graph graph where
   -- access functions
   getNodes :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> IO [Node]
   getArcs :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> IO [Arc]
   getNodeTypes :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> IO [NodeType]
   getArcTypes :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> IO [ArcType]

   getArcsOut :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Node -> IO [Arc]
   getArcsIn :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Node -> IO [Arc]
   getNodeLabel :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Node -> IO nodeLabel
   getNodeType :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Node -> IO NodeType
   getNodeTypeLabel :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> NodeType -> IO nodeTypeLabel

   getSource :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Arc -> IO Node
   getTarget :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Arc -> IO Node
   getArcLabel :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Arc -> IO arcLabel
   getArcType :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Arc -> IO ArcType
   getArcTypeLabel :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> ArcType -> IO arcTypeLabel

   shareGraph :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> GraphConnection nodeLabel nodeTypeLabel arcLabel arcTypeLabel
   newGraph :: GraphConnection nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> IO (graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel)

   -- Functions for changing the state.
   newNodeType :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> nodeTypeLabel -> IO NodeType
   newNode :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> NodeType -> nodeLabel -> IO Node
   newArcType :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> arcTypeLabel -> IO ArcType
   newArc :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> ArcType -> arcLabel -> Node -> Node -> IO Arc 

   -- Other updates, such as deletions should be done with the update 
   -- function.  It is also possible to add nodes, arcs, arctypes and
   -- nodetypes using update; however in this case the caller is responsible
   -- for providing a globally new label.  
   update :: graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> Update nodeLabel nodeTypeLabel arcLabel arcTypeLabel -> IO ()

   newEmptyGraph :: IO (graph nodeLabel nodeTypeLabel arcLabel arcTypeLabel)
   -- Actually newEmptyGraph can be synthesised from the above functions
   -- by synthesising a null GraphConnection and passing it to newGraph.

------------------------------------------------------------------------
-- GraphConnection
------------------------------------------------------------------------

type GraphConnection nodeLabel nodeTypeLabel arcLabel arcTypeLabel =
   (Update nodeLabel nodeTypeLabel arcLabel arcTypeLabel -> IO ())
   -> IO (GraphConnectionData nodeLabel nodeTypeLabel arcLabel arcTypeLabel)
   -- The first argument is passed back to the parent graph and 
   -- indicates where to put changes to the parent graph since the
   -- canned graph was made.

data GraphConnectionData nodeLabel nodeTypeLabel arcLabel arcTypeLabel = 
      GraphConnectionData {   
   graphState :: CannedGraph nodeLabel nodeTypeLabel arcLabel arcTypeLabel, 
      -- current state of graph
   deRegister :: IO (),
      -- disables graphUpdates
   graphUpdate :: Update nodeLabel nodeTypeLabel arcLabel arcTypeLabel 
      -> IO(),
      -- Similar to update (in class definition) except that
      -- it doesn't get echoed on graphUpdates.
   nameSourceBranch :: NameSourceBranch
      -- A source of new names.  Each graph should contain a NameSource
      -- to generate new node strings.
   }

------------------------------------------------------------------------
-- Nodes, Arcs, NodeTypes, ArcTypes.
------------------------------------------------------------------------

newtype Node = Node AtomString deriving (Eq,Ord)

instance StringClass Node where
   toString (Node atomString) = toString atomString
   fromString atomString = Node (fromString atomString)

node_tag = mkTyRep "Graph" "Node"

instance HasTyRep Node where
   tyRep _ = node_tag

newtype NodeType = NodeType AtomString deriving (Eq,Ord)

instance StringClass NodeType where
   toString (NodeType atomString) = toString atomString
   fromString atomString = NodeType (fromString atomString)

nodeType_tag = mkTyRep "Graph" "NodeType"

instance HasTyRep NodeType where
   tyRep _ = nodeType_tag

newtype Arc = Arc AtomString deriving (Eq,Ord)

instance StringClass Arc where
   toString (Arc atomString) = toString atomString
   fromString atomString = Arc (fromString atomString)

arc_tag = mkTyRep "Graph" "Arc"

instance HasTyRep Arc where
   tyRep _ = arc_tag

newtype ArcType = ArcType AtomString deriving (Eq,Ord)

instance StringClass ArcType where
   toString (ArcType atomString) = toString atomString
   fromString atomString = ArcType (fromString atomString)

arcType_tag = mkTyRep "Graph" "ArcType"

instance HasTyRep ArcType where
   tyRep _ = arcType_tag

------------------------------------------------------------------------
-- Update
------------------------------------------------------------------------

data Update nodeLabel nodeTypeLabel arcLabel arcTypeLabel =
   -- NB.  For various reasons, we decree that DeleteNode and DeleteArc should
   -- return normally, doing nothing, should the node already be deleted.
      NewNodeType NodeType nodeTypeLabel
   |  SetNodeTypeLabel NodeType nodeTypeLabel
   |  NewNode Node NodeType nodeLabel 
   |  DeleteNode Node
   |  SetNodeLabel Node nodeLabel
   |  NewArcType ArcType arcTypeLabel
   |  SetArcTypeLabel ArcType arcTypeLabel
   |  NewArc Arc ArcType arcLabel Node Node
   |  DeleteArc Arc
   |  SetArcLabel Arc arcLabel
   deriving (Read,Show)

------------------------------------------------------------------------
-- CannedGraph
------------------------------------------------------------------------

data CannedGraph nodeLabel nodeTypeLabel arcLabel arcTypeLabel =
   CannedGraph {
      updates :: [Update nodeLabel nodeTypeLabel arcLabel arcTypeLabel]
      -- This list should have the following structure:
      -- (a) a sequence of NewNodeType's
      -- (b) a sequence of NewNode's
      -- (c) a sequence of NewArcType's
      -- (d) a sequence of NewArc's.
      } deriving (Read,Show)

emptyCannedGraph :: CannedGraph nodeLabel nodeTypeLabel arcLabel arcTypeLabel
emptyCannedGraph = CannedGraph {
   updates = []
   }
