(* *********************************************************************** *)
(* Project: CATS                                                           *)
(* Author: Pascal Schmidt, University of Bremen (Student)                  *)
(* Purpose of this file: parsing and conversion of ATerms format           *)
(* *********************************************************************** *)

(* $Id: aterms.sml,v 1.1 2004/02/13 11:29:07 7till Exp $ *)

(* todo:

   comment => comment_line
     sml-110-Version der ATerm Library: Datenstruktur anpassen, sobald von
     Amsterdamern Rueckmeldung kommt  

  ATerm-textual sharing-Algorithmus an Mark

  Evtl. an den AS-Baum nur FCasEnv dranhaengen???
  
ATerm-Ausgabe ueber XML
  val lookup_multi: 'a list table * key -> 'a list
  val make_multi: (key * 'a) list -> 'a list table
  val dest_multi: 'a list table -> (key * 'a) list

   ATerm-Eingabe ueber XML
   Doku dafuer 
   automatische Doku-Erzeugung fuer Aterms
   
   DTD nach /home/cofi/www/CASL/xml

  Test (inkl. Performance) groesserer Bsp. (einlesen)

*)


signature ATermFunctions
=
sig

  (* basic aterm datatype for storage of ATerms                    *)
  (* last aterm list in AFun, AAppl and AList contains annotations *)

  datatype aterm =   AFun of (afun * aterm list)
                   | AAppl of (afun * aterm list * aterm list)
                   | AList   of (aterm list * aterm list)
  and afun =         AId of string 
                   | AString  of string
                   | ANumber of int

  (* read_casfix and load_casfix do all steps of parsing, read     *)
  (*  takes an aterm string, load takes a filename                 *)

  val read_casfix      : string -> GlobalEnv.global_env * AS.LIB_DEFN
  val load_casfix      : string -> GlobalEnv.global_env * AS.LIB_DEFN

  (* These four functions duplicate the interface of the old       *)
  (* routines from mkstring.sml                                    *)
  
  val mkA              : AS.LIB_DEFN -> string   (* ATerm w/o HTML *)
  val mkAT             : AS.LIB_DEFN -> string   (* ATerm w/ HTML  *)
  val mkA_file         : string -> AS.LIB_DEFN -> unit
  
  val mkA_genv         : GlobalEnv.global_env * AS.LIB_DEFN -> string
  val mkAT_genv        : GlobalEnv.global_env * AS.LIB_DEFN -> string
  val mkA_genv_file    : string -> GlobalEnv.global_env * AS.LIB_DEFN -> unit

  val S_GLOBAL_ENV     : GlobalEnv.global_env -> AS.LIB_DEFN -> aterm
  val S_F_GLOBAL_ENV   : GlobalEnv.f_global_env -> aterm
  
  val conv_at : aterm -> aterm
  val reconv_at : aterm -> aterm
end


structure AT : ATermFunctions
=
struct

open Utils AS GlobalEnv ATDef TextIO LocalEnv Symbols ArchTypes

fun outs (id:AS.SIMPLE_ID) = (writeln ("ATerming "^BasicPrint.print_SIMPLE_ID id);id)

(* useful exceptions *)

exception ATerms
exception ATerms_UnexpectedTerm
exception ATerms_WrongType
exception ATerms_CannotConvert
exception ATerms_ConvertFailure


(* helper functions *)

(* val eparse = no_assoc:PARSING_ANNO *)

fun init []     = raise ERROR
  | init [h]    = []
  | init (h::t) = h :: (init t)

fun strip s = (String.implode o init o tl o String.explode) s

fun at_esc [] = []
  | at_esc ("\\"::t) = (explode "\\\\") @ (at_esc t)
  | at_esc ("\""::t) = (explode "\\\"") @ (at_esc t)
  | at_esc ("'"::t)  = (explode "\\'")  @ (at_esc t)
  | at_esc (h::t)    = h :: (at_esc t)

fun at_escape s = (implode o at_esc o explode) s

fun at_unesc [] = []
  | at_unesc ("\\"::("\\"::t)) = "\\" :: (at_unesc t)
  | at_unesc ("\\"::("\""::t)) = "\"" :: (at_unesc t)
  | at_unesc ("\\"::("'"::t))  = "'"  :: (at_unesc t)
  | at_unesc (h::t)            = h    :: (at_unesc t)

fun at_unescape s = (implode o at_unesc o explode) s



(* functions for annotation conversion *)

fun conv_at (AFun (x,a))    = AFun (x,conv_ann a)
  | conv_at (AAppl (x,y,a)) = AAppl (x,(map conv_at) y,conv_ann a)
  | conv_at (AList (x,a))   = AList ((map conv_at) x,conv_ann a)

and conv_ann x = (map conv_anno) x

and conv_anno (AFun (x,a))    = AList ([AFun (x,[]),AFun (x,a)],[])
  | conv_anno (AAppl (x,y,a)) = AList ([AFun (x,[]),AAppl (x,y,a)],[])
  | conv_anno _               = raise ATerms_ConvertFailure


fun reconv_at (AFun (x,a))    = AFun (x,reconv_ann a)
  | reconv_at (AAppl (x,y,a)) = AAppl (x,(map reconv_at) y,reconv_ann a)
  | reconv_at (AList (x,a))   = AList ((map reconv_at) x,reconv_ann a)

and reconv_ann x = (map reconv_anno) x

and reconv_anno (AList ([AFun (i,j),x],[])) = x
  | reconv_anno _ = raise ATerms_ConvertFailure



(* functions for CasFix to AS tree conversion *)

fun AS_SIMPLE_ID (AAppl (AId "SIMPLE-ID",[AAppl (AId "WORDS",[AFun (AString
                  id,ann1)],ann2)],ann3))
                 = ((id), (None:int option)):SIMPLE_ID
  | AS_SIMPLE_ID (AAppl (AId "WORDS",[AFun (AString id,ann1)],ann2))
                 = ((id), (None:int option)):SIMPLE_ID
  | AS_SIMPLE_ID _ = raise ATerms_UnexpectedTerm

and AS_LIB_ITEM (AAppl (AId "spec-defn",[n,g,s],[ann1]))
                = spec_defn ( (AS_SPEC_NAME n),(AS_GENERICITY g),(AS_SPEC s),
                  AS_ANNO_LIST ann1 )
  | AS_LIB_ITEM (AAppl (AId "view-defn",[n,g,t,AAppl (AId "SYMB-MAP-ITEMS*",
                 [AList (l,ann1)],ann2)],[ann3]))
                = view_defn ( (AS_VIEW_NAME n), (AS_GENERICITY g),
                              (AS_VIEW_TYPE t),
                              (map AS_SYMB_MAP_ITEMS) l,AS_ANNO_LIST ann3 )
  | AS_LIB_ITEM (AAppl (AId "arch-spec-defn",[n,s],[ann1]))
                = arch_spec_defn ( (AS_ARCH_SPEC_NAME n),(AS_ARCH_SPEC s),
                  AS_ANNO_LIST ann1 )
  | AS_LIB_ITEM (AAppl (AId "unit-spec-defn",[n,s],[ann1]))
                = unit_spec_defn ( (AS_SPEC_NAME n),(AS_UNIT_SPEC s),
                  AS_ANNO_LIST ann1 )
  | AS_LIB_ITEM (AAppl (AId "download-items",[n,AAppl (AId "ITEM-NAME-OR-MAP+",
             [AList (l,[])],[])],[ann1]))
                = download_items (AS_LIB_NAME n,(map AS_ITEM_NAME_OR_MAP) l,
                  AS_ANNO_LIST ann1)
  | AS_LIB_ITEM _ = raise ATerms_UnexpectedTerm

and AS_ARCH_SPEC_NAME x = AS_SIMPLE_ID x

and AS_UNIT_NAME x = AS_SIMPLE_ID x

and AS_ARCH_SPEC (AAppl (AId "basic-arch-spec",[AAppl (AId "UNIT-DECL-DEFN+",
                  [AList (l,ann1)],ann2),r],[ann3,ann4]))
                 = (basic_arch_spec ( (map AS_UNIT_DECL_DEFN) l,
                                     (AS_RESULT_UNIT r),AS_ANNO_LIST ann3 )
                                     ,AS_ANNO_LIST ann4)
  | AS_ARCH_SPEC (AAppl (AId "ARCH-SPEC-NAME",[n],[ann1]))
                 = (named_arch_spec ( (AS_ARCH_SPEC_NAME n) ),AS_ANNO_LIST ann1)
  | AS_ARCH_SPEC _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_DECL_DEFN (AAppl (AId "UNIT-DECL",[d],[ann1]))
                      = (unit_decl_case ( (AS_UNIT_DECL d) ),AS_ANNO_LIST ann1)
  | AS_UNIT_DECL_DEFN (AAppl (AId "UNIT-DEFN",[d],[ann1]))
                      = (unit_defn_case ( (AS_UNIT_DEFN d) ),AS_ANNO_LIST ann1)
  | AS_UNIT_DECL_DEFN _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_DECL (AAppl (AId "unit-decl",[n,s,i],ann1))
                 = unit_decl ( (AS_UNIT_NAME n),(AS_UNIT_SPEC s),
                               (AS_UNIT_IMPORTS i) )
  | AS_UNIT_DECL _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_IMPORTS (AAppl (AId "unit-imports",[AAppl (AId "UNIT-TERM*",
                     [AList (l,ann1)],ann2)],ann3))
                    = unit_imports ( (map AS_UNIT_TERM) l )
  | AS_UNIT_IMPORTS _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_DEFN (AAppl (AId "unit-defn",[n,e],ann1))
                 = unit_defn ( (AS_UNIT_NAME n),(AS_UNIT_EXPRESSION e) )
  | AS_UNIT_DEFN _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_SPEC (AAppl (AId "UNIT-TYPE",[t],ann1))
                 = unit_type_case (AS_UNIT_TYPE t)
  | AS_UNIT_SPEC (AAppl (AId "SPEC-NAME",[n],ann1))
                 = spec_name_case (AS_SPEC_NAME n)
  | AS_UNIT_SPEC (AAppl (AId "ARCH-SPEC",[s],ann1))
                 = arch_spec_case (AS_ARCH_SPEC s)
  | AS_UNIT_SPEC (AAppl (AId "closed",[s],ann1))
                 = closed (AS_UNIT_SPEC s)
  | AS_UNIT_SPEC _ = raise ATerms_UnexpectedTerm
                 
and AS_UNIT_TYPE (AAppl (AId "unit-type",[AAppl (AId "SPEC*", [AList
                  (l,ann1)],ann2),s],ann3))
                 = unit_type ( (map AS_SPEC) l,(AS_SPEC s) )
  | AS_UNIT_TYPE _ = raise ATerms_UnexpectedTerm
  
and AS_RESULT_UNIT (AAppl (AId "result-unit",[e],[ann1]))
                   = result_unit (AS_UNIT_EXPRESSION e,AS_ANNO_LIST ann1)
  | AS_RESULT_UNIT _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_EXPRESSION (AAppl (AId "unit-expression",[AAppl (AId
                        "UNIT-BINDING*",[AList (l,ann1)],ann2),t],ann3))
                       = unit_expression ( (map AS_UNIT_BINDING) l,
                                           (AS_UNIT_TERM t) )
  | AS_UNIT_EXPRESSION _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_BINDING (AAppl (AId "unit-binding",[n,s],ann1))
                    = unit_binding ( (AS_UNIT_NAME n),(AS_UNIT_SPEC s) )
  | AS_UNIT_BINDING _ = raise ATerms_UnexpectedTerm
  
and AS_UNIT_TERM (AAppl (AId "unit-translation",[t,r],ann1))
                 = unit_translation ( (AS_UNIT_TERM t),(AS_RENAMING r) )
  | AS_UNIT_TERM (AAppl (AId "unit-reduction",[t,r],ann1))
                 = unit_reduction ( (AS_UNIT_TERM t),(AS_RESTRICTION r) )
  | AS_UNIT_TERM (AAppl (AId "amalgamation",[AAppl (AId "UNIT-TERM+",
                  [AList (l,ann1)],ann2)],ann3))
                 = amalgamation ( (map AS_UNIT_TERM) l )
  | AS_UNIT_TERM (AAppl (AId "local-unit",[AAppl (AId "UNIT-DEFN+",[AList
                  (l,ann1)],ann2),t],ann3))
                 = local_unit ( (map AS_UNIT_DEFN) l,(AS_UNIT_TERM t) )
  | AS_UNIT_TERM (AAppl (AId "unit-appl",[n,AAppl (AId "FIT-ARG-UNIT*",
                  [AList (l,ann1)],ann2)],ann3))
                 = unit_appl ( (AS_UNIT_NAME n),(map AS_FIT_ARG_UNIT) l )
  | AS_UNIT_TERM _ = raise ATerms_UnexpectedTerm

and AS_FIT_ARG_UNIT (AAppl (AId "fit-arg-unit",[t,AAppl (AId
                     "SYMP-MAP-ITEMS*",[AList (l,ann1)],ann2)],ann3))
                    = fit_arg_unit ( (AS_UNIT_TERM t),
                                     (map AS_SYMB_MAP_ITEMS) l )
  | AS_FIT_ARG_UNIT _ = raise ATerms_UnexpectedTerm

and AS_VIEW_TYPE (AAppl (AId "view-type",[sp1,sp2],ann1))
                 = view_type ( (AS_SPEC sp1),(AS_SPEC sp2) )
  | AS_VIEW_TYPE _ = raise ATerms_UnexpectedTerm

and AS_SPEC_NAME x = AS_SIMPLE_ID x

and AS_GENERICITY (AAppl (AId "genericity",[p,i],ann1))
                  = genericity ( (AS_PARAMS p),(AS_IMPORTS i) )
  | AS_GENERICITY _ = raise ATerms_UnexpectedTerm

and AS_PARAMS (AAppl (AId "params",[AAppl (AId "SPEC*",[AList (sp,ann1)],
               ann2)],ann3))
              = params ( (map AS_SPEC) sp )
  | AS_PARAMS _ = raise ATerms_UnexpectedTerm

and AS_IMPORTS (AAppl (AId "imports",[AAppl (AId "SPEC*",[AList (sp,ann1)],
                ann2)],ann3))
               = imports ( (map AS_SPEC) sp )
  | AS_IMPORTS _ = raise ATerms_UnexpectedTerm

and AS_SPEC (AAppl (AId "BASIC-SPEC",[bs],[ann1]))
            = (basic (AS_BASIC_SPEC bs),AS_ANNO_LIST ann1)
  | AS_SPEC (AAppl (AId "union",[AAppl (AId "SPEC+",[AList (sp,ann1)],
             ann2)],[ann3]))
            = (union_spec ( (map AS_ASPEC) sp ),AS_ANNO_LIST ann3)
  | AS_SPEC (AAppl (AId "extension",[AAppl (AId "SPEC+",[AList (sp,ann1)],
             ann2)],[ann3]))
            = (extension ( (map AS_ASPEC) sp ),AS_ANNO_LIST ann3)
  | AS_SPEC (AAppl (AId "free-spec",[sp],[ann1,ann2]))
            = (free_spec (AS_SPEC sp,AS_ANNO_LIST ann1),AS_ANNO_LIST ann2)
  | AS_SPEC (AAppl (AId "cofree-spec",[sp],[ann1,ann2]))
            = (cofree_spec (AS_SPEC sp,AS_ANNO_LIST ann1),AS_ANNO_LIST ann2)
  | AS_SPEC (AAppl (AId "local-spec",[sp1,sp2],[ann1,ann2,ann3]))
            = (local_spec (AS_SPEC sp1,AS_ANNO_LIST ann1,AS_SPEC sp2,
               AS_ANNO_LIST ann2),AS_ANNO_LIST ann3)
  | AS_SPEC (AAppl (AId "translation",[sp,r],[ann1,ann2]))
            = (translation (AS_SPEC sp,AS_RENAMING r,AS_ANNO_LIST ann1),
               AS_ANNO_LIST ann2)
  | AS_SPEC (AAppl (AId "reduction",[sp,r],[ann1,ann2]))
            = (reduction (AS_SPEC sp,AS_RESTRICTION r,AS_ANNO_LIST ann1),
               AS_ANNO_LIST ann2)
  | AS_SPEC (AAppl (AId "closed-spec",[sp],[ann1,ann2]))
            = (closed_spec (AS_SPEC sp,AS_ANNO_LIST ann1),AS_ANNO_LIST ann2)
  | AS_SPEC (AAppl (AId "spec-inst",[n,AAppl (AId "FIT-ARG*",[AList (fa,ann1)],
             ann2)],[ann3]))
            = (spec_inst ( (AS_SPEC_NAME n), (map AS_FIT_ARG) fa ),
               AS_ANNO_LIST ann3)
  | AS_SPEC _ = raise ATerms_UnexpectedTerm

and AS_RENAMING (AAppl (AId "renaming",[AAppl (AId "SYMB-MAP-ITEMS+",
                 [AList (l,ann1)],ann2)],ann3))
                = renaming ( (map AS_SYMB_MAP_ITEMS) l )
  | AS_RENAMING x = raise ATerms_UnexpectedTerm
  
and AS_SYMB_MAP_ITEMS (AAppl (AId "symb-map-items",[sk,AAppl (AId
                       "SYMB-OR-MAP+",[AList (l,ann1)],ann2)],ann3))
                      = symb_map_items ( (AS_SYMB_KIND sk),
                                         (map AS_SYMB_OR_MAP) l )
  | AS_SYMB_MAP_ITEMS _ = raise ATerms_UnexpectedTerm

and AS_SYMB_KIND (AFun (AId "implicit",ann1)) = implicitk
  | AS_SYMB_KIND (AFun (AId "sorts",ann1))    = sortsk
  | AS_SYMB_KIND (AFun (AId "ops",ann1))      = opsk
  | AS_SYMB_KIND (AFun (AId "preds",ann1))    = predsk
  | AS_SYMB_KIND _                            = raise ATerms_UnexpectedTerm
  
and AS_SYMB_OR_MAP (AAppl (AId "SYMB",[s],ann1)) = symb (AS_SYMB s)
  | AS_SYMB_OR_MAP (AAppl (AId "SYMB-MAP",[sm],ann1))
                   = symb_or_map (AS_SYMB_MAP sm)
  | AS_SYMB_OR_MAP _ = raise ATerms_UnexpectedTerm

and AS_SYMB (AAppl (AId "ID",[i],ann1)) = symb_id (AS_ID i)
  | AS_SYMB (AAppl (AId "qual-id",[i,t],ann1))
            = qual_id ( (AS_ID i),(AS_TYPE t) )
  | AS_SYMB _ = raise ATerms_UnexpectedTerm

and AS_TYPE (AAppl (AId "OP-TYPE",[t],ann1)) = op_symb_type (AS_OP_TYPE t)
  | AS_TYPE (AAppl (AId "PRED-TYPE",[t],ann1))
            = pred_symb_type (AS_PRED_TYPE t)
  | AS_TYPE _ = raise ATerms_UnexpectedTerm

and AS_SYMB_MAP (AAppl (AId "symb-map",[s1,s2],ann1))
                = symb_map ( (AS_SYMB s1),(AS_SYMB s2) )
  | AS_SYMB_MAP _ = raise ATerms_UnexpectedTerm

and AS_OP_TYPE (AAppl (AId "total-op-type",[s,t],ann1))
               = total_op_type ( (AS_SORTS s),(AS_SORT t) )
  | AS_OP_TYPE (AAppl (AId "partial-op-type",[s,t],ann1))
               = partial_op_type ( (AS_SORTS s),(AS_SORT t) )
  | AS_OP_TYPE _ = raise ATerms_UnexpectedTerm

and AS_PRED_TYPE (AAppl (AId "pred-type",[s],ann1))
                 = pred_type ( (AS_SORTS s) )
  | AS_PRED_TYPE _ = raise ATerms_UnexpectedTerm

and AS_SORTS (AAppl (AId "sorts",[AAppl (AId "SORT*",[AList (l,ann1)],
              ann2)],ann3))
             = sorts ( (map AS_SORT) l )
  | AS_SORTS _ = raise ATerms_UnexpectedTerm

and AS_RESTRICTION (AAppl (AId "hide",[AAppl (AId "SYMB-ITEMS+",[AList
                    (l,ann1)],ann2)],ann3))
                   = hide ( (map AS_SYMB_ITEMS) l )
  | AS_RESTRICTION (AAppl (AId "reveal",[AAppl (AId "SYMB-MAP-ITEMS+",[AList
                    (l,ann1)],ann2)],ann3))
                   = reveal ( (map AS_SYMB_MAP_ITEMS) l )
  | AS_RESTRICTION _ = raise ATerms_UnexpectedTerm

and AS_SYMB_ITEMS (AAppl (AId "symb-items",[sk,AAppl (AId "SYMB+",[AList
                   (l,ann1)],ann2)],ann3))
                  = symb_items ( (AS_SYMB_KIND sk), (map AS_SYMB) l )
  | AS_SYMB_ITEMS _ = raise ATerms_UnexpectedTerm

and AS_FIT_ARG (AAppl (AId "fit-spec",[sp,AAppl (AId "SYMB-MAP-ITEMS*",
                [AList (l,ann1)],ann2)],[ann3]))
               = fit_spec ( (AS_SPEC sp),
                            (map AS_SYMB_MAP_ITEMS) l )
  | AS_FIT_ARG (AAppl (AId "fit-view",[vn,AAppl (AId "FIT-ARG*",
                [AList (l,ann1)],ann2)],ann3))
               = fit_view ( (AS_VIEW_NAME vn), (map AS_FIT_ARG) l )
  | AS_FIT_ARG _ = raise ATerms_UnexpectedTerm
  
and AS_VIEW_NAME x = AS_SIMPLE_ID x

and AS_BASIC_SPEC (AAppl (AId "basic-spec",[AAppl (AId "BASIC-ITEMS*", [AList
                   (l,ann1)],ann2)],ann3))
                  = basic_spec ( (map AS_BASIC_ITEMS) l )
  | AS_BASIC_SPEC _ = raise ATerms_UnexpectedTerm
  
and AS_BASIC_ITEMS (AAppl (AId "SIG-ITEMS",[i],ann1))
                   = sig_items (AS_SIG_ITEMS i)
  | AS_BASIC_ITEMS (AAppl (AId "free-datatype",[AAppl (AId "DATATYPE-DECL+",
                    [AList (l,ann1)],ann2)],[ann3]))
                   = free_datatype ( (map AS_DATATYPE_DECL) l,AS_ANNO_LIST ann3 )
  | AS_BASIC_ITEMS (AAppl (AId "cofree-datatype",[AAppl (AId "DATATYPE-DECL+",
                    [AList (l,ann1)],ann2)],[ann3]))
                   = cofree_datatype ( (map AS_DATATYPE_DECL) l,AS_ANNO_LIST ann3 )
  | AS_BASIC_ITEMS (AAppl (AId "sort-gen",[AAppl (AId "SIG-ITEMS+",
                    [AList (l,ann1)],ann2)],[ann3]))
                   = sort_gen ( (map AS_SIG_ITEMS) l,AS_ANNO_LIST ann3 )
  | AS_BASIC_ITEMS (AAppl (AId "sort-cogen",[AAppl (AId "SIG-ITEMS+",
                    [AList (l,ann1)],ann2)],[ann3]))
                   = sort_cogen ( (map AS_SIG_ITEMS) l,AS_ANNO_LIST ann3 )
  | AS_BASIC_ITEMS (AAppl (AId "var-items",[AAppl (AId "VAR-DECL+",
                    [AList (l,ann1)],ann2)],[ann3]))
                   = var_items ( (map AS_VAR_DECL) l,AS_ANNO_LIST ann3 )
  | AS_BASIC_ITEMS (AAppl (AId "local-var-axioms",[AAppl (AId "VAR-DECL+",
                    [AList (l1,ann1)],ann2),AAppl (AId "FORMULA+", [AList
                    (l2,ann3)],ann4)],[ann5]))
                   = local_var_axioms ( (map AS_VAR_DECL) l1,
                                        (map AS_L_FORMULA) l2,AS_ANNO_LIST ann5 )
  | AS_BASIC_ITEMS (AAppl (AId "axiom-items",[AAppl (AId "FORMULA+",[AList
                    (l,ann1)],ann2)],[ann3]))
                   = axiom_items ( (map AS_L_FORMULA) l,AS_ANNO_LIST ann3 )
  | AS_BASIC_ITEMS _ = raise ATerms_UnexpectedTerm
  
and AS_SIG_ITEMS (AAppl (AId "sort-items",[AAppl (AId "SORT-ITEM+",[AList
                  (l,ann1)],ann2)],[ann3]))
                 = sort_items ( (map AS_SORT_ITEM) l,AS_ANNO_LIST ann3 )
  | AS_SIG_ITEMS (AAppl (AId "op-items",[AAppl (AId "OP-ITEM+",[AList
                  (l,ann1)],ann2)],[ann3]))
                 = op_items ( (map AS_OP_ITEM) l,AS_ANNO_LIST ann3 )
  | AS_SIG_ITEMS (AAppl (AId "pred-items",[AAppl (AId "PRED-ITEM+",[AList
                  (l,ann1)],ann2)],[ann3]))
                 = pred_items ( (map AS_PRED_ITEM) l,AS_ANNO_LIST ann3 )
  | AS_SIG_ITEMS (AAppl (AId "datatype-items",[AAppl (AId "DATATYPE-DECL+",
                  [AList (l,ann1)],ann2)],[ann3]))
                 = datatype_items ( (map AS_DATATYPE_DECL) l,
                                    AS_ANNO_LIST ann3 )
  | AS_SIG_ITEMS _ = raise ATerms_UnexpectedTerm
  
and AS_SORT_ITEM (AAppl (AId "sort-decl",[AAppl (AId "SORT+",[AList
                  (l,ann1)],ann2)],[ann3]))
                 = (sort_decl ( (map AS_SORT) l ),AS_ANNO_LIST ann3):L_SORT_ITEM
  | AS_SORT_ITEM (AAppl (AId "subsort-decl",[AAppl (AId "SORT+",[AList
                  (l,ann1)],ann2),s],[ann3]))
                 = (subsort_decl ( (map AS_SORT) l,(AS_SORT s) ),
                    AS_ANNO_LIST ann3):L_SORT_ITEM
  | AS_SORT_ITEM (AAppl (AId "subsort-defn",[s1,v,s2,f],[ann1,ann2]))
                 = (subsort_defn ( (AS_SORT s1),(AS_VAR v),(AS_SORT s2),
                   (AS_FORMULA f),(AS_ANNO_LIST ann1) ),
                   AS_ANNO_LIST ann2):L_SORT_ITEM
  | AS_SORT_ITEM (AAppl (AId "iso-decl",[AAppl (AId "SORT+",[AList
                  (l,ann1)],ann2)],[ann3]))
                 = (iso_decl ( (map AS_SORT) l ),AS_ANNO_LIST ann3):L_SORT_ITEM
  | AS_SORT_ITEM x = raise ATerms_UnexpectedTerm
                 
and AS_SORT x = AS_ID x

and AS_VAR x = AS_SIMPLE_ID x

(* AS_PRED_ITEM returns L_PRED_ITEM !! *)

and AS_PRED_ITEM (AAppl (AId "pred-decl",[AAppl (AId "PRED-NAME+",[AList
                  (l,ann1)],ann2),t],[ann3]))
                 = (pred_decl ( (map AS_PRED_NAME) l,(AS_PRED_TYPE t) ),
                    AS_ANNO_LIST ann3):L_PRED_ITEM
  | AS_PRED_ITEM (AAppl (AId "pred-defn",[n,h,f],[ann1,ann2]))
                 = (pred_defn ( (AS_PRED_NAME n),(AS_PRED_HEAD h),
                   (AS_L_FORMULA f),(AS_ANNO_LIST ann1) ),AS_ANNO_LIST ann2)
                   :L_PRED_ITEM
  | AS_PRED_ITEM _ = raise ATerms_UnexpectedTerm

and AS_PRED_NAME x = AS_ID x

and AS_PRED_HEAD (AAppl (AId "pred-head",[AAppl (AId "ARG-DECL*",[AList
                  (l,ann1)],ann2)],ann3))
                 = pred_head ( (map AS_ARG_DECL) l )
  | AS_PRED_HEAD _ = raise ATerms_UnexpectedTerm

(* AS_OP_ITEM returns L_OP_ITEM !! *)

and AS_OP_ITEM (AAppl (AId "op-decl",[AAppl (AId "OP-NAME+",[AList
                (l,ann1)],ann2),t,AAppl (AId "OP-ATTR*",[AList (a,ann3)],
                ann4)],[ann5]))
               = (op_decl ( (map AS_OP_NAME) l,(AS_OP_TYPE t),
                 (map AS_OP_ATTR) a ),AS_ANNO_LIST ann5)
                 :L_OP_ITEM
  | AS_OP_ITEM (AAppl (AId "op-defn",[n,h,t],[ann1,ann2]))
               = (op_defn ( (AS_OP_NAME n),(AS_OP_HEAD h),(AS_TERM t),
                 (AS_ANNO_LIST ann1) ),AS_ANNO_LIST ann2):L_OP_ITEM
  | AS_OP_ITEM _ = raise ATerms_UnexpectedTerm
  
and AS_OP_NAME x = AS_ID x

and AS_OP_ATTR (AFun (AId "associative",ann1)) = associative
  | AS_OP_ATTR (AFun (AId "commutative",ann1)) = commutative
  | AS_OP_ATTR (AFun (AId "idempotent",ann1))  = idempotent
  | AS_OP_ATTR (AAppl (AId "unit-op-attr",[t],ann1)) =
               unit_op_attr (AS_TERM t)
  | AS_OP_ATTR _ = raise ATerms_UnexpectedTerm
  
and AS_OP_HEAD (AAppl (AId "total-op-head",[AAppl (AId "ARG-DECL*",[AList
                (l,ann1)],ann2),s],ann3))
               = total_op_head ( (map AS_ARG_DECL) l,(AS_SORT s) )
  | AS_OP_HEAD (AAppl (AId "partial-op-head",[AAppl (AId "ARG-DECL*",[AList
                (l,ann1)],ann2),s],ann3))
               = partial_op_head ( (map AS_ARG_DECL) l,(AS_SORT s) )
  | AS_OP_HEAD _ = raise ATerms_UnexpectedTerm
  
and AS_ARG_DECL (AAppl (AId "arg-decl",[AAppl (AId "VAR+",[AList (l,ann1)],
                 ann2),s],ann3))
                = arg_decl ( (map AS_VAR) l,(AS_SORT s) )
  | AS_ARG_DECL _ = raise ATerms_UnexpectedTerm
  
and AS_DATATYPE_DECL (AAppl (AId "datatype-decl",[s,AAppl (AId
                      "ALTERNATIVE+",[AList (l,ann1)],ann2)],[ann3]))
                     = datatype_decl ( (AS_SORT s),(map AS_ALTERNATIVE) l,
                       AS_ANNO_LIST ann3 )
  | AS_DATATYPE_DECL _ = raise ATerms_UnexpectedTerm
  
(* AS_ALTERNATIVE returns L_ALTERNATIVE! *)  
  
and AS_ALTERNATIVE (AAppl (AId "total-construct",[n,AAppl (AId "COMPONENTS*",
                    [AList (l,ann1)],ann2)],[ann3]))
                   = (total_construct ( (AS_OP_NAME n),(map AS_COMPONENTS) l),
                      AS_ANNO_LIST ann3 ):L_ALTERNATIVE
  | AS_ALTERNATIVE (AAppl (AId "partial-construct",[n,AAppl (AId "COMPONENTS*",
                    [AList (l,ann1)],ann2)],[ann3]))
                   = (partial_construct((AS_OP_NAME n),(map AS_COMPONENTS) l),
                     AS_ANNO_LIST ann3 )
  | AS_ALTERNATIVE (AAppl (AId "subsorts",[AAppl (AId "SORT+",[AList
                    (l,ann1)],ann2)],[ann3]))
                   = (subsort ( (map AS_SORT) l), AS_ANNO_LIST ann3 )
  | AS_ALTERNATIVE _ = raise ATerms_UnexpectedTerm

and AS_COMPONENTS (AAppl (AId "total-select",[AAppl (AId "OP-NAME+",[AList
                   (l,ann1)],ann2),s],ann3))
                  = total_select ( (map AS_OP_NAME) l,(AS_SORT s) )
  | AS_COMPONENTS (AAppl (AId "partial-select",[AAppl (AId "OP-NAME+",[AList
                   (l,ann1)],ann2),s],ann3))
                  = partial_select ( (map AS_OP_NAME) l,(AS_SORT s) ) 
  | AS_COMPONENTS (AAppl (AId "SORT",[s],ann1))
                  = sort_component (AS_SORT s)
  | AS_COMPONENTS _ = raise ATerms_UnexpectedTerm
  
and AS_L_FORMULA (AFun (AId "true",[ann1])) =
               (atom (ttrue),AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AFun (AId "false",[ann1])) =
               (atom (ffalse),AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "predication",[p,t],[ann1]))
               = (atom ( predication ( (AS_PRED_SYMB p),(AS_TERMS t) ) ),
                 AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "definedness",[t],[ann1]))
               = (atom ( definedness (AS_TERM t) ),AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "existl-equation",[t1,t2],[ann1]))
               = (atom ( existl_equation ( (AS_TERM t1),(AS_TERM t2) ) ),
                 AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "strong-equation",[t1,t2],[ann1]))
               = (atom ( strong_equation ( (AS_TERM t1),(AS_TERM t2) ) ),
                 AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "membership",[t,s],[ann1]))
               = (atom ( membership ( (AS_TERM t),(AS_SORT s) ) ),
                 AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "negation",[f],[ann1]))
               = (negation (AS_FORMULA f),AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "equivalence",[f1,f2],[ann1]))
               = (equivalence ( (AS_FORMULA f1),(AS_FORMULA f2) ),
                 AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "implication",[f1,f2],[ann1]))
               = (implication ( (AS_FORMULA f1),(AS_FORMULA f2) ),
                 AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "disjunction",[AAppl (AId "FORMULA+",[AList
                (l,ann1)],ann2)],[ann3]))
               = (disjunction ( (map AS_FORMULA) l ),AS_ANNO_LIST ann3):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "conjunction",[AAppl (AId "FORMULA+",[AList
                (l,ann1)],ann2)],[ann3]))
               = (conjunction ( (map AS_FORMULA) l ),AS_ANNO_LIST ann3):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "quantification",[q,AAppl (AId "VAR-DECL+",
                [AList (l,ann1)],ann2),f],[ann3]))
               = (quantification ( (AS_QUANTIFIER q),(map AS_VAR_DECL) l,
                 (AS_FORMULA f) ),AS_ANNO_LIST ann3):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "unparsed-formula",[f],[ann1]))
               = (AS_UNPARSED_FORMULA f,AS_ANNO_LIST ann1):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "sort-gen-ax",[AAppl (AId "SORT*",[AList (l1,
                ann1)],ann2),AAppl (AId "OP-SYMB*",[AList (l2,ann3)],ann4)],
                [ann5]))
               = (sort_gen_ax ((map AS_SORT) l1,(map AS_OP_SYMB) l2),
                  AS_ANNO_LIST ann5):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "sort-cogen-ax",[AAppl (AId "SORT*",[AList (l1,
                ann1)],ann2),AAppl (AId "OP-SYMB*",[AList (l2,ann3)],ann4)],
                [ann5]))
               = (sort_cogen_ax ((map AS_SORT) l1,(map AS_OP_SYMB) l2),
                  AS_ANNO_LIST ann5):L_FORMULA
  | AS_L_FORMULA (AAppl (AId "sort-cofree-ax",[AAppl (AId "SORT*",[AList (l1,
                ann1)],ann2),AAppl (AId "OP-SYMB*",[AList (l2,ann3)],ann4)],
                [ann5]))
               = (sort_cofree_ax ((map AS_SORT) l1,(map AS_OP_SYMB) l2),
                  AS_ANNO_LIST ann5):L_FORMULA
  | AS_L_FORMULA _ = raise ATerms_UnexpectedTerm

and AS_FORMULA (AFun (AId "true",ann1)) = atom (ttrue)
  | AS_FORMULA (AFun (AId "false",ann1)) = atom (ffalse)
  | AS_FORMULA (AAppl (AId "predication",[p,t],ann1))
               = atom ( predication ( (AS_PRED_SYMB p),(AS_TERMS t) ) )
  | AS_FORMULA (AAppl (AId "definedness",[t],ann1))
               = atom ( definedness (AS_TERM t) )
  | AS_FORMULA (AAppl (AId "existl-equation",[t1,t2],ann1))
               = atom ( existl_equation ( (AS_TERM t1),(AS_TERM t2) ) )
  | AS_FORMULA (AAppl (AId "strong-equation",[t1,t2],ann1))
               = atom ( strong_equation ( (AS_TERM t1),(AS_TERM t2) ) )
  | AS_FORMULA (AAppl (AId "membership",[t,s],ann1))
               = atom ( membership ( (AS_TERM t),(AS_SORT s) ) )
  | AS_FORMULA (AAppl (AId "negation",[f],ann1))
               = negation (AS_FORMULA f)
  | AS_FORMULA (AAppl (AId "equivalence",[f1,f2],ann1))
               = equivalence ( (AS_FORMULA f1),(AS_FORMULA f2) )
  | AS_FORMULA (AAppl (AId "implication",[f1,f2],ann1))
               = implication ( (AS_FORMULA f1),(AS_FORMULA f2) )
  | AS_FORMULA (AAppl (AId "disjunction",[AAppl (AId "FORMULA+",[AList
                (l,ann1)],ann2)],ann3))
               = disjunction ( (map AS_FORMULA) l )
  | AS_FORMULA (AAppl (AId "conjunction",[AAppl (AId "FORMULA+",[AList
                (l,ann1)],ann2)],ann3))
               = conjunction ( (map AS_FORMULA) l )
  | AS_FORMULA (AAppl (AId "quantification",[q,AAppl (AId "VAR-DECL+",
                [AList (l,ann1)],ann2),f],ann3))
               = quantification ( (AS_QUANTIFIER q),(map AS_VAR_DECL) l,
                 (AS_FORMULA f) )
  | AS_FORMULA (AAppl (AId "unparsed-formula",[f],ann1))
               = AS_UNPARSED_FORMULA f
  | AS_FORMULA (AAppl (AId "sort-gen-ax",[AAppl (AId "SORT*",[AList (l1,
                ann1)],ann2),AAppl (AId "OP-SYMB*",[AList (l2,ann3)],ann4)],
                ann5))
               = sort_gen_ax ((map AS_SORT) l1,(map AS_OP_SYMB) l2)
  | AS_FORMULA (AAppl (AId "sort-cogen-ax",[AAppl (AId "SORT*",[AList (l1,
                ann1)],ann2),AAppl (AId "OP-SYMB*",[AList (l2,ann3)],ann4)],
                ann5))
               = sort_cogen_ax ((map AS_SORT) l1,(map AS_OP_SYMB) l2)
  | AS_FORMULA (AAppl (AId "sort-cofree-ax",[AAppl (AId "SORT*",[AList (l1,
                ann1)],ann2),AAppl (AId "OP-SYMB*",[AList (l2,ann3)],ann4)],
                ann5))
               = sort_cofree_ax ((map AS_SORT) l1,(map AS_OP_SYMB) l2)
  | AS_FORMULA _ = raise ATerms_UnexpectedTerm

and AS_UNPARSED_FORMULA (AFun (AString s,ann)) = unparsed_formula
                                                 ((at_unescape) s)
  | AS_UNPARSED_FORMULA _ = raise ATerms_UnexpectedTerm

and AS_QUANTIFIER (AFun (AId "forall",ann1))          = forall
  | AS_QUANTIFIER (AFun (AId "exists",ann1))          = exists
  | AS_QUANTIFIER (AFun (AId "exists-uniquely",ann1)) = exists_uniquely
  | AS_QUANTIFIER _                             = raise ATerms_UnexpectedTerm
  
and AS_TERMS (AAppl (AId "terms",[AAppl (AId "TERM*",[AList (l,ann1)],ann2)],
              ann3)) = terms ( (map AS_TERM) l )
  | AS_TERMS _ = raise ATerms_UnexpectedTerm
  
and AS_TERM (AAppl (AId "SIMPLE-ID",[i],ann1)) = var_or_const (AS_SIMPLE_ID i)
  | AS_TERM (AAppl (AId "qual-var",[v,s],ann1))
            = qual_var ( (AS_VAR v),(AS_SORT s) )
  | AS_TERM (AAppl (AId "application",[s,t],ann1))
            = application ( (AS_OP_SYMB s),(AS_TERMS t) )
  | AS_TERM (AAppl (AId "sorted-term",[t,s],ann1))
            = sorted_term ( (AS_TERM t),(AS_SORT s) )
  | AS_TERM (AAppl (AId "cast",[t,s],ann1))
            = cast ( (AS_TERM t),(AS_SORT s) )
  | AS_TERM (AAppl (AId "conditional",[t1,f,t2],ann1))
            = conditional ( (AS_TERM t1),(AS_FORMULA f),(AS_TERM t2) )
  | AS_TERM (AAppl (AId "unparsed-term",[t],ann1))
            = AS_UNPARSED_TERM t
  | AS_TERM _ = raise ATerms_UnexpectedTerm

and AS_UNPARSED_TERM (AFun (AString s,ann)) = unparsed_term
                                              ((at_unescape) s)
  | AS_UNPARSED_TERM _ = raise ATerms_UnexpectedTerm

and AS_VAR_DECL (AAppl (AId "var-decl",[AAppl (AId "VAR+",[AList (l,ann1)],
                 ann2),s],ann3))
                = ( (map AS_VAR) l,(AS_SORT s) ):VAR_DECL
  | AS_VAR_DECL _ = raise ATerms_UnexpectedTerm
  
and AS_PRED_SYMB (AAppl (AId "PRED-NAME",[n],ann1))
                 = pred_symb ( (AS_PRED_NAME n),None )
  | AS_PRED_SYMB (AAppl (AId "qual-pred-name",[n,t],ann1))
                 = pred_symb ( (AS_PRED_NAME n),Some (AS_PRED_TYPE t) )
  | AS_PRED_SYMB _ = raise ATerms_UnexpectedTerm
  
and AS_OP_SYMB (AAppl (AId "OP-NAME",[n],ann1))
               = op_symb ( (AS_OP_NAME n),None )
  | AS_OP_SYMB (AAppl (AId "qual-op-name",[n,t],ann1))
               = op_symb ( (AS_OP_NAME n),Some (AS_OP_TYPE t) )
  | AS_OP_SYMB _ = raise ATerms_UnexpectedTerm

and AS_ID (AAppl (AId "TOKEN-ID",[t],ann1)) = simple_id (AS_TOKEN_ID t)
  | AS_ID (AAppl (AId "MIXFIX-ID",[m],ann1)) = simple_id (AS_MIXFIX_ID m)
  | AS_ID (AAppl (AId "comp-token-id",[t,AAppl (AId "ID+",[AList (l,ann1)],
           ann2)],ann3)) = compound_id ( (AS_TOKEN t),(map AS_ID) l )
  | AS_ID (AAppl (AId "comp-mixfix-id",[m,AAppl (AId "ID+",[AList (l,ann1)],
           ann2)],ann3)) = compound_id ( (AS_MIXFIX_ID m),(map AS_ID) l )
  | AS_ID (AAppl (AId "ID",[i],ann1)) = AS_ID i
  | AS_ID _ = raise ATerms_UnexpectedTerm

and AS_TOKEN_ID (AAppl (AId "TOKEN",[t],ann1)) = AS_TOKEN t
  | AS_TOKEN_ID _ = raise ATerms_UnexpectedTerm
  
and AS_TOKEN (AAppl (AId "WORDS",[AFun (AString s,ann1)],ann2))
             = ( [token (s)],(s),None ):TOKEN_OR_MIXFIX
  | AS_TOKEN (AAppl (AId "DOT-WORDS",[AFun (AString s,ann1)],ann2))
             = ( [token (s)],(s),None ):TOKEN_OR_MIXFIX
  | AS_TOKEN (AAppl (AId "SIGNS",[AFun (AString s,ann1)],ann2))
             = ( [token (s)],(s),None ):TOKEN_OR_MIXFIX
  | AS_TOKEN _ = raise ATerms_UnexpectedTerm
  
and AS_MIXFIX_ID (AAppl (AId "token-places",[AAppl (AId "TOKEN-OR-PLACE+",
                  [AList (l,ann1)],ann2)],ann3))
                 = ( (map AS_TOKEN_OR_PLACE) l,(MAKESTR l),None )
                   :TOKEN_OR_MIXFIX
  | AS_MIXFIX_ID _ = raise ATerms_UnexpectedTerm
  
and AS_TOKEN_OR_PLACE (AAppl (AId "TOKEN",[t],ann1)) = AS_TOK t
  | AS_TOKEN_OR_PLACE (AFun (AString "__",ann1)) = place
  | AS_TOKEN_OR_PLACE _ = raise ATerms_UnexpectedTerm
  
and AS_TOK (AAppl (AId "WORDS",[AFun (AString s,ann1)],ann2))
           = token (s)
  | AS_TOK (AAppl (AId "DOT-WORDS",[AFun (AString s,ann1)],ann2))
           = token (s)
  | AS_TOK (AAppl (AId "SIGNS",[AFun (AString s,ann1)],ann2))
           = token (s)
  | AS_TOK _ = raise ATerms_UnexpectedTerm
  
and MAKESTR l = String.concat ((map MKSTR) l)

and MKSTR (AAppl (AId "TOKEN",[t],ann1)) = MK_TOK t
  | MKSTR (AFun (AString "__",ann1)) = "__"
  | MKSTR _ = raise ATerms_UnexpectedTerm

and MK_TOK (AAppl (AId "WORDS",[AFun (AString s,ann1)],ann2))
           = (s)
  | MK_TOK (AAppl (AId "DOT-WORDS",[AFun (AString s,ann1)],ann2))
           = (s)
  | MK_TOK (AAppl (AId "SIGNS",[AFun (AString s,ann1)],ann2))
           = (s)
  | MK_TOK _ = raise ATerms_UnexpectedTerm

and AS_LIB_ITEM_LIST (AAppl (AId "LIB-ITEM*",[AList (l,ann1)],ann2))
                     = (map AS_LIB_ITEM) l
  | AS_LIB_ITEM_LIST _ = raise ATerms_UnexpectedTerm

and AS_ITEM_NAME_OR_MAP (AAppl (AId "ITEM-NAME",[n],[]))
          = item_name (AS_ITEM_NAME n)
  | AS_ITEM_NAME_OR_MAP (AAppl (AId "item-name-map",[n,m],[]))
          = item_name_map (AS_ITEM_NAME n,AS_ITEM_NAME m)
  | AS_ITEM_NAME_OR_MAP _ = raise ATerms_UnexpectedTerm

and AS_ITEM_NAME x = AS_SIMPLE_ID x

and AS_LIB_NAME (AAppl (AId "LIB-ID",[l],[]))
             = lib (AS_LIB_ID l)
  | AS_LIB_NAME (AAppl (AId "LIB-VERSION",[v],[]))
             = versioned_lib (AS_LIB_VERSION v)
  | AS_LIB_NAME _ = raise ATerms_UnexpectedTerm

and AS_LIB_VERSION (AAppl (AId "lib-version",[l,v],[]))
             = (AS_LIB_ID l,version(AS_VERSION v))
  | AS_LIB_VERSION _ = raise ATerms_UnexpectedTerm

and AS_VERSION (AAppl (AId "version-number",[AAppl (AId "NUMBER+",[AList
             (v,[])],[])],[]))
             = (map AS_NUMBER) v
  | AS_VERSION _ = raise ATerms_UnexpectedTerm

and AS_NUMBER (AAppl (AId "NUMBER",[AFun (AString x,[])],[])) = x
  | AS_NUMBER _ = raise ATerms_UnexpectedTerm

and AS_LIB_ID (AAppl (AId "direct-link",[u],[]))
             = url (AS_URL u)
  | AS_LIB_ID (AAppl (AId "indirect-link",[n],[]))
             = path_name (AS_PATH n)
  | AS_LIB_ID _ = raise ATerms_UnexpectedTerm

and AS_URL (AAppl (AId "URL",[AFun (AString x,[])],[])) = x
  | AS_URL _ = raise ATerms_UnexpectedTerm

and AS_PATH (AAppl (AId "PATH",[AFun (AString x,[])],[])) = x
  | AS_PATH _ = raise ATerms_UnexpectedTerm

(* ************************************************************************ *)
(* aterm output functions                                                   *)
(* ************************************************************************ *)

and S_ITEM_NAME_OR_MAP (item_name (n)) = AAppl (AId "ITEM-NAME",[S_ITEM_NAME
          n],[])
  | S_ITEM_NAME_OR_MAP (item_name_map (n,m)) = AAppl (AId "item-name-map",
          [S_ITEM_NAME n,S_ITEM_NAME m],[])
  | S_ITEM_NAME_OR_MAP (pos_ITEM_NAME_OR_MAP (_,i)) = S_ITEM_NAME_OR_MAP i
        
and S_ITEM_NAME x = AAppl (AId "SIMPLE-ID",[S_SIMPLE_ID x],[])      

and S_LIB_ITEM (spec_defn (n:AS.SIMPLE_ID,g,s,a)) = AAppl (AId "spec-defn",[S_SPEC_NAME (outs n),
                                     S_GENERICITY g,S_SPEC s],[S_ANNO_LIST a])
  | S_LIB_ITEM (view_defn (n,g,t,l,a)) = AAppl (AId "view-defn",[S_VIEW_NAME (outs n),
                                       S_GENERICITY g,S_VIEW_TYPE t,
                                       AAppl (AId "SYMB-MAP-ITEMS*",[
                                       AList ( (map S_SYMB_MAP_ITEMS) l,[])],
                                       [])],[S_ANNO_LIST a])
  | S_LIB_ITEM (unit_spec_defn (n,s,a)) = AAppl (AId "unit-spec-defn",[
                                        S_SPEC_NAME (outs n),S_UNIT_SPEC s],
                                        [S_ANNO_LIST a])
  | S_LIB_ITEM (arch_spec_defn (n,s,a)) = AAppl (AId "arch-spec-defn",[
                                        S_ARCH_SPEC_NAME (outs n),S_ARCH_SPEC s],
                                        [S_ANNO_LIST a])
  | S_LIB_ITEM (download_items (n,l,a)) = AAppl (AId "download-items",[
             S_LIB_NAME n,AAppl (AId "ITEM-NAME-OR-MAP+",[AList ((map
             S_ITEM_NAME_OR_MAP) l,[])],[])],[S_ANNO_LIST a])
  | S_LIB_ITEM (pos_LIB_ITEM (_,l)) = S_LIB_ITEM l

and S_LIB_DEFN (g,ld as lib_defn (n,l,a)) = AAppl (AId "lib-defn",[S_LIB_NAME n,
             S_LIB_ITEM_LIST l],[S_ANNO_LIST a,S_GLOBAL_ENV g ld])
  | S_LIB_DEFN (g,pos_LIB_DEFN (_,l)) = S_LIB_DEFN (g,l)


and S_LIB_DEFN2 (lib_defn (n,l,a)) = AAppl (AId "lib-defn",[S_LIB_NAME n,
             S_LIB_ITEM_LIST l],[S_ANNO_LIST a])
  | S_LIB_DEFN2 (pos_LIB_DEFN (_,l)) = S_LIB_DEFN2 l
                          
and AS_LIB_DEFN (AAppl (AId "lib-defn",[n,l],[ann1,ann2])) =
            (get_global_env [ann1,ann2],lib_defn (AS_LIB_NAME n,
             AS_LIB_ITEM_LIST l,AS_ANNO_LIST ann1))
  | AS_LIB_DEFN (AAppl (AId "lib-defn",[n,l],[ann1])) =
            (empty_global_env,lib_defn (AS_LIB_NAME n,
             AS_LIB_ITEM_LIST l,AS_ANNO_LIST ann1))
  | AS_LIB_DEFN _ = raise ATerms_UnexpectedTerm

and S_LIB_NAME (lib (l)) = AAppl (AId "LIB-ID",[S_LIB_ID l],[])
  | S_LIB_NAME (versioned_lib (l,version(v))) = AAppl (AId "LIB-VERSION",
                                       [S_LIB_VERSION (l,v)],[])
  | S_LIB_NAME (versioned_lib (l,pos_VERSION (r,x))) = S_LIB_NAME
                                       (versioned_lib (l,x))
  | S_LIB_NAME (pos_LIB_NAME (_,l)) = S_LIB_NAME l

and S_LIB_VERSION (l,v) = AAppl (AId "lib-version",[S_LIB_ID l,
                          S_VERSION v],[])

and S_VERSION v = AAppl (AId "version-number",[AAppl (AId "NUMBER+",[AList
                  ((map S_NUMBER) v,[])],[])],[])

and S_NUMBER x = AAppl (AId "NUMBER",[AFun (AString (x),[])],[])

and S_LIB_ID (url (u)) = AAppl (AId "direct-link",[S_URL u],[])
  | S_LIB_ID (path_name (n)) = AAppl (AId "indirect-link",[S_PATH n],[])
  | S_LIB_ID (pos_LIB_ID (_,l)) = S_LIB_ID l

and S_URL x = AAppl (AId "URL",[AFun (AString (x),[])],[])

and S_PATH x = AAppl (AId "PATH",[AFun (AString (x),[])],[])

and S_SPEC_NAME (x:AS.SIMPLE_ID) = AAppl (AId "SIMPLE-ID",[S_SIMPLE_ID x],[])

and S_VIEW_NAME x = AAppl (AId "SIMPLE-ID",[S_SIMPLE_ID x],[])

and S_SIMPLE_ID (id,_) = AAppl (AId "WORDS",[AFun (AString (id),[])],
                         [])

and S_GENERICITY (genericity (p,i)) = AAppl (AId "genericity",[S_PARAMS p,
                                      S_IMPORTS i],[])
  | S_GENERICITY (pos_GENERICITY (_,g)) = S_GENERICITY g

and S_PARAMS (params (l)) = AAppl (AId "params",[AAppl (AId "SPEC*",[AList
                            ( (map S_SPEC) l,[] )],[])],[])
  | S_PARAMS (pos_PARAMS (_,p)) = S_PARAMS p

and S_IMPORTS (imports (l)) = AAppl (AId "imports",[AAppl (AId "SPEC*",[AList
                             ( (map S_SPEC) l,[] )],[])],[])
  | S_IMPORTS (pos_IMPORTS (_,i)) = S_IMPORTS i

and S_VIEW_TYPE (view_type (a,b)) = AAppl (AId "view-type",[S_SPEC a,
                                    S_SPEC b],[])
  | S_VIEW_TYPE (pos_VIEW_TYPE (_,v)) = S_VIEW_TYPE v

and S_SYMB_MAP_ITEMS (symb_map_items (k,l)) = AAppl (AId "symb-map-items",[
                     S_SYMB_KIND k,AAppl (AId "SYMB-OR-MAP+",[AList
                     ( (map S_SYMB_OR_MAP) l,[] )],[])],[])
  | S_SYMB_MAP_ITEMS (pos_SYMB_MAP_ITEMS (_,s)) = S_SYMB_MAP_ITEMS s

and S_SYMB_KIND (implicitk) = AFun (AId "implicit",[])
  | S_SYMB_KIND (sortsk)    = AFun (AId "sorts",[])
  | S_SYMB_KIND (opsk)      = AFun (AId "ops",[])
  | S_SYMB_KIND (predsk)    = AFun (AId "preds",[])
  | S_SYMB_KIND (pos_SYMB_KIND (_,s)) = S_SYMB_KIND s

and S_SYMB_OR_MAP (symb (s)) = AAppl (AId "SYMB",[S_SYMB s],[])
  | S_SYMB_OR_MAP (symb_or_map (m)) = AAppl (AId "SYMB-MAP",[S_SYMB_MAP m],[])
  | S_SYMB_OR_MAP (pos_SYMB_OR_MAP (_,s)) = S_SYMB_OR_MAP s
  
and S_SYMB (symb_id (i)) = AAppl (AId "ID",[S_ID i],[])
  | S_SYMB (qual_id (i,t)) = AAppl (AId "qual-id",[S_ID i,S_TYPE t],[])
  | S_SYMB (pos_SYMB (_,s)) = S_SYMB s
  
and S_SYMB_MAP (symb_map (a,b)) = AAppl (AId "symb-map",[S_SYMB a,S_SYMB b],
                                  [])
  | S_SYMB_MAP (pos_SYMB_MAP (_,s)) = S_SYMB_MAP s

and S_TYPE (op_symb_type (t)) = AAppl (AId "OP-TYPE",[S_OP_TYPE t],[])
  | S_TYPE (pred_symb_type (t)) = AAppl (AId "PRED-TYPE",[S_PRED_TYPE t],[])
  | S_TYPE (pos_TYPE (_,t)) = S_TYPE t
                                  
and nl_remove [] = []
  | nl_remove ((#"\n")::t) = nl_remove t
  | nl_remove (x::t) = x :: (nl_remove t)
  
and n_remove x = (String.implode o nl_remove o String.explode) x

and isTOK (token (s)) = true
  | isTOK (place)     = false

and isTOKEN ((l,s,n):TOKEN_OR_MIXFIX) = ( (length l)=1 ) andalso
                                         (isTOK (hd l))

and S_TOKEN ((x):string) =
   if x="" then
     AAppl (AId "WORDS",[AFun (AString (""),[])],[])
   else
    if hd(String.explode x)=(#".") then
      AAppl (AId "DOT-WORDS",[AFun (AString (x),[])],[])
    else
    if (Char.isAlphaNum (hd(String.explode x))) then
      AAppl (AId "WORDS",[AFun (AString (x),[])],[])
    else
      AAppl (AId "SIGNS",[AFun (AString (x),[])],[])

and S_SINGLE (token (s)) = AAppl (AId "TOKEN-ID",[AAppl (AId "TOKEN",[
                           S_TOKEN s],[])],[])
  | S_SINGLE (place)     = AAppl (AId "MIXFIX-ID",[AAppl (AId "token-places",
                           [AAppl (AId "TOKEN-OR-PLACE+",[AList ([S_TOP place],
                           [])],[])],[])],[])
                           
and S_TOP (token (s)) = AAppl (AId "TOKEN",[S_TOKEN s],[])
  | S_TOP (place)     = AFun (AString "__",[])

and S_MIXFIX ((l):token_or_place list) = AAppl (AId "MIXFIX-ID",[
  AAppl (AId "token-places",[AAppl (AId "TOKEN-OR-PLACE+",[AList
  ( (map S_TOP) l,[])],[])],[])],[])

and S_TOKEN_OR_MIXFIX ((l,s,n):TOKEN_OR_MIXFIX) =
        if length(l)=1 then (S_SINGLE (hd l))
	               else (S_MIXFIX l)

and S_TOM ((l,s,n):TOKEN_OR_MIXFIX) =
  if (isTOKEN (l,s,n)) then
    "comp-token-id"
  else
    "comp-mixfix-id"

and S_SINGLE_COMP (token (s)) = (S_TOKEN s)
  | S_SINGLE_COMP (place)     = AAppl (AId "token-places", [AAppl
                                (AId "TOKEN-OR-PLACE+",[AList ([AFun (AString 
                                "__",[])],[])],[])],[])

and S_MIXFIX_COMP ((l):token_or_place list) =
  AAppl (AId "token-places",[AAppl (AId "TOKEN-OR-PLACE+",[AList
  ( (map S_TOP) l,[] )],[])],[])

and S_TOM_COMP ((l,s,n):TOKEN_OR_MIXFIX) =
  if length(l)=1 then (S_SINGLE_COMP (hd l))
                 else (S_MIXFIX_COMP l)

and S_ID (simple_id i) = (S_TOKEN_OR_MIXFIX i)
  | S_ID (compound_id (t,l)) = AAppl (AId (S_TOM t),[S_TOM_COMP t,AAppl
                                (AId "ID+",[AList ( (map (S_ID)) l,[])],
                                [])],[])

and S_SYMB_ITEMS (symb_items (k,l)) = AAppl (AId "symb-items",[S_SYMB_KIND k,
                                      AAppl (AId "SYMB+",[AList ( (map S_SYMB)
                                      l,[])],[])],[])
  | S_SYMB_ITEMS (pos_SYMB_ITEMS (_,s)) = S_SYMB_ITEMS s
                                      
and S_RENAMING (renaming (l)) = AAppl (AId "renaming",[AAppl (AId
                                "SYMB-MAP-ITEMS+",[AList
                                ( (map S_SYMB_MAP_ITEMS) l,[])],[])],[])
  | S_RENAMING (pos_RENAMING (_,r)) = S_RENAMING r
                                
and S_RESTRICTION (hide (l)) = AAppl (AId "hide",[AAppl (AId "SYMB-ITEMS+",
                               [AList ( (map S_SYMB_ITEMS) l,[])],[])],[])
  | S_RESTRICTION (reveal (l)) = AAppl (AId "reveal",[AAppl (AId
                                 "SYMB-MAP-ITEMS+",[AList
                                 ( (map S_SYMB_MAP_ITEMS) l,[])],[])],[])
  | S_RESTRICTION (pos_RESTRICTION (_,r)) = S_RESTRICTION r

and S_SPEC ((basic (b)),g) = AAppl (AId "BASIC-SPEC",[S_BASIC_SPEC b],
                                   [S_ANNO_LIST g])
  | S_SPEC ((translation (s,r,a)),g) = AAppl (AId "translation",[S_SPEC s,
                                 S_RENAMING r],[S_ANNO_LIST a,S_ANNO_LIST g])
  | S_SPEC ((reduction (s,r,a)),g) = AAppl (AId "reduction",[S_SPEC s,
                               S_RESTRICTION r],[S_ANNO_LIST a,S_ANNO_LIST g])
  | S_SPEC ((union_spec (l)),g) = AAppl (AId "union",[AAppl (AId "SPEC+",[AList
                              ( (map S_ASPEC) l,[])],[])],[S_ANNO_LIST g])
  | S_SPEC ((extension (l)),g) = AAppl (AId "extension",[AAppl (AId "SPEC+",[AList
                             ( (map S_ASPEC) l,[])],[])],[S_ANNO_LIST g])
  | S_SPEC ((free_spec (s,a)),g) = AAppl (AId "free-spec",[S_SPEC s],
                                    [S_ANNO_LIST a,S_ANNO_LIST g])
  | S_SPEC ((cofree_spec (s,a)),g) = AAppl (AId "cofree-spec",[S_SPEC s],
                                    [S_ANNO_LIST a,S_ANNO_LIST g])
  | S_SPEC ((local_spec (a,x,b,y)),g) = AAppl (AId "local-spec",[S_SPEC a,S_SPEC b],
                                    [S_ANNO_LIST x,S_ANNO_LIST y,S_ANNO_LIST g])
  | S_SPEC ((closed_spec (s,a)),g) = AAppl (AId "closed-spec",[S_SPEC s],
                                 [S_ANNO_LIST a,S_ANNO_LIST g])
  | S_SPEC ((spec_inst (s,l)),g) = AAppl (AId "spec-inst",[S_SPEC_NAME s,AAppl
                               (AId "FIT-ARG*",[AList ( (map S_FIT_ARG) l,[])],
                               [])],[S_ANNO_LIST g])
  | S_SPEC (pos_SPEC (_,_,s),g) = S_SPEC (s,g)
                               
and S_FIT_ARG (fit_spec (s,l)) = AAppl (AId "fit-spec",[S_SPEC s,AAppl (AId
                                 "SYMB-MAP-ITEMS*",[AList
                                 ( (map S_SYMB_MAP_ITEMS) l,[])],[])],[])
  | S_FIT_ARG (fit_view (n,l)) = AAppl (AId "fit-view",[S_VIEW_NAME n,AAppl
                                 (AId "FIT-ARG*",[AList ( (map S_FIT_ARG) l,
                                 [])],[])],[])
  | S_FIT_ARG (pos_FIT_ARG (_,f)) = S_FIT_ARG f
                                 
and S_BASIC_SPEC (basic_spec (l)) = AAppl (AId "basic-spec",[AAppl (AId
                                    "BASIC-ITEMS*",[AList
                                    ( (map S_BASIC_ITEMS) l,[])],[])],[])
  | S_BASIC_SPEC (pos_BASIC_SPEC (_,b)) = S_BASIC_SPEC b
                                    
and S_BASIC_ITEMS (sig_items (i)) = AAppl (AId "SIG-ITEMS",[S_SIG_ITEMS i],[])
  | S_BASIC_ITEMS (free_datatype (l,a)) = AAppl (AId "free-datatype",[AAppl (AId
                                        "DATATYPE-DECL+",[AList
                                        ( (map S_DATATYPE_DECL) l,[])],[])],
                                        [S_ANNO_LIST a])
  | S_BASIC_ITEMS (cofree_datatype (l,a)) = AAppl (AId "cofree-datatype",[AAppl (AId
                                        "DATATYPE-DECL+",[AList
                                        ( (map S_DATATYPE_DECL) l,[])],[])],
                                        [S_ANNO_LIST a])
  | S_BASIC_ITEMS (sort_gen (l,a)) = AAppl (AId "sort-gen",[AAppl (AId
                                   "SIG-ITEMS+",[AList
                                   ( (map S_SIG_ITEMS) l,[])],[])],
                                   [S_ANNO_LIST a])
  | S_BASIC_ITEMS (sort_cogen (l,a)) = AAppl (AId "sort-cogen",[AAppl (AId
                                   "SIG-ITEMS+",[AList
                                   ( (map S_SIG_ITEMS) l,[])],[])],
                                   [S_ANNO_LIST a])
  | S_BASIC_ITEMS (var_items (l,a)) = AAppl (AId "var-items",[AAppl (AId
                                    "VAR-DECL+",[AList ( (map S_VAR_DECL) l,
                                    [])],[])],[S_ANNO_LIST a])
  | S_BASIC_ITEMS (local_var_axioms (d,l,a)) = AAppl (AId "local-var-axioms",[
         AAppl (AId "VAR-DECL+",[AList ( (map S_VAR_DECL) d,[])],[]),
         AAppl (AId "FORMULA+",[AList ( (map S_L_FORMULA) l,[])],[])],
         [S_ANNO_LIST a])
  | S_BASIC_ITEMS (axiom_items (l,a)) = AAppl (AId "axiom-items",[AAppl (AId
                                      "FORMULA+",[AList ( (map S_L_FORMULA) l,
                                      [])],[])],[S_ANNO_LIST a])
  | S_BASIC_ITEMS (pos_BASIC_ITEMS (_,b)) = S_BASIC_ITEMS b
                                      
and S_SIG_ITEMS (sort_items (l,a)) = AAppl (AId "sort-items",[AAppl (AId
                                   "SORT-ITEM+",[AList ( (map S_L_SORT_ITEM) l,
                                   [])],[])],[S_ANNO_LIST a])
  | S_SIG_ITEMS (op_items (l,a)) = AAppl (AId "op-items",[AAppl (AId
                                 "OP-ITEM+",[AList ( (map S_L_OP_ITEM) l,
                                 [])],[])],[S_ANNO_LIST a])
  | S_SIG_ITEMS (pred_items (l,a)) = AAppl (AId "pred-items",[AAppl (AId
                                   "PRED-ITEM+",[AList ( (map S_L_PRED_ITEM) l,
                                   [])],[])],[S_ANNO_LIST a])
  | S_SIG_ITEMS (datatype_items (l,a)) = AAppl (AId "datatype-items",[AAppl (AId
                                       "DATATYPE-DECL+",[AList
                                       ( (map S_DATATYPE_DECL) l,[])],[])],
                                       [S_ANNO_LIST a])
  | S_SIG_ITEMS (pos_SIG_ITEMS (_,s)) = S_SIG_ITEMS s
                                       
and S_L_SORT_ITEM (s,l) = S_SORT_ITEM l s

and S_SORT_ITEM la (sort_decl (l)) = AAppl (AId "sort-decl",[AAppl (AId "SORT+",
                                  [AList ( (map S_SORT) l,[])],[])],
                                  [S_ANNO_LIST la])
  | S_SORT_ITEM la (subsort_defn (a,v,b,f,x)) = AAppl (AId "subsort-defn",
                                  [S_SORT a,S_VAR v,S_SORT b,S_FORMULA []
                                  f],[S_ANNO_LIST x,S_ANNO_LIST la])
  | S_SORT_ITEM la (subsort_decl (l,s)) = AAppl (AId "subsort-decl",[AAppl
                                  (AId "SORT+",[AList ( (map S_SORT) l,[])],
                                  []),S_SORT s],[S_ANNO_LIST la])
  | S_SORT_ITEM la (iso_decl (l)) = AAppl (AId "iso-decl",[AAppl (AId "SORT+",
                                 [AList ( (map S_SORT) l,[])],[])],
                                 [S_ANNO_LIST la])
  | S_SORT_ITEM la (pos_SORT_ITEM (_,s)) = S_SORT_ITEM la s
                                 
and S_L_PRED_ITEM (p,a) = S_PRED_ITEM a p

and S_PRED_ITEM a (pred_decl (l,t)) = AAppl (AId "pred-decl",[AAppl (AId
                                    "PRED-NAME+",[AList ( (map S_PRED_NAME) l,
                                    [])],[]),S_PRED_TYPE t],
                                    [S_ANNO_LIST a])
  | S_PRED_ITEM a (pred_defn (n,h,f,la)) = AAppl (AId "pred-defn",[S_PRED_NAME
                                      n,S_PRED_HEAD h,S_L_FORMULA f],
                                      [S_ANNO_LIST la,S_ANNO_LIST a])
  | S_PRED_ITEM a (pos_PRED_ITEM (_,p)) = S_PRED_ITEM a p
                                      
and S_L_OP_ITEM (i,p) = S_OP_ITEM p i

and S_OP_ITEM p (op_decl (l,t,a)) = AAppl (AId "op-decl",[
          AAppl (AId "OP-NAME+",[AList ( (map S_OP_NAME) l,[])],[]),
          S_OP_TYPE t,
          AAppl (AId "OP-ATTR*",[AList ( (map S_OP_ATTR) a,[])],[])],
          [S_ANNO_LIST p])
  | S_OP_ITEM p (op_defn (n,h,t,a)) = AAppl (AId "op-defn",[S_OP_NAME n,
          S_OP_HEAD h,S_TERM t],[S_ANNO_LIST a,S_ANNO_LIST p])
  | S_OP_ITEM p (pos_OP_ITEM (_,i)) = S_OP_ITEM p i
          
and S_OP_ATTR (associative) = AFun (AId "associative",[])
  | S_OP_ATTR (commutative) = AFun (AId "commutative",[])
  | S_OP_ATTR (idempotent)  = AFun (AId "idempotent",[])
  | S_OP_ATTR (unit_op_attr (t)) = AAppl (AId "unit-op-attr",[S_TERM t],[])
  | S_OP_ATTR (pos_OP_ATTR (_,a)) = S_OP_ATTR a
  
and S_DATATYPE_DECL (datatype_decl (s,l,la)) = AAppl (AId "datatype-decl",[
       S_SORT s,AAppl (AId "ALTERNATIVE+",[AList ( (map S_L_ALTERNATIVE) l,[])],
       [])],[S_ANNO_LIST la])
  | S_DATATYPE_DECL (pos_DATATYPE_DECL (_,d)) = S_DATATYPE_DECL d

and S_L_ALTERNATIVE (x,a) = S_ALTERNATIVE a x
       
and S_ALTERNATIVE x (total_construct (n,l)) = AAppl (AId "total-construct",[
       S_OP_NAME n,AAppl (AId "COMPONENTS*",[AList ( (map S_COMPONENTS) l,[])
       ],[])],[S_ANNO_LIST x])
  | S_ALTERNATIVE x (partial_construct (n,l)) = AAppl (AId "partial-construct",
       [S_OP_NAME n,AAppl (AId "COMPONENTS*",[AList ((map S_COMPONENTS) l,[])
       ],[])],[S_ANNO_LIST x])
  | S_ALTERNATIVE x (subsort (l)) = AAppl (AId "subsorts",[AAppl (AId
       "SORT+",[AList ( (map S_SORT) l,[])],[])],[S_ANNO_LIST x])
  | S_ALTERNATIVE x (pos_ALTERNATIVE (_,a)) = S_ALTERNATIVE x a

and S_COMPONENTS (total_select (l,s)) = AAppl (AId "total-select",[AAppl
        (AId "OP-NAME+",[AList ( (map S_OP_NAME) l,[])],[]),S_SORT s],[])
  | S_COMPONENTS (partial_select (l,s)) = AAppl (AId "partial-select",[AAppl
        (AId "OP-NAME+",[AList ( (map S_OP_NAME) l,[])],[]),S_SORT s],[])
  | S_COMPONENTS (sort_component (s)) = AAppl (AId "SORT",[S_SORT s],[])
  | S_COMPONENTS (pos_COMPONENTS (_,c)) = S_COMPONENTS c
  
and S_L_FORMULA (f,x) = (S_FORMULA x f)

(* Bremen parser splits CASL formula into FORMULA and ATOM *)

and S_ATOM x (ttrue) = AFun (AId "true",[S_ANNO_LIST x])
  | S_ATOM x (ffalse) = AFun (AId "false",[S_ANNO_LIST x])
  | S_ATOM x (predication (s,t)) = AAppl (AId "predication",[S_PRED_SYMB s,
                                   S_TERMS t],[S_ANNO_LIST x])
  | S_ATOM x (definedness (t)) = AAppl (AId "definedness",[S_TERM t],
                                 [S_ANNO_LIST x])
  | S_ATOM x (existl_equation (a,b)) = AAppl (AId "existl-equation",[S_TERM a,
                                       S_TERM b],[S_ANNO_LIST x])
  | S_ATOM x (strong_equation (a,b)) = AAppl (AId "strong-equation",[S_TERM a,
                                       S_TERM b],[S_ANNO_LIST x])
  | S_ATOM x (membership (t,s)) = AAppl (AId "membership",[S_TERM t,S_SORT s],
                                  [S_ANNO_LIST x])

and S_FORMULA x (quantification (q,l,f)) = AAppl (AId "quantification",[
        S_QUANTIFIER q,AAppl (AId "VAR-DECL+",[AList ( (map S_VAR_DECL) l,[])],
        []),S_FORMULA [] f],[S_ANNO_LIST x])
  | S_FORMULA x (pred_quantification (q,l,f)) = AAppl (AId
        "pred-quantification",[S_QUANTIFIER q,S_FORMULA [] f],[S_ANNO_LIST x])
  | S_FORMULA x (conjunction (l)) = AAppl (AId "conjunction",[AAppl (AId
        "FORMULA+",[AList ( (map (S_FORMULA [])) l,[])],[])],[S_ANNO_LIST x])
  | S_FORMULA x (disjunction (l)) = AAppl (AId "disjunction",[AAppl (AId
        "FORMULA+",[AList ( (map (S_FORMULA [])) l,[])],[])],[S_ANNO_LIST x])
  | S_FORMULA x (implication (a,b)) = AAppl (AId "implication",[S_FORMULA 
        [] a,S_FORMULA [] b],[S_ANNO_LIST x])
  | S_FORMULA x (equivalence (a,b)) = AAppl (AId "equivalence",[S_FORMULA
         [] a,S_FORMULA [] b],[S_ANNO_LIST x])
  | S_FORMULA x (negation (f)) = AAppl (AId "negation",[S_FORMULA [] f],
        [S_ANNO_LIST x])
  | S_FORMULA x (unparsed_formula (s)) = AAppl (AId "unparsed-formula",[AFun
        (AString (((at_escape o n_remove) s)),[])],[S_ANNO_LIST x])
  | S_FORMULA x (sort_gen_ax (l1,l2)) = AAppl (AId "sort-gen-ax",[AAppl (AId
        "SORT*",[AList ((map S_SORT) l1,[])],[]),AAppl (AId "OP-SYMB*",[AList
        ((map S_OP_SYMB) l2,[])],[])],[S_ANNO_LIST x])
  | S_FORMULA x (sort_cogen_ax (l1,l2)) = AAppl (AId "sort-cogen-ax",[AAppl (AId
        "SORT*",[AList ((map S_SORT) l1,[])],[]),AAppl (AId "OP-SYMB*",[AList
        ((map S_OP_SYMB) l2,[])],[])],[S_ANNO_LIST x])
  | S_FORMULA x (sort_cofree_ax (l1,l2)) = AAppl (AId "sort-cofree-ax",[AAppl (AId
        "SORT*",[AList ((map S_SORT) l1,[])],[]),AAppl (AId "OP-SYMB*",[AList
        ((map S_OP_SYMB) l2,[])],[])],[S_ANNO_LIST x])
  | S_FORMULA x (atom a) = (S_ATOM x a)
  | S_FORMULA x (pos_FORMULA (_,_,f)) = S_FORMULA x f

and S_QUANTIFIER (forall) = AFun (AId "forall",[])
  | S_QUANTIFIER (exists) = AFun (AId "exists",[])
  | S_QUANTIFIER (exists_uniquely) = AFun (AId "exists-uniquely",[])
  | S_QUANTIFIER (pos_QUANTIFIER (_,q)) = S_QUANTIFIER q

and S_TERMS (terms (l)) = AAppl (AId "terms",[AAppl (AId "TERM*",[AList
                          ( (map S_TERM) l,[])],[])],[])
  | S_TERMS (pos_TERMS (_,t)) = S_TERMS t
                          
and S_TERM (var_or_const (v)) = AAppl (AId "SIMPLE-ID",[S_SIMPLE_ID v],[])
  | S_TERM (qual_var (v,s)) = AAppl (AId "qual-var",[S_VAR v,S_SORT s],[])
  | S_TERM (application (s,t)) = AAppl (AId "application",[S_OP_SYMB s,
                                 S_TERMS t],[])
  | S_TERM (sorted_term (t,s)) = AAppl (AId "sorted-term",[S_TERM t,
                                 S_SORT s],[])
  | S_TERM (cast (t,s)) = AAppl (AId "cast",[S_TERM t,S_SORT s],[])
  | S_TERM (conditional (a,f,b)) = AAppl (AId "conditional",[S_TERM a,
                                   S_FORMULA [] f,S_TERM b],[])
  | S_TERM (unparsed_term (s)) = AAppl (AId "unparsed-term",[AFun (AString
                                 (((at_escape o n_remove) s)),
                                 [])],[])
  | S_TERM (pos_TERM (_,_,t)) = S_TERM t
                                 
and S_VAR_DECL (l,s) = AAppl (AId "var-decl",[AAppl (AId "VAR+",[AList
                         ( (map S_VAR) l,[])],[]),S_SORT s],[])

and S_PRED_SYMB (pred_symb (n,None))  = 
            AAppl (AId "PRED-NAME",[S_PRED_NAME n],[])
  | S_PRED_SYMB (pred_symb (n,Some t)) = 
            AAppl (AId "qual-pred-name",[S_PRED_NAME n, S_PRED_TYPE t],[])
  | S_PRED_SYMB (pos_PRED_SYMB (_,p)) = S_PRED_SYMB p

and S_PRED_HEAD (pred_head (l)) = AAppl (AId "pred-head",[AAppl (AId
                                  "ARG-DECL*",[AList ( (map S_ARG_DECL) l,
                                  [])],[])],[])
  | S_PRED_HEAD (pos_PRED_HEAD (_,p)) = S_PRED_HEAD p
                                  
and S_PRED_TYPE (pred_type (s)) = AAppl (AId "pred-type",[S_SORTS s],[])
  | S_PRED_TYPE (pos_PRED_TYPE (_,p)) = S_PRED_TYPE p
                             
and S_OP_SYMB (op_symb (n,None)) = 
           AAppl (AId "OP-NAME",[S_OP_NAME n],[])
  | S_OP_SYMB (op_symb (n,Some t)) = 
           AAppl (AId "qual-op-name",[S_OP_NAME n,S_OP_TYPE t],[])
  | S_OP_SYMB (pos_OP_SYMB (_,s)) = S_OP_SYMB s
                           
and S_OP_HEAD (total_op_head (l,s)) = AAppl (AId "total-op-head",[AAppl
                (AId "ARG-DECL*",[AList ( (map S_ARG_DECL) l,[])],[]),
                S_SORT s],[])
  | S_OP_HEAD (partial_op_head (l,s)) = AAppl (AId "partial-op-head",[AAppl
                (AId "ARG-DECL*",[AList ( (map S_ARG_DECL) l,[])],[]),
                S_SORT s],[])
  | S_OP_HEAD (pos_OP_HEAD (_,h)) = S_OP_HEAD h
                
and S_OP_TYPE (total_op_type (a,b)) = AAppl (AId "total-op-type",[S_SORTS a,
                                      S_SORT b],[])
  | S_OP_TYPE (partial_op_type (a,b)) = AAppl (AId "partial-op-type",[
                                        S_SORTS a,S_SORT b],[])
  | S_OP_TYPE (pos_OP_TYPE (_,t)) = S_OP_TYPE t
                                        
and S_ARG_DECL (arg_decl (v,s)) = AAppl (AId "arg-decl",[AAppl (AId
         "VAR+",[AList ( (map S_VAR) v,[])],[]),S_SORT s],[])
  | S_ARG_DECL (pos_ARG_DECL (_,a)) = S_ARG_DECL a
         
and S_SORTS (sorts (l)) = AAppl (AId "sorts",[AAppl (AId "SORT*",[AList
                          ( (map S_SORT) l,[])],[])],[])
  | S_SORTS (pos_SORTS (_,s)) = S_SORTS s

and S_UNIT_NAME x = AAppl (AId "SIMPLE-ID",[S_SIMPLE_ID x],[])

and S_ARCH_SPEC_NAME x = AAppl (AId "SIMPLE-ID",[S_SIMPLE_ID x],[])

and S_PRED_NAME x = AAppl (AId "ID",[S_ID x],[])

and S_VAR x = AAppl (AId "SIMPLE-ID",[S_SIMPLE_ID x],[])

and S_OP_NAME x = AAppl (AId "ID",[S_ID x],[])

and S_SORT x = S_ID x

and S_LIB_ITEM_LIST l = AAppl (AId "LIB-ITEM*",[AList ( (map S_LIB_ITEM) l,
                        [])],[])

and S_ARCH_SPEC ((basic_arch_spec (l,r,a)),g) = AAppl (AId "basic-arch-spec",[
       AAppl (AId "UNIT-DECL-DEFN+",[AList ( (map S_UNIT_DECL_DEFN) l,[])],
       []),S_RESULT_UNIT r],[S_ANNO_LIST a,S_ANNO_LIST g])
  | S_ARCH_SPEC ((named_arch_spec (n)),g) = AAppl (AId "ARCH-SPEC-NAME",[
       S_ARCH_SPEC_NAME n],[S_ANNO_LIST g])
  | S_ARCH_SPEC (pos_ARCH_SPEC (_,a),g) = S_ARCH_SPEC (a,g)
       
and S_UNIT_DECL_DEFN ((unit_decl_case (d)),a) = AAppl (AId "UNIT-DECL",
                                            [S_UNIT_DECL d],[S_ANNO_LIST a])
  | S_UNIT_DECL_DEFN ((unit_defn_case (d)),a) = AAppl (AId "UNIT-DEFN",
                                            [S_UNIT_DEFN d],[S_ANNO_LIST a])
  | S_UNIT_DECL_DEFN (pos_UNIT_DECL_DEFN (_,u),a) = S_UNIT_DECL_DEFN (u,a)
                                            
and S_UNIT_DECL (unit_decl (n,s,i)) = AAppl (AId "unit-decl",[S_UNIT_NAME n,
                                      S_UNIT_SPEC s,S_UNIT_IMPORTS i],[])
  | S_UNIT_DECL (pos_UNIT_DECL (_,u)) = S_UNIT_DECL u
                                      
and S_UNIT_IMPORTS (unit_imports (l)) = AAppl (AId "unit-imports",[AAppl
            (AId "UNIT-TERM*",[AList ( (map S_UNIT_TERM) l,[])],[])],[])
  | S_UNIT_IMPORTS (pos_UNIT_IMPORTS (_,u)) = S_UNIT_IMPORTS u
            
and S_UNIT_DEFN (unit_defn (n,e)) = AAppl (AId "unit-defn",[S_UNIT_NAME n,
                                    S_UNIT_EXPRESSION e],[])
  | S_UNIT_DEFN (pos_UNIT_DEFN (_,u)) = S_UNIT_DEFN u
                                    
and S_UNIT_SPEC (unit_type_case (t)) = AAppl (AId "UNIT-TYPE",[
                                       S_UNIT_TYPE t],[])
  | S_UNIT_SPEC (spec_name_case (n)) = AAppl (AId "SPEC-NAME",[
                                       S_SPEC_NAME n],[])
  | S_UNIT_SPEC (arch_spec_case (a)) = AAppl (AId "ARCH-SPEC",[
                                       S_ARCH_SPEC a],[])
  | S_UNIT_SPEC (closed (s)) = AAppl (AId "closed",[S_UNIT_SPEC s],[])
  | S_UNIT_SPEC (pos_UNIT_SPEC (_,u)) = S_UNIT_SPEC u 
  
and S_UNIT_TYPE (unit_type (l,s)) = AAppl (AId "unit-type",[AAppl (AId
       "SPEC*",[AList ( (map S_SPEC) l,[])],[]),S_SPEC s],[])
  | S_UNIT_TYPE (pos_UNIT_TYPE (_,u)) = S_UNIT_TYPE u
       
and S_RESULT_UNIT (result_unit (r,a)) = AAppl (AId "result-unit",[
                                      S_UNIT_EXPRESSION r],[S_ANNO_LIST a])
  | S_RESULT_UNIT (pos_RESULT_UNIT (_,r)) = S_RESULT_UNIT r
                                      
and S_UNIT_EXPRESSION (unit_expression (l,t)) = AAppl (AId "unit-expression",
       [AAppl (AId "UNIT-BINDING*",[AList ( (map S_UNIT_BINDING) l,[])],[]),
       S_UNIT_TERM t],[])
  | S_UNIT_EXPRESSION (pos_UNIT_EXPRESSION (_,u)) = S_UNIT_EXPRESSION u
       
and S_UNIT_BINDING (unit_binding (n,s)) = AAppl (AId "unit-binding",[
                                          S_UNIT_NAME n,S_UNIT_SPEC s],[])
  | S_UNIT_BINDING (pos_UNIT_BINDING (_,u)) = S_UNIT_BINDING u
                                          
and S_UNIT_TERM (unit_translation (t,r)) = AAppl (AId "unit-translation",[
                                           S_UNIT_TERM t,S_RENAMING r],[])
  | S_UNIT_TERM (unit_reduction (t,r)) = AAppl (AId "unit-reduction",[
                                         S_UNIT_TERM t,S_RESTRICTION r],[])
  | S_UNIT_TERM (amalgamation (l)) = AAppl (AId "amalgamation",[AAppl (AId
                      "UNIT-TERM+",[AList ( (map S_UNIT_TERM) l,[])],[])],[])
  | S_UNIT_TERM (local_unit (l,t)) = AAppl (AId "local-unit",[AAppl (AId
      "UNIT-DEFN+",[AList ( (map S_UNIT_DEFN) l,[])],[]),S_UNIT_TERM t],[])
  | S_UNIT_TERM (unit_appl (n,l)) = AAppl (AId "unit-appl",[S_UNIT_NAME n,
      AAppl (AId "FIT-ARG-UNIT*",[AList ( (map S_FIT_ARG_UNIT) l,[])],[])],[])
  | S_UNIT_TERM (pos_UNIT_TERM (_,_,u)) = S_UNIT_TERM u
      
and S_FIT_ARG_UNIT (fit_arg_unit (t,l)) = AAppl (AId "fit-arg-unit",[
                                          S_UNIT_TERM t,AAppl (AId
                                          "SYMB-MAP-ITEMS*",[AList
                                          ( (map S_SYMB_MAP_ITEMS) l,[])],[])],
                                          [])
  | S_FIT_ARG_UNIT (pos_FIT_ARG_UNIT (_,f)) = S_FIT_ARG_UNIT f



(* ************************************************************************ *)
(* annotation handling                                                      *)
(* ************************************************************************ *)

and S_ANNO (number_anno (i)) = AAppl (AId "number-anno",[S_ID i],[])
  | S_ANNO (floating_anno (i,j)) = AAppl (AId "floating-lit-anno",
                                   [S_ID i,S_ID j],[])
  | S_ANNO (string_anno (i,j)) = AAppl (AId "string-anno",
                                 [S_ID i,S_ID j],[])
  | S_ANNO (list_anno (i,j,k)) = AAppl (AId "list-anno",
                                 [S_ID i,S_ID j,S_ID k],[])
  | S_ANNO (display_anno (i,s)) = AAppl (AId "display-anno",
                                [S_ID i,S_STRING s],[])
  | S_ANNO (prec_anno (b,i,j)) = AAppl (AId "prec-anno",
                                 [S_BOOL b,AAppl (AId "ID+",[AList
                                 ( (map S_ID) i,[])],[]),AAppl (AId "ID+",
                                 [AList ( (map S_ID) j,[])],[])],[])
  | S_ANNO (lassoc_anno (i)) = AAppl (AId "left-assoc-anno",
                               [AAppl (AId "ID+",[AList
                                 ( (map S_ID) i,[])],[])],[])
  | S_ANNO (rassoc_anno (i)) = AAppl (AId "right-assoc-anno",
                               [AAppl (AId "ID+",[AList
                                 ( (map S_ID) i,[])],[])],[])
  | S_ANNO (conservative) = AFun (AId "conservative",[])
  | S_ANNO (definitional) = AFun (AId "definitional",[])
  | S_ANNO (mono)         = AFun (AId "mono",[])
  | S_ANNO (implies)      = AFun (AId "implies",[])
  | S_ANNO (comment (c)) = AAppl (AId "comment",[S_STRING c],[])
  | S_ANNO (comment_line (c)) = AAppl (AId "comment",[S_STRING c],[])
  | S_ANNO (label_anno (i)) = AAppl (AId "label-anno",[S_ID i],[])
  | S_ANNO (unparsed_anno (t)) = AAppl (AId "unparsed-anno",[S_STRING t],[])
  | S_ANNO (pos_ANNO (_,a)) = S_ANNO a

and S_ANNO_LIST l = AAppl (AId "ANNO*",[AList ((map S_ANNO) l,[])],[])

and AS_ANNO_LIST (AAppl (AId "ANNO*",[AList (l,[])],[])) =
                    (map AS_ANNO) l
  | AS_ANNO_LIST _ = raise ATerms_UnexpectedTerm

and AS_ANNO (AAppl (AId "number-anno",[x],ann1)) =
                        number_anno (AS_ID x)
  | AS_ANNO (AAppl (AId "floating-anno",[x,y],ann1)) =
                        floating_anno (AS_ID x,AS_ID y)
  | AS_ANNO (AAppl (AId "string-anno",[x,y],ann1)) =
                        string_anno (AS_ID x,AS_ID y)
  | AS_ANNO (AAppl (AId "list-anno",[x,y,z],ann1)) =
                        list_anno (AS_ID x,AS_ID y,AS_ID z)
  | AS_ANNO (AAppl (AId "display-anno",[x,y],ann1)) =
                        display_anno (AS_ID x,AS_STRING y)
  | AS_ANNO (AAppl (AId "prec-anno",[b,AAppl (AId "ID+",[AList
            (l1,ann1)],ann2),AAppl (AId "ID+",[AList (l2,ann3)],ann4)],ann5)) =
                        prec_anno (AS_BOOL b,(map AS_ID) l1,(map AS_ID) l2)
  | AS_ANNO (AAppl (AId "left-assoc-anno",[AAppl (AId "ID+",[AList
            (l1,ann1)],ann2)],ann3)) =
                        lassoc_anno ((map AS_ID) l1)
  | AS_ANNO (AAppl (AId "right-assoc-anno",[AAppl (AId "ID+",[AList
            (l1,ann1)],ann2)],ann3)) =
                        rassoc_anno ((map AS_ID) l1)
  | AS_ANNO (AFun (AId "conservative",ann1)) = conservative
  | AS_ANNO (AFun (AId "definitional",ann1)) = definitional
  | AS_ANNO (AFun (AId "mono",ann1)) = mono
  | AS_ANNO (AFun (AId "implies",ann1))      = implies
  | AS_ANNO (AAppl (AId "comment",[c],[])) = comment (AS_STRING c)
  | AS_ANNO (AAppl (AId "label-anno",[i],[])) = label_anno (AS_ID i)
  | AS_ANNO _ = raise ATerms_UnexpectedTerm

and S_BOOL (true)  = AFun (AId "bool-true",[])
  | S_BOOL (false) = AFun (AId "bool-false",[])

and AS_BOOL (AFun (AId "bool-true",ann1))  = true
  | AS_BOOL (AFun (AId "bool-false",ann1)) = false
  | AS_BOOL _ = raise ATerms_UnexpectedTerm

and S_STRING (s) = AAppl (AId "STRING",[AFun (AString (s),[])],[])

and AS_STRING (AAppl (AId "STRING",[AFun (AString s,ann1)],ann2)) = s
  | AS_STRING _ = raise ATerms_UnexpectedTerm

and S_LINE_NUMBER ((Some n):LINE_NUMBER) = AAppl (AId "line-number",[AFun
              (AString ((Int.toString n)),[])],[])
  | S_LINE_NUMBER ((None):LINE_NUMBER) = AFun (AId "empty-line-number",[])
  
and AS_LINE_NUMBER (AAppl (AId "line-number",[AFun (AString s,[])],[])) =
              (Some (get_int ((Int.fromString) s))):LINE_NUMBER
  | AS_LINE_NUMBER (AFun (AId "empty-line-number",[])) = (None):LINE_NUMBER
  | AS_LINE_NUMBER _ = raise ATerms_UnexpectedTerm
  
and get_int (SOME n) = n
  | get_int (NONE)   = raise ATerms_WrongType

and get_global_env [] = empty_global_env
  | get_global_env (h::t) = AS_GLOBAL_ENV h
                            handle exn => get_global_env t

and get_anno [] = raise ATerms_WrongType
  | get_anno (h::t) = AS_ANNO h
                      handle exn => get_anno t



(* ************************************************************************ *)
(* global_env handling                                                      *)
(* ************************************************************************ *)

(* Use LIB_DEFN to lookup the LIB_ITEMs in the global env
   in the order of their occurrence in the input text *)

and get_names (item_name n) = n
  | get_names (item_name_map (n1,n2)) = n2
  | get_names (pos_ITEM_NAME_OR_MAP (_,i)) = get_names i
  
and lib_item_name (spec_defn (n,_,_,_)) = [n]
  | lib_item_name (view_defn  (n,_,_,_,_)) = [n]
  | lib_item_name (arch_spec_defn (n,_,_)) = [n]
  | lib_item_name (unit_spec_defn  (n,_,_)) = [n]
  | lib_item_name (download_items  (_,nl,_)) = map get_names nl
  | lib_item_name (pos_LIB_ITEM (_,i)) = lib_item_name i

and lookup_lib_item1 genv n =
    case Symtab_sid.lookup (genv,n) of
       Some x => Some (n,x)
     | None => None

and lookup_lib_item genv ls =
    mapfilter (lookup_lib_item1 genv) (lib_item_name ls)
    
and make_list (lib_defn (n,l,_)) genv =
    flat (map (lookup_lib_item genv) l)
  | make_list (pos_LIB_DEFN (_,l)) genv = make_list l genv

and S_GLOBAL_ENV (x,a) l = AAppl (AId "casenv",[S_GLOBAL_TABLE
             (make_list l x)],[S_ANNO_LIST a])

and AS_GLOBAL_ENV (AAppl (AId "casenv",[x],[ann1])) =
             (Symtab_sid.make (AS_GLOBAL_TABLE x),AS_ANNO_LIST ann1)

(* ???
and S_GLOBAL_ENV x l = AAppl (AId "casenv",[S_GLOBAL_TABLE
             (make_list l x)],[])

and AS_GLOBAL_ENV (AAppl (AId "casenv",[x],[])) =
             Symtab_sid.make (AS_GLOBAL_TABLE x)
*)

  | AS_GLOBAL_ENV _ = raise ATerms_UnexpectedTerm

and S_GLOBAL_TABLE x = AAppl (AId "global-entry-tuple*",[AList
             ((map S_GLOBAL_TUPLE) x,[])],[])

and AS_GLOBAL_TABLE (AAppl (AId "global-entry-tuple*",[AList (x,[])],[])) =
             (map AS_GLOBAL_TUPLE) x
  | AS_GLOBAL_TABLE _ = raise ATerms_UnexpectedTerm

and S_GLOBAL_TUPLE (k,v) = AAppl (AId "global-entry-tuple",[S_SIMPLE_ID (outs k),
             S_GLOBAL_ENTRY v],[])
             
and AS_GLOBAL_TUPLE (AAppl (AId "global-tuple-tuple",[k,v],[])) =
             (AS_SIMPLE_ID k,AS_GLOBAL_ENTRY v)
  | AS_GLOBAL_TUPLE _ = raise ATerms_UnexpectedTerm

and S_GLOBAL_ENTRY (spec_defn_env (g,s)) = AAppl (AId "spec-defn-env",
             [S_GENERICITY_ENV g,S_SPEC_LENV s],[])
  | S_GLOBAL_ENTRY (view_defn_env (g,s,m,t)) = AAppl (AId "view-defn-env",
             [S_GENERICITY_ENV g,S_SPEC_LENV s,S_MORPHISM m,S_SPEC_LENV t],[])
  | S_GLOBAL_ENTRY (arch_spec_defn_env (x)) = AAppl (AId "arch-spec-defn-env",
             [S_ARCH_GENV_ENTRY x],[])
  | S_GLOBAL_ENTRY (unit_spec_defn_env (x)) = AAppl (AId "unit-spec-defn-env",
             [S_UNIT_GENV_ENTRY x],[])
  
and AS_GLOBAL_ENTRY (AAppl (AId "spec-defn-env",[g,s],[])) =
            spec_defn_env (AS_GENERICITY_ENV g,AS_SPEC_LENV s)
  | AS_GLOBAL_ENTRY (AAppl (AId "view-defn-env",[g,s,m,t],[])) =
            view_defn_env (AS_GENERICITY_ENV g,AS_SPEC_LENV s,AS_MORPHISM m,
                           AS_SPEC_LENV t)
  | AS_GLOBAL_ENTRY (AAppl (AId "arch-spec-defn-env",[x],[])) =
            arch_spec_defn_env (AS_ARCH_GENV_ENTRY x)
  | AS_GLOBAL_ENTRY (AAppl (AId "unit-spec-defn-env",[x],[])) =
            unit_spec_defn_env (AS_UNIT_GENV_ENTRY x)
  | AS_GLOBAL_ENTRY _ = raise ATerms_UnexpectedTerm

and S_ARCH_GENV_ENTRY x = S_ARCH_SIG x

and AS_ARCH_GENV_ENTRY x = AS_ARCH_SIG x

and S_UNIT_GENV_ENTRY x = S_UNIT_SIG x

and AS_UNIT_GENV_ENTRY x = AS_UNIT_SIG x

and S_SIGN (e) = AAppl (AId "signature",[S_LOCAL_ENV e],[])
                   
and AS_SIGN (AAppl (AId "signature",[e],[])) = (AS_LOCAL_ENV e)
  | AS_SIGN _ = raise ATerms_UnexpectedTerm

and S_GENERICITY_ENV ((a,b,c):genericity_env) = AAppl (AId "genericity-env",
            [S_SPEC_LENV a,
             AAppl (AId "spec-lenv*",[AList ((map S_SPEC_LENV) b,[])],[]),
             S_SIGN c],
            [])

and AS_GENERICITY_ENV (AAppl (AId "genericity-env",[a,
            AAppl (AId "spec-lenv*",[AList (b,[])],[]),
            c],[] )) =
            (AS_SPEC_LENV a,(map AS_SPEC_LENV) b,AS_SIGN c):genericity_env
  | AS_GENERICITY_ENV _ = raise ATerms_UnexpectedTerm         

and S_SPEC_LENV (SPEC_ENV (a,b,c)) = AAppl (AId "spec-lenv",[
            S_SIGN a,S_SIGN b,S_SPEC_ENV c],[])
            
and AS_SPEC_LENV (AAppl (AId "spec-lenv",[a,b,c],[])) =
            SPEC_ENV (AS_SIGN a,AS_SIGN b,AS_SPEC_ENV c)
  | AS_SPEC_LENV _ = raise ATerms_UnexpectedTerm

and S_EXT_SIGN (e1,e2,l) = AAppl (AId "ext-signature",[S_SIGN e1,
            S_SIGN e2,AAppl (AId "formula*",[AList ((map
            S_L_FORMULA) l,[])],[])],[])

and AS_EXT_SIGN (AAppl (AId "ext-signature",[e1,e2,AAppl (AId 
            "formula*",[AList (l,[])],[])],[])) =
            (AS_SIGN e1,AS_SIGN e2,(map AS_L_FORMULA) l)
  | AS_EXT_SIGN _ = raise ATerms_UnexpectedTerm

and S_SPEC_ENV senv = S_SPEC_ENV_with_ANNO(senv,[])

and S_SPEC_ENV_with_ANNO (basic_env (e,l),an) = AAppl (AId "basic-env",[S_SIGN e,
            AAppl (AId "formula*",[AList ((map S_L_FORMULA) l,[])],[])],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (translate_env (s,m),an) = AAppl (AId "translate-env",[
            S_SPEC_ENV s,S_MORPHISM m],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (derive_env (s,m),an) = AAppl (AId "derive-env",[
            S_SPEC_ENV s,S_MORPHISM m],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (union_env (l),an) = AAppl (AId "union-env",[AAppl (AId
            "spec-env*",[AList ((map S_SPEC_ENV) l,[])],[])],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (extension_env (l),an) = AAppl (AId "extension-env",[AAppl (AId
            "spec-env*",[AList ((map S_SPEC_ENV_with_ANNO) l,[])],[])],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (free_spec_env (s),an) = AAppl (AId "free-spec-env",[
            S_SPEC_ENV s],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (cofree_spec_env (s),an) = AAppl (AId "cofree-spec-env",[
            S_SPEC_ENV s],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (closed_spec_env (s),an) = AAppl (AId "closed-spec-env",[
            S_SPEC_ENV s],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (spec_inst_env (n,s,m,l),an) = AAppl (AId "spec-inst-env",
            [S_SPEC_NAME n,S_SPEC_ENV s,S_MORPHISM m,AAppl (AId "spec-env*",
            [AList ((map S_SPEC_ENV) l,[])],[])],[S_ANNO_LIST an])
  | S_SPEC_ENV_with_ANNO (dummy_spec_env,an) = AFun (AId "dummy-spec-env",[S_ANNO_LIST an])

and AS_SPEC_ENV (AAppl (AId "basic-env",[e,AAppl (AId "formula*",[AList 
            (l,[])],[])],[])) =
            basic_env (AS_SIGN e, (map AS_L_FORMULA) l)
  | AS_SPEC_ENV (AAppl (AId "translate-env",[s,m],[])) =
            translate_env (AS_SPEC_ENV s,AS_MORPHISM m)
  | AS_SPEC_ENV (AAppl (AId "derive-env",[s,m],[])) =
            derive_env (AS_SPEC_ENV s,AS_MORPHISM m)
  | AS_SPEC_ENV (AAppl (AId "union-env",[AAppl (AId "spec-lenv*",[AList (l,[])]
            ,[])],[])) =
            union_env ( (map AS_SPEC_ENV) l)
  | AS_SPEC_ENV (AAppl (AId "extension-env",[AAppl (AId "spec-lenv*",[AList
            (l,[])],[])],[])) =
            extension_env ( (map (fn x => (AS_SPEC_ENV x,[]))) l)
  | AS_SPEC_ENV (AAppl (AId "free-spec-env",[s],[])) =
            free_spec_env (AS_SPEC_ENV s)
  | AS_SPEC_ENV (AAppl (AId "cofree-spec-env",[s],[])) =
            cofree_spec_env (AS_SPEC_ENV s)
  | AS_SPEC_ENV (AAppl (AId "closed-spec-env",[s],[])) =
            closed_spec_env (AS_SPEC_ENV s)
  | AS_SPEC_ENV (AAppl (AId "spec-inst-env",[n,s,m,AAppl (AId 
            "spec-lenv*",[AList (l,[])],[])],[])) =
            spec_inst_env (AS_SPEC_NAME n,AS_SPEC_ENV s,AS_MORPHISM m,
                           (map AS_SPEC_ENV) l)
  | AS_SPEC_ENV (AFun (AId "dummy-spec-env",[])) = dummy_spec_env
  | AS_SPEC_ENV _ = raise ATerms_UnexpectedTerm

and S_MORPHISM ((s,f,p):morphism) = AAppl (AId "morphism",
            [S_SORT_MAP s,S_FUN_MAP f,S_PRED_MAP p],[])
            
and AS_MORPHISM (AAppl (AId "morphism",[s,f,p],[])) =
            (AS_SORT_MAP s,AS_FUN_MAP f,AS_PRED_MAP p):morphism
  | AS_MORPHISM _ = raise ATerms_UnexpectedTerm

and S_SORT_MAP x = AAppl (AId "sort-map",[S_SORT_TABLE
             (Symtab_id.dest x)],[])

and AS_SORT_MAP (AAppl (AId "sort-map",[x],[])) =
             Symtab_id.make (AS_SORT_TABLE x)
  | AS_SORT_MAP _ = raise ATerms_UnexpectedTerm

and S_SORT_TABLE x = AAppl (AId "sort-entry-tuple*",[AList
             ((map S_SORT_TUPLE) x,[])],[])

and AS_SORT_TABLE (AAppl (AId "sort-entry-tuple*",[AList (x,[])],[])) =
             (map AS_SORT_TUPLE) x
  | AS_SORT_TABLE _ = raise ATerms_UnexpectedTerm

and S_SORT_TUPLE (k,v) = AAppl (AId "sort-entry-tuple",[S_ID k,
             S_SORT v],[])
             
and AS_SORT_TUPLE (AAppl (AId "sort-entry-tuple",[k,v],[])) =
             (AS_ID k,AS_SORT v)
  | AS_SORT_TUPLE _ = raise ATerms_UnexpectedTerm

and S_FUN_MAP x = AAppl (AId "fun-map",[S_FUN_TABLE
             (Symtab_id.dest x)],[])

and AS_FUN_MAP (AAppl (AId "fun-map",[x],[])) =
             Symtab_id.make (AS_FUN_TABLE x)
  | AS_FUN_MAP _ = raise ATerms_UnexpectedTerm

and S_FUN_TABLE x = AAppl (AId "fun-entry-tuple*",[AList
             ((map S_FUN_TUPLE) x,[])],[])

and AS_FUN_TABLE (AAppl (AId "fun-entry-tuple*",[AList (x,[])],[])) =
             (map AS_FUN_TUPLE) x
  | AS_FUN_TABLE _ = raise ATerms_UnexpectedTerm

and S_FUN_TUPLE (k,v) = AAppl (AId "fun-entry-tuple",[S_ID k,AAppl (AId "funmap*",
             [AList((map S_FMAP) v,[])],[])],[])
             
and AS_FUN_TUPLE (AAppl (AId "fun-entry-tuple",[k,AAppl (AId "funmap*",[AList
            (v,[])],[])],[])) =
            (AS_ID k,(map AS_FMAP) v)
  | AS_FUN_TUPLE _ = raise ATerms_UnexpectedTerm

and S_FMAP (t,n,_) = AAppl (AId "funmap",[S_OP_TYPE t,S_OP_NAME n],[]) (*?BK?*)

and AS_FMAP (AAppl (AId "funmap",[t,n],[])) =
             (AS_OP_TYPE t,AS_OP_NAME n,true)  (*?BK?*)
  | AS_FMAP _ = raise ATerms_UnexpectedTerm

and S_PRED_MAP x = AAppl (AId "pred-map",[S_PRED_TABLE
             (Symtab_id.dest x)],[])

and AS_PRED_MAP (AAppl (AId "pred-map",[x],[])) =
             Symtab_id.make (AS_PRED_TABLE x)
  | AS_PRED_MAP _ = raise ATerms_UnexpectedTerm

and S_PRED_TABLE x = AAppl (AId "pred-entry-tuple*",[AList
             ((map S_PRED_TUPLE) x,[])],[])

and AS_PRED_TABLE (AAppl (AId "pred-entry-tuple*",[AList (x,[])],[])) =
             (map AS_PRED_TUPLE) x
  | AS_PRED_TABLE _ = raise ATerms_UnexpectedTerm

and S_PRED_TUPLE (k,v) = AAppl (AId "pred-entry-tuple",[S_ID k,AAppl (AId "predmap*",
             [AList((map S_PMAP) v,[])],[])],[])
             
and AS_PRED_TUPLE (AAppl (AId "pred-entry-tuple",[k,AAppl (AId "predmap*",[AList
            (v,[])],[])],[])) =
            (AS_ID k,(map AS_PMAP) v)
  | AS_PRED_TUPLE _ = raise ATerms_UnexpectedTerm

and S_PMAP (t,n) = AAppl (AId "predmap",[S_PRED_TYPE t,S_PRED_NAME n],[])

and AS_PMAP (AAppl (AId "predmap",[t,n],[])) =
             (AS_PRED_TYPE t,AS_PRED_NAME n)
  | AS_PMAP _ = raise ATerms_UnexpectedTerm

and S_LOCAL_ENV (s,v,f,p) = AAppl (AId "local-env",[S_SUBSORT_ENV s,
             S_VAR_ENV v,S_FUN_ENV f,S_PRED_ENV p],[])
             
and AS_LOCAL_ENV (AAppl (AId "local-env",[s,v,f,p],[])) =
             (AS_SUBSORT_ENV s,AS_VAR_ENV v,AS_FUN_ENV f,AS_PRED_ENV p)
  | AS_LOCAL_ENV _ = raise ATerms_UnexpectedTerm

and S_SORT_ENTRY l = AAppl (AId "sort*",[AList ((map S_SORT) l,[])],[])

and AS_SORT_ENTRY (AAppl (AId "sort*",[AList (l,[])],[])) =
             (map AS_SORT) l
  | AS_SORT_ENTRY _ = raise ATerms_UnexpectedTerm

and S_FUN_ENTRY l = AAppl (AId "op-type*",[AList ((map S_OP_TYPE) l,[])],[])

and AS_FUN_ENTRY (AAppl (AId "op-type*",[AList (l,[])],[])) =
             (map AS_OP_TYPE) l
  | AS_FUN_ENTRY _ = raise ATerms_UnexpectedTerm

and S_PRED_ENTRY l = AAppl (AId "pred-type*",[AList ((map S_PRED_TYPE) l,[])],
             [])

and AS_PRED_ENTRY (AAppl (AId "pred-type*",[AList (l,[])],[])) =
             (map AS_PRED_TYPE) l
  | AS_PRED_ENTRY _ = raise ATerms_UnexpectedTerm
             
and S_VAR_ENTRY (x:var_entry) = S_SORT x

and AS_VAR_ENTRY x = (AS_SORT x):var_entry

and S_SUBSORT_ENV x = AAppl (AId "subsort-env",[S_SUBSORT_TABLE
             (Symtab_id.dest x)],[])

and AS_SUBSORT_ENV (AAppl (AId "subsort-env",[x],[])) =
             Symtab_id.make (AS_SUBSORT_TABLE x)
  | AS_SUBSORT_ENV _ = raise ATerms_UnexpectedTerm

and S_SUBSORT_TABLE x = AAppl (AId "subsort-entry-tuple*",[AList
             ((map S_SUBSORT_TUPLE) x,[])],[])

and AS_SUBSORT_TABLE (AAppl (AId "subsort-entry-tuple*",[AList (x,[])],[])) =
             (map AS_SUBSORT_TUPLE) x
  | AS_SUBSORT_TABLE _ = raise ATerms_UnexpectedTerm

and S_SUBSORT_TUPLE (k,v) = AAppl (AId "subsort-entry-tuple",[S_ID k,
             S_SORT_ENTRY v],[])
             
and AS_SUBSORT_TUPLE (AAppl (AId "subsort-entry-tuple",[k,v],[])) =
             (AS_ID k,AS_SORT_ENTRY v)
  | AS_SUBSORT_TUPLE _ = raise ATerms_UnexpectedTerm

and S_VAR_ENV x = AAppl (AId "var-env",[S_VAR_TABLE
             (Symtab_sid.dest x)],[])

and AS_VAR_ENV (AAppl (AId "var-env",[x],[])) =
             Symtab_sid.make (AS_VAR_TABLE x)
  | AS_VAR_ENV _ = raise ATerms_UnexpectedTerm

and S_VAR_TABLE x = AAppl (AId "var-entry-tuple*",[AList
             ((map S_VAR_TUPLE) x,[])],[])

and AS_VAR_TABLE (AAppl (AId "var-entry-tuple*",[AList (x,[])],[])) =
             (map AS_VAR_TUPLE) x
  | AS_VAR_TABLE _ = raise ATerms_UnexpectedTerm

and S_VAR_TUPLE (k,v) = AAppl (AId "var-entry-tuple",[S_SIMPLE_ID k,
             S_VAR_ENTRY v],[])
             
and AS_VAR_TUPLE (AAppl (AId "var-entry-tuple",[k,v],[])) =
             (AS_SIMPLE_ID k,AS_VAR_ENTRY v)
  | AS_VAR_TUPLE _ = raise ATerms_UnexpectedTerm

and S_FUN_ENV x = AAppl (AId "fun-env",[S_FUNE_TABLE
             (Symtab_id.dest x)],[])

and AS_FUN_ENV (AAppl (AId "fun-env",[x],[])) =
             Symtab_id.make (AS_FUNE_TABLE x)
  | AS_FUN_ENV _ = raise ATerms_UnexpectedTerm

and S_FUNE_TABLE x = AAppl (AId "fun-entry-tuple*",[AList
             ((map S_FUNE_TUPLE) x,[])],[])

and AS_FUNE_TABLE (AAppl (AId "fun-entry-tuple*",[AList (x,[])],[])) =
             (map AS_FUNE_TUPLE) x
  | AS_FUNE_TABLE _ = raise ATerms_UnexpectedTerm

and S_FUNE_TUPLE (k,v) = AAppl (AId "fun-entry-tuple",[S_ID k,
             S_FUN_ENTRY v],[])
             
and AS_FUNE_TUPLE (AAppl (AId "fun-entry-tuple",[k,v],[])) =
             (AS_ID k,AS_FUN_ENTRY v)
  | AS_FUNE_TUPLE _ = raise ATerms_UnexpectedTerm

and S_PRED_ENV x = AAppl (AId "pred-env",[S_PREDE_TABLE
             (Symtab_id.dest x)],[])

and AS_PRED_ENV (AAppl (AId "pred-env",[x],[])) =
             Symtab_id.make (AS_PREDE_TABLE x)
  | AS_PRED_ENV _ = raise ATerms_UnexpectedTerm

and S_PREDE_TABLE x = AAppl (AId "pred-entry-tuple*",[AList
             ((map S_PREDE_TUPLE) x,[])],[])

and AS_PREDE_TABLE (AAppl (AId "pred-entry-tuple*",[AList (x,[])],[])) =
             (map AS_PREDE_TUPLE) x
  | AS_PREDE_TABLE _ = raise ATerms_UnexpectedTerm

and S_PREDE_TUPLE (k,v) = AAppl (AId "pred-entry-tuple",[S_ID k,
             S_PRED_ENTRY v],[])
             
and AS_PREDE_TUPLE (AAppl (AId "pred-entry-tuple",[k,v],[])) =
             (AS_ID k,AS_PRED_ENTRY v)
  | AS_PREDE_TUPLE _ = raise ATerms_UnexpectedTerm



(* ************************************************************************ *)
(* f_global_env                                                             *)
(* ************************************************************************ *)

and S_F_GLOBAL_ENV (x,a) = AAppl (AId "fcasenv",[S_GE_TABLE
                       (Symtab_sid.dest x)],[S_ANNO_LIST a])
                       
and AS_F_GLOBAL_ENV (AAppl (AId "fcasenv",[x],[ann1])) =
                     (Symtab_sid.make (AS_GE_TABLE x),AS_ANNO_LIST ann1)
  | AS_F_GLOBAL_ENV _ = raise ATerms_UnexpectedTerm


and S_GE_TABLE x = AAppl (AId "f-global-entry-tuple*",[AList
                   ((map S_GE_TUPLE) x,[])],[])

and AS_GE_TABLE (AAppl (AId "f-global-entry-tuple*",[AList (x,[])],[])) =
             (map AS_GE_TUPLE) x
  | AS_GE_TABLE _ = raise ATerms_UnexpectedTerm

and S_GE_TUPLE (k,v) = AAppl (AId "f-global-entry-tuple",[S_SIMPLE_ID k,
             S_F_GLOBAL_ENTRY v],[])
             
and AS_GE_TUPLE (AAppl (AId "f-global-entry-tuple",[k,v],[])) =
             (AS_SIMPLE_ID k,AS_F_GLOBAL_ENTRY v)
  | AS_GE_TUPLE _ = raise ATerms_UnexpectedTerm


and S_F_GLOBAL_ENTRY (f_spec_defn_env (ge,le)) =
                      AAppl (AId "f-spec-defn-env",[S_F_GENERICITY_ENV ge,
                      S_EXT_SIGN le],[])
  | S_F_GLOBAL_ENTRY (f_view_defn_env (ge,le,mo,ke)) =
                      AAppl (AId "f-view-defn-env",[S_F_GENERICITY_ENV ge,
                      S_EXT_SIGN le,S_MORPHISM mo,S_EXT_SIGN ke],[])
  | S_F_GLOBAL_ENTRY (f_arch_spec_defn_env) =
                     AFun (AId "f-arch-spec-defn-env",[])
  | S_F_GLOBAL_ENTRY (f_unit_spec_defn_env) =
                     AFun (AId "f-unit-spec-defn-env",[])
                     
and AS_F_GLOBAL_ENTRY (AAppl (AId "f-spec-defn-env",[ge,le],[])) =
                      f_spec_defn_env (AS_F_GENERICITY_ENV ge,
                      AS_EXT_SIGN le)
  | AS_F_GLOBAL_ENTRY (AAppl (AId "f-view-defn-env",[a,b,c,d],[])) =
                      f_view_defn_env (AS_F_GENERICITY_ENV a,
                      AS_EXT_SIGN b,AS_MORPHISM c,AS_EXT_SIGN d)
  | AS_F_GLOBAL_ENTRY (AFun (AId "f-arch-spec-defn-env",[])) =
                      f_arch_spec_defn_env
  | AS_F_GLOBAL_ENTRY (AFun (AId "f-unit-spec-defn-env",[])) =
                      f_unit_spec_defn_env
  | AS_F_GLOBAL_ENTRY _ = raise ATerms_UnexpectedTerm


and S_F_GENERICITY_ENV (a,b) = AAppl (AId "f-genericity-env",
                               [S_EXT_SIGN a,S_EXT_LIST b],[])

and AS_F_GENERICITY_ENV (AAppl (AId "f-genericity-env",[a,b],[])) =
                        (AS_EXT_SIGN a,AS_EXT_LIST b)
  | AS_F_GENERICITY_ENV _ = raise ATerms_UnexpectedTerm


and S_EXT_LIST x = AAppl (AId "ext-signature*",[AList
                   ((map S_EXT_SIGN) x,[])],[])
                   
and AS_EXT_LIST (AAppl (AId "ext-signature*",[AList (x,[])],[])) =
                (map AS_EXT_SIGN) x
  | AS_EXT_LIST _ = raise ATerms_UnexpectedTerm



(* ************************************************************************ *)
(* symmap things                                                            *)
(* ************************************************************************ *)

and S_UNIT_SIG (a,b) = AAppl (AId "unit-sig",[S_COMP_SUB_SIGS a,
                       S_SUB_SIG b],[])

and AS_UNIT_SIG (AAppl (AId "unit-sig",[a,b],[])) =
                (AS_COMP_SUB_SIGS a,AS_SUB_SIG b)
  | AS_UNIT_SIG _ = raise ATerms_UnexpectedTerm


and S_ARCH_SIG (a,b) = AAppl (AId "arch-sig",[S_NAMED_UNIT_SIGS a,
                       S_UNIT_SIG b],[])

and AS_ARCH_SIG (AAppl (AId "arch-sig",[a,b],[])) =
                (AS_NAMED_UNIT_SIGS a,AS_UNIT_SIG b)
  | AS_ARCH_SIG _ = raise ATerms_UnexpectedTerm


and S_COMP_SUB_SIGS l = AAppl (AId "comp-sub-sigs",[AAppl (AId "sub-sig*",
                        [AList ((map S_SUB_SIG) l,[])],[])],[])

and AS_COMP_SUB_SIGS (AAppl (AId "comp-sub-sigs",[AAppl (AId "sub-sigs*",
                     [AList (l,[])],[])],[])) = (map AS_SUB_SIG) l
  | AS_COMP_SUB_SIGS _ = raise ATerms_UnexpectedTerm
  

and S_NAMED_UNIT_SIGS x = AAppl (AId "named-unit-sigs",[S_UTS_TABLE
                          (Symtab_sid.dest x)],[])

and AS_NAMED_UNIT_SIGS (AAppl (AId "named-unit-sigs",[x],[])) =
                       Symtab_sid.make (AS_UTS_TABLE x)
  | AS_NAMED_UNIT_SIGS _ = raise ATerms_UnexpectedTerm


and S_UTS_TABLE x = AAppl (AId "unit-tag-sig-tuple*",[AList
                    ((map S_UTS_TUPLE) x,[])],[])

and AS_UTS_TABLE (AAppl (AId "unit-tag-sig-tuple*",[AList (x,[])],[])) =
                 (map AS_UTS_TUPLE) x
  | AS_UTS_TABLE _ = raise ATerms_UnexpectedTerm

and S_UTS_TUPLE (k,v) = AAppl (AId "unit-tag-sig-tuple",[S_SIMPLE_ID k,
                        S_UNIT_TAG_SIG v],[])
             
and AS_UTS_TUPLE (AAppl (AId "unit-tag-sig-tuple",[k,v],[])) =
                 (AS_SIMPLE_ID k,AS_UNIT_TAG_SIG v)
  | AS_UTS_TUPLE _ = raise ATerms_UnexpectedTerm


and S_UNIT_TAG_SIG (a,b) = AAppl (AId "unit-tag-sig",[S_COMP_TAG_SIGS a,
                           S_TAG_SIG b],[])
                           
and AS_UNIT_TAG_SIG (AAppl (AId "unit-tag-sig",[a,b],[])) =
                    (AS_COMP_TAG_SIGS a,AS_TAG_SIG b)
  | AS_UNIT_TAG_SIG _ = raise ATerms_UnexpectedTerm


and S_COMP_TAG_SIGS l = AAppl (AId "comp-tag-sigs",[AAppl (AId "tag-sig*",
                        [AList ((map S_TAG_SIG) l,[])],[])],[])
                       
and AS_COMP_TAG_SIGS (AAppl (AId "comp-tag-sigs",[AAppl (AId "tag-sig*",
                     [AList (l,[])],[])],[])) = (map AS_TAG_SIG) l
  | AS_COMP_TAG_SIGS _ = raise ATerms_UnexpectedTerm


and S_TAG_SIG (a,b) = AAppl (AId "tag-sig",[S_SUB_SIG a,S_TAGGING_MAP b],[])

and AS_TAG_SIG (AAppl (AId "tag-sig",[a,b],[])) =
               (AS_SUB_SIG a,AS_TAGGING_MAP b)
  | AS_TAG_SIG _ = raise ATerms_UnexpectedTerm


and S_TAGGING_MAP x = AAppl (AId "tagging-map",[S_RSL_TABLE
                          (Symtab_esym.dest x)],[])

and AS_TAGGING_MAP (AAppl (AId "tagging-map",[x],[])) =
                   Symtab_esym.make (AS_RSL_TABLE x)
  | AS_TAGGING_MAP _ = raise ATerms_UnexpectedTerm


and S_RSL_TABLE x = AAppl (AId "ref-sym-list-tuple*",[AList
                    ((map S_RSL_TUPLE) x,[])],[])

and AS_RSL_TABLE (AAppl (AId "ref-sym-list-tuple*",[AList (x,[])],[])) =
                 (map AS_RSL_TUPLE) x
  | AS_RSL_TABLE _ = raise ATerms_UnexpectedTerm

and S_RSL_TUPLE (k,v) = AAppl (AId "ref-sym-list-tuple",[S_E_SYMBOL k,
                        S_REF_SYM_LIST v],[])
             
and AS_RSL_TUPLE (AAppl (AId "ref-sym-list-tuple",[k,v],[])) =
                 (AS_E_SYMBOL k,AS_REF_SYM_LIST v)
  | AS_RSL_TUPLE _ = raise ATerms_UnexpectedTerm


and S_E_SYMBOL (SYM_ESYMBOL (x)) = AAppl (AId "sym-esymbol",[S_TSYMBOL x],[])
  | S_E_SYMBOL (EMB_ESYMBOL (a,b)) = AAppl (AId "emb-esymbol",[S_SORT a,
                                     S_SORT b],[])
                                     
and AS_E_SYMBOL (AAppl (AId "sym-esymbol",[x],[])) =
                           SYM_ESYMBOL (AS_TSYMBOL x)
  | AS_E_SYMBOL (AAppl (AId "emb-esymbol",[a,b],[])) =
                           EMB_ESYMBOL (AS_SORT a,AS_SORT b)
  | AS_E_SYMBOL _ = raise ATerms_UnexpectedTerm

and S_TSYMBOL (SORT_SYMBOL (s)) = AAppl (AId "sort-symbol",[S_SORT s],[])
  | S_TSYMBOL (TOTAL_FUN_SYMBOL (a,(b,c))) = AAppl (AId "total-fun-symbol",
                           [S_OP_NAME a,S_SORTSORT (b,c)],[])
  | S_TSYMBOL (PARTIAL_FUN_SYMBOL (a,(b,c))) = AAppl (AId "partial-fun-symbol",
                           [S_OP_NAME a,S_SORTSORT (b,c)],[])
  | S_TSYMBOL (PRED_SYMBOL (n,l)) = AAppl (AId "pred-symbol",[S_PRED_NAME n,
                           AAppl (AId "sort*",[AList ((map S_SORT) l,
                           [])],[])],[])

and AS_TSYMBOL (AAppl (AId "sort-symbol",[s],[])) = SORT_SYMBOL (AS_SORT s)
  | AS_TSYMBOL (AAppl (AId "total-fun-symbol",[a,b],[])) =
                           TOTAL_FUN_SYMBOL (AS_OP_NAME a,AS_SORTSORT b)
  | AS_TSYMBOL (AAppl (AId "partial-fun-symbol",[a,b],[])) =
                           PARTIAL_FUN_SYMBOL (AS_OP_NAME a,AS_SORTSORT b)
  | AS_TSYMBOL (AAppl (AId "pred-symbol",[a,AAppl (AId "sort*",[AList (l,[])],
                           [])],[])) =
                           PRED_SYMBOL (AS_OP_NAME a,(map AS_SORT) l)
  | AS_TSYMBOL _ = raise ATerms_UnexpectedTerm
  
  
and S_SORTSORT (a,b) = AAppl (AId "sorts-sort",[AAppl (AId "sort*",[AList
                       ((map S_SORT) a,[])],[]),S_SORT b],[])
                       
and AS_SORTSORT (AAppl (AId "sort-sorts",[AAppl (AId "sort*",[AList
                       (a,[])],[]),b],[])) =
                       ((map AS_SORT) a,AS_SORT b)
  | AS_SORTSORT _ = raise ATerms_UnexpectedTerm


and S_REF_SYM_LIST l = AAppl (AId "ref-sym*",[AList ((map S_REF_SYM) l,
                       [])],[])
                       
and AS_REF_SYM_LIST (AAppl (AId "ref-sym*",[AList (l,[])],[])) =
                       (map AS_REF_SYM) l
  | AS_REF_SYM_LIST _ = raise ATerms_UnexpectedTerm
                       

and S_REF_SYM (a,b) = AAppl (AId "ref-sym",[S_UNIT_NAME a,S_E_SYMBOL b],[])

and AS_REF_SYM (AAppl (AId "ref-sym",[a,b],[])) =
                       (AS_UNIT_NAME a,AS_E_SYMBOL b)
  | AS_REF_SYM _ = raise ATerms_UnexpectedTerm


and S_SUB_SIG x = AAppl (AId "sub-sig",[S_LOCAL_ENV x],[])

and AS_SUB_SIG (AAppl (AId "sub-sig",[x],[])) = AS_LOCAL_ENV x
  | AS_SUB_SIG _ = raise ATerms_UnexpectedTerm


(* ************************************************************************ *)
(* wrapper                                                                  *)
(* ************************************************************************ *)

and S_ASPEC (s,a) = AAppl (AId "spec-list",[S_SPEC s],[S_ANNO_LIST a])

and AS_ASPEC (AAppl (AId "spec-list",[s],[a]))
                  = (AS_SPEC s,AS_ANNO_LIST a)
  | AS_ASPEC _ = raise ATerms_UnexpectedTerm

(* ************************************************************************ *)
(* user functions                                                           *)
(* ************************************************************************ *)

fun read_casfix x = (AS_LIB_DEFN o reconv_at o ATLib.string_to_aterm) x

fun load_casfix x = (AS_LIB_DEFN o reconv_at o ATLib.file_to_aterm) x

fun mkAT l = TextForm.format_text TextForm.HTML_courier 
             ((ATLib.aterm_to_string o conv_at o S_LIB_DEFN2) l)

fun mkA x = TextForm.format_text TextForm.Text_plain 
            ((ATLib.aterm_to_string o conv_at o S_LIB_DEFN2) x)

fun mkAT_genv (g,x) = TextForm.format_text TextForm.HTML_courier 
             ((ATLib.aterm_to_string o conv_at o S_LIB_DEFN) (g,x))

fun mkA_genv (g,x) = TextForm.format_text TextForm.Text_plain 
            ((ATLib.aterm_to_string o conv_at o S_LIB_DEFN) (g,x))

fun mkA_file      f x     = ATLib.aterm_to_file f ((conv_at o S_LIB_DEFN2) x)

fun mkA_genv_file f (g,x) = ATLib.aterm_to_file f ((conv_at o S_LIB_DEFN) (g,x))

end
