(* *********************************************************************** *)
(*									   *)
(* Project: HOL-CASL 							   *)
(* Author: Kolyang & Till Mossakowski, University of Bremen		   *)
(* Date: 09.06.97				 			   *)
(* Purpose of this file: Encoding of CFOL-encoded CASL into Isabelle/HOL   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)

(* This module contains an encoding of the CFOL-representation
   generated by encode.sml into Isabelle/HOLs internal term datastructure.
   We use CFOL instead of SOL as basis, because we generate the induction
   axioms using meta-quantification and meta-implication, improving
   the direct applicability of the generated axioms.

   We also generate different kinds of print syntaxes,
   leaving out the type information or not, leaving out
   subsort injections or not, etc. 

   todo:
   alle Axiomnamen nach ML-Ids konvertieren
   alle Induktions-Variablennamen nach Isabelle-IDs
   bei Sorten-Umbenennung auch Axiomnamen umbenennen (wichtig fr Induktion)
     (oder lieber gleich Axiome nach Form durchsuchen?)
   Encode SortGen with partial functions (see TCS paper)

   Klammerung in IsaWin: ((12)3)
   Prezedenzen in /home/till/CASL/HOL98-CASL/examples/in0t2.casl
   bei Goal mit := und @
   Goal-Ausgabe: (Verbesserung des Pretty-Printing in encode)
      strikte Erweiterungen mit "_" am Anfang,
      d.h. f:s->s und _f:?s->?s
   Leerzeichen bei Mixfix-WORDS einfuegen
   Evtl. auch metavars xxx in Goal zulassen?
   Metavariablen und ?-Sorten fuehren zu Konfusion
   Fenster mit Wahlmoeglichkeiten bei mehrdeutigem Goal

Bei Ersetzung von Elem durch Nat

Reading Demos/datatypes.casl
Analyzing from Basic/Numbers...
Analyzing spec List...
Analyzing spec Tree...
 
No type for ?[List] found
 
uncaught exception ERROR
  raised at: /home/till/CASL/CATS/HOL-CASL/basic_encode.sml:94.70-94.75
             goals.ML:258.17      

*)

(* Global variable determining whether subsort projections should be
   used or not. Using them can lead to lots of axioms, which may
   disturb when proving with IsaWin. Using them not disables the
   possibility of using casts in the input specifications. *)
val use_projections = ref true;

val the_axiom = ref refl;

structure BasicEncodeIsabelle :
sig
val encode_basic_CFOL_HOL : string -> theory -> LocalEnv.sign * (AS.FORMULA * string) list -> theory
val encode_ax_CFOL_HOL : string -> theory -> AS.L_FORMULA -> term * AS.ANNO list
val encode_flat_SubPCFOL_HOL : string ->
       theory -> GlobalEnv.ext_signature -> theory
val encode_flat_CFOL_HOL : string ->
       theory -> GlobalEnv.ext_signature -> theory


val ShowInjections : bool ref;
val ShowProfiles  : bool  ref;
end
= 
struct

open Theory AS LocalEnv GlobalEnv Subsorts BasicPrint BasicEncode;


(* *********************************************************************** *)
(*									   *)
(*	Encode CFOL into HOL		 				   *)	
(*									   *)
(* *********************************************************************** *)

val AxNumber = ref(0);
val ShowInjections = ref(false);
val ShowProfiles = ref(false);


fun printtyp (Type (s,_)) = s
  | printtyp _ = ""

fun try_typ s sg nil = None
  | try_typ s sg (t::rest) =
    Some (Sign.certify_typ sg t)
    handle _ => try_typ s sg rest

fun SORT_to_typ1 (name,th) (s:SORT):typ option = 
    let val str = print_ID s
        (*val _ = writeln("Looking for "^str)*)
        fun make_typ n = Type (n^"."^str,[])
        val ancestors = name::map (Sign.name_of o sign_of) (ancestors_of th)
        val typlist = map make_typ ancestors
        (*val _ = seq (writeln o printtyp) typlist*) 
    in
    try_typ str (sign_of th) typlist
    end

fun SORT_to_typ (name,th) s =
    the (SORT_to_typ1 (name,th) s)
    handle _ => (writeln ("No type for "^print_ID s^" found"); raise ERROR)

val bool_type = Type ("bool",[])

val TVnumber = ref(0);
fun newTVnumber(TVnumber) = (TVnumber := !TVnumber+1; !TVnumber);
val alpha = TVar  (("'a"^string_of_int (newTVnumber(TVnumber)),0),["term"]) 

fun type_of_equals name t = 
    let val t1=SORT_to_typ name t
    in [t1,t1]--->bool_type
    end
val bin_conn_type = [bool_type,bool_type]--->bool_type
val un_conn_type = bool_type-->bool_type
fun quant_type name t = (SORT_to_typ name t -->bool_type)-->bool_type

fun bin_conj (t1,t2:term):term =(Const ("op &",bin_conn_type) $ t1 $ t2);
fun bin_disj (t1,t2:term):term =(Const ("op |",bin_conn_type) $ t1 $ t2);

(**************** CFOL-->HOL: Sorts ****************)

fun get_mixfix_sort_name name =
case explode name of
	"$"::rest => Delimfix (implode rest)
	| _ => Delimfix name;
	 
fun SORT_to_arities (th, s:SORT) = 
     add_arities [((print_ID s),[],"term")] th;

fun encode_arities (th:theory) (sortlist:SORT list) =
  foldl SORT_to_arities (th,sortlist);
   
fun encode_sort name (th:theory,s:SORT) =
    case SORT_to_typ1 (name,th) s of
    Some _ => th
    | None => add_types [((print_ID s),0,NoSyn (*get_mixfix_sort_name s*))] th
	
fun encode_sorts name (th:theory) (sortlist:SORT list):theory =
	foldl (encode_sort name) (th,sortlist)


(**************** CFOL-->HOL: Functions ****************)

(* Add the profile to a function name *)

fun print_f1 nil = ""
  | print_f1 (s::xs) = print_ID s^"$"^print_f1 xs

fun remove_special1 "." = "$dot"
  | remove_special1 "_" = "'_"
  | remove_special1 x = x

val remove_special = map remove_special1

val remove_specials = implode o remove_special o explode


fun remove_dot1 "." = "_dot_"
  | remove_dot1 x = x

val remove_dot = map remove_dot1

val remove_dots = implode o remove_dot o explode

fun make_fun_name (f:OP_NAME,t:OP_TYPE):string = 
case (remove_specials (print_ID f), t) of
  (name, total_op_type(s, s1)) =>
     (case Utils.get_sorts s of 
        [] => name ^"$"^(print_SORTS1 [s1])
       | ss => name ^"$"^ (print_f1 ss) ^"$"^(print_SORTS1 [s1])
      )
|_ => "NOT_A_FUNCTION";


fun make_fun_type name (f:OP_NAME,total_op_type(s, s1):OP_TYPE):typ = 
  (map (SORT_to_typ name) (Utils.get_sorts s)) ---> (SORT_to_typ name) s1

fun add_star x = "*"^x
fun show_args nil = ""
  | show_args (arg::rest) = arg^concat(map add_star rest)


fun quote_isabelle "_" = "'_"
  | quote_isabelle "'" = "''"
  | quote_isabelle "(" = "'("
  | quote_isabelle ")" = "')"
  | quote_isabelle "/" = "'/"
  | quote_isabelle "." = "'."
  | quote_isabelle x = x

val quote_isabelles = concat o (map quote_isabelle) o explode;

fun quote_isabelle1 "'" = "''"
  | quote_isabelle1 "(" = "'("
  | quote_isabelle1 ")" = "')"
  | quote_isabelle1 "/" = "'/"
  | quote_isabelle1 "." = "'."
  | quote_isabelle1 x = x


fun is_underscore x = x = "_"

fun underscore x =
   let val (l1,l2) = take_prefix is_underscore x
       val trans_und = 
           case l1 of 
             (* no leading underscores *)
             [] => []
             (* one underscore, just quote it *)
            | [_] => ["'","_"]
             (* more than one = two underscores, collapse into one *)
            | _   => ["_"]
   in
   trans_und@underscore2 l2
   end

and underscore2 nil = nil
  | underscore2 x = 
    let val (l1,l2) = take_prefix ((op not) o is_underscore) x
    in
    case l1 of 
      [] => []
      | _ => (case l2 of
              [] => l1
              | _ => l1 @ [" "] @ underscore l2) 
    end

fun to_sml_tk1 "<" = "&lt;"
  | to_sml_tk1 x = x

val to_sml_tk = map to_sml_tk1



fun same_name1 (longname,shortname) =
    fst (take_prefix (fn x => not (x=":")) (explode longname)) = explode shortname;

fun same_name (simple_id (_,longname,_),simple_id (_,shortname,_)) =
    same_name1 (longname,shortname)
  | same_name (compound_id ((_,longname,_),comps1),
               compound_id ((_,shortname,_),comps2)) =
   (* IDEquality.ID1list_ord (comps1,comps2) = EQUAL andalso*)
    same_name1 (longname,shortname)


(* ??? is this necessary ? *)

fun is_mixfix1 nil = false
  | is_mixfix1 ("_"::"_"::_) = true
  | is_mixfix1 (_::rest) = is_mixfix1 rest
  
fun is_mixfix (simple_id(_,s,_)) = is_mixfix1 (explode s)
  | is_mixfix (compound_id((_,s,_),_)) = is_mixfix1 (explode s)

fun get_mixfix_fun_name (showInjections,showTypes) (an:ANNO list) 
	(f:OP_NAME,total_op_type(sorts s, s1):OP_TYPE):mixfix =
let val prio = 28
    fun const_prio x = prio
    val res_prio = prio-1
    fun const_u x = ",_"
    fun const_u1 x = "_"
    val f_id = print_ID f
    val qual_name' = 
       if showTypes 
       then f_id^":"^show_args [(print_SORTS1 s)]^(if s=nil then "" else "->")^ (print_SORTS1 [s1])
       else f_id
    val is_number = case get_number_an an of
        Utils.Some (number_anno n) => same_name(f,n)
        | _ => false
    val qual_name = if is_number 
                    then concat (map const_u1 s)
                    else qual_name'  
in case f_id of
   "inj$" => if showInjections then (Delimfix (print_SORTS1 s^">"^(print_SORTS1 [s1])))
             else Delimfix ""
   | "proj$" => (Delimfix ("_ as "^implode(tl(explode((print_SORTS1 [s1]))))))
   | _ =>
   	(case s of
   	 [] => (* Just a constant, i.e. no mixfix treatment necessary *)
               Delimfix (quote_isabelles qual_name)
   	 | (x::xs) => 
              if is_number 
              then Mixfix (qual_name, map (K 0) s, 1000 ) 
              else if is_mixfix f  
              then  (* in case of Mixfix, e.g. "__+__" becomes  "_+_",
                       because Isabelle's placeholder for Mixfix arguments is "_". *)
                    (Mixfix  ((implode o (map quote_isabelle1) o underscore o explode) qual_name, 
                               map const_prio s , 
                               res_prio )  ) 
              else                           
                    (* otherwise just quote the underscores to prevent the from becoming
                       argument placeholders, e.g. "hello_world" becomes "hello'_world". *)
		    (Mixfix  ( quote_isabelles qual_name
                                 ^"'(_"^concat (map const_u xs)^"')", 
                               map const_prio s , 
                               res_prio )  )
   	)
end

fun flatten_list (a, []) = []
 | flatten_list (a, (l::ls)) = (a,l)::flatten_list (a,ls);

(* just take one profile *) 
fun encode_fun1 name an ambigous (th, f) =
 let val fname = make_fun_name f
     (*val x = writeln("Encoding fun "^ fname)*)
     val ftype = make_fun_type name f
 in
   th
   |> add_consts_i 
         [(fname, ftype, get_mixfix_fun_name (false,false) an f)]
   |> add_modesyntax_i ("Inj",false)
         [(fname, ftype, get_mixfix_fun_name (true,false) an f)]
   |> add_modesyntax_i ("Types",false)
         [(fname, ftype, get_mixfix_fun_name (false,true) an f)]
   |> add_modesyntax_i ("InjTypes",false)
         [(fname, ftype, get_mixfix_fun_name (true,true) an f)]
   |> add_modesyntax_i ("AutoTypes",false)
         [(fname, ftype, get_mixfix_fun_name (false,ambigous) an f)]
   |> add_modesyntax_i ("InjAutoTypes",false)
         [(fname, ftype, get_mixfix_fun_name (false,ambigous) an f)]
 end;

fun is_ambigous (x::nil) = false
|   is_ambigous nil = false
|   is_ambigous l = true

fun encode_fun name an ((th:theory), (funlist: (OP_NAME * fun_entry))):theory =
let val liste1 = flatten_list funlist 
in  foldl (encode_fun1 name an (is_ambigous (snd funlist)))(th, liste1)
end; 


fun encode_funs name an (th:theory) (funlist: Fun_list):theory =
foldl (encode_fun (name,th) an) (th,funlist);



(**************** CFOL-->HOL: Predicates ****************)

(* Add the profile to a predicate name *)
fun make_pred_name (name:PRED_NAME,t:PRED_TYPE):string =
let val pred_type(s) = t
in remove_specials (print_ID name)
    ^"$"^(print_f1 (Utils.get_sorts s))^"$bool"
end;

fun make_pred_type name (f:PRED_NAME,pred_type s:PRED_TYPE):typ = 
    (map (SORT_to_typ name) (Utils.get_sorts s)) ---> bool_type


fun get_mixfix_pred_name (showInjections,showTypes)
	(f:PRED_NAME,pred_type sl:PRED_TYPE):mixfix =
let val s = Utils.get_sorts sl
    fun get_res nil = nil
    |   get_res ("$"::rest) = nil
    |   get_res (x::rest) = x::get_res rest
    fun const0 x = 0
    fun const_u x = ",_"
    val f_id = print_ID f
    val qual_name = if showTypes then f_id^":pred("^show_args [ print_SORTS1 s]^")"
    	  	    else f_id
in case explode f_id of
   ("m"::("e"::("m"::"b"::"$"::rest))) => (Delimfix ("_ in "^(implode(get_res rest))))
   | ("d"::"e"::"f"::"$"::rest) => (Delimfix ("def _ "))  (* defined <-| def *)
   | ("e"::"e"::"q"::"$"::rest) => (Mixfix ("_ =e= _",[0,0],0))
   | _ =>
   	(case s of
         [] =>  Delimfix (quote_isabelles qual_name)
   	 | (x::xs) => 
             if is_mixfix f 
             then (Mixfix ((implode o (map quote_isabelle1) o underscore o explode) qual_name, 
                           map const0 s , 
                           0 )  )
             else (Mixfix (quote_isabelles qual_name
                             ^"'(_"^concat (map const_u xs)^"')", 
                           map const0 s , 
                           0 )  )
   	)
end   

fun encode_pred1 name ambigous (th, f) =
 let val pname = make_pred_name f
     (*val _ = writeln("Encoding "^pname);*)
     val ptype = make_pred_type name f
 in
   th
   |> add_consts_i 
         [(pname, ptype, get_mixfix_pred_name (false,false) f)]
   |> add_modesyntax_i ("Inj",false)
         [(pname, ptype, get_mixfix_pred_name (true,false) f)]
   |> add_modesyntax_i ("Types",false)
         [(pname, ptype, get_mixfix_pred_name (false,true) f)]
   |> add_modesyntax_i ("InjTypes",false)
         [(pname, ptype, get_mixfix_pred_name (true,true) f)]
   |> add_modesyntax_i ("AutoTypes",false)
         [(pname, ptype, get_mixfix_pred_name (false,ambigous) f)]
   |> add_modesyntax_i ("InjAutoTypes",false)
         [(pname, ptype, get_mixfix_pred_name (false,ambigous) f)]
 end;





fun encode_pred name ((th:theory), (predlist: (PRED_NAME * pred_entry) )):theory =
let val liste1 = flatten_list predlist 
in  foldl (encode_pred1 name (is_ambigous (snd predlist))) (th, liste1)
end;
fun encode_preds name an (th:theory) (predlist: Pred_list):theory =
   foldl (encode_pred (name,th)) (th,predlist);



(**************** CFOL-->HOL: Axioms ****************)

(* Auxiliary functions *)



fun big_and l =
   if null l 
   then Const ("True",bool_type)
   else Utils.nefoldl bin_conj l

(* Sort generation constraints *)

local
    fun make_pred name srts s =
        let val t = SORT_to_typ name s
            val ind = find_index (fn t => t=s) srts
        in Free("P"^BasisLibrary.Int.toString ind,t-->bool_type)
        end
    fun full_pred name srts s =
        let val t = SORT_to_typ name s
            val ind = find_index (fn t => t=s) srts
        in Free("P"^BasisLibrary.Int.toString ind,t-->bool_type)
           $ Free("x"^BasisLibrary.Int.toString ind,t)
        end
    fun mk_x1 name (s,n) = ("x"^BasisLibrary.Int.toString n,SORT_to_typ name s)
    fun mk_x name (s,n) = Free("x"^BasisLibrary.Int.toString n,SORT_to_typ name s)
    fun get_prem name srts (s,n) =
        if s mem srts 
        then let val t = SORT_to_typ name s
             in Some (make_pred name srts s $ Free("x"^BasisLibrary.Int.toString n,t))
             end
        else None
    fun get_ind_hyp name srts (op_symb (f,Utils.Some(ftype as (total_op_type (sorts args,res))))) =
          if res mem srts then 
          let val fname = make_fun_name (f,ftype)
              val ftype = make_fun_type name (f,ftype)
              val args_nos = Utils.zip (args,1 upto length args)
              val concl = make_pred name srts res $ 
                         ( foldl (op $) (Free(fname,ftype), map (mk_x name) args_nos) )
              val prems = mapfilter (get_prem name srts) args_nos
              val phi = if null prems then concl
                        else Const ("op -->", bin_conn_type) $ (big_and prems) $ concl
          in
          Some (list_all_free (map (mk_x1 name) args_nos,HOLogic.mk_Trueprop phi))
          end
          else None
       | get_ind_hyp name srts (op_symb (f,Utils.Some(ftype as (partial_op_type (sorts args,res))))) =
         if res mem srts then
          (writeln "Warning: no correct encoding of sort generation with partial functions yet";
                              None)
         else None

in
fun trans_sort_gen name (srts,funs) =
    let val ind_hyp = mapfilter (get_ind_hyp name srts) funs
        val ind_concl = HOLogic.mk_Trueprop (big_and (map (full_pred name srts) srts))
    in
    Logic.list_implies (ind_hyp,ind_concl)
    end
end;      	      


(* Terms *)

fun trans_TERM name preds (a:TERM):(term*SORT)= 
case a of
   qual_var ( x,s)  => (Free(print_vars [x],SORT_to_typ name s),s)
 | application (op_symb(f,Utils.Some ftype), terms tt) => 
                  let val fname= make_fun_name (f, ftype)
                      val HOL_ftype = make_fun_type name (f,ftype)
                  in (foldl (op $)(Free(fname,HOL_ftype), map (fst o (trans_TERM name preds))  tt),Utils.get_res ftype)
                  end
 | sorted_term (t,s) => (fst (trans_TERM name preds t),s)
 | conditional (T1,Phi,T2) =>
   let val t1 = trans_TERM name preds T1
       val t2 = trans_TERM name preds T2
       val phi = trans_FORMULA name preds Phi
       val HOL_Iftype = [bool_type, SORT_to_typ name (snd t1), SORT_to_typ name (snd t2)] ---> SORT_to_typ name (snd t2)
   in  ((Const ("If",HOL_Iftype) $ phi $ (fst t1) $ (fst t2)), snd t1)
			 (* attenton with the sort type of  this term *)
			 (* raise ERROR  *)
   end
 | pos_TERM (_,_,t) => trans_TERM name preds t
 (*   If :: "[bool, ?'a, ?'a] => ?'a"
 *)

(* Atomic formulas *)

and trans_ATOM name preds (a:ATOM):term =
case a of
    (predication (pred_symb(p,Utils.Some t),terms ts))   =>
        let val pname = if (p,t) mem preds 
                        then remove_specials (print_ID p)
                        else make_pred_name (p, t)
            val HOL_ptype = make_pred_type name (p,t)
        in foldl (op $)(Free(pname, HOL_ptype), map (fst o (trans_TERM name preds)) ts)
        end  	
  | (existl_equation (T1 , T2)) => 
       let val (t1,s1) = trans_TERM name preds T1
           val (t2,s2) = trans_TERM name preds T2
       in (Const ("op =", type_of_equals name s1) $ t1 $ t2)
       end
  | (strong_equation (T1 , T2)) => 
       let val (t1,s1) = trans_TERM name preds T1
           val (t2,s2) = trans_TERM name preds T2
       in (Const ("op =", type_of_equals name s1) $ t1 $ t2)
       end
  | ttrue => Const ("True",bool_type)
  | ffalse => Const ("False",bool_type)
  | t => raise (WRONG (4,its_an_atom t))


(* First-order formulas *)

and trans_FORMULA name preds (phi:FORMULA):term =

let fun get_var_decl ([], s) = []
      | get_var_decl (v::ls, s)= (print_SIMPLE_ID v, s)::get_var_decl(ls,s);
    fun get_var_decls vl = flat (map get_var_decl vl)

    fun quantconst forall = "All"
      | quantconst exists = "Ex"
      | quantconst exists_uniquely = "Ex1"
    fun quant name q ((v,t),phi) = (Const (q,quant_type name t) $ (absfree (v,SORT_to_typ name t, phi)))
(*    fun quant_pred name q ((p,t),phi) = 
        let val HOL_ptype = make_pred_type name (p,t)
        in (Const (q,(HOL_ptype -->bool_type)-->bool_type) $ (absfree (print_SIMPLE_ID p,HOL_ptype, phi)))
        end*)
in
case phi of
       (atom a) => trans_ATOM name preds a
  | (conjunction philist) => Utils.nefoldl bin_conj (map (trans_FORMULA name preds) philist)
  | (disjunction philist) => Utils.nefoldl bin_disj (map (trans_FORMULA name preds) philist)
  | (implication (f1,f2)) =>(Const ("op -->", bin_conn_type) $ trans_FORMULA name preds f1 $ trans_FORMULA name preds f2) 
  | (equivalence (f1,f2)) => Const ("op =", bin_conn_type) $ (trans_FORMULA name preds f1) $(trans_FORMULA name preds f2)
  | negation phi1 => (Const ("Not", un_conn_type) $ trans_FORMULA name preds phi1)
  | quantification (q ,vlist, f) =>  
                foldr (quant name (quantconst q)) ((get_var_decls vlist),trans_FORMULA name preds f)
  | pred_quantification (forall ,vlist, f) =>  
               (* foldr (quant_pred name (quantconst q)) ((get_var_decls vlist),*) (trans_FORMULA name (preds@vlist) f)
  | sort_gen_ax x => raise ERROR
  | unparsed_formula x => raise ERROR
  | pos_FORMULA(_,_,phi) => trans_FORMULA name preds phi
end;


fun del_white_space s =
    let fun is_white_space c = ord c mem [9,10,13,32];
        fun del_initial_ws s =
            snd (take_prefix is_white_space s)
    in implode (rev (del_initial_ws (rev (del_initial_ws (explode s))))) 
    end

fun is_alpha c = ("A"<= c andalso c<="Z")
    orelse ("a"<= c andalso c<="z")

fun is_ml_char c = ("A"<= c andalso c<="Z")
    orelse ("a"<= c andalso c<="z")
    orelse ("0"<= c andalso c<="9")
    orelse (c="'") orelse (c="_")

fun syntax_ok [] = false
  | syntax_ok (c::rest) = 
    is_alpha c
    andalso Utils.forall is_ml_char rest
  
fun is_ml s = syntax_ok (explode s)


(* ??? This has to be improved !!!!*)

fun replace_non_ids " " = "_"
  | replace_non_ids "!" = "Exclam"
  | replace_non_ids "\"" = "_"
  | replace_non_ids "#" = "Sharp"
  | replace_non_ids "$" = "_"
  | replace_non_ids "%" = "Percent"
  | replace_non_ids "&" = "Amp"
  | replace_non_ids "(" = "_"
  | replace_non_ids ")" = "_"
  | replace_non_ids "*" = "x"
  | replace_non_ids "+" = "Plus"
  | replace_non_ids "," = "_"
  | replace_non_ids "-" = "Minus"
  | replace_non_ids "." = "Dot"
  | replace_non_ids "/" = "Div"
  | replace_non_ids ":" = "_"
  | replace_non_ids ";" = "_"
  | replace_non_ids "<" = "Lt"
  | replace_non_ids "=" = "Eq"
  | replace_non_ids ">" = "Gt"
  | replace_non_ids "?" = "Q"
  | replace_non_ids "@" = "At"
  | replace_non_ids "[" = "_"
  | replace_non_ids "\\" = "Back"
  | replace_non_ids "]" = "_"
  | replace_non_ids "^" = "Hat"
  | replace_non_ids "`" = "'"
  | replace_non_ids "{" = "Cur"
  | replace_non_ids "|" = "Bar"
  | replace_non_ids "}" = "Ruc"
  | replace_non_ids "~" = "Tilde"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "AE"
  | replace_non_ids "" = "C"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "D"
  | replace_non_ids "" = "N"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "x"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "Y"
  | replace_non_ids "" = "F"
  | replace_non_ids "" = "ss"
  | replace_non_ids "" = "_"
  | replace_non_ids "" = "!"
  | replace_non_ids "" = "c"
  | replace_non_ids "" = "Lb"
  | replace_non_ids "" = "o"
  | replace_non_ids "" = "Yen"
  | replace_non_ids "" = "Bar1"
  | replace_non_ids "" = "Paragraph"
  | replace_non_ids "" = "\""
  | replace_non_ids "" = "Copyright"
  | replace_non_ids "" = "a"
  | replace_non_ids "" = "\""
  | replace_non_ids "" = "not"
  | replace_non_ids "" = "-"
  | replace_non_ids "" = "Regmark"
  | replace_non_ids "" = "_"
  | replace_non_ids "" = "Degree"
  | replace_non_ids "" = "Plusminus"
  | replace_non_ids "" = "2"
  | replace_non_ids "" = "3"
  | replace_non_ids "" = "'"
  | replace_non_ids "" = "Mu"
  | replace_non_ids "" = "q"
  | replace_non_ids "" = "Dot"
  | replace_non_ids "" = "'"
  | replace_non_ids "" = "1"
  | replace_non_ids "" = "2"
  | replace_non_ids "" = "\""
  | replace_non_ids "" = "Quarter"
  | replace_non_ids "" = "Half"
  | replace_non_ids "" = "Threequarter"
  | replace_non_ids "" = "Q"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "A"
  | replace_non_ids "" = "AE"
  | replace_non_ids "" = "C"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "E"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "I"
  | replace_non_ids "" = "D"
  | replace_non_ids "" = "N"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "O"
  | replace_non_ids "" = "x"
  | replace_non_ids "" = "0"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "U"
  | replace_non_ids "" = "Y"
  | replace_non_ids "" = "F"
  | replace_non_ids "" = "ss"
  | replace_non_ids "" = "a"
  | replace_non_ids "" = "a"
  | replace_non_ids "" = "a"
  | replace_non_ids "" = "a"
  | replace_non_ids "" = "a"
  | replace_non_ids "" = "a"
  | replace_non_ids "" = "ae"
  | replace_non_ids "" = "c"
  | replace_non_ids "" = "e"
  | replace_non_ids "" = "e"
  | replace_non_ids "" = "e"
  | replace_non_ids "" = "e"
  | replace_non_ids "" = "i"
  | replace_non_ids "" = "i"
  | replace_non_ids "" = "i"
  | replace_non_ids "" = "i"
  | replace_non_ids "" = "d"
  | replace_non_ids "" = "n"
  | replace_non_ids "" = "o"
  | replace_non_ids "" = "o"
  | replace_non_ids "" = "o"
  | replace_non_ids "" = "o"
  | replace_non_ids "" = "o"
  | replace_non_ids "" = "Div1"
  | replace_non_ids "" = "o"
  | replace_non_ids "" = "u"
  | replace_non_ids "" = "u"
  | replace_non_ids "" = "u"
  | replace_non_ids "" = "u"
  | replace_non_ids "" = "y"
  | replace_non_ids "" = "f"
  | replace_non_ids "" = "y"
  | replace_non_ids x = "Error"
  


fun make_ml_chars nil = nil
  | make_ml_chars ("-"::">"::rest) = "To"::make_ml_chars rest
  | make_ml_chars (x::rest) = 
    (if is_ml_char x then x else replace_non_ids x)
    :: make_ml_chars rest

fun remove_double_underscores nil = nil
  | remove_double_underscores ("_"::"_"::rest) =
    remove_double_underscores ("_"::rest)
  | remove_double_underscores (x::rest) =
    x :: remove_double_underscores rest

fun make_ml_id id = 
    case explode id of
      nil => ""
      | (init::rest) => 
         implode ((if is_alpha init then init else "X") :: 
                  remove_double_underscores (make_ml_chars rest))

fun strip_forall (quantification (forall,vlist, phi)) = strip_forall phi
  | strip_forall (pos_FORMULA (_,_,phi)) = strip_forall phi
  | strip_forall x =  x;


fun lift_implication (Const ("op -->", t) $ f1 $ f2) = Logic.mk_implies (HOLogic.mk_Trueprop f1, HOLogic.mk_Trueprop f2)
  | lift_implication x =  HOLogic.mk_Trueprop x


fun encode_axiom name ((th:theory), ((l,axname):(FORMULA * string))):theory =
 let (*val x = writeln("Encoding "^print_FORMULA l)*)
     val axterm  = case (strip_forall l) of
           (sort_gen_ax x) => trans_sort_gen name x
          | phi =>  lift_implication (trans_FORMULA name [] phi)
     val (anno_semterm,_) = Sign.infer_types (sign_of th) 
				(K None) (K None) [] true ([axterm],propT)
     val axname1 = del_white_space axname
     val axname2 = remove_dots axname1
     val th1 = add_axioms_i [(axname2,anno_semterm )] th
     val _ = the_axiom := get_axiom th1 axname2
  in (Utils.use_strings ["val "^make_ml_id axname2^" = !the_axiom;"];
      th1)
 end;
 
fun encode_axioms name (th:theory) (l:(FORMULA * string) list):theory =
    (writeln ("\nNumber of encoded axioms: "^string_of_int (length l));
     foldl (encode_axiom (name,th)) (th, l) 
    );




	
fun encode_basic_CFOL_HOL name thy (Sigma:sign,ax:(FORMULA * string) list):theory =
  let 
  	val an = []   (* ??? Pass global annotations here !!! *)
  	val (sortlist, subsortlist,varlist,funlist,predlist) = 
  	    env_to_list Sigma
	val thy1 = encode_sorts name thy sortlist
	val thy2 = encode_arities thy1 sortlist 
	val thy3 = encode_funs name an thy2 funlist
	val thy4 = encode_preds name an thy3 predlist
	val thy5 = encode_axioms name thy4 ax 
  in    thy5 
  end

and  encode_ax_CFOL_HOL name th ((phi,n):L_FORMULA):(term * ANNO list) =
	(trans_FORMULA (name,th) [] (strip_forall phi),n)

  

  
 
(* *********************************************************************** *)
(*									   *)
(*	Overall encoding		 				   *)	
(*									   *)
(* *********************************************************************** *)

fun newnumber(AxNumber) = (AxNumber := !AxNumber+1; !AxNumber);

fun fill3 s = 
case explode s of
    [x] => ("00"^s)
    |[x,y] => ("0"^s)
    |_ =>s;

fun new_number x = fill3(string_of_int(newnumber(AxNumber)))

fun ren_ax (lab) (ax,n) =
    (ax,(lab^n))
    
fun ren_axs (lab,nil) = nil
  | ren_axs ("",ax_list) = 
    let val nums = map new_number ax_list
    in map (ren_ax "Ax") (Utils.zip (ax_list,nums))
    end
  | ren_axs (lab,[ax]) = [(ax,lab)]
  | ren_axs (lab,ax_list) =
    let val nums = map BasisLibrary.Int.toString (1 upto length ax_list)
    in map (ren_ax lab) (Utils.zip (ax_list,nums))
    end

fun get_ax_name1 nil = ""
  | get_ax_name1 (label_anno id::_) = print_ID id
  | get_ax_name1 (pos_ANNO(_,an)::rest) =
    get_ax_name1 (an::rest)
  | get_ax_name1 (_::rest) = get_ax_name1 rest

fun get_ax_name (ax,ans) =
    (get_ax_name1 ans,ax)
   
fun rename_dups ax = 
    let val ax0 = map get_ax_name ax
        val ax1 = Symtab.dest (Symtab.make_multi ax0)
        val ax2 = map ren_axs ax1
    in flat ax2
    end

fun encode_flat_SubPCFOL_HOL name thy (ESigma:ext_signature):theory =
  let 
        val (Sigma,HSigma,ax) = ESigma
        val Sigma1 = LocalEnv.signature_union (Sigma,HSigma) 
             (* Also add hidden env, since Isabelle has no hiding *)
        val x = (AxNumber := 0);
	val (Sigma2,ax2) = encode_basic_SubPCFOL_CFOL
                           (!use_projections) (Sigma1,ax)
	val ax3 = rename_dups ax2
	val thy1 = encode_basic_CFOL_HOL name thy (Sigma2,ax3)
  in    thy1
  end;

fun encode_flat_CFOL_HOL name thy (ESigma:ext_signature):theory =
  let 
        val (Sigma,HSigma,ax) = ESigma
        val Sigma1 = LocalEnv.signature_union (Sigma,HSigma) 
             (* Also add hidden env, since Isabelle has no hiding *)
        val x = (AxNumber := 0);
	val ax1 = rename_dups ax
        (*val _ = writeln("Encoded sig:\n"^BasicPrint.print_sign Sigma1^"\n\n")*)
	val thy1 = encode_basic_CFOL_HOL name thy (Sigma1,ax1)
  in    thy1
  end;
end
