(****************************************************************************)
(*                                                                          *)
(* Project: CATS                                                            *)
(* Author: Pascal Schmidt, University of Bremen (Student)                   *)
(* Created: Su, 05 Dec 1999                                                 *)
(* Purpose of this file: LaTeX pretty printing of formulas and terms        *)
(*                                                                          *)
(****************************************************************************)

(* The pretty printing of formulas and terms is done with the help
   of Isabelle's pretty printer

   Future work: instead of working with Isabelle's pretty printer,
   use the positions in the iutput text for determining line
   breaks and formatting.
*)

(****************************************************************************)
(* Uses files: as.sml CASLEnv.sml stat_ana.sml                              *)
(****************************************************************************)
(* errors:                                                                  *)
(*  - <xy> should be <x\ y>                                                 *)
(* todo list:                                                               *)
(*  - qualified op and pred applications                                    *)
(*  - reduce more brackets with left_assoc and right_assoc annos            *)
(*  - print other annotations                                               *)
(****************************************************************************)


(* $Id: pprinter.sml,v 1.1 2004/02/13 11:29:08 7till Exp $ *)

signature PrettyLaTeX
=
sig
  type precedence = (AS.token_or_place list * AS.token_or_place list list) list

  type literals = AS.token_or_place list * AS.token_or_place list *
                  AS.token_or_place list * AS.token_or_place list *
                  AS.token_or_place list * AS.token_or_place list *
                  AS.token_or_place list * AS.token_or_place list

  val pretty_axioms : (Syntax.syntax  * precedence * literals) ->
                       GlobalEnv.global_env -> string -> AS.L_FORMULA list
                      -> string

  val pretty_local_axioms : (Syntax.syntax * precedence * literals) ->
                             GlobalEnv.global_env -> string -> AS.L_FORMULA list
                            -> string

  val pretty_op_defn : (Syntax.syntax * precedence * literals) ->
                        GlobalEnv.global_env -> string -> AS.TERM
                       -> string

  val pretty_pred_defn : (Syntax.syntax * precedence * literals) ->
                          GlobalEnv.global_env -> string -> AS.L_FORMULA
                         -> string

  val pretty_subsort_defn : (Syntax.syntax * precedence * literals) ->
                             GlobalEnv.global_env -> string -> AS.FORMULA
                            -> string

  val pretty_unit : (Syntax.syntax * precedence * literals) ->
                     GlobalEnv.global_env -> string -> AS.TERM
                    -> string

  val pretty_parse_anno : (Syntax.syntax * precedence * literals) ->
                          GlobalEnv.global_env -> string -> AS.ANNO
                          -> string
  
  val pretty_id : (Syntax.syntax * precedence * literals) ->
                   GlobalEnv.global_env -> string -> int -> AS.ID
                  -> string
                    
  val get_syn : GlobalEnv.global_env -> string
                -> (Syntax.syntax * precedence * literals)

end

structure PPL (* : PrettyLaTeX *)
=
struct

open Utils Term Mixfix IsabelleParser
infix -->
infix --->
infix $
infix mem

(****************************************************************************)
(* Type declarations                                                        *)
(****************************************************************************)

type precedence = (AS.token_or_place list * AS.token_or_place list list) list

type literals = AS.token_or_place list * AS.token_or_place list *
                AS.token_or_place list * AS.token_or_place list *
                AS.token_or_place list * AS.token_or_place list *
                AS.token_or_place list * AS.token_or_place list


(****************************************************************************)
(* Useful exceptions                                                        *)
(****************************************************************************)



(* ------------------------------------------------------------------------ *)
(* DisplayAnnotation is raised if a display annotation does not terminate   *)
(*  correctly, i.e. "%!<latex>something%" (missing !)                       *)
(* ------------------------------------------------------------------------ *)

exception PPrint_DisplayAnnotation

(* ------------------------------------------------------------------------ *)
(* SortGenAx is raised when a sort_gen_ax item is found in the spec. This   *)
(*  should not be the case for any spec passed to the LaTeX formatter.      *)
(* Unsupported is raised for all other constructs not yet supported by the  *)
(*  formatter (such as architectural specification).                        *)
(* ------------------------------------------------------------------------ *)

exception PPrint_SortGenAx
exception PPrint_Unsupported

(* ------------------------------------------------------------------------ *)
(* EndOfList is raised if a list ends where it should not. CannotBeHere is  *)
(*  raised when a function case is reached which should not be reachable.   *)
(*  Some of these cases are just there to prevent SML from outputting a     *)
(*  warning.                                                                *)
(* ------------------------------------------------------------------------ *)

exception PPrint_EndOfList
exception PPrint_CannotBeHere



(****************************************************************************)
(* String handling                                                          *)
(****************************************************************************)



(* ------------------------------------------------------------------------ *)
(* split_at split a string at the specified character into a string list    *)
(* ------------------------------------------------------------------------ *)

fun is_eq ch x = if x=ch then true else false

fun split_at ch st = String.fields (is_eq ch) st 



(* ------------------------------------------------------------------------ *)
(* chkI checks for special Isabelle pretty printer character and escapes    *)
(*  them if found                                                           *)
(*  special chars: / ( ) _ '                                                *)
(* ------------------------------------------------------------------------ *)

fun chkI [] = []
  | chkI ((#"/")::t) = (String.explode "'/") @ (chkI t)
  | chkI ((#"(")::t) = (String.explode "'(") @ (chkI t)
  | chkI ((#")")::t) = (String.explode "')") @ (chkI t)
  | chkI ((#"_")::t) = (String.explode "'_") @ (chkI t)
  | chkI ((#"'")::t) = (String.explode "''") @ (chkI t)
  | chkI (h::t)      = h :: (chkI t)



(* ------------------------------------------------------------------------ *)
(* chkL checks for special LaTeX charaters and escapes them if found        *)
(*  special chars: _ ^ { } # \n ~ (newline is deleted, not escaped)         *)
(* ------------------------------------------------------------------------ *)

fun chkL [] = []
  | chkL ((#"/")::(#"/")::t) = (#"/") :: (chkL t)
  | chkL (h::t) = if (h=(#"_")) then ((String.explode "\\_") @ (chkL t))
                  else
                  if (h=(#"^")) then ((String.explode "\\symbol{94}") @
                                                               (chkL t))
                  else
                  if (h=(#"{")) then ((String.explode "\\{") @ (chkL t))
                  else
                  if (h=(#"}")) then ((String.explode "\\}") @ (chkL t))
                  else
                  if (h=(#"#")) then ((String.explode "\\#") @ (chkL t))
                  else
                  if (h=(#"~")) then ((String.explode "\\~") @ (chkL t))
                  else
                  if (h=(#"\n")) then (chkL t)
                  else
                  (h :: (chkL t))



(* ------------------------------------------------------------------------ *)
(* interface function for chkI and chkL above                               *)
(*   check for LaTeX escaping, icheck for Isabelle escaping                 *)
(* ------------------------------------------------------------------------ *)

fun check x = (String.implode o chkL o String.explode) x 

fun icheck x = (String.implode o chkI o String.explode) x



(****************************************************************************)
(* General helper functions                                                 *)
(****************************************************************************)



(* ------------------------------------------------------------------------ *)
(* constT and freeT are used to construct Isabelle terms. They save a lot   *)
(*  of typing                                                               *)
(* ------------------------------------------------------------------------ *)

fun constT x = (Const (x,dummyT))
fun freeT  x = (Free  (x,dummyT))



(* ------------------------------------------------------------------------ *)
(* last and init are standard list functions. They are either not there in  *)
(*  SML or overwritten by some part of the Isabelle/CASL system             *)
(* ------------------------------------------------------------------------ *)

fun last []     = raise PPrint_EndOfList
  | last [h]    = h
  | last (h::t) = last t

fun init []     = []
  | init [h]    = []
  | init (h::t) = h :: (init t)



(* ------------------------------------------------------------------------ *)
(* unlink and relink are used to split a string at the newline(s) and       *)
(*  to later put it back together again                                     *)
(* ------------------------------------------------------------------------ *)

fun relink []     = ""
  | relink [h]    = h
  | relink (h::t) = h ^ "\n" ^ (relink t)
  
fun unlink s = split_at (#"\n") s



(****************************************************************************)
(* ID processing                                                            *)
(****************************************************************************)



(* ------------------------------------------------------------------------ *)
(* TTermsType is the argument type for symbol applications, i.e.            *)
(*   some_operation ( arg1, arg2, ... , argx )                              *)
(* ------------------------------------------------------------------------ *)

val TTermsType = Type ("CASL98Pure.TTerms",[])



(* ------------------------------------------------------------------------ *)
(* get_stropt converts a string option to string, None becomes the          *)
(*  empty string                                                            *)
(* ------------------------------------------------------------------------ *)

fun get_stropt (Some s) = s
  | get_stropt None     = ""



(* ------------------------------------------------------------------------ *)
(* pl_filt marks places __ in a string by replacing them with ASCII         *)
(*  character #255. This may be insecure if this is a legal input           *)
(*  character in CASL (which it is not).                                    *)
(* ------------------------------------------------------------------------ *)

fun pl_filt ("_"::("_"::t)) = "\255" :: (pl_filt t)
  | pl_filt (x::t)          = (x::(pl_filt t))
  | pl_filt []              = []



(* ------------------------------------------------------------------------ *)
(* beautify does the same thing as beautify2, only for token_or_place lists *)
(*  obtained from a display annotation. This works by first stripping the   *)
(*  spaces from front and back of tokens, then inserting the logical length *)
(*  and finally adding one space to the end                                 *)
(* ------------------------------------------------------------------------ *)

fun r_space []       = []
  | r_space (" "::t) = r_space t
  | r_space x        = x

fun rem_space "" = ""
  | rem_space t  = (implode o r_space o rev o r_space o rev o explode) t

fun add_space "" = ""
  | add_space t  = t ^ "{}"

fun log_len "" = ""
  | log_len t  = if ((Char.isAlpha o hd o String.explode) t) then
                             t
                           else
                             "(1)" ^ t

fun beautify x = (map (add_space o log_len o rem_space)) x



(* ------------------------------------------------------------------------ *)
(* beautify2 tries to set off non-math symbols in applications, and inserts *)
(*  a logical length of 1 for math symbols                                  *)
(* ------------------------------------------------------------------------ *)

fun beautify2 [] = []
  | beautify2 (""::(x::t)) = if x<>"" then
                               if ((Char.isAlpha o hd o String.explode) x) then
                                 "" :: (beautify2 (("~"^x)::t))
                               else
                                 "" :: (beautify2 (("(1)"^x)::t))
                             else
                               "" :: (beautify2 (x::t))
  | beautify2 (x::(""::t)) = if x<>"" then
                               if ((Char.isAlpha o hd o String.explode) x)
                                  orelse ((hd o String.explode) x)=(#"~") then
                                 (x^"~") :: (beautify2 (""::t))
                               else
                                 x :: (beautify2 (""::t))
                             else
                               x :: (beautify2 (""::t))
  | beautify2 (h::t) = h :: (beautify2 t)                               



(* ------------------------------------------------------------------------ *)
(* add_breaks adds Isabelle linebreak symbols to operations, so that line   *)
(*  breaking inside of mixfix applications becomes possible                 *)
(* ------------------------------------------------------------------------ *)

fun l_is_prefix l = (last(l)="") andalso (hd(l)<>"")

fun l_is_postfix l = (last(l)<>"") andalso (hd(l)="")

fun l_is_outfix l = (last(l)<>"") andalso (hd(l)<>"")

fun add_brks_a [] = []
  | add_brks_a (""::t)       = ["","/"] @ (add_brks_a t)
  | add_brks_a ("["::t)      = ["/","["] @ (add_brks_a t)
  | add_brks_a (h::t)        = h :: (add_brks_a t)

fun add_brks_b [] = []
  | add_brks_b (""::t)       = ["/",""] @ (add_brks_b t)
  | add_brks_b ("["::t)      = ["/","["] @ (add_brks_b t)
  | add_brks_b (h::t)        = h :: (add_brks_b t)

fun add_breaks [] = []
  | add_breaks l  = if (l_is_prefix l) then
                      (add_brks_a (init l)) @ [""]
                    else
                    if (l_is_postfix l) then
                      "" :: (add_brks_b (tl l))
                    else
                    if (l_is_outfix l) then
                      (take (2,l)) @ (add_brks_b (drop (2,l)))
                    else (* infix *)
                      "" :: (add_brks_b (tl l))



(* ------------------------------------------------------------------------ *)
(* cadd_breaks inserts line break symbols for compound_ids                  *)
(* ------------------------------------------------------------------------ *)

fun cadd_breaks []                 = []
  | cadd_breaks ("["::"\\'_\\'_"::t) = 
                        ["[","\\'_\\'_","/"] @ (cadd_breaks t)
  | cadd_breaks ("\\'_\\'_"::"]"::t) =
                        ["/","\\'_\\'_","]"] @ (cadd_breaks t)
  | cadd_breaks ("\\'_\\'_"::t) =
                        ["/","\\'_\\'_"] @ (cadd_breaks t)
  | cadd_breaks (h::t) = h :: (cadd_breaks t)



(* ------------------------------------------------------------------------ *)
(* mk_nice? adds some space between adjacent place symbols                  *)
(* ------------------------------------------------------------------------ *)

and mk_nice [] = []
  | mk_nice ("\255"::("\255"::t)) = ["\255","~"] @ (mk_nice ("\255" :: t))
  | mk_nice (h::t) = h :: (mk_nice t)

and mknice s = (implode o mk_nice o explode) s

and mknice2 [] = []
  | mknice2 (""::(""::t)) = ["","~"] @ (mknice2 (""::t))
  | mknice2 (h::t) = h :: (mknice2 t)

and mk_nice3 [] = []
  | mk_nice3 ("_"::("_"::("_"::("_"::t)))) = ["_","_","~"] @
                                             (mk_nice3 (["_","_"] @ t))
  | mk_nice3 ("_"::("_"::(" "::("_"::("_"::t))))) = ["_","_","~"] @
                                                    (mk_nice3 (["_","_"] @ t))
  | mk_nice3 (h::t) = h :: (mk_nice3 t)
  
and mknice3 s = (implode o mk_nice3 o explode) s



(* ------------------------------------------------------------------------ *)
(* pl_space inserts space between adjacent places. This is used on          *)
(*   token_or_place lists when converted to string by strtom                *)
(* ------------------------------------------------------------------------ *)

and pl_space []                      = []
  | pl_space (AS.place::AS.place::t) = [AS.place,AS.token "~"] @
                                       pl_space (AS.place::t)
  | pl_space (h::t)                  = h :: (pl_space t)



(* ------------------------------------------------------------------------ *)
(* do_place inserts empty strings into a list of strings, for string        *)
(*  lists generated by split_at (#"\255"), used in list_conv. This way,     *)
(*  places are converted to empty strings in listconv below.                *)
(* ------------------------------------------------------------------------ *)

and do_place []           = []
  | do_place [h]          = [h]
  | do_place (""::t)      = "" :: (do_place t)
  | do_place (h::(""::t)) = [h,""] @ (do_place t)
  | do_place (h::t)       = [h] @ [""] @ (do_place t)



(* ------------------------------------------------------------------------ *)
(* listconv takes a latex string s from a display anno, i.e. "__\\div__"    *)
(*  and converts it to a string list with empty strings, in this            *)
(*  example case ["","\\div",""], places are converted to empty strings.    *)
(*  If the LaTeX string is empty, the token_or_place list in t is used      *)
(*  instead, with checking whether it is a literal syntax special symbol    *)
(*                                                                          *)
(* This function does the main work with regard to display annotations!     *)
(* ------------------------------------------------------------------------ *)

and listconv d (s,t) = if (anno_exists s) then  
                         (add_breaks o beautify o do_place o
                          (split_at (#"\255")) o mknice 
                          o implode o pl_filt o ev_filt o explode) s
                       else
                         strtom t



(* ------------------------------------------------------------------------ *)
(* str_tom and strtom are used to convert a TOKEN_OR_MIXFIX to LaTeX,       *)
(*  places are converted to empty strings                                   *)
(* ------------------------------------------------------------------------ *)

and str_tom (AS.token (t)) = icheck (check t)
  | str_tom (AS.place)     = ""

and strtom l = (add_breaks o beautify2 o cadd_breaks) ((map str_tom) 
                                                        (pl_space l))



(* ------------------------------------------------------------------------ *)
(* comp_to_tok generates a token_or_mixfix list out of a compound_id        *)
(*  the brackets [] are inserted as additional tokens, and places inside    *)
(*  the compound are made invisible to the rest of the code by conversion   *)
(*  to regular tokens                                                       *)
(* ------------------------------------------------------------------------ *)

and comp_to_tok (tok,l) = tok @ [AS.token "["] @ (flat ((map toksub) l))
                          @ [AS.token "]"]

and toksub (AS.simple_id (tok,_,_)) = comp_filter tok
  | toksub (AS.compound_id ( (tok,_,_),l )) = comp_to_tok (comp_filter tok,l)

and comp_filter x = (map comp_filt) x

and comp_filt (AS.token (t)) = AS.token t
  | comp_filt (AS.place)     = AS.token "__"


(* ------------------------------------------------------------------------ *)
(* da_filt2 filters an ID list for display annotations. It takes the        *)
(*  token_or_place list, which is needed for LaTeXing op and pred           *)
(*  applications for mixfix. Display annotations for LaTeX are converted    *)
(*  to the string list format needed for formatting symbol application.     *)
(* ------------------------------------------------------------------------ *)

and da_filt2 d (AS.simple_id (tok,_,_),ann) = (tok,listconv d 
                                                        (get_stropt ann,tok))
  | da_filt2 d (AS.compound_id ( (tok,_,_),l ),ann) = ((comp_to_tok (tok,l)),
                                                     listconv d 
                                                           (get_stropt ann,
                                                     (comp_to_tok (tok,l))))



(* ------------------------------------------------------------------------ *)
(* ev_filt functions are used to filter out a LaTeX annotation from a       *)
(*  display annotation.                                                     *)
(* ------------------------------------------------------------------------ *)

and ev_filt3 x ("<"::t) = x
  | ev_filt3 x ("%"::t) = x
  | ev_filt3 x (h::t)   = ev_filt3 (x @ [h]) t
  | ev_filt3 x []       = raise PPrint_DisplayAnnotation

and ev_filt2 ("<"::("l"::("a"::("t"::("e"::("x"::(">"::t))))))) = 
                 (ev_filt3 [] t)
  | ev_filt2 (h::t) = ev_filt2 t
  | ev_filt2 _ = []

and ev_filt ("%"::("!"::t)) = (ev_filt2 t)
  | ev_filt _               = []



(* ------------------------------------------------------------------------ *)
(* anno_exists returns true if the string x is a non-empty LaTeX display    *)
(*  annotation                                                              *)
(* ------------------------------------------------------------------------ *)

and anno_exists x = (((implode o ev_filt o explode) x)>"")



(* ------------------------------------------------------------------------ *)
(* da_list2 returns a list of (token or mixfix list * string list) pairs    *)
(*  from the current global environment                                     *)
(* ------------------------------------------------------------------------ *)

and da_list2 genv name = (map (da_filt2 (genv,name)))
                                     (GlobalEnv.get_ids genv (name,None))
                  handle ERROR => (map (da_filt2 (genv,name))) []



(* ------------------------------------------------------------------------ *)
(* mangle_tom converts a token_or_mixfix list to an identifier suitable     *)
(*  for Isabelle syntax rules, somewhat similar to the name mangling done   *)
(*  by C++ compilers                                                        *)
(* ------------------------------------------------------------------------ *)

and mangle_t (AS.token (t)) = (check t)
  | mangle_t (AS.place)     = "__"
  
and mangle_tom t = "__PPL__" ^ (String.concat o (map mangle_t)) t

and mangle_tom2 t = " " ^ (check ((String.concat o (map mangle_t)) t))



(* ------------------------------------------------------------------------ *)
(* convert_slist converts a string list with empty-string-for-place into    *)
(*  an Isabelle pretty printing string                                      *)
(* ------------------------------------------------------------------------ *)

and convert_s "" = "_"
  | convert_s x  = x

and convert_slist x = let
                        val conv = (String.concat o (map convert_s)) x
                      in
                        if ((Char.isDigit o hd o String.explode) conv) then
                          "( " ^ conv ^ ")"  (* else screw up number consts *)
                        else
                          "(" ^ conv ^ ")"
                      end



(* ------------------------------------------------------------------------ *)
(* count_places takes a token_or_mixfix list and counts the number of       *)
(*  places                                                                  *)
(* ------------------------------------------------------------------------ *)

and count_places [] = 0
  | count_places ((AS.token (x))::t) = (count_places t)
  | count_places (AS.place::t)       = 1 + (count_places t)



(* ------------------------------------------------------------------------ *)
(* da_list return a list of triples of argument number * Isabelle           *)
(*   identifier * Isabelle pretty printer string                            *)
(* da_list2 return the same for applications without arguments, i.e.        *)
(*   constants                                                              *)
(* ------------------------------------------------------------------------ *)

and da_flt (x,y) = (count_places x,mangle_tom x,convert_slist y)

and da_flt2 (x,y) = (count_places x,mangle_tom2 x,convert_slist y)

and da_list genv name = (map da_flt) (da_list2 genv name)

and da_list_2 genv name = (map da_flt2) (da_list2 genv name)



(* ------------------------------------------------------------------------ *)
(* maketype generates a list of the form [TTermType,...,TTermType]          *)
(* ------------------------------------------------------------------------ *)

and maketype 0 = []
  | maketype x = TTermType :: (maketype (x-1))



(* ------------------------------------------------------------------------ *)
(* makefix does the same for priority pri                                   *)
(* ------------------------------------------------------------------------ *)

and makefix 0 pri = []
  | makefix x pri = pri :: (makefix (x-1) pri)



(* ------------------------------------------------------------------------ *)
(* addu adds the argument placeholder to non-mixfix symbols (short for add  *)
(*  underscore)                                                             *)
(* ------------------------------------------------------------------------ *)

and addu x = ((implode o init o explode) x) ^ "/'(_'))"



(* ------------------------------------------------------------------------ *)
(* makesyn adds the the type and mixfix information needed for Isabelle's   *)
(*  syntax rules                                                            *)
(* ------------------------------------------------------------------------ *)

and printrule (tm,pr) = print ("Rule: " ^ tm ^ " Print: " ^ pr ^ "\n")

and makesyn (nr,tm,pr) = if (nr>0) then
                           ( (*printrule (tm,pr);*)
                           (tm,(maketype nr) ---> TTermType,
                            Mixfix (pr,(makefix nr 30),30))
                           )
                         else
                           ( (*printrule (tm,pr);*)
                           (tm,[TTermsType] ---> TTermType,
                            Delimfix (addu pr))
                           )

and makesyn2 (nr,tm,pr) = if (nr>0) then
                            (* Note: this rule(s) should never be actually
                                     used!! *)
                            (tm,(maketype nr) ---> TTermType,
                             Mixfix (pr,(makefix nr 30),30))
                          else
                            (tm,TTermType,
                             Delimfix pr)

and syn_list genv name = (map makesyn) (da_list genv name)

and syn_list2 genv name = (map makesyn2) (da_list_2 genv name)



(* ------------------------------------------------------------------------ *)
(* extend_syn takes a syntax, the global environment and a spec name and    *)
(*  extends the syntax to include all IDs used in the spec                  *)
(* extend_syn_const creates the syntax extensions for constant operators    *)
(*  and predications                                                        *)
(* ------------------------------------------------------------------------ *)

and extend_syn syn genv name = Syntax.extend_const_gram syn
                               ("latex",true) (syn_list genv name)
                               
and extend_syn_const syn genv name = Syntax.extend_const_gram syn
                                 ("latex",true) (syn_list2 genv name)



(* ------------------------------------------------------------------------ *)
(* extend_syn_lit adds rules for list literals to the syntax                *)
(* ------------------------------------------------------------------------ *)

and extend_syn_lit syn genv name = 
                          if (has_list_lit genv name) then
                            Syntax.extend_const_gram syn ("latex",true)
                            (lit_rules genv name)
                          else
                            syn

and has_list_lit genv name = ll_pred (GlobalEnv.get_annos genv (name,None))
                             handle _ => ll_pred []

and ll_pred []                          = false
  | ll_pred ((AS.list_anno (a,b,c))::t) = true
  | ll_pred (h::t)                      = ll_pred t
                                   
and lit_rules genv name = [lit_op_decl genv name,lit_op_use genv name,
                           lit_op_nil genv name]

and lit_op_decl genv name = (mangle_tom (get_lit_tom genv name),
                             [TTermType] ---> TTermType,
                             Mixfix ((get_lit_str genv name),[30],30))

and lit_op_use genv name =  ("LatexLLBra",
                             [TTermType] ---> TTermType,
                             Mixfix ((get_lit_str genv name),[30],30))
                             
and lit_op_nil genv name =  ("LatexLLNil",TTermType,
                              Delimfix (get_lit_nil genv name))

and get_lit_tom genv name = (id_to_tom o idunwrap o ll_filter)
                             (GlobalEnv.get_annos genv (name,None))
                             handle _ => []

and ll_filter []                          = (AS.simple_id ([],"",None),None)
  | ll_filter ((AS.list_anno (a,b,c))::t) = a
  | ll_filter (h::t)                      = ll_filter t

and get_lit_str genv name = (convert_slist o strtom) (get_lit_tom genv name)

and get_lit_nil genv name = let
                              val tom = get_lit_tom genv name
                            in
                              "(" ^ (get_1st tom) ^ (get_3rd tom) ^ ")"
                            end

and get_1st [AS.token (t),_,_] = icheck (check t)
  | get_1st _                  = raise PPrint_CannotBeHere
  
and get_3rd [_,_,AS.token (t)] = icheck (check t)
  | get_3rd _                  = raise PPrint_CannotBeHere



(* ------------------------------------------------------------------------ *)
(* append_to_syn adds rules for predications and operations to the syntax,  *)
(*  both with and without arguments (constants)                             *)
(* ------------------------------------------------------------------------ *)

and append_to_syn syn genv name = extend_syn_const
                                  (extend_syn (extend_syn_lit syn genv name) 
                                   genv name) genv name



(* ------------------------------------------------------------------------ *)
(* get_literals returns the token_or_place lists for defined literals       *)
(* ------------------------------------------------------------------------ *)

and get_literals genv name = let
                               val anlist = GlobalEnv.get_annos genv (name,None)
                                            handle _ => []
                             in
                               (get_number_lit anlist,
                                get_fl_dot_lit anlist,
                                get_fl_exp_lit anlist,
                                get_st_nil_lit anlist,
                                get_st_con_lit anlist,
                                get_li_bra_lit anlist,
                                get_li_emp_lit anlist,
                                get_li_con_lit anlist)
                             end

and get_x_lit l f g = let
                        val alist = filter f l
                      in
                        if (alist=[]) then
                          []
                        else
                          (id_to_tom o idunwrap o g o hd) alist
                      end

and get_number_lit l = get_x_lit l is_number_lit number_id

and is_number_lit (AS.number_anno (i)) = true
  | is_number_lit _                    = false

and number_id (AS.number_anno (i)) = i
  | number_id _                    = raise PPrint_CannotBeHere

and get_fl_dot_lit l = get_x_lit l is_float_lit fl_dot_id

and is_float_lit (AS.floating_anno (i,j)) = true
  | is_float_lit _                        = false

and fl_dot_id (AS.floating_anno (i,j)) = i
  | fl_dot_id _                        = raise PPrint_CannotBeHere

and get_fl_exp_lit l = get_x_lit l is_float_lit fl_exp_id

and fl_exp_id (AS.floating_anno (i,j)) = j
  | fl_exp_id _                        = raise PPrint_CannotBeHere

and get_st_nil_lit l = get_x_lit l is_string_lit st_nil_id

and is_string_lit (AS.string_anno (i,j)) = true
  | is_string_lit _                      = false

and st_nil_id (AS.string_anno (i,j)) = i
  | st_nil_id _                      = raise PPrint_CannotBeHere

and get_st_con_lit l = get_x_lit l is_string_lit st_con_id

and st_con_id (AS.string_anno (i,j)) = j
  | st_con_id _                      = raise PPrint_CannotBeHere

and get_li_bra_lit l = get_x_lit l is_list_lit li_bra_id

and is_list_lit (AS.list_anno (i,j,k)) = true
  | is_list_lit _                      = false
  
and li_bra_id (AS.list_anno (i,j,k)) = i
  | li_bra_id _                      = raise PPrint_CannotBeHere  

and get_li_emp_lit l = get_x_lit l is_list_lit li_emp_id

and li_emp_id (AS.list_anno (i,j,k)) = j
  | li_emp_id _                      = raise PPrint_CannotBeHere  

and get_li_con_lit l = get_x_lit l is_list_lit li_con_id

and li_con_id (AS.list_anno (i,j,k)) = k
  | li_con_id _                      = raise PPrint_CannotBeHere  



(* ------------------------------------------------------------------------ *)
(* get_symb_ord returns the precendence order for symbols                   *)
(* ------------------------------------------------------------------------ *)

and c_po (i,l) = ((id_to_tom o idunwrap) i,(map (id_to_tom o idunwrap)) l)

and conv_po x = (map c_po) x

and get_symb_ord genv name = let
                               val idlist = GlobalEnv.get_ids genv (name,None)
                                            handle _ => []
                               val anlist = GlobalEnv.get_annos genv (name,None)
                                            handle _ => []
                             in
                               conv_po (IsabelleMixfixParser.get_prec_ord
                                        idlist anlist)
                             end



(****************************************************************************)
(* formula handling                                                         *)
(****************************************************************************)



(* ------------------------------------------------------------------------ *)
(* The following functions convert formulas from as.sml to Isabelle terms.  *)
(*  Everything should be obvious when taking a look at CASL98Pure.thy and   *)
(*  CASL98.thy                                                              *)
(* ------------------------------------------------------------------------ *)

and p_formula d (AS.atom (x)) = ((constT "LatexAtom") $ (p_atom d x))
  | p_formula d (AS.quantification (q,l,f)) = (p_quan d (q,l,f))
  | p_formula d (AS.conjunction (l)) = (p_conj d l)
  | p_formula d (AS.disjunction (l)) = (p_disj d l)
  | p_formula d (AS.implication (a,b)) = (p_impl d (a,b))
  | p_formula d (AS.equivalence (a,b)) = (p_equi d (a,b))
  | p_formula d (AS.negation (x)) = (p_nega d x)
  | p_formula d (AS.sort_gen_ax (_,_)) = ( print "Error: sort_gen_ax found.";
                                      raise PPrint_SortGenAx )
  | p_formula d (AS.unparsed_formula (x)) = (p_unpf d x)



and p_unpf d x = ((constT "LatexUnpF") $
               ((constT "simple_id") $ (freeT ("\\verb'" ^ x ^ "'"))))

and p_quan d ((AS.forall),l,f) = (((constT "LatexForall") $ (p_vdlist l)) $
                            (p_formula d f))
  | p_quan d ((AS.exists),l,f) = (((constT "LatexExists") $ (p_vdlist l)) $
                            (p_formula d f))
  | p_quan d ((AS.exists_uniquely),l,f) = (((constT "LatexExistU") $
                                          (p_vdlist l)) $ (p_formula d f))

and p_vdlist l = ((constT "simple_id") $ (freeT (L_VAR_DECLS l)))



(* ------------------------------------------------------------------------ *)
(* The following functions for formatting a var_decl list are taken from    *)
(*  mklatex.sml and don't respect display annotation for sort names.        *)
(* ------------------------------------------------------------------------ *)

and L_VAR_DECLS [] = ""
  | L_VAR_DECLS [h] = (L_VAR_DECL h)
  | L_VAR_DECLS (h::t) = (L_VAR_DECL h) ^ "; " ^ (L_VAR_DECLS t)

and L_VAR_DECL (l,s,a) = (L_V_LIST l) ^ " : " ^ (L_SORT s)

and L_V_LIST [] = ""
  | L_V_LIST [h] = (L_VAR h)
  | L_V_LIST (h::t) = (L_VAR h) ^ ", " ^ (L_V_LIST t)

and L_VAR ((x):AS.VAR) = (L_SIMPLE_ID x)

and L_SORT ((x):AS.SORT) = (L_ID x)

and TP_STRING []     = ""
  | TP_STRING (h::t) = (TP_WORD h) ^ (TP_STRING t)

and TP_WORD (AS.place)     = "\\_\\_"
  | TP_WORD (AS.token (s)) = (check s)

and L_SIMPLE_ID ((x,a):AS.SIMPLE_ID) = (check x)

and L_TOM ((l,s,n):AS.TOKEN_OR_MIXFIX) = (TP_STRING l)

and L_ID1 (AS.simple_id (x)) = (L_TOM x)   
  | L_ID1 (AS.compound_id (x,l)) = (L_TOM x) ^ "[" ^ (L_ID1_LIST l) ^ "]"

and L_ID1_LIST [] = ""
  | L_ID1_LIST (h::t) = (L_ID1 h) ^ (L_ID1_LIST t)

and L_ID ((x,a):AS.ID) = (L_ID1 x)



and conj_brack (AS.disjunction (_))              = true
  | conj_brack (AS.quantification (_,_,_))       = true
  | conj_brack (AS.atom (_))                     = false
  | conj_brack (AS.implication (_,_))            = true
  | conj_brack (AS.equivalence (_,_))            = true
  | conj_brack _                              = false

and conj_ebrack (AS.disjunction (_))              = true
  | conj_ebrack (AS.quantification (_,_,_))       = false
  | conj_ebrack (AS.atom (_))                     = false
  | conj_ebrack (AS.implication (_,_))            = true
  | conj_ebrack (AS.equivalence (_,_))            = true
  | conj_ebrack _                              = false

and p_conj d [] = raise PPrint_EndOfList
  | p_conj d [h] = if (conj_ebrack h) then
                      ((constT "parFormula") $ (p_formula d h))
                    else
                      (p_formula d h)
  | p_conj d (h::t) = if (conj_brack h) then
                      (((constT "LatexConj") $
                      ((constT "parFormula") $ (p_formula d h))) $
		      (p_conj d t))
                    else
                      (((constT "LatexConj") $
                      (p_formula d h)) $ (p_conj d t))

and disj_brack (AS.conjunction (_))              = true
  | disj_brack (AS.quantification (_,_,_))       = true
  | disj_brack (AS.atom (_))                     = false
  | disj_brack (AS.implication (_,_))            = true
  | disj_brack (AS.equivalence (_,_))            = true
  | disj_brack _                              = false

and disj_ebrack (AS.conjunction (_))              = true
  | disj_ebrack (AS.quantification (_,_,_))       = false
  | disj_ebrack (AS.atom (_))                     = false
  | disj_ebrack (AS.implication (_,_))            = true
  | disj_ebrack (AS.equivalence (_,_))            = true
  | disj_ebrack _                              = false

and p_disj d []     = raise PPrint_EndOfList
  | p_disj d [h]    = if (disj_ebrack h) then
                      ((constT "parFormula") $ (p_formula d h))
                    else
                      (p_formula d h)
  | p_disj d (h::t) = if (disj_brack h) then
                      (((constT "LatexDisj") $
                      ((constT "parFormula") $ (p_formula d h))) $
		      (p_disj d t))
                    else
                      (((constT "LatexDisj") $
                      (p_formula d h)) $ (p_disj d t))

and impl_rbrack (AS.atom (_)) = false
  | impl_rbrack _          = false

and impl_lbrack (AS.atom (_)) = false
  | impl_lbrack (AS.implication (_,_)) = true
  | impl_lbrack _          = false

and p_impl d (a,b) = if (impl_lbrack a) then
                     if (impl_rbrack b) then
                       (((constT "LatexImpl")  $
                       ((constT "parFormula") $ (p_formula d a))) $
                       ((constT "parFormula") $ (p_formula d b)))
                     else
                       (((constT "LatexImpl")  $
                       ((constT "parFormula") $ (p_formula d a))) $
                       (p_formula d b))
                   else
                     if (impl_rbrack b) then
                       (((constT "LatexImpl") $
                       (p_formula d a)) $ ((constT "parFormula") $
		       (p_formula d b)))
                     else
                       (((constT "LatexImpl") $
                       (p_formula d a)) $ (p_formula d b))

and equi_brack (AS.equivalence (_,_)) = true
  | equi_brack (AS.implication (_,_)) = true
  | equi_brack (AS.atom (_))          = false
  | equi_brack _                   = false

and p_equi d (a,b) = if (equi_brack a) then
                     if (equi_brack b) then
                       (((constT "LatexEqui")  $
                       ((constT "parFormula") $ (p_formula d a))) $
                       ((constT "parFormula") $ (p_formula d b)))
                     else
                       (((constT "LatexEqui")  $
                       ((constT "parFormula") $ (p_formula d a))) $
                       (p_formula d b))
                   else
                     if (equi_brack b) then
                       (((constT "LatexEqui") $
                       (p_formula d a)) $ ((constT "parFormula") $
		       (p_formula d b)))
                     else
                       (((constT "LatexEqui") $
                       (p_formula d a)) $ (p_formula d b))

and nega_brack (AS.negation (_)) = true
  | nega_brack _                 = false

and p_nega d (x) = if (nega_brack x) then
                   ((constT "LatexNega") $
                   ((constT "parFormula") $ (p_formula d x)))
                 else
                   ((constT "LatexNega") $ (p_formula d x))



(* ------------------------------------------------------------------------ *)
(* atom handling                                                            *)
(* ------------------------------------------------------------------------ *)

and p_atom d (AS.ttrue) = (constT "LatexTrue")
  | p_atom d (AS.ffalse) = (constT "LatexFalse")
  | p_atom d (AS.predication (s,(AS.terms t))) = (p_pred_appl d (s,t))
  | p_atom d (AS.definedness (t)) = ((constT "LatexDefi") $ (p_term2 d t))
  | p_atom d (AS.existl_equation (a,b)) = (((constT "LatexEEqu") $
                                     (p_term2 d a)) $ (p_term2 d b))
  | p_atom d (AS.strong_equation (a,b)) = (((constT "LatexEqua") $
                                     (p_term2 d a)) $ (p_term2 d b))
  | p_atom d (AS.membership (t,s)) = (((constT "LatexMemb") $ (p_term2 d t)) $
                                   (p_sort_id s))



(* ------------------------------------------------------------------------ *)
(* p_pred_appl does the predicate application. First, if we have a constant *)
(*  we have no args and need to use mangle_tom2. Else, if we have a mixfix  *)
(*  operation, we need to use "recursive application" to form the Isabelle  *)
(*  term. If the symbol is non-mixfix, we can simply apply it to a TTerms   *)
(*  construct.                                                              *)
(* ------------------------------------------------------------------------ *)

and p_pred_appl d (s,[]) = (pred_const d s)
  | p_pred_appl d (s,l)  = if (pred_is_mixfix s) then
   (p_some_terms d (pred_symb_tom s) (constT (mangle_tom (pred_symb_tom s))) 
        ((map (rewrite_term d)) l))
                         else
                           ((constT (mangle_tom (pred_symb_tom s))) $
                           (p_tterms d ((map (rewrite_term d)) l) ))



(* ------------------------------------------------------------------------ *)
(* p_some_term takes the symbol x and applies a list of arguments to it     *)
(*  this is special because (f x y) is ((f x) y) in Isabelle                *)
(* ------------------------------------------------------------------------ *)

and p_some_terms d outer x []  = raise PPrint_EndOfList
  | p_some_terms d outer x [h] = (x $ (p_bterm d outer h))
  | p_some_terms d outer x y   = ((p_some_terms d outer x (init y))) $ (p_bterm
                                 d outer (last y))



and pred_is_mixfix s = AS.place mem (pred_symb_tom s)

and pred_const d s = constT (mangle_tom2 (pred_symb_tom s))

and pred_symb_tom (AS.pred_symb (name,_)) = (id_to_tom (idunwrap name))



(* ------------------------------------------------------------------------ *)
(* use (id_to_tom o idunwrap) to convert IDs from the AS tree to the        *)
(*  token_or_mixfix list format we use in this file                         *)
(* ------------------------------------------------------------------------ *)

and idunwrap ((id,an):AS.ID) = id

and id_to_tom (AS.simple_id (tom,_,_)) = tom
  | id_to_tom (AS.compound_id ( (tom,_,_),l )) = comp_to_tok (tom,l)
  


and p_sort_id (s) = ((constT "simple_id") $ (freeT (L_SORT s)))

and p_simple_id (s,lnr) = ((constT "simple_id") $ (freeT (check s)))



and p_simp (s,lnr) = ((constT "simple_id") $ (freeT s))

and p_label l = p_simp ((concat o (map L_ID)) l,None)

and p_l_formula d (f,([],e)) = (p_formula d f)
  | p_l_formula d (f,(l,e))  = (((constT "LatexLabel") $ (p_label l)) $
                               (p_formula d f))



(****************************************************************************)
(* term handling                                                            *)
(****************************************************************************)



(* ------------------------------------------------------------------------ *)
(* The following function translate AS terms to Isabelle terms. Take a look *)
(*  at CASL98Pure.thy and CASL98.thy for the corresponding syntax rules     *)
(* ------------------------------------------------------------------------ *)

(* ------------------------------------------------------------------------ *)
(* precedence code                                                          *)
(* ------------------------------------------------------------------------ *)

and is_token (AS.token (x)) = true
  | is_token (AS.place)     = false
  
and is_place (AS.token (x)) = false
  | is_place (AS.place)     = true

and is_outfix d l  = (AS.place mem l)  andalso
                     (is_token (hd l)) andalso (is_token (last l))

and is_postfix d l = (is_place (hd l)) andalso (is_token (last l))

and is_prefix  d l = (is_token (hd l)) andalso (is_place (last l))



(* ------------------------------------------------------------------------ *)
(* standard precedence values taken from CASL Summary, Appendix C           *)
(* ------------------------------------------------------------------------ *)

and get_pri d l = if (AS.place mem l) then
                  if (is_postfix d l) then
                    3
                  else
                    if (is_prefix d l) then
                      2
                    else
                      1
                else
                  4



(* ------------------------------------------------------------------------ *)
(* op_needs_paren returns true if we need brackets around "inner" symbol    *)
(*  application                                                             *)
(* ------------------------------------------------------------------------ *)

and op_needs_paren d outer inner = 
    if (is_outfix d outer) orelse (is_outfix d inner) then
      false
    else
      (get_pri d outer)>=(get_pri d inner)



(* ------------------------------------------------------------------------ *)
(* needs_paren does basically the same, but also respects precedence rules  *)
(*  set by annotations                                                      *)
(* ------------------------------------------------------------------------ *)

and needs_paren d outer (AS.application (s,AS.terms [])) = false
  | needs_paren d outer (AS.application (s,_)) =
                   if (check_is_lit d s (op_symb_tom s)) then
                     false
                   else
                     if (in_pri d outer (op_symb_tom s)) then
                       calc_pri d outer (op_symb_tom s)
                     else
                       op_needs_paren d outer (op_symb_tom s)
  | needs_paren d outer _                      = false



(* ------------------------------------------------------------------------ *)
(* the following four functions make lookups in the precendence order       *)
(* ------------------------------------------------------------------------ *)

and in_prio [] outer inner         = false
  | in_prio ((i,l)::t) outer inner = if ((i=outer)andalso(inner mem l)) orelse
                                        ((i=inner)andalso(outer mem l))
                                     then
                                       true
                                     else
                                        in_prio t outer inner  

and in_pri (genv,name,po,li) outer inner = in_prio po outer inner

and calc_prio [] outer inner = raise PPrint_EndOfList
  | calc_prio ((i,l)::t) outer inner = if ((i=outer)andalso(inner mem l)) then
                                         true
                                       else
                                       if ((i=inner)andalso(outer mem l)) then
                                         false
                                       else
                                         calc_prio t outer inner

and calc_pri (genv,name,po,li) outer inner = calc_prio po outer inner



(* ------------------------------------------------------------------------ *)
(* literal code                                                             *)
(* ------------------------------------------------------------------------ *)


(* ------------------------------------------------------------------------ *)
(* check_lit is used to apply rules for literal constucts that take args,   *)
(*  check_lit2 is used for those without args (constants)                   *)
(* NOTE: the term rewriting functions insert some pseudo-applications into  *)
(*   the syntax tree to make formatting string and list literals possible   *)
(* ------------------------------------------------------------------------ *)

and check_is_lit (genv,name,po,li) sym l = check_is_li li sym l

and check_lit (genv,name,po,li) sym l = check_li li sym l

and check_lit2 (genv,name,po,li) sym l = check_li2 li sym l

and check_is_li li sym l = let
                             val (num,flo1,flo2,str1,str2,li1,li2,li3) = li
                             val llist = [num,flo1,flo2]
                             val inlist = (l<>[]) andalso (l mem llist)
                             val is1 = is_special_op_symb sym "StrBegin"
                             val is2 = is_special_op_symb sym "StrCons"
                             val is3 = is_special_op_symb sym "StrEmpty"
                             val is4 = is_special_op_symb sym "StrEnd"
                             val is5 = is_special_op_symb sym "LiBegin"
                             val is6 = is_special_op_symb sym "LiCons"
                             val is7 = is_special_op_symb sym "LiEmpty"
                             val is8 = is_special_op_symb sym "LiEnd"
                           in
                             inlist orelse is1 orelse is2 orelse is3 orelse
                             is4 orelse is5 orelse is6 orelse is7 orelse is8
                           end

and check_li li sym l = let
                      val (num,flo1,flo2,str1,str2,li1,li2,li3) = li
                      val llist = [num,flo1,flo2]
                    in
                      if ((l mem llist) andalso (l<>[])) then
                        if l=num  then (constT "LatexLNum")  else
                        if l=flo1 then (constT "LatexLDot")  else
                        if l=flo2 then (constT "LatexLE")    else
                        raise PPrint_CannotBeHere
                      else
                         if (is_special_op_symb sym "StrBegin") then
                           (constT "LatexLStr") else
                         if (is_special_op_symb sym "StrEnd") then
                           (constT "LatexLSEnd") else
                         if (is_special_op_symb sym "StrCons") then
                           (constT "LatexLNum") else
                         if (is_special_op_symb sym "LiBegin") then
                           (constT "LatexLLBra") else
                         if (is_special_op_symb sym "LiEnd") then
                           (constT "LatexLSEnd") else
                         if (is_special_op_symb sym "LiCons") then
                           (constT "LatexLLCons")
                        else
                          (constT (mangle_tom l))
                    end

and check_li2 li sym l = let
                       val (num,flo1,flo2,str1,str2,li1,li2,li3) = li
                     in
                       if (is_special_op_symb sym "StrEmpty") then
                         (constT "LatexLEStr") else
                       if (is_special_op_symb sym "LiEmpty") then
                         (constT "LatexLLNil")
                       else
                         (constT (mangle_tom2 l))
                     end



(* ------------------------------------------------------------------------ *)
(* p_bterm is used for recursive symbol applications, where there may be a  *)
(*  need to put brackets around some applications                           *)
(* ------------------------------------------------------------------------ *)

and p_bterm d outer x = if (needs_paren d outer x) then
                          ((constT "parTerm") $ (p_term d x))
                        else
                          (p_term d x)



(* ------------------------------------------------------------------------ *)
(* p_term2 converts an AS term to an Isabelle term, taking literal syntax   *)
(*  into account                                                            *)
(* ------------------------------------------------------------------------ *)

and p_term2 d x = p_term d (rewrite_term d x)



(* ------------------------------------------------------------------------ *)
(* p_term converts an AS term to an Isabelle term, but MUST NOT be called   *)
(*  directly! Use p_term2 - otherwise the tree rewrite operation will be    *)
(*  missing, resulting in garbage if literal syntax operations are present  *)
(* ------------------------------------------------------------------------ *)

and p_term d (AS.var_or_const (x)) = ((constT "LatexVarConst") $
                                     (p_simple_id x))
  | p_term d (AS.qual_var (x,s)) = (((constT "LatexQualVar") $ (p_simple_id x))
                                   $ (p_sort_id s))
  | p_term d (AS.application (s,(AS.terms t))) = (p_op_appl d (s,t))
  | p_term d (AS.sorted_term (t,s)) = (((constT "LatexSorted") $ (p_term d t))
                                      $ (p_sort_id s))
  | p_term d (AS.cast (t,s)) = (((constT "LatexCast") $ (p_term d t)) $
                             (p_sort_id s))
  | p_term d (AS.conditional (t1,f,t2)) = ((((constT "LatexWhen") $ (p_term d
                                       t1)) $ (p_formula d f)) $ (p_term d t2))
  | p_term d (AS.unparsed_term (s)) = (p_unpt d s)



and p_unpt d x = ((constT "LatexUnpT") $
               ((constT "simple_id") $ (freeT ("\\verb'" ^ x ^ "'"))))



(* ------------------------------------------------------------------------ *)
(* operation application is much similar to predicate application (above),  *)
(*  but this time we also take into account literal syntax                  *)
(* ------------------------------------------------------------------------ *)

and p_op_appl d (s,[]) = (op_const d s)
  | p_op_appl d (s,l)  = if ((op_is_mixfix s) orelse
                             (check_is_lit d s (op_symb_tom s))) then
    (p_some_terms d (op_symb_tom s) (check_lit d s (op_symb_tom s)) l)
                         else
                           ((constT (mangle_tom (op_symb_tom s))) $
                           (p_tterms d l))

and op_is_mixfix s = AS.place mem (op_symb_tom s)

and op_const d s = (check_lit2 d s (op_symb_tom s))

and op_symb_tom (AS.op_symb (name,_)) = (id_to_tom (idunwrap name))



(* ------------------------------------------------------------------------ *)
(* p_tterms constructs a TTerms object for use in normal symbol application *)
(*  (non-mixfix)                                                            *)
(* ------------------------------------------------------------------------ *)

and p_tterms d [] = raise PPrint_EndOfList
  | p_tterms d [h] = ((constT "_idT") $ (p_term d h))
  | p_tterms d (h::t) = (((constT "_consT") $ (p_term d h)) $ (p_tterms d t))



(****************************************************************************)
(* annotation handling                                                      *)
(****************************************************************************)

(* ------------------------------------------------------------------------ *)
(* only the precedence anno is handled here, the others are in mklatex.sml  *)
(* ------------------------------------------------------------------------ *)

and p_parse_anno (AS.prec_anno (b,i,j)) = if b then
                                              (p_prec_anno i j)
                                            else
                                              (p_noprec_anno i j)
  | p_parse_anno _ = raise PPrint_CannotBeHere

and p_prec_anno l1 l2 = (((constT "LatexPrec") $ (p_id_list l1)) $
                          (p_id_list l2))

and p_noprec_anno l1 l2 = (((constT "LatexNPrec") $ (p_id_list l1)) $
                            (p_id_list l2))

and (* p_id_list [h] = ((constT "LatexSingle") $ (p_idterm h))
  |*) p_id_list l   = ((constT "LatexList") $ (p_list_ids l))

and p_list_ids [] = raise PPrint_EndOfList
  | p_list_ids [h] = ((constT "_idT") $ (p_idterm h))
  | p_list_ids (h::t) = (((constT "_consT") $ (p_idterm h)) $
                          (p_list_ids t))

and p_idterm t = if (AS.place mem ((id_to_tom o idunwrap) t)) then
                     p_idplace ((id_to_tom o idunwrap) t)
                   else
                     p_idconst t

and p_idconst t = (constT (mangle_tom2 ((id_to_tom o idunwrap) t)))

and p_idplace l = p_place_appl ((constT o mangle_tom) l) (count_places l)



(* ------------------------------------------------------------------------ *)
(* p_place_appl applicates a mixfix symbol to a list of places, this causes *)
(*  output similar to the symbol declaration                                *)
(* ------------------------------------------------------------------------ *)

and p_place_appl symbl 0 = raise PPrint_EndOfList
  | p_place_appl symbl 1 = (symbl $ (p_genplace ()))
  | p_place_appl symbl n = ((p_place_appl symbl (n-1)) $ (p_genplace ()))

and p_genplace () = (constT "LatexPlace")



(****************************************************************************)
(* term rewriting                                                           *)
(****************************************************************************)

(* ------------------------------------------------------------------------ *)
(* The following functions provide term rewriting for output of the string  *)
(*  and list literals                                                       *)
(* ------------------------------------------------------------------------ *)

and rewrite_term d x = rewrite_list d (rewrite_string d x)



(* ------------------------------------------------------------------------ *)
(* Handling of character literals in strings                                *)
(* ------------------------------------------------------------------------ *)

and is_char_lit [AS.token (t)] = (((hd o explode) t)="'") andalso
                                 (((hd o rev o explode) t)="'")
  | is_char_lit _              = false

and handle_char (AS.application (AS.op_symb (n,t),AS.terms [])) =
                   if is_char_lit( (id_to_tom o idunwrap) n) then
                     AS.application (AS.op_symb
                     (rewrite_char ((id_to_tom o idunwrap) n),t),
                      AS.terms [])
                   else
                     AS.application (AS.op_symb (n,t),AS.terms [])
  | handle_char x = x

and rewrite_char [AS.token (t)] = (AS.simple_id ([AS.token ((icheck o check o
                                   implode o tl o rev o tl o
                                   explode) t)],"",None),None)
  | rewrite_char _ = raise PPrint_CannotBeHere



(* ------------------------------------------------------------------------ *)
(* Isolate the string symbols                                               *)
(* ------------------------------------------------------------------------ *) 

and get_string_tok x = let
                         val (num,flo1,flo2,str1,str2,li1,li2,li3) = x
                       in
                         (str1,str2)
                       end

and rewrite_string (genv,name,po,li) x = (rew_term
                                          ("StrBegin","StrCons","StrEnd",
                                           "StrEmpty",handle_char) 
                                          false (get_string_tok li) x)



(* ------------------------------------------------------------------------ *)
(* Rewriting is mainly concerned with applications, but we also need to     *)
(*  recursively check all other terms, as they may also contain appls.      *)
(*                                                                          *)
(* x in the following code is a bool flag: if it is true, we operate inside *)
(*  of a literal construct                                                  *)
(* y here is a tuple of the form                                            *)
(*  (begin_label,cons_label,end_label,nil_label,handle_fun)                 *)
(* ------------------------------------------------------------------------ *)

and rew_term y x ids (AS.application (i,j))   = rew_appl y x ids i j
  | rew_term y x ids (AS.sorted_term (i,j))   = AS.sorted_term
                                                (rew_term y false ids i,j)
  | rew_term y x ids (AS.cast (i,j))          = AS.cast
                                                (rew_term y false ids i,j)
  | rew_term y x ids (AS.conditional (i,j,k)) = AS.conditional
                                                (rew_term y false ids i,j,
                                                 rew_term y false ids k)
  | rew_term y x ids whatever                 = whatever



(* ------------------------------------------------------------------------ *)
(* str_to_op_symb takes a string and constructs a pseudo-application from   *)
(*  it                                                                      *)
(* is_special_op_symb checks whether an op_symb is such a pseudo-construct  *)
(* ------------------------------------------------------------------------ *)

and str_to_op_symb x = AS.op_symb ((AS.simple_id ([],x,None),None),
                                   None)

and is_special_op_symb (AS.op_symb ((AS.simple_id ([],x,None),None),None)) y =
                         x=y
  | is_special_op_symb _ _ = false


(* ------------------------------------------------------------------------ *)
(* replace "string" in the following comment(s) by "string/list"            *)
(* ------------------------------------------------------------------------ *)
(* rew_appl does the main work for string literals. If it finds an id2      *)
(*  application (string concat), it does the follwing:                      *)
(*   - outside a string construct: inserts a "StrBegin" pseudo-application, *)
(*                                calls rew_1st_str                         *)
(*   - inside a string construct: calls rew_nth_str                         *)
(*  If it finds an id1 application (empty string), it calls:                *)
(*   - outside a string construct: replaces it with a "StrEmpty" pseudo-    *)
(*                                 application                              *)
(*   - inside a string construct: error, this should be handled in nth_str  *)
(*  Other applications are returned unchanged, but their arguments are      *)
(*   first checked for string constructs                                    *)
(* ------------------------------------------------------------------------ *)

and rew_appl (beg,con,en,ni,fu) x (id1,id2) (AS.op_symb (n,t))(AS.terms [a,b])=
         if ((id_to_tom o idunwrap) n)=id2 then
           if x then
             rew_nth_str (beg,con,en,ni,fu) (AS.op_symb (n,t)) (id1,id2) a b
           else
             if (ends_in_nil (id1,id2) b) then
               AS.application (str_to_op_symb beg,
                               rew_1st_str (beg,con,en,ni,fu)
                                           (AS.op_symb (n,t)) (id1,id2) a b)
             else
               AS.application (AS.op_symb (n,t),AS.terms [rew_term
                                                          (beg,con,en,ni,fu)
                                                          false (id1,id2) a,
                                                          rew_term
                                                          (beg,con,en,ni,fu)
                                                          false (id1,id2) b])
         else
           AS.application (AS.op_symb (n,t),AS.terms [rew_term
                                                      (beg,con,en,ni,fu)
                                                      false (id1,id2) a,
                                                      rew_term 
                                                      (beg,con,en,ni,fu)
                                                      false (id1,id2) b])
  | rew_appl (beg,con,en,ni,fu) x (id1,id2) (AS.op_symb (n,t)) (AS.terms []) =
         if ((id_to_tom o idunwrap) n)=id1 then
           if x then
             raise PPrint_CannotBeHere
           else
             AS.application (str_to_op_symb ni,AS.terms [])
         else
           AS.application (AS.op_symb (n,t),AS.terms [])
  | rew_appl (beg,con,en,ni,fu) x (id1,id2) sym (AS.terms tree) =
           AS.application (sym,AS.terms ((map (rew_term (beg,con,en,ni,fu) 
                                          false (id1,id2))) tree))


and ends_in_nil (id1,id2) (AS.application (AS.op_symb (n,t),AS.terms [a,b]))
        = if ((id_to_tom o idunwrap) n)=id2 then
            ends_in_nil (id1,id2) b
          else
            false
  | ends_in_nil (id1,id2) (AS.application (AS.op_symb (n,t),AS.terms []))
        = (((id_to_tom o idunwrap) n)=id1)
  | ends_in_nil (id1,id2) _ = false



(* ------------------------------------------------------------------------ *)
(* rew_1st_str is called when a string is begun. If the second argument to  *)
(*  the string constructor is the empty string, this means we only have     *)
(*  to format the first argument, discarding the empty string symbol. If    *)
(*  it was not the empty string, then we need to format both parts, which   *)
(*  means we have to keep the string constructor symbol in the tree.        *)
(*                                                                          *)
(* rew_str is called recursively with the bool flag set to true, as we are  *)
(*  now INSIDE of a string                                                  *)
(* ------------------------------------------------------------------------ *)

and rew_1st_str x sym (id1,id2) a b = let
                                        val (_,_,_,_,f) = x 
                                      in
                                        if (is_empty id1 b) then
                                          AS.terms [rew_term x true (id1,id2) 
                                          (f a)]
                                        else
                                          AS.terms [rew_term x true (id1,id2)
                                          (AS.application (sym,AS.terms [a,b]
                                           ))]
                                      end



(* ------------------------------------------------------------------------ *)
(* rew_nth_str is called in the middle of a string. If the second argument  *)
(*  to the string constructor is the empty string, then the constructor     *)
(*  need to be replaced with a "StrEnd" pseudo-application, as we do not    *)
(*  want the empty string symbol "" to appear at the end of a string. If    *)
(*  this was not the end of the string, the constructor is kept.            *)
(*                                                                          *)
(* calls to rew_str from here are made with the bool flag set to true, we   *)
(*  are inside of a string                                                  *)
(* ------------------------------------------------------------------------ *)

and rew_nth_str (beg,con,en,ni,f) sym (id1,id2) a b = if (is_empty id1 b) then
                                      AS.application (str_to_op_symb en,
                                       AS.terms [(f a)])
                                    else
                                      AS.application
                                        (str_to_op_symb con,AS.terms
                                         [(f a),rew_term (beg,con,en,ni,f) 
                                          true (id1,id2) b])

and is_empty id1 (AS.application (AS.op_symb (x,_),_)) =
              ((id_to_tom o idunwrap) x)=id1
  | is_empty id1 _ = false



(* ------------------------------------------------------------------------ *)
(* Isolate the list symbols                                                 *)
(* ------------------------------------------------------------------------ *) 

and get_list_tok x = let
                       val (num,flo1,flo2,str1,str2,li1,li2,li3) = x
                     in
                       (li2,li3)
                     end

and identity x = x

and rewrite_list (genv,name,po,li) x = (rew_term
                                        ("LiBegin","LiCons","LiEnd",
                                         "LiEmpty",identity)
                                        false (get_list_tok li) x)



(****************************************************************************)
(* pretty printing                                                          *)
(****************************************************************************)

(* ------------------------------------------------------------------------ *)
(* This section contains the functions that actually call the Isabelle      *)
(*  pretty printing system                                                  *)
(* ------------------------------------------------------------------------ *)


fun do_til []       = []
  | do_til (" "::t) = "~" :: (do_til t)
  | do_til (h::t)   = h :: t

fun do_tilde x = (implode o do_til o explode) x

fun filter_wspace []              = []
  | filter_wspace (" "::(" "::t)) = filter_wspace (" "::t)
  | filter_wspace (h::t)          = h :: (filter_wspace t)

fun whitesp s = (implode o filter_wspace o explode) s

fun linebrks []     = []
  | linebrks [h]    = [h]
  | linebrks (h::t) = (h ^ "\\\\{}") :: (linebrks t)

fun linebreaks f = (relink o linebrks o unlink) f

fun get_syn genv name = (append_to_syn IsabelleParser.CASL98 genv name,
                         get_symb_ord genv name, get_literals genv name)

fun get_ft (syn,po,li) genv name f =
                ( Printer.print_mode := ["latex"];
                  Pretty.setmargin 90;
                  let
                    val res = Syntax.pretty_term syn false
                              (p_l_formula (genv,name,po,li) f)
                  in
                    Printer.print_mode := [];
                    Pretty.setmargin 80;
                    res
                  end )

fun pretty_formula (syn,po,li) genv name linelen f =
                ( Printer.print_mode := ["latex"];
                  Pretty.setmargin linelen;
                  let
                    val res = Pretty.string_of
                              (Syntax.pretty_term
                              syn false (p_l_formula (genv,name,po,li) f))
                  in
                    ( Printer.print_mode := [];
                      Pretty.setmargin 80;
                      (whitesp o linebreaks) res )
                  end )

fun pretty_term (syn,po,li) genv name linelen t =
                ( Printer.print_mode := ["latex"];
                  Pretty.setmargin linelen;
                  let
                    val res = Pretty.string_of
                              (Syntax.pretty_term
                              syn false (p_term2 (genv,name,po,li) t))
                  in
                    ( Printer.print_mode := [];
                      Pretty.setmargin 80;
                      (whitesp o linebreaks) res )
                  end )

fun pretty_anno (syn,po,li) genv name linelen t =
                ( Printer.print_mode := ["latex"];
                  Pretty.setmargin linelen;
                  let
                    val res = Pretty.string_of
                              (Syntax.pretty_term
                              syn false (p_parse_anno t))
                  in
                    ( Printer.print_mode := [];
                      Pretty.setmargin 90;
                      (whitesp o linebreaks) res )
                  end )

fun pretty_id (syn,po,li) genv name linelen t =
                ( Printer.print_mode := ["latex"];
                  Pretty.setmargin linelen;
                  let
                    val res = Pretty.string_of
                              (Syntax.pretty_term
                              syn false (p_idterm t))
                  in
                    ( Printer.print_mode := [];
                      Pretty.setmargin 90;
                      (whitesp o linebreaks) res )
                  end )


fun pretty_axioms syn genv name []     = ""
  | pretty_axioms syn genv name (h::t) = (pretty_formula syn genv name 70 h)
                                         ^ ";\\\\\n" ^
				         (pretty_axioms syn genv name t)


fun pretty_local_axioms syn genv name [] = ""
  | pretty_local_axioms syn genv name (h::t) = "\\I\\. \\(\\[ " ^
                                        (pretty_formula syn genv name 70 h)
                                        ^ "\\]\\)\n" ^
                                        (pretty_local_axioms syn genv name t)
                                           
fun pretty_subsort_defn syn genv name f = pretty_formula syn genv name 70
                                          (f,AS.empty_label)

fun pretty_pred_defn syn genv name f = pretty_formula syn genv name 45 f

fun pretty_unit syn genv name t = pretty_term syn genv name 30 t

fun pretty_op_defn syn genv name t = pretty_term syn genv name 40 t

fun pretty_parse_anno syn genv name t = pretty_anno syn genv name 70 t

end

(****************************************************************************)
(* Music played during development:                                         *)
(*   Blind Guardian, Theatre of Tragedy, Metallica, JBO, Weird Al Yankovic  *)
(****************************************************************************)
