(* *********************************************************************** *)
(*									   *)
(* Project : CATS 							   *)
(* Author: Till Mossakowski, University of Bremen			   *)
(* Date: 09.05.2000				 			   *)
(* Purpose of this file: Renaming of signatures and axioms		   *)
(*			 						   *)	
(*									   *)
(* *********************************************************************** *)


(* Signatures and axioms can be renamed along signature morphisms
   Position info is deleted while renaming a formula 
*)

structure Rename :
sig
  val rename_L_FORMULA : GlobalEnv.morphism -> AS.L_FORMULA -> AS.L_FORMULA
  val rename_axs : GlobalEnv.morphism -> AS.L_FORMULA list -> AS.L_FORMULA list
  val rename_sign : GlobalEnv.morphism -> LocalEnv.sign -> LocalEnv.sign
end

= struct
open Utils AS LocalEnv GlobalEnv

(*********** Renaming a formula along a morphism *****************)

fun rename_SORT (sortmap,_,_) s =
    case Symtab_id.lookup(sortmap,s) of
      Some t => t
      | None => s;

fun rename_OP_NAME (morphism as (_,opmap,_)) f =
    case Symtab_id.lookup(opmap,f) of
                  Some [(_,h,_)] => h
                  | _ => f

fun rename_OP_TYPE morphism (total_op_type (args,res)) =
    total_op_type (sorts (map (rename_SORT morphism) (get_sorts args)),rename_SORT morphism res)
  | rename_OP_TYPE morphism (partial_op_type (args,res)) =
    partial_op_type (sorts (map (rename_SORT morphism) (get_sorts args)),rename_SORT morphism res)
  | rename_OP_TYPE morphism (pos_OP_TYPE (_,t)) =
    rename_OP_TYPE morphism t

fun rename_PRED_NAME (morphism as (_,_,predmap)) p =
    case Symtab_id.lookup(predmap,p) of
                     Some [(_,q)] => q
                     | _ => p

fun rename_PRED_TYPE morphism (pred_type args) =
    pred_type (sorts (map (rename_SORT morphism) (get_sorts args)))
  | rename_PRED_TYPE morphism (pos_PRED_TYPE (_,t)) =
    rename_PRED_TYPE morphism t

fun rename_OP_SYMB morphism (op_symb (f,t)) =
    op_symb(rename_OP_NAME morphism f,
            apsome (rename_OP_TYPE morphism) t)
  | rename_OP_SYMB morphism (pos_OP_SYMB(_,sy)) =
    rename_OP_SYMB morphism sy

fun rename_PRED_SYMB morphism (pred_symb (p,t)) =
    pred_symb(rename_PRED_NAME morphism p,
              apsome (rename_PRED_TYPE morphism) t)
  | rename_PRED_SYMB morphism (pos_PRED_SYMB(_,sy)) =
    rename_PRED_SYMB morphism sy

fun rename_VAR_DECL morphism (vlist,srt) =
	(vlist,rename_SORT morphism srt)

fun rename_VAR_DECL_list morphism l =
    map (rename_VAR_DECL morphism) l
    
fun rename_TERM morphism (a:TERM):TERM= 
case a of
    qual_var (x,s) => qual_var(x,rename_SORT morphism s)
 |  var_or_const x  => var_or_const x  
 | application (opsym, ts) => 
   application (rename_OP_SYMB morphism opsym,
                rename_TERMS morphism ts)
 | sorted_term (t, s) =>  
      sorted_term (rename_TERM morphism t,rename_SORT morphism s)
 | cast (t,s) =>  
      cast (rename_TERM morphism t,rename_SORT morphism s)
 | conditional (t1,phi,t2) =>
      conditional (rename_TERM morphism t1,
                   rename_FORMULA morphism phi,
                   rename_TERM morphism t2)
 | unparsed_term s => unparsed_term s
 | pos_TERM (_,_,t) => rename_TERM morphism t
 
and rename_TERMS morphism ts:TERMS = 
    terms (map (rename_TERM morphism) (get_terms ts))



and rename_ATOM morphism (a:ATOM):ATOM =
case a of
  (predication (psymb,ts)) =>
   predication (rename_PRED_SYMB morphism psymb,rename_TERMS morphism ts)
 | (definedness T) =>  
    definedness (rename_TERM morphism T) 
 | (existl_equation (t1 , t2)) =>  
      existl_equation (rename_TERM morphism t1,rename_TERM morphism t2)
 | (strong_equation (t1 , t2)) =>  
      strong_equation (rename_TERM morphism t1,rename_TERM morphism t2)
 | membership (t, s)  => 
      membership (rename_TERM morphism t,rename_SORT morphism s)
 | ttrue => ttrue
 | ffalse => ffalse 
  
and rename_FORMULA morphism (phi:FORMULA):FORMULA =
case phi of
 quantification (quant ,vlist, f) =>  
      quantification (quant, rename_VAR_DECL_list morphism vlist, rename_FORMULA morphism f)
 | conjunction ff =>  conjunction (map (rename_FORMULA morphism) ff)
 | disjunction ff => disjunction (map (rename_FORMULA morphism) ff)
 | implication (f1, f2) => implication (rename_FORMULA morphism f1, rename_FORMULA morphism f2)
 | equivalence (f1, f2) => equivalence (rename_FORMULA morphism f1, rename_FORMULA morphism f2)
 | negation f =>  negation (rename_FORMULA morphism f)
 | atom a => atom (rename_ATOM morphism a)
 | sort_gen_ax (s,ops) => sort_gen_ax (map (rename_SORT morphism) s,map (rename_OP_SYMB morphism) ops)
 | unparsed_formula s => unparsed_formula s
 | pos_FORMULA (_,_,phi) => rename_FORMULA morphism phi

			   
fun rename_L_FORMULA morphism (phi,l) = (rename_FORMULA morphism phi,l)
fun rename_axs morphism axs = map (rename_L_FORMULA morphism) axs


(*********** Renaming a signature along a morphism *****************)

fun rename_sub morphism (s,sublist) =
    (rename_SORT morphism s,map (rename_SORT morphism) sublist)

fun rename_var morphism (v,s) =
    (v,rename_SORT morphism s)

fun rename_fun morphism (f,types) =
    (rename_OP_NAME morphism f,map (rename_OP_TYPE morphism) types)

fun rename_pred morphism (p,types) =
    (rename_PRED_NAME morphism p,map (rename_PRED_TYPE morphism) types)

fun rename_sign morphism sign = 
    let val local_list = env_to_list sign
        val (_,sublist,varlist,funlist,predlist) = local_list
    in list_to_env
       ([],
        map (rename_sub morphism) sublist,
        map (rename_var morphism) varlist,
        map (rename_fun morphism) funlist,
        map (rename_pred morphism) predlist)
    end
end
