(* ----------------------------------------------------------------------- *)
(* xml_types.sml                          Pascal Schmidt <pharao90@tzi.de> *)
(* ----------------------------------------------------------------------- *)
(* datatypes for xml handling                                              *)
(* ----------------------------------------------------------------------- *)

(* $Id: xml_types.sml,v 1.1 2004/02/13 11:29:21 7till Exp $ *)

structure xml_types
=
struct

  datatype xmldata = pcdata of string
                   | entity_def of (string * xmldata)
                   | entity_ref of string
                   | share_ref of int
                   | tag of (string * (string * string) list * xmldata list)
                   | empty_tag of (string * (string * string) list)

  fun xml2string (pcdata (s)) = s
    | xml2string (tag (s,l,[pcdata (t)])) = "<" ^ s ^ (att2string l) ^ ">" ^
                                           t ^ "</" ^ s ^ ">"
    | xml2string (tag (s,l,x)) = "<" ^ s ^ (att2string l) ^">" ^
                                 (String.concat ((map xml2string) x)) ^ "</" 
                                 ^ s ^ ">"
    | xml2string (empty_tag (s,l)) = "<" ^ s ^ (att2string l) ^ "/>"  
    | xml2string (entity_def (e,t)) = "<!ENTITY "^e^" \""
                                      ^(xml2string t)^"\">"
    | xml2string (entity_ref (e)) = "&"^e^";"
    | xml2string (share_ref (i)) = ""

  and xml2str x = (xmlhead()) ^ (xml2string x) 

  and sxml2str (l,x) = (xmlhead()) ^ (String.concat ((map xml2string) l))
                       ^ "]>" ^ (xml2string x)

  and xmlhead () = "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>"^
                  "<!DOCTYPE casl SYSTEM \"casl.dtd\" ["
 
  and att2string l = String.concat ((map attr2string) l)
  
  and attr2string (t,v) = " " ^ t ^ "=\"" ^ v ^ "\""

  fun xml2file x fname = let
                           val st = TextIO.openOut fname
                         in
                           ( TextIO.output(st,xmlhead());
                             xml2f st x;
                             TextIO.closeOut st )
                         end

  and sxml2file (l,x) fname = let
                                val st = TextIO.openOut fname
                              in
                                ( TextIO.output(st,xmlhead());
                                  (map (xml2f st)) l;
                                  TextIO.output(st,"]>");
                                  xml2f st x;
                                  TextIO.closeOut st )
                              end 
  
  and xml2f st (pcdata (s)) = ( TextIO.output(st,s); st )
    | xml2f st (tag (s,l,[pcdata (t)])) = ( TextIO.output(st,"<"^ s^
                                          (att2string l)^">"^t^"</"^ s^">");
                                          st )
    | xml2f st (tag (s,l,x)) = ( TextIO.output(st,"<"^s^(att2string l)^">");
                               (map (xml2f st)) x;
                               TextIO.output(st,"</"^s^">"); st )
    | xml2f st (empty_tag (s,l)) = ( TextIO.output(st,
                                   "<"^s^(att2string l)^"/>"); st )
    | xml2f st (entity_def (e,t)) = ( TextIO.output(st,"<!ENTITY "^e^" \"");
                                    (xml2f st t);
                                    TextIO.output(st,"\">"); st )
    | xml2f st (entity_ref (e)) = ( TextIO.output(st,"&"^e^";"); st )
    | xml2f st (share_ref (i)) = st

end
