% Polymerization Reactor Model
%
function[sys,x0,str,ts] = sfunc(t,x,u,flag);

% coefficients for calculation of cp and H
c1 = [2.7851E-3 -9.18E-6 2.93E-8 2.25E-3];
% coefficients for calculation of density Rhom
d1 = [-263.7 6.8265 -0.0143];
% coefficients for Flory-Huggins parameter phi
f1 = [4.1477 -3.063E-2 6E-5];
% coefficients for calculation of q(MW)
q1 = [0.00313 8.821];
    
switch flag,
   %% initialization %%    
case 0,
   [sys,x0,str,ts] = mdlInitializeSizes;   
   %% derivatives %%
case 1,
   sys = mdlDerivatives(t,x,u,c1,d1,f1);
   %% update and terminate
case {2, 9},
   sys = []; % do nothing
   %% output
case 3,
   sys = mdlOutputs(t,x,u,d1,f1,q1);
otherwise
   error(['unhandled flag = ',num2str(flag)]);
end

%% mdlInitializeSizes 
%% Return the sizes, initial conditions, and sample time for the S-function

function [sys,x0,str,ts] = mdlInitializeSizes

sizes = simsizes;
sizes.NumContStates  = 7;
sizes.NumDiscStates  = 0;
sizes.NumOutputs     = 4;
sizes.NumInputs      = 2;
sizes.DirFeedthrough = 0;
sizes.NumSampleTimes = 1;

sys = simsizes(sizes);
str = [];
ts  = [0 0]; % continuous sample time [period offset]
x0  = [0.9151 0.3325 2.585E-5 9.102E-6 5E-5 0.8360 333.5];

%% end mdlInitialize sizes

%% mdlDerivatives
%% Compute derivatives for continuous states

function sys = mdlDerivatives(t,x,u,c1,d1,f1)
%
% state parameters
%      
M    = x(1);
conv = x(2);
xH2  = x(3);
xcna = x(4);
xca  = x(5);
E    = x(6);
Ta   = x(7);
%
% inputs
%
xH2in = u(1);
Tj    = u(2);
%
% calculate the monomer fraction
%
xm = 1 - conv;
%
% calculation of the real temperature, this is done
% via an iterative loop with first order filter with
% time constant of 0.05 hrs
%
cpm = c1(1)+c1(2)*Ta+c1(3)*Ta^2;
cp  = xm*cpm + (1-xm)*c1(4);
T   = E/(M*cp);
dT  = 20*(T - Ta);
%
% calculation of density monomer
%
Rhop  = 900.0;
Rhom  = d1(1)+d1(2)*T+d1(3)*T^2;
Rho   = Rhop*Rhom/(Rhop*xm+(1-xm)*Rhom);
% 
% calculation of specific heat and enthalpy
%
Tref  = 173.0;
Tin   = 330.0;
H     = (c1(1)*(T-Tref)+(c1(2)/2)*(T^2-Tref^2)+(c1(3)/3)*(T^3-Tref^3)) ...
        *xm+c1(4)*(T-Tref)*(1-xm);
Hin   = (c1(1)*(Tin-Tref)+(c1(2)/2)*(Tin^2-Tref^2)+(c1(3)/3) ...
        *(Tin^3-Tref^3))*xm;
%
% Flory-Huggins approximation
%
phi = f1(1)+f1(2)*T+f1(3)*T^2;

%   
% kinetic constants and rates
%
Vr    = 0.001855;
H2    = M*xH2/Vr;
ki0   = 7.632E6;
ki1   = 5.4E8;
Eacti = 37E3;
R     = 8.314;
ki    = (ki0+ki1*H2)*exp(-Eacti/(R*T));
Ri    = ki*M*xcna;
kp0   = 8.172E7;
kp1   = 1.080E9;
Eactp = 45E3;
kp    = (kp0+kp1*H2^0.4)*exp(-Eactp/(R*T));
Rp    = kp*M*xca*phi*Rhom;
kd0   = 1.1664E4;
Eactd = 23.5E3; 
kd    = kd0*exp(-Eactd/(R*T));
Rd    = kd*M*xca;
%
% calculation of outlet flow
%
Fin   = 1.0;
F     = (Fin/Rhom+Rp*(1/Rhop-1/Rhom)-(dT*M*xm ...
        *((d1(2)+2*d1(3)*T)/Rhom^2)))*Rho;
%
% differential equations
% 
sys(1) = Fin-F;                       % overall mass balance
sys(2) = -(Fin*(1.0-xm)-Rp)/M;        % monomer conversion balance
sys(3) = Fin*(xH2in-xH2)/M;           % hydrogen component balance
xcnain = 200E-6;
sys(4) = (Fin*(xcnain-xcna)-Ri)/M;    % balance for non -activated catalyst
xcain  = 0.0;
sys(5) = (Fin*(xcain-xca)+Ri-Rd)/M;   % balance for activated catalyst
delH   = -2.033;
U      = 1.62;
Ar     = 0.0961;
sys(6) = Fin*Hin-F*H-Rp*delH-U*Ar*(T-Tj); % energy balance reactor
sys(7) = dT;                          % iterative temperature calculation

%% end mdlDerivatives

%% mdlOutputs
%% Return the output vector for the S-function

function output = mdlOutputs(t,x,u,d1,f1,q1)

output(1) = x(1);
output(2) = x(2);
output(3) = x(7);
%
% calculation of density monomer
%
T     = x(7);
Rhom  = d1(1)+d1(2)*T+d1(3)*T^2;
%
% Flory-Huggins approximation
%
phi = f1(1)+f1(2)*T+f1(3)*T^2;
%
% chain termination probability and conversion
%
M    = x(1);
xH2  = x(3);
Vr   = 0.001855;
H2   = M*xH2/Vr;
q    = q1(1)+q1(2)*(H2/(Rhom*phi));
%
output(4) = q;
%
% end mdlOutputs