% File SISO_ID.m, used for parametric model identification
% of single input-single output systems

echo on,clc
% This case study concerns data collected on a gas furnace as
% described in G.P. Box and G.M. Jenkins: Time series analysis,
% forecasting and control, Holden-Day 1976, p. 381

load boxj

% Vector MethFd contains 296 readings of the Methane Feed to 
% gas furnace, CO2Conc contains readings of the carbon dioxide 
% concentration in the off gases
% The sampling interval is 9 seconds.

clc

z = [CO2Conc MethFd];

% Plot the measurements

idplot(z, 1:296, 9.0), pause  % Press ENTER to continue
clc
% Remove the constant levels and make the data zero-mean:

z = dtrend(z);

% First estimate the impulse response of the system by
% correlation analysis to get some idea of the system dynamics

ir = cra(z); pause  % Press ENTER to continue

% We can form the corresponding step response by integrating
% the impulse response:

stepr = cumsum(ir);
clf reset, plot(stepr), title('Step response from correlation analysis'),
xlabel('time(samples)'), pause

% Use the least squares method to find an ARX model with two 
% poles, one zero, and three delays:
% The model structure is: A(z) y(t) = B(z) u(t-nk) + e(t)
% A(z) =  1 + a1*z^(-1) + a2*z^(-2) + ... + ana*z^(-na)
% B(z) = b1 + b2*z^(-1) + b2*z^(-2) + ... + bnb*z^(-nb)
% na = order of A; nb = order of B; nk = delay
% nn = [na nb nk]; th = arx(z2, nn);

th = arx(z,[2 2 3]);
th = sett(th,9.0);      % Set the correct sampling interval.
pause  % Presss ENTER to continue

% Calculate and plot the residuals for the model obtained by
% the least sqaures method:

[A1,B1,C1,D1,F1] = polyform(th);
disp(A1);
disp(B1); pause  % Press ENTER to continue
clc

[e,r] = resid(z,th), pause;
plot(e), title('Residuals, Least Squares Method'), pause;
clc;

% Use the Instrumental Variable method to find a model of 
% the same order

iv = iv4(z,[2 2 3]);
[A2,B2,C2,D2,F2] = polyform(iv);
disp(A2);
disp(B2); pause  % Press ENTER to continue
clc

% Calculate and plot residuals for the model obtained by IV4:

[e,r] = resid(z,iv), pause;
plot(e), title('Residuals, IV method'), pause;  % Press ENTER to continue
clc;

% Compute a second order ARMAX model
% The general model structure is:
% A(z) y(t) = B(z) u(t-nk) + C(z) e(t)
% C(z) = 1 + c1*z^(-1) + c2*z^(-2) + .... +cnc*z^(-nc)

am = armax(z,[2 2 2 3]);
[A3,B3,C3,D3,F3] = polyform(am);
disp(A3);
disp(B3);
disp(C3); pause  % Press ENTER to continue
clc

% How good are the models? One way to find out is to simulate 
% them and compare the model output with measured output. We
% will do this for the Least Squares model.

ud = dtrend(MethFd); yd = dtrend(CO2Conc);
yh = idsim(ud,th); 
pause  % Press ENTER to continue
plot([yh yd]);
legend('simulated','measured');
xlabel('Time (samples)'); 
ylabel('y,ypred');
pause  % Press ENTER to continue
clc

% The agreement is quite good. Compute the poles and zeros of
% the model. Don't bother to identify a noise model

zpth = th2zp(th);
zpplot(zpth);
legend('bx','poles','go','zeros',0);
pause  % Press ENTER to continue
clc

% We can compute the step response of the model as follows.

step = ones(20,1);
mstepr = idsim(step,th);
save c:\matlabR13\work\sisowts.dat mstepr -ascii

% This step response can be plotted and compared with the step 
% response from the correlation analysis

pause,  % Press ENTER to continue
plot([stepr mstepr]);
legend('measured','simulated');
xlabel('Time (samples)');
pause  % Press ENTER to continue