# Alkylation Reactor Optimization
# Reference: Sauer, Colville and Burwick, Hydrocarbon Processing 1964
#            Bracken, J. and McCormick, G.P., chaapter 4 in Selected
#            applications of non-linear programming, John Wiley, New York, 1968

set COMP;			
param Cost{COMP};		# in ECU per barrel
param Flowmin{COMP};		# minimum flows
param Flowmax{COMP};		# maximum flows
param Flowinit{COMP};		# initial values of flows
param ACIDstrinit > 0;		# minimum acid strength starting value
param ACIDstrlo > 0;		# minimum acid strength
param ACIDstrhi > 0;		# maximum acid strength
param MONinit > 0;		# minimum motor octane number starting value
param MONlo > 0;		# minimum motor octane number
param MONhi > 0;		# maximum motor octane number
param Extratinit > 0;		# minimum ext. IC4_Olefins ratio starting value
param Extratlo > 0;		# minimum ext. IC4_Olefins ratio
param Extrathi > 0;		# maximum ext. IC4_Olefins ratio
param ADFinit > 0;		# minimum acid dilution factor starting value
param ADFlo > 0;		# minimum acid dilution factor
param ADFhi > 0;		# maximum acid dilution factor
param APNinit > 0;		# minimum alkylate performance number starting value
param APNlo > 0;		# minimum alkylate performance number
param APNhi > 0;		# maximum alkylate performance number
var Flow{i in COMP} >= Flowmin[i], <= Flowmax[i], := Flowinit[i];
				# component flows
var ALKest >= 0;		# estimated alkylate flow
var MONest >= 0;		# estimated motor octane number
var MON >= MONlo, <= MONhi, := MONinit;	# motor octane number
var ADFest >= 0;		# estimated acid dilution factor
var ADF >= ADFlo, <= ADFhi, := ADFinit;	# acid dilution factor
var APNest >= 0;		# estimated alkylate performance number
var APN >= APNlo, <= APNhi, := APNinit;
				# alkylate performance number
var Extrat >= Extratlo, <= Extrathi, := Extratinit;
				# external IC4 to olefins ratio
var ACIDstr >= ACIDstrlo, <= ACIDstrhi, := ACIDstrinit;
				# acid strength

maximize profit:
   Cost["Alkylate"] * Flow["Alkylate"] * MON - Cost["Olefin"] * Flow["Olefin"]
   - Cost["IC4R"] * Flow["IC4R"] - Cost["Acid"] * Flow["Acid"]
   - Cost["IC4"] * Flow["IC4"];

# *** ALKYLATE FLOW ***
# estimated alkylate flow:
subject to AlkylFlowEst:
  ALKest = Flow["Olefin"]*(1.12+0.13167*Extrat-0.00667*Extrat*Extrat);

# alkylate flow upper limit:
subject to AlkylFlowHigh:
   Flow["Alkylate"] <= 100*ALKest/98;

# alkylate flow lower limit:
subject to AlkylFlowLow:
   Flow["Alkylate"] >= 98*ALKest/100;

# *** OCTANE NUMBER ***
# estimated motor octane number:
subject to OctaneEst:
   MONest = 86.35+1.098*Extrat-0.038*Extrat*Extrat+0.325*(ACIDstr-89);

# motor octane number upper limit:
subject to OctaneHigh:
   MON <= 100*MONest/98;

# motor octane number lower limit:
subject to OctaneLow:
   MON >= 98*MONest/100;

# *** ACID DILUTION FACTOR ***
# estimated acid dilution factor:
subject to DilutionEst:
   ADFest = 35.82-0.222*APN;

# acid dilution factor upper limit:
subject to DilutionHigh:
   ADF <= 10*ADFest/9;

# acid dilution factor lower limit:
subject to DilutionLow:
   ADF >= 9*ADFest/10;

# *** ALKYLATE PERFORMANCE NUMBER ***
# estimated alkylate performance number
subject to APNEst:
   APNest = 3*MON-133.0;

# alkylate performance upper limit:
subject to APNHigh:
   APN <= 100*APNest/98;

# alkylate performane lower limit:
subject to APNLow:
   APN >= 98*APN/100;

# *** OTHER EQUATIONS ***
# external IC4 to olefins ratio
subject to ExtIC4Olefins:
   Extrat = (Flow["IC4R"]+Flow["IC4"])/Flow["Olefin"];

# IC4 balance:
subject to IC4balance:
   Flow["IC4"] = 1.22*Flow["Alkylate"] - Flow["Olefin"];

# Acid strength:
subject to AcidStrength:
   ACIDstr = 98000*Flow["Acid"]/(Flow["Alkylate"]*ADF+1000*Flow["Acid"]);