## Alkylation plant optimization based on data from:
# 	Sauer, Colville and Burwick
#	Hydrocarbon Processing, 43, 84, 1964
#
# Naming convention for variables:
# Y******  = vol fraction of stream
# Yf2****  = vol fraction of stream 2
# F?       = flowrate of stream ?, bbl/day for hydrocarbon streams
#
# &*&**&*&*&*&*&*&*&*&* PARAMETER DEFINITION *&*&*&*&*&*&*&*&*&*&*&*
# *** -------------------- Butane splitter ----------------------***

param F9hi >= 0;
param F10lo >= 0;
param SNoTr >= 0;			# splitter theoretical trays
param SNM_i := 0.8*SNoTr;
param SRelVol >= 0;       		# splitter relative volatility
param SVaporhi >= 0;                    # maximum splitter vapor overhead
param SRFRatio_i >= 0;
param Yf9iC4lo >= 0;
param Yf9iC4hi >= 0;
param Yf9iC4_i >= 0;
param Yf10iC4lo >= 0;
param Yf10iC4hi >= 0;
param Yf10iC4_i >= 0;
param Yf11iC4lo >= 0;
param Yf11iC4hi >= 0;
param Yf11iC4_i >= 0;
param Yf11nC4_i >= 0;

#
# *** -------------- End Butane splitter definition ------------ ***
#
# *** ------------------- Olefin feed definition --------------- ***
#

param F1lo >= 0;
param F1hi >= 0;
param F1_i >= 0;
param Yf1C3_i >= 0;
param Yf1nC4_i >= 0;
param Yf1Oleflo >= 0;
param Yf1Olef_i >= 0;
param Yf2iC4_i >= 0;

#
# *** ----------------- End olefin feed definition ------------- ***
#
# *** ----------------------- Reactor -------------------------- ***
#

param Acidstr_i >= 0;
param RecRatio_i >= 0;
param MOctane_i >= 0;
param F3Acid_i >= 0;
param Adilute_i >= 0;

# coefficients in model

param YInter >= 0;			# alkyl yield intercept
param YLinear >= 0;			# alkyl yield linear term
param YQuad <= 0;			# alkyl yield quadratic term
param OInter >= 0;			# alkyl octane intercept
param OYf4iC4 >= 0;			# alkyl octane IC4 term
param OAcidstr >= 0;			# alkyl octane acid strength term

param Acidstrhi >= 0;
param Acidstrlo >= 0;
param RecRatiohi >= 0;
param RecRatiolo >= 0;
param Ratiohi >= 0;
param Ratiolo >= 0;

#
# *** ------------------- End reactor definition --------------- ***
#
# *** ------------------- DIB tower definition ----------------- ***
#

param F6lo >= 0;
param Yf6iC4_i >= 0;
param Yf6nC4lo >= 0;
param Yf6nC4hi >= 0;
param Yf8iC4_i >= 0;
param Yf8nC4_i >= 0;
param Yf8nC4lo >= 0;
param Yf8nC4hi >= 0;
param Yf8Alkyl_i >= 0;

#
# *** ------------------ End DIB tower definition -------------- ***
#
# *** ---------------------- Gasoline blend -------------------- ***
#

param F12_i >= 0;
param F13_i >= 0;
param F14hi >= 0;
param F14_i >= 0;
param F15hi >= 0;
param F15_i >= 0;
param F16_i >= 0;
param F17hi >= 0;
param F17_i >= 0;
param RVPiC4 >= 0;			# rvp of iC4
param RVPnC4 >= 0;			# rvp of nC4
param RVPAlkyl >= 0;			# rvp of alkylate
param RVPRef >= 0;			# rvp of reformate
param RVPLsr >= 0;			# rvp lsr-naphta
param RVPMax >= 0;			# maximum RVP specification
param RVPMin >= 0;			# minimum RVP specification
param OctiC4 >= 0;			# octane of iC4
param OctnC4 >= 0;			# octane of nC4
param OctRef >= 0;			# octane of reformate
param OctLsr >= 0;			# octane of lsr
param OctMax >= 0;			# maximum octane specification
param OctMin >= 0;			# minimum octane specification

#
# *** ------------------ End blending definition --------------- ***
#
# *** -------- Economics definition, $ per flow unit ----------- ***
#

param PriceQ >= 0;			# price reboiler energy, $/bbl C4 vapor
param PriceF1 >= 0;			# price olefin feed
param PriceF3 >= 0;			# price acid makeup
param PriceF9 >= 0;			# price mixed butane feed
param PriceF14 >= 0;			# price reformate
param PriceF15 >= 0;			# price lsr-naphta
param PriceF5 >= 0;			# price propane product
param PriceF7 >= 0;			# price dib side stream
param PriceF16 >= 0;			# price gasoline
param PriceF17 >= 0;			# price butane sales
param PriceInf >= 0;			# penalty on blend slack, selected to
					# prevent infeasibility

#
# *** ----------- Calculation of initial values ---------------- ***
#

param Yf1iC4_i   := 1 - Yf1C3_i - Yf1nC4_i - Yf1Olef_i;
param F9_i       := F1_i/3.0;
param F10_i      := F9_i * Yf9iC4_i;
param F11_i      := F9_i - F10_i;
param SReflux_i  := F10_i * SRFRatio_i;
param SVapor_i   := F10_i + SReflux_i;
param SRM_i      := SRFRatio_i - 2.0;
param F6_i       := 10.0 * F1_i * Yf1Olef_i;
param Yf6nC4_i   := 1 - Yf6iC4_i;
param F6iC4_i    := F6_i * Yf6iC4_i;
param F6nC4_i    := F6_i * Yf6nC4_i;
param F2_i       := F10_i + F6_i;
param Yf2nC4_i   := 1. - Yf2iC4_i;
param Ratio_i    := F2_i/F1_i;
param F4_i       := F1_i + F2_i;
param Yf4C3_i    := Yf1C3_i/Ratio_i;
param Yf4Alkyl_i := Yf1Olef_i/Ratio_i;
param Yf4nC4_i   := (F1_i * Yf1nC4_i + F2_i * Yf1nC4_i)/(F1_i + F2_i);
param Yf4iC4_i   := (F1_i * Yf1iC4_i + F2_i * Yf2iC4_i)/(F1_i + F2_i);
param F5_i       := F4_i * Yf4C3_i;
param F7_i       := F4_i * Yf4nC4_i;
param F8_i       := F4_i * Yf4Alkyl_i;

# &*&*&*&*&*&*&*&*&*&*&* VARIABLE DEFINITION *&*&*&*&*&*&*&*&*&*&*&*
# *** -------------------- Butane splitter ----------------------***

var   F9 >= 0, <= F9hi, := F9_i;	# mixed c4 feed to splitter
var   F10 >= F10lo, :=F10_i;		# splitter distillate product
var   F11 >= 0, := F11_i;		# splitter bottoms product
var   Yf9iC4 >= Yf9iC4lo, <= Yf9iC4hi, := Yf9iC4_i;
					# ic4 in splitter feed
var   Yf10iC4 >= Yf10iC4lo, <= Yf10iC4hi, := Yf10iC4_i;
					# ic4 in splitter distillate product
var   Yf11iC4 >= Yf11iC4lo, <= Yf11iC4hi, := Yf11iC4_i;
					# ic4 in splitter bottoms product
var   Yf11nC4 >= 0, := Yf11nC4_i;	# nc4 in splitter bottoms product
var   SReflux >= 0, := SReflux_i;	# splitter reflux flow
var   SRFRatio >= 0, := SRFRatio_i;	# splitter reflux ratio
var   SRM >= 0, := SRM_i;		# splitter minmum reflux
var   SVapor >= 0, <= SVaporhi, := SVapor_i;
					# splitter vapor overhead
var   SNM >=0, := SNM_i;       		# splitter minimum number of trays

# *** ------------------- Olefin feed definition --------------- ***

var   F1 >= F1lo, <= F1hi, := F1_i;	# olefin feed
var   Yf1C3 >= 0, <= 1, := Yf1C3_i;	# propane in olefin feed
var   Yf1iC4 >= 0, <= 1, := Yf1iC4_i;	# ic4 in olefin feed
var   Yf1nC4 >= 0, <= 1, := Yf1nC4_i;	# nc4 in olefin feed
var   Yf1Olef >= Yf1Oleflo, <= 1, := Yf1Olef_i;
					# olefin (c4) in olefin feed
var   F2 >= 0, := F2_i;			# recycle plus makeup
var   Yf2iC4 >= 0, <= 1, := Yf2iC4_i;	# ic4 in recycle plus makeup
var   Yf2nC4 >= 0, <= 1, := Yf2nC4_i;	# nc4 in recycle plus makeup

# *** ----------------------- Reactor -------------------------- ***

var   Acidstr >= Acidstrlo, <= Acidstrhi, := Acidstr_i;
                                        # Acidstr  = acid strength
var   RecRatio >= RecRatiolo, <= RecRatiohi, := RecRatio_i;
                                        # RecRatio = ratio of F2 to F1
var   Ratio >= Ratiolo, <= Ratiohi, := Ratio_i;
					# ratio of total iC4 to olefin in F1+F2
var   F4 >= 0, := F4_i;			# reactor hydrocarbon effluent
var   Yf4iC4 >= 0, <= 1, := Yf4iC4_i;	# iC4 in reactor effluent
var   Yf4nC4 >= 0, <= 1, := Yf4nC4_i;	# nC4 in reactor effluent
var   Yf4Alkyl >= 0, <=1, := Yf4Alkyl_i;
					# alkylate in reactor effluent
var   Yf4C3 >= 0, <= 1, := Yf4C3_i;	# C3 in reactor effluent
var   MOctane >= 0, :=MOctane_i;        # octane of alkylate (ron+mon)/2
var   Adilute >= 0, := Adilute_i;	# acid dilution
var   F3Acid >= 0, := F3Acid_i;		# acid flow

# *** ------------------- DIB tower definition ----------------- ***

var   F5 >= 0, := F5_i;			# propane product
var   F8 >= 0, := F8_i;			# DIB bottoms (alkylate) flow
var   F6 >= F6lo, := F6_i;		# recycle flow
var   F6iC4 >= 0, := F6iC4_i;		# total flow of iC4 in recycle, bbl/day
var   F6nC4 >= 0, := F6nC4_i;		# total flow of nC4 in recycle, bbl/day
var   Yf6iC4 >= 0, <= 1, := Yf6iC4_i;	# iC4 in recycle
var   F7 >= 0, := F7_i;			# DIB side stream product
var   Yf8nC4 >= Yf8nC4lo, <= Yf8nC4hi, := Yf8nC4_i;
					# nC4 in alkylate
var   Yf8iC4 >= 0, <=1, := Yf8iC4_i;	# iC4 in alkylate
var   Yf8Alkyl >= 0, <= 1, := Yf8Alkyl_i;
					# alkylate in DIB bottoms flow
var   Yf6nC4 >= Yf6nC4lo, <= Yf6nC4hi, := Yf6nC4_i;
					# nC4 in recycle

# *** ---------------------- Gasoline blend -------------------- ***

var   F12 >= 0, := F12_i;		# C4 to blend
var   F13 >= 0, := F13_i;		# alkylate to blend
var   F14 >= 0, <= F14hi, := F14_i;	# reformate in blend
var   F15 >= 0, <= F15hi, := F15_i;	# lsr-naphta in blend
var   F16 >= 0, := F16_i;		# gasoline product in blend
var   F17 >= 0, <= F17hi, := F17_i;	# C4 to sales


# *** -----------------miscellaneous variables ----------------- ***
var   Intmed1 >= 0;			# intermediate in rvp calc
var   Intmed2 >= 0;			# intermediate in rvp calc
var   Intmed3 >= 0;			# intermediate in oct calc
var   Intmed4 >= 0;			# intermediate in oct calc
var   Intmed5 >= 0;			# intermediate in profit calc
var   Intmed6 >= 0;			# intermediate in profit calc

var   SL1 >= 0, := 0;			# slack in rvp
var   SL2 >= 0, := 0;			# slack in rvp
var   SL3 >= 0, := 0;			# slack in oct
var   SL4 >= 0, := 0;			# slack in oct

var   Steam >= 0;			# steam flow*price to splitter and DIB

# *&*&*&*&*&*&*&*&*&*&* MODEL DEFINITION *&*&*&*&*&*&*&*&*&*&*&*&*&*
#
# *** Economic objective ***
maximize operating_profit:
	Intmed6-Intmed5-PriceInf*(SL1+SL2+SL3+SL4)-Steam;
#
# *** Splitter ***
#
subject to SMinRefx:			# splitter min reflux
	SRM = (Yf10iC4/Yf9iC4 - SRelVol*(1.-Yf10iC4)/(1.-Yf9iC4))/
		(SRelVol-1.);

subject to SMinTray:			# splitter min tray
	(SNoTr-SNM)/(SNoTr+1) = 0.75*(1.-((SRFRatio-SRM)/
		(SRFRatio+1.))**.5668);

subject to SSEP:			# splitter separation
	SNM = log((Yf10iC4/(1.-Yf10iC4))*((1.-Yf11iC4)/Yf11iC4))/
		log(SRelVol);

subject to SMatBal:			# splitter material balance
	F9 = F10 + F11;

subject to SKeyBal:			# splitter balance on key component
	F9*Yf9iC4 = F10*Yf10iC4+F11*Yf11iC4;

subject to SRRatio:			# splitter reflux ratio
	SRFRatio = SReflux/F10;

subject to SReboil:			# splitter vapor rate
	SVapor = SReflux + F10;

subject to Feas1:			# splitter ensure feasible calc
	SNM <= SNoTr - 0.1;

subject to Feas2:			# splitter ensure feasible calc
	SRFRatio - 0.1 >= SRM;

subject to BottnC4:			# nc4 in F11
	Yf11nC4 = 1. - Yf11iC4;

# *** Mix recycle ***
	
subject to MixF2:			# material balance F6+F10
	F2 = F10 + F6;

subject to ConcF2iC4:			# ic4 in combined F6+F10
	Yf2iC4 = (F10*Yf10iC4+F6*Yf6iC4)/(F10+F6);

subject to ConcF2nC4:			# nc4 in combined F6+F10
	Yf2nC4 = 1. - Yf2iC4;

# *** Reactor ***

subject to RMasBal:			# vol flow of reactor effluent
	F4 = F1*Yf1C3+F1*Yf1nC4+F2*Yf2nC4+F4*Yf4iC4+F4*Yf4Alkyl;

subject to RC3Bal:			# reactor effluent c3
	F1*Yf1C3 = F4*Yf4C3;

subject to RnC4Bal:			# reactoreffluent nc4
	F1*Yf1nC4+F2*Yf2nC4 = F4*Yf4nC4;

subject to RAlkBal:			# reactor effluent alkylate
 	F4*Yf4Alkyl = F1*Yf1Olef*(YInter+YLinear*Ratio+YQuad*sqrt(Ratio));


subject to RiC4Bal:			# reactor effluent ic4
	F4*Yf4iC4 = F1*Yf1iC4+F2*Yf2iC4-1.22*F4*Yf4Alkyl/2;

subject to RecycleR:			# calc recycle ratio
	RecRatio = F2/F1;

subject to ReacRatio:			# calc total mixed feed ic3 to olefin
					# ratio 
	(F2*Yf2iC4+F1*Yf1iC4)/(F1*Yf1Olef) = Ratio;

subject to Octane:			# alkylate octane
	MOctane = OInter+OYf4iC4*(Yf4iC4-0.4)+OAcidstr*(Acidstr-89);

subject to Aciddil:			# reactor acid dilution
	Adilute = 65.35 - 0.666*MOctane;

subject to AcidFlow:			# reactor acid flow
	F3Acid = F4*Yf4Alkyl*Adilute*Acidstr/(1000.*(99-Acidstr));

# *** DIB tower ***

subject to PropFlow:
	F5 = Yf4C3*F4;

subject to AlkylFlow:
	F8*Yf8Alkyl = F4*Yf4Alkyl;

subject to AlkylAlkyl:			# fraction alkylate in F8
	Yf8Alkyl = 1.0-(Yf8iC4+Yf8nC4);

subject to RiC4:			# bbl/day ic4 recycle
	F6iC4 = F4*Yf4iC4-F8*Yf8iC4;

subject to RecFlow:
	F6 = F6iC4+F6nC4;

subject to RecynC4:
	Yf6nC4 = F6nC4/F6;

subject to RecyiC4:
	Yf6iC4 = 1. - Yf6nC4;

subject to nButFlow:
	F7 = F4*Yf4nC4 - F6*Yf6nC4 - F8*Yf8nC4;

# *** Gasoline blending ***

subject to MinF12:
	F12+F17 >= F11;

subject to MaxF12:
	F12+F17 <= F11;

subject to MinF13:
	F13 >= F8;

subject to MaxF13:
	F13 <= F8;

subject to InterMed1:
	Intmed1 = F12*(RVPiC4*Yf11iC4+RVPnC4*Yf11nC4);

subject to InterMed2:
	Intmed2 = F13*(RVPiC4*Yf8iC4+RVPnC4*Yf8nC4+RVPAlkyl*Yf8Alkyl);

subject to MaxRVP:			# rvp max spec
	RVPMax*F16 >= Intmed1+Intmed2+F14*RVPRef+F15*RVPLsr-SL1;

subject to MinRVP:			# rvp min spec
	RVPMin*F16 <= Intmed1+Intmed2+F14*RVPRef+F15*RVPLsr+SL2;

subject to InterMed3:
	Intmed3 = F12*(OctiC4*Yf11iC4+OctnC4*Yf11nC4);

subject to InterMed4:
	Intmed4 = F13*(OctiC4*Yf8iC4+OctnC4*Yf8nC4+MOctane*Yf8Alkyl);

subject to MaxOct:			# octane max spec
	OctMax*F16 >= Intmed3+Intmed4+F14*OctRef+F15*OctLsr-SL3;

subject to MinOct:			# octane min spec
	OctMin*F16 <= Intmed3+Intmed4+F14*OctRef+F15*OctLsr +SL4;

subject to BlndFlow:			# gasoline product flow
	F16 = F12+F13+F14+F15;

# *** Economics ***

subject to Energy:			# reboiler steam
	Steam = PriceQ*(SVapor+F4*(Yf4iC4+Yf4nC4)*5);

subject to InterMed5:
	Intmed5 = PriceF1*F1+PriceF3*F3Acid+PriceF9*F9+PriceF14*F14+
		PriceF15*F15;

subject to InterMed6:
	Intmed6 = PriceF5*F5+PriceF7*F7+PriceF16*F16+PriceF17*F17;


