# BOILER/TURBOGENERATOR SYSTEM OPTIMIZATION USING LINEAR PROGRAMMING
# Reminder:
# slack - difference between a variable's value (or the value of the body of
#         a constraint) and its nearest constraint value
# rc    - change in value of the objective function for a unit change in 
#         variable
# dual  - (also called marginal value or shadow price)
#         dual price for a constraint gives the slope of the cost curve at 
#         the constraint's current value

# definition of turbine variables:
# flow rates are in lb/hr, power in kW

set Turb;                             # turbine number
set Pres;                             # pressure levels
param Pmin{Turb}>=0;                  # minimum turbine power
param Pmax{Turb}>=0;                  # maximum turbine power
param FImax{Turb}>=0;                 # maximum turbine inlet flow
param FCmax{Turb}>=0;                 # maximum turbine condensate flow
param FELmax{Turb}>=0;                # maximum turbine to 62 psig flowrate
param Fint{Turb}>=0;                  # turbine internal flow limitation
param Enth{Pres}>=0;                  # steam enthalpies
var P{j in Turb}>=Pmin[j],<=Pmax[j];  # turbine power
var FI{j in Turb}>=0, <=FImax[j];     # turbine inlet flow
var FC{Turb}>=0;                      # turbine condensate flow
var FEH{Turb}>=0;                     # turbine exit flow to 195 psi header
var FEL{Turb}>=0;                     # turbine exit flow to 62 psi header


# steam header data:
var FHPS >=0;                         # flowrate through 635 psig header
var FMPS >=0;                         # flowrate through 195 psig header
var FLPS >=0;                         # flowrate through 62 psig header

# bypass flow data:
var FB1 >=0;                          # flowrate from 635 to 195 psig
var FB2 >=0;                          # flowrate from 195 to 62 psig 

# power data:
var PP >=0;                           # purchased power
var EP >=0;                           # excess power
param Econv>=0;                       # conversion kW to Btu/hr
param PPbase>=0;                      # base purchased power
param Pdemand>=0;                     # demand electric power

# utility costs:
param C1 >=0;                         # steam costs 625 psig
param C2 >=0;                         # cost purchased electric power
param C3 >=0;                         # demand penalty

# objective function:
minimize cost:
        C1*FHPS + C2*PP+ C3*EP;

# ------------process constraints-------------
# maximum condensate flow for turbine 1:
subject to MaxCondFlow:
        FC["T01"] <= FCmax["T01"];
    
# maximum 62 psi exhaust for turbine 2:
subject to Max62Exh:
        FEL["T02"] <= FELmax["T02"];

# internal flow limitation turbine 1:
subject to Max_Int_Flo:
        FI["T01"] - FEH["T01"] <= Fint["T01"];

# -------------material balances--------------
# 635 psig steam mass balance:
subject to Mbal635:
        FHPS - sum{j in Turb} FI[j] - FB1 = 0;

# 195 psig steam balance:
subject to Mbal195:
        sum{j in Turb} FEH[j] + FB1 - FB2 - FMPS = 0;

# 62 psig steam balance:
subject to Mbal62:
        sum{j in Turb} FEL[j] + FB2 - FLPS = 0;

# turbine mass balance:
subject to MbalT{j in Turb}:
        FI[j] - FEH[j] - FEL[j] - FC[j] = 0;

# -------------energy balances---------------
# turbine energy balance:
subject to EbalT{j in Turb}:
        Enth["HP"]*FI[j] - Enth["MP"]*FEH[j] - Enth["LP"]*FEL[j] -
        Enth["FW"]*FC[j] - Econv*P[j] = 0;

# power constraints:
subject to Pbase:
        PP + EP >= PPbase;

subject to Pexcess:
        PP + sum{j in Turb} P[j] - Pdemand = EP;