# model file CRACKER.MOD
# taken from Edgar,T.F. and Himmelblau, D.M., Optimization of
#            chemical processes, Mc-Graw-Hill, 1988
# Thermal Cracker optimization problem

# user entry parameters:
set   comp;                          # feed flows plus recycle flows
set   products;                      # cracking products
param Prod_Val{products}>=0;         # value of products
set   feeds within {comp};           # feed components
param Feed_Cost{feeds}>=0;           # feed costs
param Fuel_Req{feeds}>=0;            # feed fuel requirements
param Yield{products,feeds}>=0;      # yield of products from feeds
param CrackCap>=0;                   # thermal cracking capacity
param EthyProcLim>=0;                # ethylene processing limitation
param PropyProcLim>=0;               # propylene processing limitation
param HvalGas>=0;                    # heating value natural gas
param HvalMeth>=0;                   # heating value methane
param HvalOil>=0;                    # heating value gas oil
param HeatLoss>=0;                   # heat losses

#internal program parameters:
var   Prod_Sales>=0;                 # product sales
var   Feed_Costs>=0;                 # feed costs
var   Energy_Costs>=0;               # energy costs
var   Ethy_Cap>=0;                   # ethylene capacity
var   Ethy_Val>=0;                   # value ethylene
var   Propy_Cap>=0;                  # propylene capacity
var   Propy_Val>=0;                  # value propylene
var   Buta_Val>=0;                   # value butadiene
var   Gaso_Val>=0;                   # value gasoline
var   THV>=0;                        # total heating value
var   FFC>=0;                        # fuel for cracking

var   Flow{comp}>=0;                 # flow of components


# -----------------------objective definition:----------------------
maximize profit:
         Prod_Sales - Feed_Costs - Energy_Costs;

subject to ProductSales:
         Prod_Sales = Ethy_Val + Propy_Val + Buta_Val + Gaso_Val;

subject to EthyleneCapacity:
         Ethy_Cap = sum {i in feeds} Yield["ethylene",i] * Flow[i] +
                    Yield["ethylene","ethane"]  * Flow["C2_rec"] +
                    Yield["ethylene","propane"] * Flow["C3_rec"];

subject to EthyleneValue:
         Ethy_Val = Prod_Val["ethylene"] * Ethy_Cap;


subject to PropyleneCapacity:
         Propy_Cap = sum {i in feeds} Yield["propylene",i] * Flow[i] +
                     Yield["propylene","ethane"]  * Flow["C2_rec"] +
                     Yield["propylene","propane"] * Flow["C3_rec"];

subject to PropyleneValue:
         Propy_Val = Prod_Val["propylene"] * Propy_Cap;

subject to ButadieneValue:
         Buta_Val = Prod_Val["butadiene"]*(sum {i in feeds} Yield["butadiene",i]
                    * Flow[i] + Yield["butadiene","ethane"]  * Flow["C2_rec"] +
                                Yield["butadiene","propane"] * Flow["C3_rec"]);

subject to GasolineValue:
         Gaso_Val = Prod_Val["gasoline"]*(sum {i in feeds} Yield["gasoline",i]*
                    Flow[i] + Yield["gasoline","ethane"]  * Flow["C2_rec"] +
                              Yield["gasoline","propane"] * Flow["C3_rec"]);


subject to FeedCosts:
         Feed_Costs = sum {i in feeds} Feed_Cost[i] * Flow[i];

subject to EnergyCost:
         Energy_Costs = Prod_Val["methane"] * Flow["fuel"];

# ----------------------Process Limitations---------------------
# Cracker Capacity Constraint:
subject to CrackerCap:
         1.1*(Flow["ethane"]  + Flow["C2_rec"]) +
         0.9*(Flow["propane"] + Flow["C3_rec"]) +
         0.9* Flow["gas_oil"] + Flow["ngas"] <= CrackCap;

# Ethylene processing limitation:
subject to EthyleneLim:
         Ethy_Cap <= EthyProcLim;


# Propylene processing limitation:
subject to PropyleneLim:
         Propy_Cap <= PropyProcLim;

# Ethane recycle:
subject to EthaneRec:
         sum {i in feeds} Yield["ethane",i] * Flow[i] -
         (1 - Yield["ethane","ethane"]) * Flow["C2_rec"] +
         Yield["ethane","propane"] * Flow["C3_rec"] = 0;

# Propane recycle:
subject to PropaneRec:
         sum {i in feeds} Yield["propane",i] * Flow[i] -
         (1 - Yield["propane","propane"]) * Flow["C3_rec"] = 0;

# total heating value:
subject to TotalHeatVal:
         THV = HvalGas*Flow["fuel"]+HvalMeth*(sum {i in feeds} Yield["methane",i]
               * Flow[i] + Yield["methane","ethane"] * Flow["C2_rec"]
               + Yield["methane","propane"] * Flow["C3_rec"]) +
               + HvalOil*sum{i in feeds} Yield["fuel_oil",i] * Flow[i];

# fuel for cracking:
subject to HeatCracking:
         FFC = sum {i in feeds} Fuel_Req[i] * Flow[i] + Fuel_Req["ethane"] *
               Flow["C2_rec"] + Fuel_Req["propane"] * Flow["C3_rec"];

# heat constraint:
subject to HeatBal:
         FFC + HeatLoss = THV;
