#  THE BOILER AND STEAM SYSTEM IS BASED ON CLARK, J. AND N. HELMICK,
#  HOW TO OPTIMIZE THE DESIGN OF STEAM SYSTEMS, CHEM. ENG.,
#  PP. 152-164, (MAR 10, 1980)


set PROD;			# equipment which generates a steam supply
set USER;			# equipment which receives a steam supply
set BOILER;			# identifies the individual boilers
set LIN;			# defines the number of linearization segments in
				# every boiler steam-fuel relationship.
set TURB within {USER};		# identifies the individual turbines
set MatBal within {PROD,USER};	# equipment that has material balances enforced
set Net within {PROD,USER};	# defines the flow network

param GasCost >= 0;		# cost of fuel gas
param OilCost >= 0;		# cost of fuel oil, ECU per GJ
param ConvHP >= 0;		# convert hp to kJ per hour
param EffT01E >= 0;		# efficiency turbine 1 extraction section
param EffT01C >= 0;		# efficiency turbine 1 condensing section
param EffT02 >= 0;		# efficiency turbine 2
param EffT03 >= 0;		# efficiency turbine 3
param EffT04 >= 0;		# efficiency turbine 4
param EffT05 >= 0;		# efficiency turbine 5

param DHT01E >= 0;		# delta enthalpy turbine 1 extraction
param DHT01C >= 0;		# delta enthalpy turbine 1 condensing
param DHT02 >= 0;		# delta enthalpy turbine 2
param DHT03 >= 0;		# delta enthalpy turbine 3
param DHT04 >= 0;		# delta enthalpy turbine 4
param DHT05 >= 0;		# delta enthalpy turbine 5
param PCap >= 0;		# process plant throughput factor range

param LPExport >= 0;		# amount of steam exported from LP header

param BSFLin{BOILER,LIN} > 0;	# defines steam-fuel relationships for boilers 
				# in several linearization segments, kJ/kgSTM

param Work{TURB};		# HP requirement compressor

param MaxSteam >= 0;		# maximum steam usage


var Flow{PROD,USER} >= 0;	# flow into equipment from equipment, kJ/hr 
var Totalin{USER} >= 0;		# total flow of steam to an equipment
var Totalout{PROD} >= 0;	# total flow of steam from an equipment
var Fuel{BOILER} >= 0;		# fuel consumed by individual boilers
var FuelGas{BOILER} >= 0;	# fuel gas fired in each boiler, kJ/hr
var FuelOil{BOILER} >= 0;	# fuel oil fired in each boiler, kJ/hr
var FuelCost >= 0;		# cost of firing boilers, ECU/hr
var TotFuel >= 0;		# total fuel consumed by all boilers
var WorkT01R >= 0;		# recycle HP turbine 1
var WminT01 >= 0;		# minimum HP requirement turbine 1

# steam produced from boiler "K" in the linearization segment "L"
var Steam{BOILER,LIN} >= 0;	# kJ/hr

# TotalFGas = total fuel gas fired in all boilers, TotalFOil = total fuel
# oil fired in all boilers, kJ/hr
var TotalFGas >= 0, <= 1265.0 * PCap;
var TotalFOil >= 0;

minimize operating_cost:	# cost of fuels
	FuelCost;

# link boiler flows to header using the boiler balance:
subject to BoilerBal:
	Flow["BOILER","HPHEADER"] = sum{l in LIN} sum{k in BOILER} Steam[k,l];

subject to BoilBal{k in BOILER, l in LIN}:
	Steam[k,l] <= MaxSteam;

subject to MassBalance{(i,j) in MatBal}:
	Totalin[j] = Totalout[i];

subject to Flowin{j in USER}:
	sum{(i,j) in Net} Flow[i,j] = Totalin[j];

subject to Flowout{i in PROD}:
	sum{(i,j) in Net} Flow[i,j] = Totalout[i];

subject to BoilerFuel{k in BOILER}:	# heat for each boiler:
	Fuel[k] = sum{l in LIN} Steam[k,l] * BSFLin[k,l];

subject to FuelBalance{k in BOILER}:	# fuel to each boiler:
	Fuel[k] = FuelGas[k] + FuelOil[k];

subject to GasBalance:			# total fuel gas in boilers
	TotalFGas = sum{k in BOILER} FuelGas[k];

subject to OilBalance:			# total fuel oil in boilers
	TotalFOil = sum{k in BOILER} FuelOil[k];

subject to RecycleT01:			# recycle turbine 1
	Work["TURBINE01"] * PCap + WorkT01R = WminT01;


subject to Turb01TotHP:			# energy balance for turbine 1 
	WminT01 * ConvHP = Flow["TURBINE01","LPHEADER"] * DHT01E * EffT01E +
		Flow["TURBINE01","CONDENS"] * DHT01C * EffT01C;

subject to Turb02TotHP:			# energy balance for turbine 2
	Work["TURBINE02"] * ConvHP * PCap = Flow["HPHEADER","TURBINE02"] * 
                 DHT02 * EffT02;

subject to Turb03TotHP:			# energy balance for turbine 3
	Work["TURBINE03"] * ConvHP * PCap = Flow["HPHEADER","TURBINE03"] * 
                 DHT03 * EffT03;

subject to Turb04TotHP:			# energy balance for turbine 4
	Work["TURBINE04"] * ConvHP * PCap = Flow["HPHEADER","TURBINE04"] * 
                 DHT04 * EffT04;

subject to Turb05TotHP:			# energy balance for turbine 5
	Work["TURBINE05"] * ConvHP * PCap = Flow["MPHEADER","TURBINE05"] * 
                 DHT05 * EffT05;

subject to FuelPrice:			# total cost of fuels
	FuelCost = GasCost * TotalFGas + OilCost * TotalFOil;
