# Reactor model acc. to B.S. Dayal & J.F. McGregor
# Recursive exponentially weighted PLS
# J. Proc. Cont., vol. 7, no. 3, pp. 169-179,1997

set varin;
set varout;

param Ah   >= 0;  # heat transfer area, m2
param cAf  >= 0;  # concentration species A, kg/m3
param cps  >= 0;  # specific heat capacity species, J/kg.K
param cpw  >= 0;  # specific heat capacity water, J/kg.K
param Ea   >= 0;  # activation energy, J/mol
param dHr  <= 0;  # heat of reaction, J/kg
param k0   >= 0;  # reaction rate constant, s-1
param R    >= 0;  # gas law constant, J/mol.K
param Ti   >= 0;  # inlet cooling water temperature, K
param Tf   >= 0;  # species feed temperature, K
param U    >= 0;  # overall heat transfer coefficient, W/m2.K
param V    >= 0;  # reactor volume, m3
param Vcj  >= 0;  # volume cooling jacket, m3
param rhos >= 0;  # density species, kg/m3
param rhow >= 0;  # density water, kg/m3

param Cf  >= 0;   # cost feed, Euro/kg feed
param Cpu >= 0;   # cost purification, Euro/kg product
param Cu  >= 0;   # cost utilities, Euro/kg feed
param Cpr >= 0;   # value product, Euro/kg product

param Inputinit{varin} >= 0;
param Inputmin{varin} >= 0;
param Inputmax{varin} >= 0;
param Outputinit{varout} >= 0;
param Outputmin{varout};
param Outputmax{varout} >= 0;

var Input{i in varin}, >= Inputmin[i], <= Inputmax[i];
var Output{i in varout}, >= Outputmin[i], <= Outputmax[i];

var Product_Value >= 0;
var Feed_Cost  >= 0;
var Purif_Cost >= 0;
var Util_Cost  >= 0;

maximize profit:
    Product_Value - Feed_Cost - Purif_Cost - Util_Cost;

subject to Income:
	Product_Value = Input["Fin"]*rhos*Output["C"]*Cpr;

subject to Expense_1:
	Feed_Cost = Input["Fin"]*rhos*Cf;

subject to Expense_2:
	Purif_Cost = Input["Fin"]*rhos*Output["C"]*Cpu;

subject to Expense_3:
	Util_Cost = Input["Fin"]*rhos*Cu;

subject to RxMassBal:
	Input["Fin"]*cAf*Output["C"] - V*k0*exp(-Ea/(R*Output["T"]))
    *cAf*(1-Output["C"]) = 0;

subject to RxEnergyBal:
	rhos*cps*Input["Fin"]*(Tf-Output["T"]) - 
    V*dHr*k0*exp(-Ea/(R*Output["T"]))*cAf*(1-Output["C"]) + 
    U*Ah*(Output["Tcj"]-Output["T"]) = 0;

subject to CwEergyBal:
	Input["Fw"]*cpw*(Ti-Output["Tcj"]) + 
    U*Ah*(Output["T"]-Output["Tcj"]) = 0;




