#include <sstream>
#include "HPI.h"
#include "evm6xdll.h"

using namespace Qureshi;
using namespace std;

HPI::HPI(string filename) : m_programFile(filename), m_hEvm(NULL), m_hHpi(NULL), m_hEvent(NULL)
{
}

HPI::~HPI()
{
  ::CloseHandle(m_hEvent);
  evm6x_hpi_close(m_hHpi);
  evm6x_close(m_hEvm);
}

pair<int,int> HPI::init()
{
  if (INVALID_HANDLE_VALUE == (m_hEvm = evm6x_open(0, FALSE)))
	throw runtime_error("Failed to open connection to EVM.");

  evm6x_set_timeout(m_hEvm, 1000);
  evm6x_abort_write(m_hEvm);
  evm6x_clear_message_event(m_hEvm);

  ostringstream ostr;
  ostr << EVM6X_GLOBAL_MESSAGE_EVENT_BASE_NAME << "0";
  if (NULL == (m_hEvent = ::CreateEvent(NULL, FALSE, FALSE, ostr.str().c_str())))
    throw runtime_error("Could not open event!");

  evm6x_reset_dsp(m_hEvm, HPI_BOOT);
  evm6x_init_emif(m_hEvm, NULL);
  char *sCOFF = (char *)m_programFile.c_str();
  if (!evm6x_coff_load(m_hEvm, NULL, sCOFF, FALSE, FALSE, FALSE))
    throw runtime_error(string("Failed to load ").append(m_programFile).c_str());
  evm6x_unreset_dsp(m_hEvm);

  if (NULL == (m_hHpi = evm6x_hpi_open(m_hEvm))) 
	throw runtime_error("Failed to initialize HPI.");

  ::Sleep(500);

  return this->handshake();
}

void HPI::sendImage(const mxArray *pImage, int nLevels)
{
  int nr = mxGetM(pImage), nc = mxGetN(pImage);
  unsigned char *pPixels = this->colMajor2rowMajor(pImage, nr, nc);
  unsigned long ulBytes = nr*nc,
	            expectedBytes = ulBytes;
  
  // write directly to EVM memory
  if (!evm6x_hpi_write(m_hHpi, (PULONG)pPixels, (PULONG)&ulBytes, (ULONG)m_pTargetInputBuf))
    throw runtime_error("Failed to write image data over HPI interface.");

  if (ulBytes != expectedBytes) {
    ostringstream ostr;
    ostr << "HPI only wrote " << ulBytes << " bytes, expecting " << expectedBytes;
    throw runtime_error(ostr.str().c_str());
  }

  // inform EVM that we're done transferring data by telling 
  // it the # of wavelet decompositions to use for the 
  // multi-scale edge detection.
  if (!evm6x_send_message(m_hEvm, (PULONG)&nLevels))
    throw runtime_error("Failed to signal to EVM that write completed!");
}

void HPI::readImage(mxArray *pImage)
{
  int wait = 10000; // give it some time to process the image
  while (!::WaitForSingleObject(m_hEvent, 1))
    if (0 == --wait)
	  throw runtime_error("Error or timed out waiting for handshake from EVM");

  // already know the # of bytes to read, however we're
  // using this data mainly as a signal from the target 
  // that the processed data can be now read now over 
  // the HPI interface
  unsigned long nBytes;
  for (int ii=0; ii<200; ++ii) {
    if (!evm6x_mailbox_read(m_hEvm, 2, (unsigned long *)&nBytes)) {
	  ::Sleep(100);
	}
	else
	  break;
  }
  if (200 == ii)
    throw runtime_error("Failed to retrieve # bytes to read");
  // read the image data (scratch buffer used because we transpose the data as we shove into the mxArray)
  unsigned char *pReadBuf = &m_buf[0];
  if (!evm6x_hpi_read(m_hHpi, (PULONG)pReadBuf, &nBytes, (ULONG)m_pTargetOutputBuf))
    throw runtime_error("Failed to read image data over the HPI interface.");

  int nr = mxGetM(pImage), nc = mxGetN(pImage);
  unsigned long expectedBytes = nr*nc;
  if (nBytes != expectedBytes) {
    ostringstream ostr;
    ostr << "HPI only read " << nBytes << " bytes, expecting " << expectedBytes;
    throw runtime_error(ostr.str().c_str());
  }

  // marshal into MATLAB mxArray object (convert row-major to col-major)
  unsigned char *pWriteBuf = (unsigned char *)mxGetPr(pImage);
  for (int iCol=0; iCol<nc; ++iCol)
	for (int iRow=0; iRow<nr; ++iRow)
	  *pWriteBuf++ = pReadBuf[iRow*nc + iCol];
}

unsigned char *HPI::colMajor2rowMajor(const mxArray *pImage, int nr, int nc)
{
  m_buf.resize(nr*nc); // ensure there is enough memory
  unsigned char *pSrc = (unsigned char *)mxGetPr(pImage),
	            *pDst = &m_buf[0];
  for (int ii=0; ii<m_buf.size(); ++ii)
    pDst[(ii%nr)*nc + (ii/nr)] = *pSrc++;
  return &m_buf[0];
}

pair<int, int> HPI::handshake()
{
  int wait = 1000;
  while (!::WaitForSingleObject(m_hEvent, 1))
    if (0 == --wait)
	  throw runtime_error("Error or timed out waiting for handshake from EVM");

  unsigned long dims;
  if (!evm6x_mailbox_read(m_hEvm, 2, (unsigned long *)&dims))
	throw runtime_error("Failed to retrieve image dimensions");
  unsigned short xSize = dims>>16, ySize = dims;

  if (!evm6x_retrieve_message(m_hEvm, (unsigned long *)&m_pTargetInputBuf))
	throw runtime_error("Error retrieving target input image buffer mem address.");
  ::Sleep(100);
  if (!evm6x_retrieve_message(m_hEvm, (unsigned long *)&m_pTargetOutputBuf))
	throw runtime_error("Error retrieving target output image buffer mem address.");

  m_imageRowsCols = pair<int,int>(xSize, ySize);
  m_buf.resize(m_imageRowsCols.first*m_imageRowsCols.second);
  return m_imageRowsCols;
}