#include "stdafx.h"
#include <algorithm>
#include "MedianFilter.h"

using namespace std;

MedianFilter::MedianFilter(int kernelSize)
: m_kernelSize(kernelSize), 
  m_khw(kernelSize/2) // assumed odd kernelSize
{
	// initialize timing infrastructure
	LARGE_INTEGER freq;
	m_freq = (!::QueryPerformanceFrequency(&freq)) ? 0 : freq.QuadPart;
}

float MedianFilter::processImage(Image8bpp *pNoisyImage, 
	  							 Image8bpp *pProcessedImage, 
							     Implementation impl)
{
	if (pNoisyImage->getHeight()!=pProcessedImage->getHeight() ||
		pNoisyImage->getWidth()!=pProcessedImage->getWidth())
		throw runtime_error("invalid args: images must be same size!");

	// begin timing block
	m_stop.QuadPart = 0;
	if (!::QueryPerformanceCounter(&m_start))
		m_start.QuadPart = 0;

		if (IPP == impl)
			this->processImageIpp(pNoisyImage, pProcessedImage);
		else if (QSORT == impl)
			this->processImageQsort(pNoisyImage, pProcessedImage);
		else if (STL_SORT == impl)
			this->processImageSTLsort(pNoisyImage, pProcessedImage);
		else if (FIXED_3x3 == impl)
			this->processImage3x3(pNoisyImage, pProcessedImage);
		else if (FIXED_5x5 == impl)
			this->processImage5x5(pNoisyImage, pProcessedImage);

	// end timing block
	if (m_start.QuadPart)
        if (!::QueryPerformanceCounter(&m_stop))
			m_stop.QuadPart = 0;

	return this->calcMicroSecs(m_start, m_stop);
}

void MedianFilter::processImageIpp(Image8bpp *pNoisyImage, Image8bpp *pProcessedImage)
{
	int nr = pNoisyImage->getHeight(),
		nc = pNoisyImage->getWidth();
	IppiSize mask = {m_kernelSize, m_kernelSize};
	IppiPoint anchor = {m_khw, m_khw};
	IppiSize dstRoi = {nc-2*m_khw, nr-2*m_khw}; // just the interior region

	// offset pointers to start within the interior of the image
	int offset = nc*m_khw + m_khw;
	IppStatus status = 
		ippiFilterMedian_8u_C1R(pNoisyImage->getPixels() + offset,
		                        pNoisyImage->getStride(),
								pProcessedImage->getPixels() + offset,
								pProcessedImage->getStride(),
								dstRoi, // leaves boundary pixels alone
								mask,
								anchor);

	if (ippStsNoErr != status)
		throw runtime_error("median filter operation failed");
}

void MedianFilter::processImageQsort(Image8bpp *pNoisyImage, Image8bpp *pProcessedImage)
{
	vector<Ipp8u> neighborhood(m_kernelSize*m_kernelSize);
	Ipp8u *pNeighborhood;
	const int nr = pNoisyImage->getHeight(),
		      nc = pNoisyImage->getWidth(),
		      stride = pNoisyImage->getStride(),
		      nks = m_kernelSize*m_kernelSize,
		      iMedian = nks/2,
		      nBackwards = m_kernelSize*stride,
		      noisyNextRowOffset = 2*m_khw + (stride-nc),
		      outputNextRowOffset = m_khw + (stride-nc);
	Ipp8u *pNoisy = pNoisyImage->getPixels(),
	      *pOutput = pProcessedImage->getPixels() + m_khw*stride; 


	// march through the interior portion of the image
	for (int iRow=m_khw; iRow<nr-m_khw; ++iRow) {
		pOutput += m_khw;
		for (int iCol=m_khw; iCol<nc-m_khw; ++iCol, pNoisy++, pOutput++) {
			// splice out the current neighborhood
			pNeighborhood = &neighborhood[0];
			for (int kk=0; kk<m_kernelSize; ++kk, pNoisy+=nc, pNeighborhood+=m_kernelSize) 
				memcpy(pNeighborhood, pNoisy, m_kernelSize*sizeof(Ipp8u));
			pNoisy -= nBackwards;

			qsort(&neighborhood[0], nks, sizeof(Ipp8u), &MedianFilter::compare);
			*pOutput = neighborhood[iMedian]; // output pixel is the median value
		}
		pNoisy += noisyNextRowOffset;   // incr ptrs in prep
		pOutput += outputNextRowOffset; // for next scan-line
	}
}

int MedianFilter::compare(const void *pElem1, const void *pElem2)
{
	Ipp8u *pPixel1 = (Ipp8u *)pElem1,
		  *pPixel2 = (Ipp8u *)pElem2;

	return (*pPixel1<*pPixel2) ? -1 : ( (*pPixel1==*pPixel2) ? 0 : 1 );
}

void MedianFilter::processImageSTLsort(Image8bpp *pNoisyImage, Image8bpp *pProcessedImage)
{
	vector<Ipp8u> neighborhood(m_kernelSize*m_kernelSize);
	Ipp8u *pNeighborhood;
	const int nr = pNoisyImage->getHeight(),
		      nc = pNoisyImage->getWidth(),
		      stride = pNoisyImage->getStride(),
		      nks = m_kernelSize*m_kernelSize,
		      iMedian = nks/2,
		      nBackwards = m_kernelSize*stride,
		      noisyNextRowOffset = 2*m_khw + (stride-nc),
		      outputNextRowOffset = m_khw + (stride-nc);
	Ipp8u *pNoisy = pNoisyImage->getPixels(),
	      *pOutput = pProcessedImage->getPixels() + m_khw*stride; 


	// march through the interior portion of the image
	for (int iRow=m_khw; iRow<nr-m_khw; ++iRow) {
		pOutput += m_khw;
		for (int iCol=m_khw; iCol<nc-m_khw; ++iCol, pNoisy++, pOutput++) {
			// splice out the current neighborhood
			pNeighborhood = &neighborhood[0];
			for (int kk=0; kk<m_kernelSize; ++kk, pNoisy+=nc, pNeighborhood+=m_kernelSize) 
				memcpy(pNeighborhood, pNoisy, m_kernelSize*sizeof(Ipp8u));
			pNoisy -= nBackwards;

			sort(neighborhood.begin(), neighborhood.end());
			*pOutput = neighborhood[iMedian]; // output pixel is the median value
		}
		pNoisy += noisyNextRowOffset;   // incr ptrs in prep
		pOutput += outputNextRowOffset; // for next scan-line
	}
}

#define PIX_SWAP(a,b) { temp=(a);(a)=(b);(b)=temp; }
#define PIX_SORT(a,b) { if ((a)>(b)) PIX_SWAP((a),(b)); }

void MedianFilter::processImage3x3(Image8bpp *pNoisyImage, Image8bpp *pProcessedImage)
{
	const int nr = pNoisyImage->getHeight(),
		      nc = pNoisyImage->getWidth(),
		      stride = pNoisyImage->getStride(),
		      nBackwards = 3*stride,
		      noisyNextRowOffset = 2 + (stride-nc),
		      outputNextRowOffset = 1 + (stride-nc);
	Ipp8u temp, neighborhood[9], *pNeighborhood;
	Ipp8u *pNoisy = pNoisyImage->getPixels(),
	      *pOutput = pProcessedImage->getPixels() + stride; 
	Ipp8u *p = neighborhood;

	// march through the interior portion of the image
	for (int iRow=1; iRow<nr-1; ++iRow) {
		pOutput += 1;
		for (int iCol=1; iCol<nc-1; ++iCol, pNoisy++, pOutput++) {
			// splice out the current neighborhood
			pNeighborhood = neighborhood;
			for (int kk=0; kk<3; ++kk, pNoisy+=nc, pNeighborhood+=3) 
				memcpy(pNeighborhood, pNoisy, 3*sizeof(Ipp8u));
			pNoisy -= nBackwards;

			// minimum exchange filter that partially sorts a 9-element 
			// array such that the central element contains the median value,
			// in theory it is impossible to get the median using a fewer 
			// number of comparisons
			PIX_SORT(p[1], p[2]) ; PIX_SORT(p[4], p[5]) ; PIX_SORT(p[7], p[8]) ;
			PIX_SORT(p[0], p[1]) ; PIX_SORT(p[3], p[4]) ; PIX_SORT(p[6], p[7]) ;
			PIX_SORT(p[1], p[2]) ; PIX_SORT(p[4], p[5]) ; PIX_SORT(p[7], p[8]) ;
			PIX_SORT(p[0], p[3]) ; PIX_SORT(p[5], p[8]) ; PIX_SORT(p[4], p[7]) ;
			PIX_SORT(p[3], p[6]) ; PIX_SORT(p[1], p[4]) ; PIX_SORT(p[2], p[5]) ;
			PIX_SORT(p[4], p[7]) ; PIX_SORT(p[4], p[2]) ; PIX_SORT(p[6], p[4]) ;
			PIX_SORT(p[4], p[2]) ; 
			*pOutput = (p[4]) ;
		} // end (for each column)
		pNoisy += noisyNextRowOffset;   // incr ptrs in prep
		pOutput += outputNextRowOffset; // for next scan-line
	} // end (for each row)
}

void MedianFilter::processImage5x5(Image8bpp *pNoisyImage, Image8bpp *pProcessedImage)
{
	const int nr = pNoisyImage->getHeight(),
		      nc = pNoisyImage->getWidth(),
		      stride = pNoisyImage->getStride(),
		      nBackwards = 5*stride,
		      noisyNextRowOffset = 4 + (stride-nc),
		      outputNextRowOffset = 2 + (stride-nc);
	Ipp8u temp, neighborhood[25], *pNeighborhood;
	Ipp8u *pNoisy = pNoisyImage->getPixels(),
	      *pOutput = pProcessedImage->getPixels() + 2*stride; 
	Ipp8u *p = neighborhood;

	// march through the interior portion of the image
	for (int iRow=2; iRow<nr-2; ++iRow) {
		pOutput += 2;
		for (int iCol=2; iCol<nc-2; ++iCol, pNoisy++, pOutput++) {
			// splice out the current neighborhood
			pNeighborhood = neighborhood;
			for (int kk=0; kk<5; ++kk, pNoisy+=nc, pNeighborhood+=5) 
				memcpy(pNeighborhood, pNoisy, 5*sizeof(Ipp8u));
			pNoisy -= nBackwards;

			// optimized search for the median value in a list of 25 elements,
			// again in theory it is impossible to do this with fewer comparisons
			PIX_SORT(p[0], p[1]) ;   PIX_SORT(p[3], p[4]) ;   PIX_SORT(p[2], p[4]) ;
			PIX_SORT(p[2], p[3]) ;   PIX_SORT(p[6], p[7]) ;   PIX_SORT(p[5], p[7]) ;
			PIX_SORT(p[5], p[6]) ;   PIX_SORT(p[9], p[10]) ;  PIX_SORT(p[8], p[10]) ;
			PIX_SORT(p[8], p[9]) ;   PIX_SORT(p[12], p[13]) ; PIX_SORT(p[11], p[13]) ;
			PIX_SORT(p[11], p[12]) ; PIX_SORT(p[15], p[16]) ; PIX_SORT(p[14], p[16]) ;
			PIX_SORT(p[14], p[15]) ; PIX_SORT(p[18], p[19]) ; PIX_SORT(p[17], p[19]) ;
			PIX_SORT(p[17], p[18]) ; PIX_SORT(p[21], p[22]) ; PIX_SORT(p[20], p[22]) ;
			PIX_SORT(p[20], p[21]) ; PIX_SORT(p[23], p[24]) ; PIX_SORT(p[2], p[5]) ;
			PIX_SORT(p[3], p[6]) ;   PIX_SORT(p[0], p[6]) ;   PIX_SORT(p[0], p[3]) ;
			PIX_SORT(p[4], p[7]) ;   PIX_SORT(p[1], p[7]) ;   PIX_SORT(p[1], p[4]) ;
			PIX_SORT(p[11], p[14]) ; PIX_SORT(p[8], p[14]) ;  PIX_SORT(p[8], p[11]) ;
			PIX_SORT(p[12], p[15]) ; PIX_SORT(p[9], p[15]) ;  PIX_SORT(p[9], p[12]) ;
			PIX_SORT(p[13], p[16]) ; PIX_SORT(p[10], p[16]) ; PIX_SORT(p[10], p[13]) ;
			PIX_SORT(p[20], p[23]) ; PIX_SORT(p[17], p[23]) ; PIX_SORT(p[17], p[20]) ;
			PIX_SORT(p[21], p[24]) ; PIX_SORT(p[18], p[24]) ; PIX_SORT(p[18], p[21]) ;
			PIX_SORT(p[19], p[22]) ; PIX_SORT(p[8], p[17]) ;  PIX_SORT(p[9], p[18]) ;
			PIX_SORT(p[0], p[18]) ;  PIX_SORT(p[0], p[9]) ;   PIX_SORT(p[10], p[19]) ;
			PIX_SORT(p[1], p[19]) ;  PIX_SORT(p[1], p[10]) ;  PIX_SORT(p[11], p[20]) ;
			PIX_SORT(p[2], p[20]) ;  PIX_SORT(p[2], p[11]) ;  PIX_SORT(p[12], p[21]) ;
			PIX_SORT(p[3], p[21]) ;  PIX_SORT(p[3], p[12]) ;  PIX_SORT(p[13], p[22]) ;
			PIX_SORT(p[4], p[22]) ;  PIX_SORT(p[4], p[13]) ;  PIX_SORT(p[14], p[23]) ;
			PIX_SORT(p[5], p[23]) ;  PIX_SORT(p[5], p[14]) ;  PIX_SORT(p[15], p[24]) ;
			PIX_SORT(p[6], p[24]) ;  PIX_SORT(p[6], p[15]) ;  PIX_SORT(p[7], p[16]) ;
			PIX_SORT(p[7], p[19]) ;  PIX_SORT(p[13], p[21]) ; PIX_SORT(p[15], p[23]) ;
			PIX_SORT(p[7], p[13]) ;  PIX_SORT(p[7], p[15]) ;  PIX_SORT(p[1], p[9]) ;
			PIX_SORT(p[3], p[11]) ;  PIX_SORT(p[5], p[17]) ;  PIX_SORT(p[11], p[17]) ;
			PIX_SORT(p[9], p[17]) ;  PIX_SORT(p[4], p[10]) ;  PIX_SORT(p[6], p[12]) ;
			PIX_SORT(p[7], p[14]) ;  PIX_SORT(p[4], p[6]) ;   PIX_SORT(p[4], p[7]) ;
			PIX_SORT(p[12], p[14]) ; PIX_SORT(p[10], p[14]) ; PIX_SORT(p[6], p[7]) ;
			PIX_SORT(p[10], p[12]) ; PIX_SORT(p[6], p[10]) ;  PIX_SORT(p[6], p[17]) ;
			PIX_SORT(p[12], p[17]) ; PIX_SORT(p[7], p[17]) ;  PIX_SORT(p[7], p[10]) ;
			PIX_SORT(p[12], p[18]) ; PIX_SORT(p[7], p[12]) ;  PIX_SORT(p[10], p[18]) ;
			PIX_SORT(p[12], p[20]) ; PIX_SORT(p[10], p[20]) ; PIX_SORT(p[10], p[12]) ;

			*pOutput = p[12];
		} // end (for each column)
		pNoisy += noisyNextRowOffset;   // incr ptrs in prep
		pOutput += outputNextRowOffset; // for next scan-line
	} // end (for each row)
}

float MedianFilter::calcMicroSecs(LARGE_INTEGER start, LARGE_INTEGER stop)
{
	if (0==m_freq) // performance counter not available
		return 0;
	else {
		float delta = stop.QuadPart-start.QuadPart;
		return (delta/m_freq) * 1000000.f;
	}
}