function J = mmse(I, sz)
%
% MMSE   Minimum Mean-Squared Error adaptive filter
%    J = MMSE(I, sz)
%
%    J: filtered image
%    I: input image, must be UINT8 or UINT16
%    sz: size of the neighborhood used to calculate local pixel stats
%

dI = double(I);

% compute local statistics
ncoeffs = sz*sz;
u = filter2(ones(sz)./ncoeffs, dI); % local mean
sigmasq = filter2(ones(sz)./ncoeffs, dI.^2); 
sigmasq = sigmasq - u.^2; % local variance

% from the local variances estimate the noise variance
noise_variance = mean2(sigmasq);

% the guts of the MMSE adaptive filter
% (linearly interpolate b/w input pixel and local mean,
% depending on the magnitude of the local variance)
alpha = noise_variance ./ sigmasq;
% must be careful to handle the case where noise variance
% much greater than local variance - in this case the output
% pixel should be the local mean (alpha = 1)
ii = find(alpha > 1); 
alpha(ii) = 1;
% linearly interpolate
J = (1-alpha).*dI + alpha.*u;

% cast to uint8 or uint16, but for those pixels that are 
% out-of-range replace with the local mean
if isa(I, 'uint8')
    J = clamp_pixel_values(J, u, 255);
    J = uint8(J + 0.5);
elseif isa(I, 'uint16')
    J = clamp_pixel_values(J, u, 65535);
    J = uint16(J + 0.5);
end

function J = clamp_pixel_values(J, u, pmax)

% those that are too low receive the local mean pixel value
ilow = find(J < 0);
J(ilow) = u(ilow);

% those that are too high receive the local mean pixel value
ihi = find(J > pmax);
J(ihi) = u(ihi);