function dB = psnr(I, J)
%
% PSNR   calculate peak signal-to-noise ratio between two images
%    dB = PSNR(I, J)
%
%    dB: peak SNR, in decibels
%    I: input image 1, must be UINT8, UINT16, or double
%    J: input image 2, must be same type as input image 1
%
% NOTE: if I is of type double, then pixel values should lie within
%       the range [0,1]. Formula is 10log10(p/RMS), where p is the
%       max pixel value (1, 255, or 65535) and RMS is the root
%       mean squared error difference between I and J.
%

if size(I) ~= size(J)
    error('size of both images must be the same.');
end
N = length(I(:));

% promote to double
if isa(I, 'uint8') & isa(J, 'uint8')
    p = 255;
    I = double(I);
    J = double(J);
elseif isa(I, 'uint16') & isa(J, 'uint16')
    p = 65535;
    I = double(I);
    J = double(J);
elseif isa(I, 'double') & isa(J, 'double')
    p = 1.0;
else
    error('Both images must be of the same type (uint8, uint16, or double)');
end

% RMS between I and J
dIJ = I-J;
dIJsquared = dIJ.^2;
rms = sqrt(mean2(dIJsquared)); 

% return peak signal-to-noise ratio in decibels
dB = 10 * log10(p / rms);