function J = sam(I, LP, HP, sz, k)
%
% SAM   Signal Adaptive Mean adaptive filter
%    J = SAM(I, LP, HP, sz, k)
%
%    J: filtered image
%    I: input image, must be UINT8 or UINT16
%    LP: low-pass filter kernel
%    HP: high-pss filter kernel
%    sz: size of the neighborhood used to calculate local pixel stats
%    k: constant (see 4.6.3)
%

dI = double(I);
[M N] = size(I);
Ilpf = filter2(LP, dI);
Ihpf = filter2(HP, dI);

% compute local statistics
u = filter2(ones(sz)./prod(sz), dI);
sigmasq = filter2(ones(sz)./prod(sz), dI.^2);
sigmasq = sigmasq - u.^2;

% from the local variances estimate the noise variance
noise_variance = mean2(sigmasq);

% find those pixels that meet the SAM criterion
ii = find(k*noise_variance < sigmasq);
Kweights = zeros(M, N);
Kweights(ii) = 1 - (k .* (noise_variance ./ sigmasq(ii)));

J = Ilpf + Kweights.*Ihpf;

% clip to valid range
if isa(I, 'uint8')
    pmax = 255;
elseif isa(I, 'uint16')
    pmax = 65535;
else
    pmax = 1;
end

ii = find(J < 0);
J(ii) = 0;
ii = find(J > pmax);
J(ii) = pmax;

% return J back in original form
if isa(I, 'uint8')
    J = uint8(J);
elseif isa(I, 'uint16')
    J = uint16(J);
end