function J = segment_triangle(I)
%
% SEGMENT_TRIANGLE   image segmentation using the triangle algorithm
%    J = SEGMENT_TRIANGLE(I)
%
%    J: binary segmented image, will be of type LOGICAL
%    I: input image
%

I = double(I);
J = logical(zeros(size(I))); % return matrix

% image histogram
max_pixel = max(I(:));
bins = 0:max_pixel;
H = histc(I(:),bins)';
% zero-phase smoothing of the histogram (so peaks don't shift)
H_smoothed = conv(H, [1 1 1 1 1]./5);
H(2:length(H)) = H_smoothed(5:length(H_smoothed)-1);

% find bmax (most popular bin) and bmin
[y2, bmax] = max(H);
bmax = bmax - 1;
ii = find(H);
if isempty(ii)
    error('completely empty histogram');
end
bmin1 = ii(1) - 1; % 1st non-zero bin
bmin2 = ii(length(ii)) - 1; % last non-zero bin
if (bmax-bmin1) > (bmin2-bmax) % choose the one farthest away from max 
    bmin = bmin1;
    bins = [bmin:bmax-1];
else
    bmin = bmin2;
    bins = [bmax+1:bmin];
end
y1 = H(bmin+1);

m = (y2-y1) / (bmax-bmin); % slope of the line from bmin to bmax
% fit a line with slope m through the point (bmin, H[bmin])
b = -m*bmin + y1;
mperp = -(bmax-bmin) / (y2-y1); % slope of the perpendicular

dmax = realmin;
T = bins(1);
for bin = bins
    % we have one point P1 = (bin, H[bin])
    % find another point P2 lying somewhere on the line
    % connecting (bmin, H[bin]) & (bmax, H[bmax]) such
    % that P1P2 is perpendicular to it.
    
    % set the two equations equal to one another, and
    % solve for x
    x = (H(bin+1) - mperp*bin - b) / (m-mperp);
    % now solve for y
    y = mperp*(x-bin) + H(bin+1);
    
    % distance d
    d = sqrt( (bin-x)^2 + (H(bin+1)-y)^2 );
    if d>dmax
        dmax = d;
        T = bin;
    end
end

% segmentation
indices = find(I > T);
J(indices) = true;