function K = denoise_daub4_universal(I, N, type, mult)
%
% DENOISE_DAUB4_UNIVERSAL   wavelet denoising with D4 wavelet and universal
%                           threshold
%    K = DENOISE_DAUB4_UNIVERSAL(I, N, type, mult)
%    K = DENOISE_DAUB4_UNIVERSAL(I, N, type)
%    K = DENOISE_DAUB4_SUBBAND_ADAPTIVE(I, N)
%
%    K: denoised image
%    I: input image
%    N: # wavelet levels
%    type: 'H' for hard threshold, 'S' for soft threshold
%    mult: scalar multiplier
%
% NOTE: if not specified, hard thresholding with a multiplier of 1.0 is
%       assumed. Donoho and Johnstone's "universal threshold" is used
%       for thresholding in the wavelet domain.
%

I = double(I);

% default to mult = 1
if nargin<4, mult = 1;, end

% default to hard thresholding
if nargin<3, type='H';, end

% sanity checks omitted (check for double, square matrix, etc.)
[nrows ncols] = size(I);

% N-level D4 forward wavelet transform on the image I
num_levels = N;
J = I;
for level = 1:num_levels
    sz = nrows/2^(level-1); % assumes # rows = # cols
    H = daub4_basis(sz);
    P = permutation_matrix(sz);
    J(1:sz,1:sz) = P*H*J(1:sz,1:sz)*H'*P';
end

% denoising step (i.e. thresholding in the wavelet domain)
lambda = mult*universal_threshold(J, N);

m = nrows/(2^N);
n = ncols/(2^N);
if type == 'H' % hard threshold
  
    J(:,n+1:ncols) = hard_thresh(J(:,n+1:ncols), lambda);
    J(m+1:nrows,1:n) = hard_thresh(J(m+1:nrows,1:n), lambda);
    
elseif type == 'S' % soft threshold

    J(:,n+1:ncols) = soft_thresh(J(:,n+1:ncols), lambda);
    J(m+1:nrows,1:n) = soft_thresh(J(m+1:nrows,1:n), lambda);
    
end

% N-level D4 inverse wavelet transform
K = J;
sz = nrows/2^num_levels;
for level = 1:num_levels
    sz = sz*2;
    H = daub4_basis(sz);
    P = permutation_matrix(sz);
    K(1:sz,1:sz) = inv(P*H)*K(1:sz,1:sz)*inv(H'*P');   
end

function lambda = universal_threshold(J, N)

% 1st order of business is to estimate the variance

% extract the finest HH subband
[nr nc] = size(J);
m = nr/2;
n = nc/2;
HH = J(m+1:nr,n+1:nc);

% sigma estimate
sigma = median( abs( HH(:) ) ) / .6745;

lambda = sigma*sqrt(2*log(nr*nc));

function thresholded = hard_thresh(detail_coefs, T)

% simple, "keep or kill": anything above T keep and
% anything below T set to zero.
thresholded = (abs(detail_coefs) > T) .* detail_coefs;

function thresholded = soft_thresh(detail_coefs, T)

% In the text we define the soft threshold operator as so:
%
%            x-T, if x >= T
% Tsoft(x) = 0,   if x < T
%            x+T, if x <= -T
%
% An equivalent means of stating the above is to set to zero 
% any coefficient below T (in absolute value), and
%
% Tsoft(x) = sgn(x)(|x|-T), if |x|>T 

absd = abs(detail_coefs);
thresholded = sign(detail_coefs).*(absd >= T).*(absd - T); 