// import_grayscale_image.cpp
//

#include <windows.h>
#include <vector>
#include <gdiplus.h>
#include "mex.h"

using namespace std;
using namespace Gdiplus;

BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    return TRUE;
}

static void print_gdiplus_err_and_exit(Status s)
{
	switch (s) {
		case Ok:
			mexErrMsgTxt("no error");
		case GenericError:
			mexErrMsgTxt("generic error");
		case InvalidParameter:
			mexErrMsgTxt("invalid parameter");
		case OutOfMemory:
			mexErrMsgTxt("out of memory");
		case ObjectBusy:
			mexErrMsgTxt("object busy");
		case InsufficientBuffer:
			mexErrMsgTxt("insufficient buffer");
		case NotImplemented:
			mexErrMsgTxt("not implemented");
		case WrongState:
			mexErrMsgTxt("wrong state");
		case Aborted:
			mexErrMsgTxt("aborted");
		case FileNotFound:
			mexErrMsgTxt("file not found");
		case ValueOverflow:
			mexErrMsgTxt("value overflow");
		case AccessDenied:
			mexErrMsgTxt("access denied");
		case UnknownImageFormat:
			mexErrMsgTxt("unknown image format");
		case FontFamilyNotFound:
			mexErrMsgTxt("font family not found");
		case FontStyleNotFound:
			mexErrMsgTxt("font style not found");
		case NotTrueTypeFont:
			mexErrMsgTxt("not true type font");
		case UnsupportedGdiplusVersion:
			mexErrMsgTxt("unsupported GDI+ version");
		case GdiplusNotInitialized:
			mexErrMsgTxt("GDI+ not initialized");
		case PropertyNotFound:
			mexErrMsgTxt("property not found");
		case PropertyNotSupported:
			mexErrMsgTxt("property not supported");
		default:
			{
				string sErr;
				LPVOID lpMsgBuf;
				if (!FormatMessage( 
					FORMAT_MESSAGE_ALLOCATE_BUFFER | 
					FORMAT_MESSAGE_FROM_SYSTEM | 
					FORMAT_MESSAGE_IGNORE_INSERTS,
					NULL,
					::GetLastError(),
					MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
					(LPTSTR) &lpMsgBuf,
					0,
					NULL ))
				{
					sErr = "Unable to extract Win32 error code";
				}
				else
                    sErr = (LPCTSTR)lpMsgBuf;

				LocalFree(lpMsgBuf);
				mexErrMsgTxt(sErr.c_str());
			}
	}
}

void read_image(const char *filename, int buflen, mxArray *plhs[])
{
	// ASCII -> UNICODE conversion
	vector<wchar_t> wName(buflen+1);
	mbstowcs (&wName[0], filename, buflen);
	wName[buflen] = L'\0';

	// use this temporary object purely to deal with
	// the machinations of performing file I/O to parse
	// the binary image data
	Status s = Ok;
	Bitmap bmp(&wName[0]);
	if (Ok != (s = bmp.GetLastStatus()))
		print_gdiplus_err_and_exit(s);

	// allocate MATLAB matrix for gray-scale image
	plhs[0] = mxCreateNumericMatrix(bmp.GetHeight(), bmp.GetWidth(), mxUINT8_CLASS, mxREAL);
	unsigned char *pOutput = (unsigned char *)mxGetPr(plhs[0]);
	
	BitmapData bmpData;
	Rect rect(0, 0, bmp.GetWidth(), bmp.GetHeight());
	PixelFormat fmt = bmp.GetPixelFormat();
	if (PixelFormat24bppRGB == fmt) { // convert from RGB to grayscale

		s = bmp.LockBits(&rect, ImageLockModeRead, PixelFormat24bppRGB, &bmpData);
		if (Ok != s)
			print_gdiplus_err_and_exit(s);

		// color conversion here (remember MATLAB is column-major which is opposite from C)
		unsigned char *pInput  = (unsigned char*)bmpData.Scan0;
		for (UINT iCol=0; iCol<bmp.GetWidth(); ++iCol) {
			for (UINT iRow=0; iRow<bmp.GetHeight(); ++iRow) {
				unsigned char *pRGB = &pInput[iRow*bmpData.Stride + iCol];
				*pOutput++ = (unsigned char)(pRGB[0]*0.114 + pRGB[1]*0.587 + pRGB[2]*0.299);
			}
		}

	} else if (PixelFormat8bppIndexed == fmt) { // color conversion not necessary

		// get to the raw bits comprising the temporary GDI+ bitmap object
		s = bmp.LockBits(&rect, ImageLockModeRead, PixelFormat8bppIndexed, &bmpData);
		if (Ok != s)
			print_gdiplus_err_and_exit(s);

		// copy from temporary GDI+ object into MATLAB matrix
		unsigned char *pInput  = (unsigned char*)bmpData.Scan0;
		for (UINT iCol=0; iCol<bmp.GetWidth(); ++iCol)
			for (UINT iRow=0; iRow<bmp.GetHeight(); ++iRow)
				*pOutput++ = pInput[iRow*bmpData.Stride + iCol];

	} else
		mexErrMsgTxt("Only 8bpp indexed or 24bpp RGB images supported.");
}

void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) 
{
	int buflen = 0;
	char *filename = NULL;

    /* Check for proper number of arguments */
    if (nrhs != 1)
		mexErrMsgTxt("One input argument required."); 
    else if (nlhs != 1)
		mexErrMsgTxt("One output argument required."); 

	if (!mxIsChar(prhs[0]))
		mexErrMsgTxt("IMPORT_GRAYSCALE_IMAGE requires a string input argument.");

	/* extract the filename string from the Matlab object */
	buflen = (mxGetM(prhs[0]) * mxGetN(prhs[0])) + 1;
	filename = (char *)mxCalloc(buflen, sizeof(char));
    if(0 != mxGetString(prhs[0], filename, buflen)) 
		mexWarnMsgTxt("Not enough space. String is truncated.");

	/* initialize GDI+ */
	GdiplusStartupInput gdiplusStartupInput;
	ULONG_PTR gdiplusToken;
	if (Ok != GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL))
		mexErrMsgTxt("Failed to initialize GDI+");

	read_image(filename, buflen, plhs);

	/* shutdown GDI+ */
	GdiplusShutdown(gdiplusToken); 
}