#include <common.h>
#include <board.h> /* EVM library */
#include <pci.h>
#include <img_median_3x3.h>

// each video frame is 240 rows by 320 cols
#define X_SIZE 240
#define Y_SIZE 320
#define N_PIXELS (X_SIZE*Y_SIZE)

#pragma DATA_ALIGN (in_img, 8);
#pragma DATA_SECTION (in_img, "SBSRAM");
unsigned char in_img[N_PIXELS];

#pragma DATA_ALIGN (out_img, 8);
#pragma DATA_SECTION (out_img, "SBSRAM"); 
unsigned char out_img[N_PIXELS]; /* filtered image */

/* 
 * Faster than memset(), count must be a multiple of  
 * 8 and greater than or equal to 32
 */
void memclear( void * ptr, int count )
{
  long * lptr = ptr;
  _nassert((int)lptr%8==0);
  #pragma MUST_ITERATE (32);
  for (count>>=3; count>0; count--)
    *lptr++ = 0;
}

/* 3x3 median filter */
void medfilt()
{
  int irow = 0;
  unsigned char *pin = in_img+Y_SIZE,
                *pout = out_img+Y_SIZE;
  for (; irow<X_SIZE-2; ++irow, pin+=Y_SIZE, pout+=Y_SIZE)
    IMG_median_3x3(pin, Y_SIZE, pout);
}

/* host|target handshake, transmit image size and mem addrs */
void handshake()
{
  unsigned int image_size = (X_SIZE<<16) | Y_SIZE;
  int sts;
  /*
   * target will initialize HPI, therefore we
   * should keep attempting to send the image
   * dimensions until the HPI message is successfully
   * sent.
   */
  do {
    sts = amcc_mailbox_write(2, image_size);
  } while (ERROR == sts);

  /* 
   * comm is up now, now send input buf mem addr 
   * and processed image buf mem addr
   */
  pci_message_sync_send((unsigned int)in_img, FALSE);
  pci_message_sync_send((unsigned int)out_img, FALSE);     
}

/* wait for anything from the host */
void wait_4_frame()
{
  unsigned int msg;
  pci_message_sync_retrieve(&msg);
}

/* tell host that N_PIXELS worth of data ready */
void processing_complete()
{
  int sts;
  unsigned int bytes = N_PIXELS;
    	
  do {
    sts = amcc_mailbox_write(2, bytes);
  } while (ERROR == sts);
}

int main(void)
{
  evm_init(); /* initialize the board */
  pci_driver_init(); /* call before using any PCI code */
  handshake(); /* comm init */
  
  memclear(out_img, Y_SIZE);
  memclear(out_img+N_PIXELS-Y_SIZE, Y_SIZE);

  for (;;) {
    /* wait for host to signal that they have sent a frame. */ 
    wait_4_frame();
    
    /* process the video frame */
    medfilt();
    
    /* signal to host that denoised frame ready */
    processing_complete();
  }
}
